/***************************************************
  This file is distributed as part of Sula PrimeriX
  (http://members.xoom.com/sprimerix).
****************************************************/


/*
   script.c     - named connections. This doesn't handle Guile (Scheme) scripts 
   that are interpreted *internally* by the Scheme interpreter. Instead,
   this is for named connections, that is, for external programs  and TCP/IP
   connections which may be used an extension to _sula_ .
   See doc/PROGRAMMING for more info.

   Author: Tano Fotang, 1998

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free 
   Software Foundation; either version 2 of the License , or (at your option) 
   any later version.

   See the file sula/COPYING for details.

 */

#include <time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <fnmatch.h>
#include <ctype.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "spx.h"
#include "nc.h"
#include "parser.h"
#include "dcc.h"
#include "setting.h"

#include "cmd.h"
#include "hooks.h"

Script_control *script_form=NULL;
extern int MAX_ON;
S_Pipe *s_pipe=NULL;
short sp_count=0;
size_t nc_read_bufsize;

#define SP_ALLOCATE	5	//  # of connection buffers to allocate each time

static void remove_script_numhooks(int, NumHook *);	// for all numerics

static void remove_script_hook(NumHook * h, int sp);	// for one numeric

static int remove_sp(int fd);
static void remove_script_numhooks(int sp, NumHook * h)
{
  //remove script from all numeric lists h=numTree 
  if (h)
  {
    remove_script_numhooks(sp, h->left);
    remove_script_hook(h, sp);
    remove_script_numhooks(sp, h->right);
  }
}
static void remove_script_hook(NumHook * h, int sp)
{
//remove script from  a numeric's list 
  On *p;

  if (!h)
    return;
  p = h->start;
  while (p != h->end)
    if (p->cmd.sp == sp)
    {
      On *q = p->next;

      free(p->pattern);
      if (q == h->end)
        h->end = p;
      else
        *p = *q;
      free(q);
      continue;
    }
    else
      p = p->next;

}

typedef struct TCtrlType
{
  int type;
  char *name;
}
CtrlType;

#define sc_CONTINUE	(0)
#define sc_CMD		(1)
#define sc_CTCP		(2)
#define sc_EVENT	(3)
#define sc_REVENT	(4)
#define sc_SIGCONT	(5)
#define sc_SLEEPING	(6)
#define sc_DCC		(7)
#define sc_REGEXEVENT   (8)     //caseless
#define sc_CREGEXEVENT   (9)    //case-sensitive
#define sc_CEVENT	(10)	// case-sensitive

CtrlType ctrltype[] =
{
  {sc_CEVENT, "_CEVENT_"},
  {sc_CMD, "_CMD_"},
  {sc_CONTINUE, "_CONTINUE_"},

#if REGEX_HOOKS
  {sc_CREGEXEVENT, "_CREGEXEVENT_"},
#endif
  {sc_CTCP, "_CTCP_"},
  {sc_DCC, "_DCC_"},
  {sc_EVENT, "_EVENT_"},
#if REGEX_HOOKS
  {sc_REGEXEVENT, "_REGEXEVENT_"},
#endif
  {sc_REVENT, "_REVENT_"},
  {sc_SIGCONT, "_SIGCONT_"},
  {sc_SLEEPING, "_SLEEPING_"}
};
int nctrltypes = sizeof(ctrltype) / sizeof(CtrlType);
static int lookup_ctrl(const char *name)
{
  CtrlType *p = ctrltype;
  register int low,
       high,
       mid,
       cond;

  low = 0;
  high = nctrltypes - 1;
  while (low <= high)
  {
    mid = (low + high) / 2;
    if ((cond = strcasecmp(name, (p + mid)->name)) < 0)
      high = mid - 1;
    else if (cond > 0)
      low = mid + 1;
    else
      return (p + mid)->type;
  }
  return -1;
}
static void work_command(int, const char *inp, int ctrl_cmd);

#if USE_XFORMS
static void sp_cb(int fd, void *data)
#else
static void sp_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
//  read from  script connections (sockets and stream pipes 
{
  char *line;
  int n;
  static void parse_sp(int, char *line);

  line = alloca(sizeof(char) * (nc_read_bufsize + 1));

  if (line == NULL)
  {
    error(ERR_SYS, "Please reduce NC buffer size");
    return;
  }
  if ((n = read(fd, line, nc_read_bufsize)) < 0)
  {
    if (errno != EAGAIN)
    {
      error(ERR_SYS, "read error from %d", fd);
      n = remove_sp(fd);
      assert(!n);
    }
    return;
  }
  if (n == 0)
  {
    n = remove_sp(fd);
    assert(n == 0);
    return;
  }
  line[n] = '\0';
  if (line[n-1] == '\n'){
      line[--n] = '\0';
      if(n==0) return;
  }
  for (n = 0; n < sp_count; n++)
    if (s_pipe[n].fd == fd)
      break;
  assert(n != sp_count);
  if (!check_msg_hook(NC_RAW, n, "%s %s", s_pipe[n].name, line))
  {
    char *buf,
        *pp,
        *ptr = &line[0];
    int i;
    buf = malloc(sizeof(char) * (strlen(line) + 1));

    while (*ptr)
    {
      pp = strchr(ptr, '\n');
      if (pp != NULL)
      {
        i = pp - ptr;
        strncpy(buf, ptr, i);
        *(buf + i) = 0;
        if (*buf)
          parse_sp(n, buf);
        strshift((char *) line, i + 1);
        ptr = &line[0];
      }
      else if (*ptr)
      {
        parse_sp(n, ptr);
        break;
      }
    }
    free(buf);
  }
}


static void work_input(int sp, const char *ww, const char *inp)
{
// inp is trashed if ctcp reply

  if (!check_msg_hook(NC, sp, "%s %s", s_pipe[sp].name, inp))
  {
    char *s;
    int w;
    char *buf;

    buf = (char *) inp;
    w = strtol(ww, &s, 0);
    if (*s)
      say(buf, -1, 1);
    else
    {
      rmstr(buf, 0);
      if (*buf == 0)
        say(ww, -1, 1);
      else if (*buf!='/')
        say(buf, w, 0);
      else if((buf[1]!='r' && buf[1]!='R') || strncasecmp(buf, "/reply ", 7))
        parse_input(w, buf, 0);
      else
      {
        /* CTCP replies are a special case  */
        int n = strlen(buf) - 1;
        char *tmp, *cmd;

        /*  /reply xtr SFINGER @iLogin: xxx  */
        if (*(buf + n) == '\n')
          *(buf + n) = '\0';
        cmd = nextword(buf, 1); //  nick

        if (*cmd != 0)
        {
          n = find_pos(buf, 2);
          if (n < 0)
            return;
          else
            buf += n;
        }
        else
          return;
        tmp = alloca(sizeof(char) *
                     (strlen(buf) + 10) + strlen(cmd));

        sprintf(tmp, "/reply %s %s", cmd, buf);
        parse_input(w, tmp, 0);
        free(cmd);
      }
    }
  }
}
static void parse_sp(int sp, char *inp)
{
  /*if(!check_msg_hook(NC_RAW, sp, "%s %s", s_pipe[sp].name,inp)) */
  {
    char *cmd;
    char *buf = (char *) inp;
    int ctrl_cmd;

    cmd = nextword(buf, 0);
    if (*cmd == '_')
    {
      if ((ctrl_cmd = lookup_ctrl(cmd)) > -1)
      {
        work_command(sp, buf, ctrl_cmd);
        free(cmd);
        return;
      }
    }
    work_input(sp, cmd, buf);
    free(cmd);
  }
}

static void add_type(int fd, const char *inp, int type);
static void work_command(int i, const char *inp, int ctrl_cmd)
{
#define RM_BY_ID	0
#define RM_BY_SNR	1
  static void rm_event(int id, int fd, int flag);

  char msg[64],
       refresh_display = 1;
  char *buf = alloca(sizeof(char) * (strlen(inp) + 1));

  strcpy(buf, inp);
  msg[0] = '\0';
  switch (ctrl_cmd)
   {
     case sc_CONTINUE:
       {
         if ((s_pipe[i].status & SCR_SLEEPING))
         {
           s_pipe[i].status &= ~SCR_SLEEPING;

#if DEBUG
           sprintf(msg, "Connection %s now ready",
                   s_pipe[i].name);
#endif
         }
         break;
       }
     case sc_SIGCONT:
       {
         if (s_pipe[i].status & SCR_STOPPED)
         {
           s_pipe[i].status &= ~SCR_STOPPED;
#if DEBUG
           sprintf(msg, "Connection %s continued", s_pipe[i].name);
#endif
         }
         break;
       }
     case sc_SLEEPING:
       {
         if (!(s_pipe[i].status & SCR_SLEEPING))
         {
           s_pipe[i].status |= SCR_SLEEPING;
#if DEBUG
           sprintf(msg, "Connection %s sleeping",
                   s_pipe[i].name);
#endif
         }
         break;
       }
     case sc_REVENT:
       {
         int n;

         if (sscanf(inp, "%*s %d", &n) > 0)
           rm_event(n, i, RM_BY_SNR);
         break;
       }
     case sc_EVENT:
     case sc_CEVENT:
     case sc_REGEXEVENT:
     case sc_CREGEXEVENT:
     case sc_CTCP:
     case sc_CMD:
       {
         char *a = nextword(buf, 1);

         if (*a == 0)
           say(inp, -1, 1);
         else
           add_type(i, buf, ctrl_cmd);
         free(a);
       }
       break;
     case sc_DCC:
       {
         /*
            _DCC_ ident to CHAT s_add port 

            /connect fserv localhost 6700
            /fserv USER tano
            /fserv PASS tano12
            /on dcc_send_request *  /fserv accept $1 $3-
            /on private_msg "* * * !callin" /fserv conn $1

          */
         char *s;
         int w;
         char *cmd;

         /*
            _DCC_ w to SEND|CHAT args 
          */
         cmd = nextword(buf, 1);
         if (*cmd == 0)
           break;
         w = strtol(cmd, &s, 0);	// not sscanf(buf,"%d",&w) cos of 12hallo 

         if (*s || w < 0)
           say(buf, -1, 1);
         else
         {
           u_long sin_addr;
           u_short sin_port;
           char file_req = -1;
           char *nick = alloca(sizeof(char) * (strlen(buf) + 1));
           char *bb = alloca(sizeof(char) * (strlen(buf) + 16));
           char *txt = alloca(sizeof(char) * (strlen(buf) + 16));
           S_Pipe *sp = &s_pipe[i];

           free(cmd);
           if (win_invalid(w))
           {
             // >= win_count || winstruct[w].chanwin == NULL) {
             error(0, "[%s %d] bad window %d from %s\n",
                   __FILE__, __LINE__, w, sp->name);
             return;
           }
           /*
              _DCC_ ident to CHAT s_add port 
            */
           if (sscanf(buf, "%*s %*s %[^!]!%*s CHAT %lu %hu", nick,
                      &sin_addr, &sin_port) == 3)
             file_req = 0;
           else
             /*
                _DCC_ <ident> <to> SEND <txt>
                <txt>=filename s_addr port cksum(1) ext_flag
              */
           if (sscanf(buf, "%*s %*s %[^!]!%*s SEND %[^\n]\n", nick, txt) == 2)
             file_req = 1;
           if (file_req < 0)
           {
             say2(0, -1, 1, "Invalid req \"%s\" from %s\n",
                  buf, sp->name);
             return;
           }
           if (file_req)
             sprintf(bb, "PRIVMSG %s :%cDCC SEND %s%c\n",
                     nick, 0x01, txt, 0x01);
           else
             sprintf(bb, "PRIVMSG %s :%cDCC CHAT chat %lu %u%c\n",
                     nick, 0x01, sin_addr, sin_port, 0x01);
           sendto_server(&winstruct[w], bb);
#if DEBUG
           fputs(bb, stderr);
#endif
           return;
         }
         free(cmd);
       }
       break;
     default:
       sprintf(msg, "Error: unimplemented cmd %d", ctrl_cmd);
       refresh_display = 0;
       break;
   }                            //  swicth ctrl_cmd

  if (msg[0])
    {
    if (script_form)
     {
      fit_object_label(script_form->status, msg);
      if (refresh_display)
      {
        list_hooks(1);
        list_hooks(0);
      }
     }
#if DEBUG
    else
      fprintf(stderr, "-%s\n", msg);
#endif
  }
  return;
}
static void add_type(int i, const char *inp, int type)
{
  char *str = (char *) inp;
  char *cmdname;

  cmdname = nextword(str, 1);
  if (!*cmdname)
  {
    free(cmdname);
    return;
  }
  if (type == sc_CMD || type == sc_CTCP)
  {
    S_Pipe *p = &s_pipe[i];
    int flag;
    Cmd c;
    spCmd *s;

    flag = type == sc_CMD ? CMD_FD : CMD_CTCP;
#ifdef _GUILE
    if (type == sc_CMD && *cmdname == '(')
    {
      fprintf(stderr, "invalid cmd name \"%s\" from \"%s\"\n",
              cmdname, p->name);
      return;
    }
#endif
    s = my_malloc(sizeof(spCmd));
    s->name = strdup(cmdname);
    s->type = flag;
    s->next = p->cmd;
    p->cmd = s;
    c.name = strdup(p->cmd->name);
    c.type = CMD_FD | flag;
    c.help = (char *) my_malloc(sizeof(char) * strlen(str));

    if (sscanf(str, "%*s %*s %[^\n]\n", c.help) < 1)
    {
      free(c.help);
      c.help = NULL;
    }
    c.cmd.sp = i;
    if (flag == CMD_FD)
      insert_fd_cmd(&c);
    else
      insert_ctcp_cmd(&c);
    free(cmdname);
    return;
  }                             //   is is_normal..

  else
  {
    free(cmdname);
    cmdname = nextword(str, 2);
    if (!*cmdname)
    {
      free(cmdname);
      return;
    }
    if (type == sc_EVENT || type == sc_CEVENT ||
        type == sc_REGEXEVENT || type == sc_CREGEXEVENT)
    {

      On w;
      int num,
           flag;
      void *hooktype;

      if (sscanf(str, "%*s %d", &w.ser_number) < 1)
      {
        free(cmdname);
        return;
      }
      w.pattern = (char *) malloc(sizeof(char) * strlen(str));

      w.flag = HK_FD | HK_ICASE;
      if (type == sc_REGEXEVENT || type == sc_CREGEXEVENT)
        w.flag |= HK_REGEX;
      if (type == sc_CREGEXEVENT && type == sc_CEVENT)
        w.flag &= ~HK_ICASE;
      if (sscanf(str, "%*s %*d %*s %d", &flag) < 1)
      {
        if (sscanf(str, "%*s %*d %*s %[^\n]\n", w.pattern) < 1)
          strcpy(w.pattern, "*");
      }
      else
      {
        if (flag)
          w.flag |= HK_EXCLUSIVE;
        if (sscanf(str, "%*s %*d %*s %*d %[^\n]\n", w.pattern) < 1)
          strcpy(w.pattern, "*");
      }
      w.cmd.sp = i;
      w.id = -1;
      hooktype = lookup_type(cmdname, &num);
      if (!num)
      {
        MsgNr t = *((MsgNr *) hooktype);

        if (t != INVALID)
        {
          add_hook(t, &w);
          if (script_form)
            list_hooks(1);
        }
#if DEBUG
        else
          fprintf(stderr,
                  "[%s %d] Unknown: %s\n",
                  __FILE__, __LINE__, cmdname);
#endif
      }
      else
      {
        num = *((int *) hooktype);
        if (num > 0)
        {
          add_numhook(num, &w);
          if (script_form)
            list_hooks(1);
        }
      }
      free(w.pattern);
    }
  }                             //  else

}

static jmp_buf env;
static void rm_event(int id, int sp, int flag)
{
/*
   remove 1 event
 */
  static void rm_event_numhooks(int snr, int sp, NumHook * h, int flag);
  register int j;
  volatile int i;
  S_Pipe *p;

  i = sp;
  p = &s_pipe[i];
  for (j = 0; j < MAX_ON; j++)
  {
    On *q = want[j].start;

    while (q != want[j].end)
      if (q->flag & HK_FD && q->cmd.sp == i &&
          (flag == RM_BY_ID ? (q->id == id) : (q->ser_number == id))
         )
      {
        On *r = q->next;

        free(q->pattern);
        if (r == want[j].end)
          want[j].end = q;
        else
          *q = *r;
        free(r);
        return;
      }
      else
        q = q->next;
  }
  if (setjmp(env) != 0)         /*  in order to jump out of the numerics tree */
    return;
  rm_event_numhooks(id, i, numTree, flag);
}
static void rm_event_numhooks(int id, int sp, NumHook * h, int flag)
{
/*
   remove numeric event 
 */
  if (h)
  {
    On *p;

    rm_event_numhooks(id, sp, h->left, flag);
    p = h->start;
    while (p != h->end)
      if (p->flag & HK_FD && p->cmd.sp == sp &&
          (flag == RM_BY_ID ? (p->id == id) : (p->ser_number == id))
         )
      {
        On *q = p->next;

        free(p->pattern);
        if (q == h->end)
          h->end = p;
        else
          *p = *q;
        free(q);
        longjmp(env, 1);
      }
      else
        p = p->next;
    rm_event_numhooks(id, sp, h->right, flag);
  }
}

static int remove_sp(int fd)
{
/*
   remove a script
 */
  int i,
       j,
       do_it = 0;
  S_Pipe *p;

  if (fd == -1)
    return 0;

  for (i = 0; i < sp_count; i++)
    if (s_pipe[i].fd == fd)
    {
      do_it = 1;                //   or indent for ever...

      break;
    }
  if (!do_it)
    return (-1);
  check_msg_hook(NC_LOST, i, "%s", s_pipe[i].name);
#if USE_XFORMS
  fl_remove_io_callback(fd, FL_READ, sp_cb);
#else
  spx_remove_input(fd);
#endif
  close(fd);
  p = &s_pipe[i];
  p->fd = -1;
  free(p->name);
  switch (p->type)
   {
     case CON_SCRIPT:
       free(p->to.proc->sys_cmd);
       break;
     case CON_SERVER:
       free(p->to.server.name);
       if (p->to.server.args)
         free(p->to.server.args);
       break;
     case CON_DCC:
       free(p->to.dcc.nick);
       break;
   }
  while (p->cmd != NULL)
  {                             /*
                                   remove cmds
                                 */
    spCmd *q = p->cmd->next;

    if (p->cmd->type == CMD_FD)
      remove_fd_cmd(p->cmd->name);
    else
      remove_ctcp_cmd(p->cmd->name);
    free(p->cmd->name);
    free(p->cmd);
    p->cmd = q;
  }
  /*
     remove message hooks
   */
  for (j = 0; j < MAX_ON; j++)
  {
    On *q = want[j].start;

    while (q != want[j].end)
    {
      if ((q->flag & HK_FD) && q->cmd.sp == i)
      {
        On *r = q->next;

        free(q->pattern);
#if REGEX_HOOKS
        if (q->flag & HK_REGEX)
        {
          regfree(q->preg);
          free(q->preg);
        }
#endif
        if (r == want[j].end)
          want[j].end = q;
        else
          *q = *r;
        free(r);
        continue;
      }
      else
        q = q->next;
    }
  }
  remove_script_numhooks(i, numTree);
  if (script_form)
  {
    list_hooks(1);
    list_hooks(0);
  }
  return 0;
}

static void sp_allocate(void)
{
  register int i;

  if (s_pipe == NULL)
    s_pipe = my_malloc(sizeof(S_Pipe) * SP_ALLOCATE);
  else
    s_pipe = my_realloc(s_pipe, (sp_count + SP_ALLOCATE) * sizeof(S_Pipe));
  for (i = sp_count; i < sp_count + SP_ALLOCATE; i++)
  {
    s_pipe[i].fd = -1;
    s_pipe[i].cmd = NULL;
  }
  sp_count += SP_ALLOCATE;
}
int add_to_sp(S_Pipe * s)
{
  int i;

  errno = 0;
  if (!s_pipe)
    sp_allocate();
  do
  {
    for (i = 0; i < sp_count; i++)
      if (s_pipe[i].fd == -1)
      {
        Cmd *cm;

        if (!(cm = find_fd_cmd(s->name, 1)))
        {
          Cmd c;
          spCmd *sc;
          S_Pipe *p;

          s_pipe[i] = *s;
          p = &s_pipe[i];
          sc = malloc(sizeof(spCmd));
          sc->name = strdup(p->name);
          sc->type = CMD_FD;
          sc->next = p->cmd;
          p->cmd = sc;
          c.name = strdup(p->cmd->name);
          c.type = CMD_FD;
          if (p->type == CON_SCRIPT)
            c.help = strdup(p->to.proc->sys_cmd);
          else if (p->type == CON_SERVER)
          {
            c.help = malloc(sizeof(char) * (strlen(p->to.server.name) + 10));

            sprintf(c.help, "%s %hu", p->to.server.name, p->to.server.port);
          }
          else
            c.help = strdup("named connection");
          c.cmd.sp = i;
          insert_fd_cmd(&c);
        }                       //   if(!find..)

        else
        {
          fprintf(stderr,
                  "Pick different name for connection. \"%s\" exists:"
                  "%s- %s. Stop.\n", s->name, cm->name, cm->help ?
                  cm->help : "");
          return -1;
        }
        fcntl(s_pipe[i].fd, F_SETFL, O_NONBLOCK);
        spx_add_io_callback(s_pipe[i].fd, SPX_READ, sp_cb, 0);
        return 0;
      }
    sp_allocate();
  }
  while (1);
}
int write_to_sp(int sp, const char *buf)
{
  size_t ll;

  assert(sp > -1 && sp < sp_count && s_pipe[sp].fd > -1);
  if (s_pipe[sp].status & (SCR_STOPPED | SCR_SLEEPING | SCR_SUSPENDED))
    return (-2);
  ll = strlen(buf);
  if (writen(s_pipe[sp].fd, buf, ll) != ll)
   {
    if (errno == EINTR || errno == EAGAIN)
      return 0;
    else
    {
      int err = errno;

      if (s_pipe[sp].type == CON_SCRIPT)
        error(ERR_SYS, "[%s %d] write error to %s (pid=%d)",
              __FILE__, __LINE__, s_pipe[sp].to.proc->sys_cmd,
              s_pipe[sp].to.proc->pid);
      else
        error(ERR_SYS, "[%s %d] write error to %s",
              __FILE__, __LINE__, s_pipe[sp].name);
      remove_sp(s_pipe[sp].fd);
      return err;
    }
   }
  return 0;
}

S_Pipe *make_connection(const char *name, const char *server,
                        u_short port, const char *args)
{
  int newfd;
  S_Pipe *a;

  newfd = tcp_open((char *) server, 0, NULL, port, 30);
  if (newfd < 0)
    return NULL;
  a = malloc(sizeof(S_Pipe));
  if (!a)
    return NULL;
  a->fd = newfd;
  a->type = CON_SERVER;
  a->cmd = NULL;
  a->name = strdup(name);
  a->to.server.port = port;
  a->to.server.name = strdup(server);
  a->to.server.sin_addr = tcp_rem_addr.sin_addr;
  if (args)
  {
    a->to.server.args = strdup(args);
    writestr(newfd, "%s\n", args);
  }
  else
    a->to.server.args = NULL;
  if (add_to_sp(a))
  {
    close(a->fd);
    a->fd = -1;
    free(a->to.server.name);
    if (a->to.server.args)
      free(a->to.server.args);
    free(a->name);
    a = NULL;
  }
  else
  {
    int i;

    for (i = 0; i < sp_count; i++)
      if (s_pipe[i].fd == a->fd)
        return &s_pipe[i];
  }
  //should not be reached
  assert(0);
  return NULL;                  /* shut up noisy compiler */
}

void launch_prog(const char *name, const char *prog)
{
  int fd[2];
  pid_t pid;
  register int i;
  char *tmp;

  tmp = strrchr(prog, '/');
  if (tmp)
  {
    tmp++;
    if (!(*tmp))
      return;
  }
  else
    tmp = (char *) prog;
  for (i = 0; i < sp_count; i++)
    if (s_pipe[i].fd != -1 && s_pipe[i].type == CON_SCRIPT)
    {
      char *pp;

      pp = strrchr(s_pipe[i].to.proc->sys_cmd, '/');
      if (!pp)
        pp = s_pipe[i].to.proc->sys_cmd;
      else
        pp++;
      if (!strcmp(pp, tmp))
      {
        char *errorstr = malloc(sizeof(char) * (strlen(tmp) + 120));

        sprintf(errorstr,
                "Program \"%s\" is already running (PID %d)"
                , tmp, s_pipe[i].to.proc->pid);
        if (script_form)
          fit_object_label(script_form->status, errorstr);
        error(0, "%s", errorstr);
        free(errorstr);
        spx_bell(0);
        return;
      }
    }
  if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd) < 0)
  {
    error(-1, "socketpair error");
    return;
  }
  TELL_WAIT();
  if ((pid = fork()) < 0)
  {
    error(-1, "fork error");
    return;
  }
  if (pid > 0)
  {
    S_Pipe s;

    close(fd[1]);
    s.name = strdup(name);
    s.type = CON_SCRIPT;
    s.fd = fd[0];
    s.to.proc = malloc(sizeof(ProcInfo));
    s.to.proc->pid = pid;
    s.to.proc->sys_cmd = strdup(prog);
    s.cmd = NULL;
    s.status = SCR_RESET;
    if (!add_to_sp(&s))
    {
      TELL_CHILD(pid);
      list_hooks(0);
    }
    else if (kill(pid, SIGKILL) < 0)
      error(-1, "kill error");
    return;
  }
  else
  {
    char dup_err = 1;
    char *argv[4];

    //fprintf(stderr,"child:%d %s\n",getpid(), prog);
    WAIT_PARENT();
    close(fd[0]);
    if (fd[1] != STDIN_FILENO)
      if (dup2(fd[1], STDIN_FILENO) != STDIN_FILENO)
        goto bye;
    if (fd[1] != STDOUT_FILENO)
      if (dup2(fd[1], STDOUT_FILENO) != STDOUT_FILENO)
        goto bye;
    if (fd[1] != STDERR_FILENO)
      if (dup2(fd[1], STDERR_FILENO) != STDERR_FILENO)
        goto bye;
    argv[0] = "sh";
    argv[1] = "-c";
    argv[2] = (char *) prog;
    argv[3] = 0;
    if (execvp("/bin/sh", argv) == -1)
      dup_err = 0;

  bye:
    fprintf(stderr, "Unable to launch %s: %s\n",
            dup_err ? "dup2" : prog, strerror(errno));
    _exit(-1);
  }
}

/*
   t h e  G U I 
 */
#if USE_XFORMS
static int close_script_form(FL_FORM * form, void *junk)
{
  fl_hide_form(fl_get_active_folder(script_form->folder));
  fl_free_form(((FD_script *) script_form->folder->u_ldata)->script);
  fl_free_form(((FD_event *) script_form->folder->u_vdata)->event);
  fl_hide_form(script_form->scriptcontrol);
  fl_free_form(script_form->scriptcontrol);
  script_form = NULL;
  return FL_OK;
}

void do_script(void)
{

  if (script_form)
  {
    fl_raise_form(script_form->scriptcontrol);
    return;
  }
  else
  {
    FD_event *eventform = create_form_event();
    FD_script *scriptform = create_form_script();

    script_form = create_form_scriptcontrol();
    fl_set_object_bw(script_form->folder, -2);
    fl_set_browser_fontsize(eventform->browser, FL_NORMAL_SIZE);
    fl_set_browser_fontstyle(eventform->browser, FL_FIXED_STYLE);
    fl_set_browser_fontsize(scriptform->browser, FL_NORMAL_SIZE);
    fl_set_browser_fontstyle(scriptform->browser, FL_FIXED_STYLE);

    fl_set_form_atclose(script_form->scriptcontrol, close_script_form,
                        (void *) &script_form);
    fl_set_form_minsize(script_form->scriptcontrol, 560, 380);
    fl_set_form_maxsize(script_form->scriptcontrol, 560, 380);
    fl_addto_tabfolder(script_form->folder, "Scripts", scriptform->script);
    fl_addto_tabfolder(script_form->folder, "Events", eventform->event);
    /*
       hang "Event" form on to folder->u_vdata and "Scripts" form on to form->u_ldata
       since none of both forms is global
     */
    fl_deactivate_object(eventform->info);
    fl_set_object_color(eventform->info, FL_INACTIVE_COL, FL_INACTIVE_COL);
    script_form->folder->u_vdata = (void *) eventform;
    script_form->folder->u_ldata = (long) scriptform;

    list_hooks(1);
    list_hooks(0);
    fl_show_form(script_form->scriptcontrol, FL_PLACE_CENTER | FL_FREE_SIZE,
                 FL_FULLBORDER,
            sula_NAME " " sula_VERSION ": Event control -testing-testing-");

    fl_do_forms();
  }
}
     #if 0
static void edit_hook(SPX_OBJ(ob), void *data)
{
  fprintf(stderr, "[%s %d] edit_hook() commented out\a\n", __FILE__, __LINE__);
/*
   REMOVED COMPLETELY.
   SErved no useful purpose. (?)
 */
}
   #endif
void hook_edit_cb(SPX_OBJ(ob), long data)
{
}
#elif USE_GTK
static void run_a_program(const char *prog, void *junk)
{
  if (prog && *prog)
  {
    char *name = strrchr(prog, '/');

    if (name)
    {
      name++;
      if (!(*name))
        return;
    }
    else
      name = (char *) prog;
    launch_prog(name, prog);
  }
}
#endif
void script_cb(SPX_OBJ(ob), SPX_DATA(data))
{
  switch ((long) data)
   {
       /*
          todo: data<20 -->script 
        */
     case 10:                  //   run

       {
#if USE_XFORMS
         char *fname;

         fl_set_cursor(script_form->scriptcontrol->window, XC_watch);
         fl_deactivate_form(script_form->scriptcontrol);
         fname = getstr2("Enter program to execute: ",
           sula_NAME " " sula_VERSION ": Create Named Connection", NULL, 1);
         if (fname)
         {
           char *name = getstr2("Enter a name for the connection: ",
                     sula_NAME " " sula_VERSION ": Run a program", NULL, 0);

           if (name)
             launch_prog(name, fname);
           free(fname);
           free(name);
         }
         fl_activate_form(script_form->scriptcontrol);
         fl_reset_cursor(script_form->scriptcontrol->window);
#elif USE_GTK
         get_filename2(sula_NAME ": Create a named connection",
                      NULL,
                      run_a_program, 0);
#endif
         return;
       }
     case 11:                  //   script suspend (not sigstop)

     case 17:                  //   script stop, sigstop 

     case 18:                  //   continue/resume

     case 12:                  //   script kill->sigkill

       {
         char refresh_display = 0;
#if USE_XFORMS
         FD_script *form = (FD_script *) script_form->folder->u_ldata;
         int browser_line = fl_get_browser(form->browser);

         if (browser_line < 1)
         {
           if (gflags & BEEP_ERR)
             spx_bell(0);
             return;
         }
         else
#elif USE_GTK
           GtkCList * clist = GTK_CLIST(
                    gtk_object_get_data(GTK_OBJECT(script_form->window), "clist2"));
         GList *p = clist->selection;
         gint row;

         if (!clist->selection)
         {
           if (gflags & BEEP_ERR)
             spx_bell(0);
             return;
         }
         else
           while (p)
#endif
           {

             char *buf;
             int sock = -1;

#if USE_XFORMS
             buf = (char *) fl_get_browser_line(form->browser, browser_line);
             if (*buf == 0 || sscanf(buf, "%*s %d", &sock) < 1
                 || sock < 0)
               return;
#elif USE_GTK
             row = GPOINTER_TO_INT(p->data);
             gtk_clist_get_text(clist, row, 1, &buf);
             sscanf(buf, "%d", &sock);
             if (sock < 0)
             {
               p = p->next;
               continue;        //   internal hook 

             }
#endif

             assert(sock < sp_count && -1 < s_pipe[sock].fd);
             buf = malloc(sizeof(char) * 256);

             buf[0] = 0;
             switch ((long) data)
              {
                case 11:
                  if (!(s_pipe[sock].status & SCR_SUSPENDED))
                  {
                    s_pipe[sock].status |= SCR_SUSPENDED;
if(debug){
                    if (s_pipe[sock].type == CON_SCRIPT)
                      sprintf(buf, "Connection '%s' suspended. New status %#x",
                              s_pipe[sock].name, s_pipe[sock].status);
}
                    refresh_display = 1;
                  }
                  break;
                case 17:
                  if (!(s_pipe[sock].status & SCR_STOPPED))
                  {
                    if (s_pipe[sock].type == CON_SCRIPT &&
                        kill(s_pipe[sock].to.proc->pid, SIGSTOP) < 0)
                      strncpy(buf, strerror(errno), 58);
                  }
                  break;
                case 12:
                  if (s_pipe[sock].type == CON_SCRIPT &&
                      kill(s_pipe[sock].to.proc->pid, SIGKILL) < 0)
                  {
                    strncpy(buf, strerror(errno), 58);
                  }
                  break;
                case 18:
                  if ((s_pipe[sock].status & SCR_STOPPED))
                  {
                    if (s_pipe[sock].type == CON_SCRIPT &&
                        kill(s_pipe[sock].to.proc->pid, SIGCONT) < 0)
                      strncpy(buf, strerror(errno), 58);
                  }
                  if ((s_pipe[sock].status & SCR_SUSPENDED))
                  {
                    s_pipe[sock].status &= (~SCR_SUSPENDED);
if(debug){
                    if (s_pipe[sock].type == CON_SCRIPT)
                      sprintf(buf, "Process %d resumed?. New status %#X",
                            s_pipe[sock].to.proc->pid, s_pipe[sock].status);
                    else
                      sprintf(buf, "%s resumed. New status %#X",
                              s_pipe[sock].name, s_pipe[sock].status);
}
                    refresh_display = 1;
                  }
                  break;
              }                 //   switch data 

             if (*buf)
               fit_object_label(script_form->status, buf);
             free(buf);
#if USE_GTK
             p = p->next;
#endif
           }                    //   else/while
           if (refresh_display)
             {
               list_hooks(1);
               list_hooks(0);
             }
         break;
       }
       /*
          data >20 -->event form
        */
     case 28:                  //   remove

       {

         int n;
         char refresh_display = 0;
#if USE_XFORMS
         FD_event *form = (FD_event *) script_form->folder->u_vdata;

         n = fl_get_browser(form->browser);

         if (n < 1)
           fit_object_label(script_form->status, "select first");
         else
#elif USE_GTK
           GtkCList * clist = GTK_CLIST(
                     gtk_object_get_data(GTK_OBJECT(script_form->window), "clist"));
         GList *pclist = clist->selection;
         gint row;

         if (!clist->selection)
         {
           if (gflags & BEEP_ERR)
             spx_bell(0);
         }
         else
           while (pclist)
#endif
           {
             char *tmp;
             char *ptr;
             void *type;
             int id;
             On *p = NULL,
                **end;

#if USE_XFORMS
             char *buf = (char *) fl_get_browser_line(form->browser, n);

             if (!buf || *buf == 0 || sscanf(buf, "%*s %d", &id) != 1)
               return;
             ptr = nextword(buf, 0);
             tmp = alloca(sizeof(char) * (strlen(ptr) + 1));

             strcpy(tmp, ptr);
             free(ptr);
             //      kludge to remove xforms format characters
             ptr = strrchr(tmp, '@');
             if (ptr)
               strcpy(tmp, *(ptr + 1) == 'C' ? (ptr + 3) : (ptr + 2));
#elif USE_GTK
             row = GPOINTER_TO_INT(pclist->data);
             id = (int) gtk_clist_get_row_data(GTK_CLIST(clist), row);
             gtk_clist_get_text(clist, row, 0, &tmp);
#endif
             type = lookup_type(tmp, &n);
             if (!n)
             {
               MsgNr t = *((MsgNr *) type);

               // on -del #msg snr pattern
               // on -del msg pattern
               if (t == INVALID)
#if USE_XFORMS
                 return;
#elif USE_GTK
               {
                 #if DEBUG
                 fprintf(stderr, "bug:unknown '%s'\n", tmp);
                 #endif
                 pclist = pclist->next;
                 continue;
               }
#endif
               p = want[t].start;
               while (p != want[t].end)
                 if (p->id == id)
                   break;
                 else
                   p = p->next;
               if (p == want[t].end)
                 p = NULL;
               end = &want[t].end;
             }
             else
             {
               NumHook *N = find_numhook(*((int *) type));

               assert(N);
               p = N->start;
               while (p != N->end)
                 if (p->id == id)
                   break;
                 else
                   p = p->next;
               if (p == N->end)
                 p = NULL;
               end = &N->end;
             }
             if (p == NULL)     // this is a bug
#if USE_XFORMS
               break;
#elif USE_GTK
             {
               pclist = pclist->next;
               continue;
             }
#endif
             if (p->flag & HK_FD)
             {
               On *q = p->next;

               free(p->pattern);
               if (q == *end)
                 *end = p;
               else
                 *p = *q;
               free(q);
               refresh_display=1;
             }
             else
             {
               ptr = malloc(sizeof(char) * (strlen(tmp)+strlen(p->pattern) + 64));

               if (p->ser_number == 0)
                 sprintf(ptr, "%s %s", tmp, p->pattern);
               else
                 sprintf(ptr, "#%s %d %s", tmp, p->ser_number, p->pattern);
               delete_hook(ptr, p->flag & HK_PREEMPTIVE ? 1 : 0, 0);
               free(ptr);
               refresh_display=1;
             }
      #if USE_GTK
             pclist = pclist->next;
     #endif
           }
           if (refresh_display)
            list_hooks(1);
       }
       break;
     case 23:                  //   info

       break;
     case 24:                  //   edit

     case 27:                  //   pause 

     case 26:                  //   continue

       {
         int n;
         char refresh_display = 0;
#if USE_XFORMS
         FD_event *form = (FD_event *) script_form->folder->u_vdata;

         n = fl_get_browser(form->browser);

         if (n < 1)
           fit_object_label(script_form->status, "select first");
         else
#elif USE_GTK
         GtkCList * clist = GTK_CLIST(
                    gtk_object_get_data(GTK_OBJECT(script_form->window), "clist"));
         GList *pclist = clist->selection;
         gint row;

         if (!clist->selection)
         {
           if (gflags & BEEP_ERR)
             spx_bell(0);
         }
         else
           while (pclist)
#endif
           {
             char *tmp;
             void *type;
             int id;
             On *p = NULL,
                **pp;

#if USE_XFORMS
             char *ptr;
             char *buf = (char *) fl_get_browser_line(form->browser, n);

             if (!buf || *buf == 0 || sscanf(buf, "%*s %d", &id) != 1)
               return;
             ptr = nextword(buf, 0);
             tmp = alloca(sizeof(char) * (strlen(ptr) + 1));

             strcpy(tmp, ptr);
             free(ptr);
             ptr = strrchr(tmp, '@');
             if (ptr)
               strcpy(tmp, *(ptr + 1) == 'C' ? (ptr + 3) : (ptr + 2));
#elif USE_GTK
             row = GPOINTER_TO_INT(pclist->data);
             id = (int) gtk_clist_get_row_data(GTK_CLIST(clist), row);
             gtk_clist_get_text(clist, row, 0, &tmp);
#endif
             type = lookup_type(tmp, &n);
             if (!n)
             {
               MsgNr t = *((MsgNr *) type);

               if (t != INVALID)
               {
                 p = want[t].start;
                 while (p != want[t].end)
                   if (p->id == id)
                     break;
                   else
                     p = p->next;
                 if (p == want[t].end)
                   p = NULL;
                 pp = &want[t].end;
               }
               else
#if USE_XFORMS
                 return;
#elif USE_GTK
               {
                 pclist = pclist->next;
                 continue;
               }
#endif
             }
             else
             {
               NumHook *N = find_numhook(*((int *) type));

               assert(N);
               p = N->start;
               while (p != N->end)
                 if (p->id == id)
                   break;
                 else
                   p = p->next;
               if (p == N->end)
                 p = NULL;
               pp = &N->end;
             }
             if (!p)
             {
             #if DEBUG
               fprintf(stderr, "[%s %d] bug here..\a\n", __FILE__, __LINE__);
             #endif
#if USE_XFORMS
               break;
#elif USE_GTK
               {
                 pclist = pclist->next;
                 continue;
               }
#endif
             }                  //   id not found 

             switch ((long)data)
              {
                case 23:       //   info

                  break;
                case 24:
                  break;
                case 27:
                  if (p->flag & HK_SUSPENDED)
                    break;
                  p->flag |= HK_SUSPENDED;
                  refresh_display=1;
                  break;
                case 26:
                  if (!(p->flag & HK_SUSPENDED))
                    break;
                  p->flag &= ~HK_SUSPENDED;
                  refresh_display=1;
                  break;
              }                 //   switch
             #if USE_GTK
             pclist = pclist->next;
             #endif
           }
           if(refresh_display) list_hooks(1);
       }
       break;
     case 25:
       {                        //   new

         //edit_hook(ob, NULL);
       }
       break;
#if USE_XFORMS
     case 6:                   //   done 

       {
         close_script_form(NULL, NULL);
         break;
       }
#endif
     case 8:                   //   update

       {
         list_hooks(1);
         list_hooks(0);
       }
       break;
     default:
       break;
   }
}
