/***************************************************
  This file is distributed as part of Sula PrimeriX
  (http://members.xoom.com/sprimerix).
****************************************************/


/*
   notify.c     - NOTIFY features

   Author: Tano Fotang, 12/1998

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free 
   Software Foundation; either version 2 of the License , or (at your option) 
   any later version. see the file COPYING for more information.

 */

#ifndef NO_EXTENDED_NOTIFY
#include "spx.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"

//#include "setting.h"

#include "server.h"
#include "hooks.h"
#include "notify.h"

char *fmt_notify_signon,
 *fmt_notify_signoff;
Notify_t NotifyListCount=0;
Notify *aNotifyList=NULL;
Notify_sent *NotifySentStart,
 *NotifySentEnd;

#define valid_index(i) valid_notify_index((i))
// compare a notify server and an irc server
#define server_comp(ns, server) ( (ns) && \
                        !strcmp((ns)->name,(server)->name) && \
                        (ns)->port==(server)->port)

static void reset_notify_flags(Notify_t k, int j)
{

  NotifyNick *p = aNotifyList[k].start;

  while (p != aNotifyList[k].end)
  {
    (p->flag)[j] = (~NOTIFY_ISONLINE) | NOTIFY_STATUS_CHANGED;
    p = p->next;
  }
}
//attaching notify groups to servers
static int add_server_to_notifygroup(Server * s, Notify_t i)
{
  Notify *nl = &aNotifyList[i];

  register int j;

  for (j = 0; j < MAX_NOTIFY_SERVERS; j++)
    if ((nl->server)[j] == NULL)
    {
      (nl->server)[j] = malloc(sizeof(NotifyServer));
      (nl->server)[j]->name = s->name;
      (nl->server)[j]->port = s->port;
      reset_notify_flags(i, j);
      return 0;
    }
  return 1;                     //no more slots

}
int attach_notify_group(Server * server, Notify_t c)
{
  // add c to t. Similar to addding a char to a string
  Notify_t *ptr;
  int newlen;

  assert(valid_index(c));
  ptr = server->notify;
  errno = 0;
  while (*ptr != NOTIFY_EOL)
  {
    if (*ptr == c)              // got it already

      return 0;
    if (*ptr == -1)
    {                           // free slot

      if (add_server_to_notifygroup(server, c))
        return -1;
      *ptr = c;
      return 0;
    }
    ptr++;
  }
  if (add_server_to_notifygroup(server, c))
    return -1;
  newlen = 1 + (ptr - server->notify);
  server->notify = realloc(server->notify, sizeof(Notify_t) * (newlen + 1));
  if (!server->notify)
    return -1;
  *(server->notify + newlen - 1) = c;
  *(server->notify + newlen) = NOTIFY_EOL;
  return 0;
}

//detaching notify groups from servers
static int notify_length(const Notify_t * n)
{
  Notify_t *p = (Notify_t *) n;

  if (n == NULL)
    return 0;
  while (*p != NOTIFY_EOL)
    p++;
  return (p - n);
}

void copy_notify(Notify_t ** dest, const Notify_t * src)
{
  if (src == NULL)
    *dest = NULL;
  else
  {
    int i = notify_length(src);

    *dest = malloc(sizeof(Notify_t) * (i + 1));
    *(*dest + i) = NOTIFY_EOL;
    while (--i > -1)
      *(*dest + i) = *(src + i);
  }
}

void remove_server_from_notifygroup(Server * s, Notify_t i)
{
  Notify *nl = aNotifyList + i;

  register int j;

  if (!valid_index(i))
    return;
  for (j = 0; j < MAX_NOTIFY_SERVERS; j++)
    if (server_comp((nl->server)[j], s))
    {
      free((nl->server)[j]);
      (nl->server)[j] = NULL;
      reset_notify_flags(i, j);
      break;
    }
}
int detach_notify_group(Server * server, Notify_t c)
{
  // remove group c from servers group list 
  Notify_t *p;

  errno = 0;
  if (!valid_index(c))
    return 1;
  remove_server_from_notifygroup(server, c);
#ifndef NO_LAZY_NOTIFY
  // we dont free the 1 byte space occupied by c in server->notify.
  // we keep it for next time. This makes attaching notify groups
  // easier and much faster.
  for (p = server->notify; *p != NOTIFY_EOL; p++)
    if (*p == c)
    {
      *p = -1;
      return 0;
    }
#else
  {
    Notify_t *s;
    Notify_t *ptr;

    ptr = server->notify;
    while (*ptr != NOTIFY_EOL)
      ptr++;
    s = malloc(sizeof(Notify_t) * ((ptr - server->notify) + 2));
    if (!s)
      return -1;
    ptr = s;
    p = server->notify;
    while (*p != NOTIFY_EOL)
    {
      if (*p != c)
        *ptr++ = *p;
      p++;
    }
    *ptr = NOTIFY_EOL;
    free(server->notify);
    copy_notify(&server->notify, s);
  }
#endif
  return 0;
}

int addto_notify(Notify_t i, const char *nick)
{

  Notify *n;
  NotifyNick *p;

  assert(valid_index(i));
  n = &aNotifyList[i];
  p = n->start;
  if (*nick == 0)
    return 2;
  while (p != n->end)
    if (!strcasecmp(p->nick, nick))
      break;
    else
      p = p->next;
  if (p == n->end)              // nick not yet in the list

  {
    register int i;

    p = n->start;
    n->start = malloc(sizeof(NotifyNick));
    if (!n->start)
      error(-2, "malloc error");
    n->start->nick = strdup(nick);
    for (i = 0; i < MAX_NOTIFY_SERVERS; i++)
      (n->start->flag)[i] = ~(NOTIFY_ISONLINE | NOTIFY_STATUS_CHANGED);
    n->start->next = p;
    return 0;
  }
  return 1;                     // nickname is already in  the group

}

int drop_from_notify(Notify_t i, const char *nick)
{
  Notify *n;
  NotifyNick *p;

  assert(valid_index(i));
  n = &aNotifyList[i];
  p = n->start;

  if (*nick == 0)
    return 2;
  while (p != n->end)
    if (!strcasecmp(p->nick, nick))
      break;
    else
      p = p->next;
  if (p == n->end)
    return 1;
  else
  {
    NotifyNick *q = p->next;

    free(p->nick);
    if (q == n->end)
      n->end = p;
    else
      *p = *q;
    free(q);
    return 0;
  }
}

static int notify_more(void)
{
  Notify_t i;

#define NOTIFY_LIST_ALLOC 2

  if (aNotifyList == NULL)
    aNotifyList = malloc(sizeof(Notify) * NOTIFY_LIST_ALLOC);
  else
    aNotifyList = realloc(aNotifyList,
                    sizeof(Notify) * (NotifyListCount + NOTIFY_LIST_ALLOC));
  if (aNotifyList == NULL)
    return -1;
  for (i = NotifyListCount; i < NotifyListCount + NOTIFY_LIST_ALLOC; i++)
    aNotifyList[i].label = NULL;
  NotifyListCount += NOTIFY_LIST_ALLOC;
  return 0;
}

Notify_t new_notify_list(const char *label)
/*return a new notify group if none exists named <label>
   (case-sensitive */
{
  Notify_t i = find_notify_list_bylabel(label);

  if (i != -1)
    return i;
  if (aNotifyList == NULL && notify_more())
    return -1;
again:
  for (i = 0; i < NotifyListCount; i++)
    if (aNotifyList[i].label == NULL)
    {
      memset((void *) aNotifyList[i].server, 0,
             MAX_NOTIFY_SERVERS * sizeof(NotifyServer *));
      aNotifyList[i].start = aNotifyList[i].end = malloc(sizeof(NotifyNick));
      if (aNotifyList[i].start == NULL)
        return -1;
      aNotifyList[i].label = strdup(label);
      return i;
    }
  if (notify_more())
    return -1;
  goto again;
}

int remove_notify_list(Notify_t i)
//delete a notify group from memory
{
  NotifyNick *pp;

  assert(valid_index(i));
// remove from server notify groups first!
  {
    register int i;

    for (i = 0; i < win_count; i++)
      if (winstruct[i].chanwin && winstruct[i].server)
      {
        Notify_t *p = winstruct[i].server->notify;

        while (*p != NOTIFY_EOL)
        {
          if (*p == i)
            *p = -1;
          p++;
        }
      }
  }
  pp = aNotifyList[i].start;
  while (pp != aNotifyList[i].end)
  {
    NotifyNick *pq = pp->next;

    free(pp->nick);
    if (pq == aNotifyList[i].end)
      aNotifyList[i].end = pp;
    else
      *pp = *pq;
    free(pq);
  }
  free(aNotifyList[i].label);
  aNotifyList[i].label = NULL;
  return 0;
}

static void make_fname(const char *buf, const char *fname)
{

  if (fname == NULL || *fname == 0)
  {
    sprintf((char *) buf, "%s/%s", prog_home, P_NOTIFY);
  }
  else
  {
    if (*fname == '~')
    {
      char *tt = expand_tilde(fname);

      sprintf((char *) buf, "%s", tt);
      free(tt);
    }
    else
      sprintf((char *) buf, "%s", fname);
  }
}
int save_notify_lists(const char *fname)
{

  FILE *fp;
  char *buf = alloca(sizeof(char) * (MAXPATHLEN + 255));

  make_fname(buf, fname);
  fp = fopen(buf, "w");
  if (!fp)
    return -1;
  else
  {
    Notify_t i;
    NotifyNick *a;
    time_t t = time(NULL);

    fprintf(fp, "%c " sula_NAME sula_VERSION
            "\n%c ==%s== Notify groups created %s\n",
            COMMENT, COMMENT, getenv("USER"), ctime(&t));
    for (i = 0; i < NotifyListCount; i++)
    {
      if (aNotifyList[i].label == NULL)
        continue;
      fprintf(fp, "\n<%s>\n", aNotifyList[i].label);
      a = aNotifyList[i].start;
      while (a != aNotifyList[i].end)
      {
        fprintf(fp, "\t%s\n", a->nick);
        a = a->next;
      }
      fprintf(fp, "</%s>\n", aNotifyList[i].label);
    }
  }
  fclose(fp);
  return 0;
}

int load_notify_lists(const char *fname)
{
  FILE *fp;
  char *buf = alloca(sizeof(char) * (MAXPATHLEN + 255));

  make_fname(buf, fname);

  fp = fopen(buf, "r");
  if (!fp)
    return -1;
  else
  {
    char **nicks,
     *label;
    int count;

    nicks = alloca(sizeof(char *) * (NOTIFY_QUEUE_LEN + 1));

    while ((label = get_string_entries(fp,
                                       &count, nicks, 
                                       NOTIFY_QUEUE_LEN, 1)))
    {
      Notify_t i = new_notify_list(label);

      if (i < 0)
      {
        i = errno;
        free(label);
        errno = i;
        fclose(fp);
        return -1;
      }
      else
      {
        char **p = nicks;

        free(label);
        while (*p)
        {
          addto_notify(i, *p);
          free(*p);
          p++;
        }
      }
    }
  }
  fclose(fp);
  return 0;
}
Notify_t find_notify_list_bylabel(const char *label)
{
  Notify *p = aNotifyList;

  while (p - aNotifyList < NotifyListCount)
    if (p->label && !strcmp(p->label, label))
      return (p - aNotifyList);
    else
      p++;
  return -1;
}

// 
// the NOTIFY command
//

static int server_to_index(Notify_t i, Server * s)
{
#define valid_flag_index(j) ((j) != -1)

  register int j;
  Notify *nl = &aNotifyList[i];

  for (j = 0; j < MAX_NOTIFY_SERVERS; j++)
    if (server_comp((nl->server)[j], s))
      return j;
  return -1;

}

#ifdef _GUILE
static SCM build_it(Notify_t i, Server * s)
{
  NotifyNick *p;

  //SCM ret=gh_list(gh_long2scm(i),SCM_UNDEFINED);
  SCM tmp = gh_list(SCM_UNDEFINED);

  p = aNotifyList[i].start;
  while (p != aNotifyList[i].end)
  {
    int idx = server_to_index(i, s);

    if (!valid_flag_index(idx))
      return SCM_BOOL_F;        // a bug really

    tmp = gh_cons(gh_list(gh_str02scm(p->nick),
                          gh_bool2scm((p->flag)[idx] & NOTIFY_ISONLINE),
                          SCM_UNDEFINED
                  ),
                  tmp);
    p = p->next;
  }
  return gh_cons(gh_long2scm(i), tmp);
}

SCM make_server_SCM_notify_list(Server * s)
{
  Notify_t *list = s->notify;
  SCM ret = gh_list(SCM_UNDEFINED);

  while (*list != NOTIFY_EOL)
  {
    if (*list != -1)
    {
      SCM tmp = build_it(*list, s);

      if (!gh_equal_p(tmp, SCM_BOOL_F))
        ret = gh_cons(tmp, ret);
    }
    list++;
  }
  return ret;
}
#endif

static int list_notify(Notify_t i, int win)
{
  NotifyNick *p;
  u_char count = 0;

  assert(valid_index(i));
  p = aNotifyList[i].start;
  say2(1, win, 0, "@v==Group: %s==", aNotifyList[i].label);
  while (p != aNotifyList[i].end)
  {
    int idx = server_to_index(i, winstruct[win].server);

    if (!valid_flag_index(idx))
      return -1;
    if ((p->flag)[idx] & NOTIFY_ISONLINE)
    {
      say2(0, win, 0, "\t%s is online", p->nick);
      count++;
    }
    p = p->next;
  }
  if (count == 0)
    say0("\tAll offline", win, 0);
  return 0;
}
static void list_server_notify(int win)
{

  if (win == -1)
  {                             // list those who are online for all connected servers

    Notify_t *list;
    register int i;

    for (i = 0; i < win_count; i++)
      if (winstruct[i].chanwin && winstruct[i].server &&
          winstruct[i].server->fd > -1)
      {
        say2(1, win, 0, "@i--Online for %s/%hu",
             winstruct[i].server->name, winstruct[i].server->port);
        list = winstruct[i].server->notify;
        while (*list != NOTIFY_EOL)
        {
          if (*list != -1)
            list_notify(*list, i);
          list++;
        }
      }
    return;
  }
  else if (!winstruct[win].server)
  {
    if (gflags & BEEP_ERR)
      spx_bell(0);
    fit_object_label(winstruct[win].chanwin->WinStatusLine,
                     "Window has no server");
    return;
  }
  else
  {

    Notify_t *list = winstruct[win].server->notify;

    while (*list != NOTIFY_EOL)
    {
      if (*list != -1)
        list_notify(*list, win);
      list++;
    }
  }
}

void notify(char *line, int win)
/* the NOTIFY command */
{
  char *opt = alloca(sizeof(char) * strlen(line));

  *opt = 0;
  sscanf(line, "%*s %s", opt);
  if (sscanf(line, "%*s %s", opt) != 1)
  {
    list_server_notify(win);
    return;
  }
  if (*opt == '-')
  {
    char *arg = alloca(sizeof(char) * strlen(line));

    *arg = 0;
    sscanf(line, "%*s %*s %[^\n]", arg);
    if (!strcasecmp(opt, "--save"))
    {
      if (save_notify_lists(arg) == -1)
        say2(0, win, -1, "(%s) Error saving notify lists: %s",
             *arg ? arg : "", strerror(errno));
      return;
    }
    else if (!strcasecmp(opt, "--load"))
    {
      if (load_notify_lists(arg))
        say2(0, win, -1, "Unable to load notify lists: %s",
             strerror(errno));
      return;
    }
    else if (!strcasecmp(opt, "--drop"))
    {
      Notify_t i;
      char *group = *arg ? arg : "default";

      i = find_notify_list_bylabel(group);
      if (i == -1)
        say2(0, win, -1, PROMPT "%s: No such notify group", group);
      else
        remove_notify_list(i);
      return;
    }
    else if (!strcasecmp(opt, "--list"))
    {
      Notify_t i;
      NotifyNick *a;

      for (i = 0; i < NotifyListCount; i++)
      {
        if (aNotifyList[i].label == NULL)
          continue;
        say2(1, win, 0, "@v==Notify group: %s==", aNotifyList[i].label);
        a = aNotifyList[i].start;
        while (a != aNotifyList[i].end)
        {
          say2(0, win, 0, "\t%s", a->nick);
          a = a->next;
        }
      }
    }
    else if (!strcasecmp(opt, "--force"))
    {
      check_notify(1);
      return;
    }
    else
      goto work;
  }
  else
  work:
  {
    char *group;
    Notify_t n;

    group = strstr(line, "--");
    if (group)
    {
      *group = 0;
      group += 2;
      unspace(group);
    }
    else
      group = "default";
    if (*group == 0)
      return;
    n = find_notify_list_bylabel(group);
    if (n == -1)
    {
      if (gflags & VERBOSE_CLIENT)
        say2(0, win, -1, PROMPT "%s: No such notify group. Creating...",
             group);
      n = new_notify_list(group);
      if (n == -1)
      {
        say2(0, win, -1, PROMPT "Unable to create %s: %s", group,
             strerror(errno));
        return;
      }
      else if (gflags & VERBOSE_CLIENT)
        say2(0, win, -1, PROMPT "%s has been created", group);
    }
    rmstr(line, 0);
    if (*line == 0)
      return;                   //--name

    else
    {
      char *ptr;
      char *pp = strdup(line),
       *argv = pp;

      while ((ptr = strsep(&argv, " \t")))
        if (*ptr == '-')
        {
          if (*++ptr)
            drop_from_notify(n, ptr);
        }
        else
        {
          if (*ptr == '+')
            ptr++;
          if (*ptr)
            addto_notify(n, ptr);
        }
      free(pp);
    }
  }
}

// ////////////////// end of notify command

// / the idle callback ///////////////////////
static int notifyTimeoutID = 0;
static void save_sent_notify(Server * s, const char *nick)
{
  Notify_sent *p = NotifySentStart;

  NotifySentStart = malloc(sizeof(Notify_sent));
  //NotifySentStart->nick = strdup(nick);
  //NotifySentStart->time = time(NULL);
  NotifySentStart->next = p;
  NotifySentStart->server = strdup(s->name);
  NotifySentStart->port = s->port;
}
static void do_one(Server * s)
{
  if (!s->connecting)
  {
#define ISON_SIZE 420
    Notify_t *p;
    char buf[ISON_SIZE + 64];
    char already_saved = 0;

    remove_sent_notifies(s);    // establish a sane state

    buf[0] = 0;
    for (p = s->notify; *p != NOTIFY_EOL; p++)
      if (*p != -1)
      {
        NotifyNick *n = aNotifyList[*p].start,
         *end = aNotifyList[*p].end;

        while (n != end)
        {
          if (strcasecmp(s->nick, n->nick))
          {
            strcat(buf, n->nick);
            if (!already_saved)
            {
              save_sent_notify(s, n->nick);
              already_saved = 1;
            }
            if (strlen(buf) > ISON_SIZE)
            {
              buf[ISON_SIZE] = 0;
              //we shall wait for more replies if our nick is the first in the reply
              writestr(s->fd, "ISON %s %s\n", s->nick, buf);
              buf[0] = 0;
              continue;
            }
            else
              strcat(buf, " ");
          }
          n = n->next;
        }
      }
    if (buf[0])
    {
      writestr(s->fd, "ISON %s\n", buf);
      buf[0] = 0;
    }
  }
}

static void send_ison(Server * p)
// check notify lists of all connected servers
{
  if (p)
  {
    Server *s;

    send_ison(p->left);
    s = search_server_tree_byname(p->g, p->name, p->port);
    assert(s);
    do_one(s);
    send_ison(p->right);
  }
}
#if USE_XFORMS
static void autocheck_notify(int id, void *junk)
#else
static gint autocheck_notify(gpointer junk)
#endif
{
  send_ison(servFdTree);
  notifyTimeoutID = 0;
#if !USE_XFORMS
  return FALSE;
#endif
}

// send notify nicknames to all connected irc servers
extern void check_notify(int force)
{
  if (force && servFdTree)
  {
    send_ison(servFdTree);
    return;
  }
  if (servFdTree == NULL || !(gflags & AUTO_CHECKNOTIFY))
  {
    if (notifyTimeoutID)
    {
      spx_remove_timeout(notifyTimeoutID);
      notifyTimeoutID = 0;
    }
  }
  else if (!notifyTimeoutID)
  {
    static char first_run = 0;

    if (!first_run)
      send_ison(servFdTree);
    first_run = 1;
    notifyTimeoutID = spx_add_timeout((notify_frequency * 1000),
                                      autocheck_notify, 0);
  }
}

// check NOTIFY hooks
static void check_notify_hooks(Winstruct * win)
// run thro notify lists for a server and report status changes
{
  Notify_t *i;

  for (i = win->server->notify; *i != NOTIFY_EOL; i++)
    if (*i != -1)
    {
      register int j;
      Notify *nl = &aNotifyList[*i];

      for (j = 0; j < MAX_NOTIFY_SERVERS; j++)
        if (server_comp((nl->server)[j], win->server))
        {
          NotifyNick *p = aNotifyList[*i].start;

          while (p != aNotifyList[*i].end)
          {
            if ((p->flag)[j] & NOTIFY_STATUS_CHANGED)
            {
              (p->flag)[j] &= ~NOTIFY_STATUS_CHANGED;
              if (!check_msg_hook((p->flag)[j] & NOTIFY_ISONLINE ?
                                  NOTIFY_SIGNON : NOTIFY_SIGNOFF,
                                  win - winstruct, "%s,%hu %s",
                                  win->server->name,
                                  win->server->port,
                                  p->nick))
              {
                char *buf;

                fmt_string = "%n%h%z$";
                buf = format_msg((p->flag)[j] & NOTIFY_ISONLINE ?
                                 fmt_notify_signon : fmt_notify_signoff,
                                 p->nick, win->server->name,
                                 win->server->port);
                if (buf)
                {
                  say(buf, win - winstruct, 0);
                  free(buf);
                }
              }
            }
            p = p->next;
          }                     //while p!=q

        }
    }                           // if (*i != -1)

}
static int present(char **nicks, const char *nick)
{
  while (*nicks)
    if (!strcasecmp(nick, *nicks++))
      return 1;
  return 0;
}

static void mark_group_nicks(Notify_t i, int j, char **nicks)
// inspect notify group i and mark people who are on- or offline on server j.
// j is an index into aNotifyList->server.
{
  NotifyNick *p = aNotifyList[i].start;

  while (p != aNotifyList[i].end)
  {
    if (present(nicks, p->nick))
    {                           //online

      if (((p->flag)[j] & NOTIFY_ISONLINE) == 0)
      {                         //wasnt online

        (p->flag)[j] |= NOTIFY_STATUS_CHANGED;
        (p->flag)[j] |= NOTIFY_ISONLINE;
      }
      else
        (p->flag)[j] &= ~NOTIFY_STATUS_CHANGED;
    }
    else
    {                           //offline

      if ((p->flag)[j] & NOTIFY_ISONLINE)
      {                         //was online

        (p->flag)[j] |= NOTIFY_STATUS_CHANGED;
        (p->flag)[j] &= ~NOTIFY_ISONLINE;
      }
      else
        (p->flag)[j] &= ~NOTIFY_STATUS_CHANGED;
    }
    p = p->next;
  }                             //while p!=q

}

static void check_online(char **nicks, Winstruct * win)
{
  Notify_t *i;

  for (i = win->server->notify; *i != NOTIFY_EOL; i++)
    if (*i != -1)
    {
      register int j;
      Notify *nl = &aNotifyList[*i];

      for (j = 0; j < MAX_NOTIFY_SERVERS; j++)
        if (server_comp((nl->server)[j], win->server))
          mark_group_nicks(*i, j, nicks);
    }                           // if (*i != -1)

  check_notify_hooks(win);      // now check all groups and say who's signed off/now

}

void remove_sent_notifies(Server * s)
{
  Notify_sent *p = NotifySentStart;

  while (p != NotifySentEnd)
    if (!strcasecmp(p->server, s->name) && p->port == s->port)
    {
      Notify_sent *q = p->next;

      //free(p->nick);
      free(p->server);
      if (q == NotifySentEnd)
        NotifySentEnd = p;
      else
        *p = *q;
      free(q);
      continue;
    }
    else
      p = p->next;
}
int requested_notify(const char *buf, Winstruct * win)
{
/*
   if auto-notifier sent ISON, check nicks against notify groups. After
   setting NOTIFY_ONLINE flag, anyone who previously had NOTIFY_ONLINE and
   whose flag hasnt been changed is considered as having signed off.

 */
  Notify_sent *p = NotifySentStart;

  while (p != NotifySentEnd)
  {
    if (                        //!strcasecmp(*nick,*p->nick) &&
         !strcasecmp(win->server->name, p->server) &&
         win->server->port == p->port)
    {
      char clear = 1;

      if (*buf == 0)
      {                         //nobody online

        char *argv[1];

        *argv = 0;
        check_online(argv, win);
      }
      else
      {
        char *argv[50];         //process at most 50 nicks in a reply

        int argc;
        char *tmp = strdup(buf);

        new_str2args(tmp, &argc, argv, 50);
        clear = strcmp(win->server->nick, argv[0]);
        check_online(argv, win);
        free(tmp);
      }
      if (clear)
        // if first nick is not ours, clear sent notifies. another kludge
        remove_sent_notifies(win->server);
      return 1;
    }
    p = p->next;
  }
  return 0;
}
#if 0
//periodically remove ISONs requests sent
#define FLUSH_PERIOD  30        //every 30 seconds
bad things happend when FLUSH_PERIOD is up immediately after
  sending ISON for notify.
void clean_sent_notifies(void)
{
  Notify_sent *p = NotifySentStart;
  time_t t = time(NULL);

  while (p != NotifySentEnd)
    if ((t - NotifySentStart->time) > FLUSH_PERIOD)
    {
      Notify_sent *q = p->next;

      //free(p->nick);
      free(p->server);
      if (q == NotifySentEnd)
        NotifySentEnd = p;
      else
        *p = *q;
      free(q);
      continue;
    }
    else
      p = p->next;
}
#endif
// / end of idle callback ///////////////////////

#endif
