/***************************************************
  This file is distributed as part of Sula PrimeriX
  (http://members.xoom.com/sprimerix).
****************************************************/


/*
   our_guile.c - Scheme support in Sula PrimeriX.

   No procedure should return SCM_UNDEFINED to the user.
   That crashes the program
   if verbose_return is set on. The fix is expensive:
   using (catch) when printing results, or doing a check first.

   Copyright (C) 1999 Tano Fotang

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free 
   Software Foundation; either version 2 of the License , or (at your option) 
   any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
   or FITNESS FOR A PARTICULAR PURPOSE. See the file LICENCE for details.

   You should have received a copy of the GNU General Public License along
   with this program; see the file COPYING.  If not, write to the Free
   Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#include "config.h"
#include "spx.h"
#include "server.h"
#include "nc.h"
#include "cmd.h"
#include "scheme.h"
#include "hooks.h"
#include "timer.h"
#include "nicks.h"
#include "notify.h"
#include "dcc.h"
#include <arpa/inet.h>          //inet_ntoa
#if USE_GTK
#include "skriptx.h"
#endif
#include <glob.h>

#define MY_NULL gh_list(SCM_UNDEFINED)
#define VAR_STEM_LEN 16         /* max length of a variable name */
#define MAX_FUNC_NAME_LEN  (VAR_STEM_LEN+12)
SCM print_obj;
ScmFunc *scm_func;
int scm_func_count;
KeyBinding *keybinding;
int bind_count;

static char var_stem[VAR_STEM_LEN];	//variable names

extern char *input_line;/* user's input. from common-chanwin_cb.c */
#if MUTABLE_IRC
extern char *Msg2;/* IRC output. from parser.c */
#endif
extern SCM gs_bind_proc(SCM Key, SCM state, SCM proc);
extern SCM gs_delete_binding_proc(SCM Key, SCM state);
extern SCM gs_query_bindings_proc(void);

extern SCM get_set_variable(const char *);
extern SCM cmd_set_var(SCM, SCM);
extern SCM create_set_variable(SCM varname, SCM type, SCM proc);
extern SCM delete_set_variable(SCM varname);

extern int ignored_p(const char *mask, const char *what);

#if USE_GTK
extern SCM gs_add_pup(SCM name, SCM proc_name, SCM opt_flag);
extern SCM gs_remove_pup(SCM name);
extern SCM gs_clear_pup(void);
extern SCM gs_set_status_bg_pixmap(SCM fname, SCM win);
extern SCM gs_set_window_bg_pixmap(SCM fname, SCM win, SCM args);
extern SCM gs_set_window_text_font(SCM fname, SCM win, SCM args);
extern SCM gs_set_input_font(SCM fname, SCM win);
extern SCM gs_set_status_font(SCM fname, SCM win);
extern SCM gs_set_window_bg_colour(SCM name, SCM win, SCM args);
extern SCM gs_set_window_fg_colour(SCM name, SCM win, SCM args);
extern SCM gs_set_input_bg_colour(SCM name, SCM win);
extern SCM gs_set_input_fg_colour(SCM name, SCM win);
extern SCM gs_set_status_bg_colour(SCM name, SCM win);
extern SCM gs_set_status_fg_colour(SCM name, SCM win);
extern SCM gs_set_console_clock_font(SCM);
extern SCM gs_set_console_clock_fg_colour(SCM);
extern SCM gs_set_console_clock_bg_colour(SCM);
#endif

#define SCM_FUNC_ALLOC 20
static void func_allocate(void)
{
  register int i;

  if (scm_func == NULL)
    scm_func = my_malloc(sizeof(ScmFunc) * SCM_FUNC_ALLOC);
  else
    scm_func = my_realloc(scm_func,
                          (scm_func_count +
                           SCM_FUNC_ALLOC) * sizeof(ScmFunc));
  for (i = scm_func_count; i < scm_func_count + SCM_FUNC_ALLOC; i++)
    scm_func[i].idx = -1;
  scm_func_count += SCM_FUNC_ALLOC;
}
extern int define_func(SCM proc)
{
  if (gh_equal_p(proc, SCM_BOOL_F))
    return -1;
#warning 31Jul99 [FIXME] Is proc really a procedure? check 1st!
  if (!scm_func)
    func_allocate();
  do
  {
    register int i;

    for (i = 0; i < scm_func_count; i++)
      if (scm_func[i].idx == -1)
      {
        char buf[MAX_FUNC_NAME_LEN + 1];

        scm_func[i].idx = i;
        if (snprintf(buf, MAX_FUNC_NAME_LEN, "%s-%d", var_stem, i) == -1)
          buf[MAX_FUNC_NAME_LEN] = 0;
        scm_func[i].func = gh_define(buf, proc);
        return i;
      }
    func_allocate();
  }
  while (1);
}
extern void remove_func(int i)
{
  if (i != -1)
  {
    char buf[MAX_FUNC_NAME_LEN + 10];

    scm_func[i].idx = -1;
    sprintf(buf, "(undefine %s-%d)", var_stem, i);
    gh_eval_str_with_catch(buf, &exception_handler);
  }
}
extern SCM call_scm_command(void *data)
{
  Scm_cmd_arg *arg = (Scm_cmd_arg *) data;

  return (gh_apply(gh_cdr(scm_func[arg->func].func), arg->args));
}

// SCM print_obj; leave alone. Never redefine!
static SCM s_sprint_command(SCM proc)
{
  if (!gh_procedure_p(proc))
  {
    scm_wrong_type_arg("s_sprint_command", 1, proc);
    return SCM_BOOL_F;
  }
  print_obj = proc;
  return SCM_BOOL_T;
}

SCM exception_handler(void *data, SCM tag, SCM rest)
{
  char *s = 0;
  int l;

#define BUFL 150
  char buf[BUFL + 1];
  SCM other;
  char *fmt, *p;
  char *pbuf;

  p = (char *) data;
  gh_defer_ints();
  memset((void *) &buf[0], 0, BUFL + 1);
  pbuf = strchr(p, '\n');
  if (pbuf)
  {
    if (pbuf - p > BUFL - 5)
      pbuf -= 5;
    strncpy(buf, p, pbuf - p);
    strcpy(&buf[pbuf - p], " ...");
  }
  else
    strncpy(buf, p, BUFL);
  say2(1, -1, 1, "@s@f@.ERROR: %s", buf);
  if (gflags & BEEP_ERR)
    spx_bell(-60);
  gh_allow_ints();

  l = 0;
  buf[0] = 0;
  /*
     other := (<subr>|#f <message> <args> <rest>)
   */
  other = gh_car(rest);
  if (!gh_boolean_p(other))
  {
    /* get name of subroutine that threw error */
    other = gh_call2(print_obj, other, SCM_BOOL_T);
    s = gh_scm2newstr(other, &l);
    if (l)
    {
      buf[0] = '[';
      strncpy(&buf[1], s, l);
      l++;
      buf[l++] = ']';
      buf[l++] = ' ';
      free(s);
    }
  }

  pbuf = &buf[l];
  fmt = gh_scm2newstr(gh_cadr(rest), &l);	/* the message */
  other = gh_caddr(rest);       /* arguments */
  p = fmt;
  while ((*pbuf = *p++))
  {
    if (*pbuf == '%')
      if (*p == 's' || *p == 'S')
      {
        char *tmp;
        SCM tt = gh_call2(print_obj, gh_car(other),
                          *p == 's' ? SCM_BOOL_T : SCM_BOOL_F);

        other = gh_cdr(other);
        tmp = gh_scm2newstr(tt, &l);
        gh_defer_ints();
        strncpy(pbuf, tmp, l);
        gh_allow_ints();
        pbuf += l;
        free(tmp);
        p++;
        continue;
      }
    pbuf++;
  }
  *pbuf = 0;
  free(fmt);
  other = gh_car(gh_cdddr(rest));	/* rest */
  if (!gh_boolean_p(other))
  {
    /* get name of subroutine that threw error */
    other = gh_call2(print_obj, other, SCM_BOOL_F);
    s = gh_scm2newstr(other, &l);
    if (l)
    {
      sprintf(pbuf, " %s", s);
      free(s);
    }
  }
  s = 0;
  s = gh_symbol2newstr(tag, &l);
  gh_defer_ints();
  say2(1, -1, 1, "@f@s@.%s: %s", s, buf);
  gh_allow_ints();
  free(s);
  return SCM_BOOL_F;
}

/********************************/
/*      Guile interface         */
/********************************/

static SCM gs_srandom(SCM use_usec)
{
  if (!gh_null_p(use_usec))
  {
    struct timeval t;

    if (gettimeofday(&t, NULL) == -1)
      srandom(time(NULL));
    else
      srandom(t.tv_usec);
  }
  else
    srandom(time(NULL));
  return gh_ulong2scm(random());
}

 /* 
    I/O watch
    Here we add socket/file descriptors to the main select(2) loop.

    The user tells us which fdes we should watch for data.
    We are also given a procedure to pass the data to:
    SCM cmd_add_io_watch(SCM fdes, SCM proc).
    To stop watching a connection, we call
    SCM cmd_stop_io_watch(SCM fd).
    The procedure <proc> should take 3 args:
    - fd, the file descriptor
    - data, the data read
    - len, the length of the data
    on error:
    data=0, len<0
    when connection is closed:
    data=0, len=0
    When an error occurs (len<=0), we stop watching the fd.
  */

typedef struct
{
  short fd;
  int proc;
  char flag;                    /* has fl_remove_io_cb been called? */
}
IO_watch;
static IO_watch *aIO_watch = NULL;
static short IO_watch_count = 0;

#define IO_WATCH_ALLOC 5
static void allocate_IO_watch(void)
{
  register int i;

  if (aIO_watch == NULL)
    aIO_watch = my_malloc(sizeof(IO_watch) * IO_WATCH_ALLOC);
  else
    aIO_watch = my_realloc(aIO_watch,
                           (IO_watch_count +
                            IO_WATCH_ALLOC) * sizeof(IO_watch));
  for (i = IO_watch_count; i < IO_watch_count + IO_WATCH_ALLOC; i++)
    aIO_watch[i].fd = -1;
  IO_watch_count += IO_WATCH_ALLOC;
}
static int add_IO_watch(int fd, SCM proc)
{
  register int i;

  for (i = 0; i < IO_watch_count; i++)
    if (aIO_watch[i].fd == fd)
      break;
  if (i != IO_watch_count)
  {                             /* clear slot */
    remove_func(aIO_watch[i].proc);
    aIO_watch[i].fd = -1;
  }
  if (!aIO_watch)
    allocate_IO_watch();
  do
  {
    for (i = 0; i < IO_watch_count; i++)
      if (aIO_watch[i].fd == -1)
      {
        aIO_watch[i].proc = define_func(proc);
        aIO_watch[i].fd = fd;
        aIO_watch[i].flag = 0;
        return i;
      }
    allocate_IO_watch();
  }
  while (1);
}

static void send_IO_watch_data(int i, const char *data, int len)
{
/*
   pass the data from a watched I/O source to the owner
 */
  Scm_cmd_arg args;
  char status[50];

  sprintf(status, "Watching port %d.", aIO_watch[i].fd);
  args.func = aIO_watch[i].proc;
  args.args = gh_list(gh_long2scm(aIO_watch[i].fd),
                      gh_str2scm((char *) data, len),
                      gh_long2scm(len), SCM_UNDEFINED);
  gh_catch(SCM_BOOL_T, &call_scm_command, &args,
           (scm_catch_handler_t) & exception_handler, (void *) &status);

}
static SCM rm_IO_watch(int fd);

#if USE_XFORMS
static void io_callback_cb(int fd, void *junk)
#else
static void io_callback_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
/* read data from a source that's being watched */
{
  register int i;

  for (i = 0; i < IO_watch_count; i++)
    if (aIO_watch[i].fd == fd)
    {
#define RD_BUF    1024
      char data[RD_BUF + 1];
      int n;

#ifndef SA_RESTART
    retry:
#endif
      n = read(fd, data, RD_BUF);
      if (n < 0)
      {
        if (errno != EAGAIN)
#ifndef SA_RESTART
          if (errno == EINTR)
            goto retry;
          else
#endif
          {
            spx_remove_io_callback(aIO_watch[i].fd, SPX_READ, io_callback_cb);
            aIO_watch[i].flag = 1;
            send_IO_watch_data(i, 0, n);
            rm_IO_watch(fd);
          }
      }
      else if (n == 0)
      {
        spx_remove_io_callback(aIO_watch[i].fd, SPX_READ, io_callback_cb);
        aIO_watch[i].flag = 1;
        send_IO_watch_data(i, 0, 0);
        rm_IO_watch(fd);
      }
      else
        send_IO_watch_data(i, data, n);
      return;
    }
  assert(i != IO_watch_count);
}
static SCM rm_IO_watch(int fd)
{

  if (fd >= 0)
  {
    register int i;

    for (i = 0; i < IO_watch_count; i++)
      if (aIO_watch[i].fd == fd)
      {
        remove_func(aIO_watch[i].proc);
        if (aIO_watch[i].flag == 0)
          spx_remove_io_callback(aIO_watch[i].fd, SPX_READ, io_callback_cb);
        aIO_watch[i].fd = -1;
        return SCM_BOOL_T;
      }
  }
  return SCM_BOOL_F;
}

static SCM cmd_stop_io_watch(SCM fd)
/* close a file descriptor/stop watching. */
{
  return rm_IO_watch(gh_scm2long(fd));
}
static SCM cmd_add_io_watch(SCM fdes, SCM proc)
{
  int fd;
  int f;

  fd = gh_scm2long(fdes);
  if (fd < 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  f = add_IO_watch(fd, proc);
  gh_allow_ints();
  if (f < 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  spx_add_io_callback(fd, SPX_READ, io_callback_cb, 0);
  gh_allow_ints();
  return SCM_BOOL_T;
}

/*
   End of I/O watch
 */

static SCM cmd_parse_colour(SCM str)
{
#if USE_GTK
  int n;
  char *name = gh_scm2newstr(str, &n);

  if (n == 0)
    return SCM_BOOL_F;
  n = spx_add_colour(name);
  free(name);
  return n > 0 ? gh_int2scm(n) : SCM_BOOL_F;
#else
  return SCM_BOOL_F;
#endif
}
static SCM cmd_max_colour(void)
{
#if USE_GTK
  return gh_int2scm(spx_max_colour);
#else
  return SCM_BOOL_F;
#endif
}
static SCM cmd_bell(SCM percent)
{
  int n = 0;

  if (!gh_null_p(percent))
    n = (int) gh_scm2long(gh_car(percent));
  gh_defer_ints();
  spx_bell(n);
  spx_xflush();
  gh_allow_ints();
  return SCM_BOOL_T;
}
static SCM cmd_delay(SCM microseconds)
{
  long n = gh_scm2long(microseconds);

  if (n < 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  n = delay(n);
  gh_allow_ints();
  return gh_int2scm(n);
}

static SCM s_ignored_p(SCM who, SCM what)
{
  int n;
  char *pat;
  char *mask = gh_scm2newstr(who, &n);

  if (n == 0)
    return SCM_BOOL_F;
  pat = gh_scm2newstr(what, &n);
  gh_defer_ints();
  n = ignored_p(mask, pat);
  gh_allow_ints();
  return gh_bool2scm(n);

}
static int get_valid_window(void)
{
  register int i;
  Winstruct *win = NULL;

  for (i = 0; i < win_count; i++)
    if (winstruct[i].chanwin)
    {
      if (!win)
        win = &winstruct[i];
      if (winstruct[i].server && winstruct[i].server->fd > -1)
        return i;
    }
  return win ? win - winstruct : -1;
}
static SCM fast_window_server(SCM win)
{
  int w = gh_scm2long(win);

  if (!win_invalid(w))
  {
    Server *s = winstruct[w].server;

    if (s != NULL)
      return gh_list(gh_long2scm(s->fd),	//fd 
                     SCM_BOOL_F, SCM_BOOL_F, gh_str02scm(s->nick),	//nick 
                     SCM_BOOL_F,
                     SCM_BOOL_F,
                     SCM_BOOL_F,
                     SCM_BOOL_F,
                     SCM_BOOL_F,
                     (s->flag & SET_AWAY) ? SCM_BOOL_T : SCM_BOOL_F,
                     SCM_BOOL_F,
                     SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_UNDEFINED);
  }
  return MY_NULL;
}
static SCM s_get_server_by_win_nr(args)
/*
   return server connected to a window 
 */
     SCM args;
{
  register int w;

  w = -1;

  if (gh_null_p(args))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(args));
  if (win_invalid(w))
    return MY_NULL;
  else
  {
    Server *s = winstruct[w].server;

    if (s == NULL)
      return MY_NULL;
    return gh_list(gh_long2scm(s->fd),	//fd 
                   gh_str02scm(s->alias),	//server "alias" 
                   gh_long2scm(s->port),	//port 
                   gh_str02scm(s->nick),	//nick 
                   gh_str02scm(s->ircname),	//realname 
                   gh_str02scm(s->mode), gh_long2scm(w), gh_str02scm(s->name),	//real server name 
                   gh_str02scm(s->lastInvite),
                   gh_bool2scm((s->flag & SET_AWAY) != 0), gh_long2scm(s->read),	//bytes read from serevr 
                   gh_long2scm(s->wrote),	//count written 
                   gh_long2scm(s->connect_time),
                   gh_long2scm(s->iNick), SCM_UNDEFINED);
  }

}

static SCM s_get_channel_list_by_win(args)
/*
   a list of all channels of this window 
 */
     SCM args;
{
  int w = -1;
  SCM list = gh_list(SCM_UNDEFINED);

  if (gh_null_p(args))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(args));
  if (!win_invalid(w))
  {
    Channel *current;
    Winstruct *win = &winstruct[w];

    current = win->chanStart->left;
    while (current != win->chanStart)
    {
      list = gh_cons(gh_list(gh_long2scm(w),
                             gh_str02scm(current->name), gh_str02scm(current->modes),	// o,v,n,t etc 
                             gh_long2scm(current->limit),
                             gh_str02scm(current->key),
                             gh_str02scm(current->topic),
                             SCM_UNDEFINED), list);
      current = current->left;
    }
  }
  return list;
}

static SCM gs_dcc_list(void)
{
/*
DCC LIST
Return a list of all DCC items. Each item is a vector.

item  ::=  #(
            0. '(is_chat? active? we_initiated?)
            1.  item id
            2.  nick
            3.  start time 
            4.  bytes written
            5.  bytes read
            6.  remote user@host
            7.  if active? remote IP
              else remote internet addr
            8.  remote port
              ;the following only available if not is_chat:
            9. file name
            10. file size
            11. file checksum
            12. window number (->server request came from))


*/
 SCM list=gh_list(SCM_UNDEFINED);
 DCCChatWin *q = dccchatwinStart;
  DCC *p;
  
  while(q!=dccchatwinEnd){
    list=gh_cons(
    gh_list_to_vector(gh_list(
                        gh_list_to_vector(gh_list(SCM_BOOL_T,
                                                SCM_BOOL_T,
                                                SCM_BOOL_F,
                                                SCM_BOOL_F,
                                                SCM_UNDEFINED)), 
                        gh_long2scm(q->id),
                        gh_str02scm(q->nick),
                        gh_long2scm(*q->starttime),
                        gh_long2scm(*q->wrote),
                        gh_long2scm(*q->read),
                        gh_str02scm(q->email),
                        gh_str02scm(q->rem_host),
                        gh_long2scm(*q->port),
                        gh_long2scm(q->w),
                        SCM_UNDEFINED)), list);
     q=q->next;
  }
  p=dccStart;
  while(p!=dccEnd){
    if(p->type&DCC_CHAT)
      list=gh_cons(
      gh_list_to_vector(gh_list(
                       gh_list_to_vector(gh_list(SCM_BOOL_T,
                                 p->type&DCC_ACTIVE?SCM_BOOL_T:SCM_BOOL_F,
                                 p->type&DCC_SENT?SCM_BOOL_T:SCM_BOOL_F,
                                 p->type&DCC_FWD?SCM_BOOL_T:SCM_BOOL_F,
                                 SCM_UNDEFINED)), 
                        gh_long2scm(p->id),
                        gh_str02scm(p->nick),
                        gh_long2scm(p->starttime),
                        gh_long2scm(0),
                        gh_long2scm(0),
                        gh_str02scm(p->peer),
                        gh_long2scm(p->inetaddr),
                        gh_long2scm(p->rport),
                        gh_long2scm(p->w),
                        SCM_UNDEFINED)), list);
    else
      list=gh_cons(
        gh_list_to_vector(gh_list(
                        gh_list_to_vector(gh_list(SCM_BOOL_F,
                                 p->type&DCC_ACTIVE?SCM_BOOL_T:SCM_BOOL_F,
                                 p->type&DCC_SENT?SCM_BOOL_T:SCM_BOOL_F,
                                 p->type&DCC_FWD?SCM_BOOL_T:SCM_BOOL_F,
                                 SCM_UNDEFINED)), 
                        gh_long2scm(p->id),
                        gh_str02scm(p->nick),
                        gh_ulong2scm(p->starttime),
                        gh_long2scm((p->type&DCC_SENT)? p->dcc.file->read: 0),
                        gh_long2scm((p->type&DCC_SENT)? 0:p->dcc.file->read),
                        gh_str02scm(p->peer),
                        gh_ulong2scm(p->inetaddr),
                        gh_long2scm(p->rport),
                        gh_long2scm(p->w),
                        gh_str02scm(p->dcc.file->name),
                        gh_ulong2scm(p->dcc.file->size),
                        gh_ulong2scm(p->dcc.file->checksum),
                        SCM_UNDEFINED)), list);
    p=p->next;
  }
  return list;
}

static SCM s_get_dcc_list_by_win(args)
/*
   a list of all DCC chats on this window
   Each Chat item is a vector. 
 */
     SCM args;
{
  int w = -1;
  SCM list;

  if (gh_null_p(args))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(args));
  list = gh_list(SCM_UNDEFINED);
  if (!win_invalid(w))
  {
    DCCChat *current;
    Winstruct *win = &winstruct[w];

    current = win->dccStart->left;
    while (current != win->dccStart)
    {
      list = gh_cons(gh_list_to_vector(gh_list(gh_long2scm(w),
                             gh_long2scm(current->id),
                             gh_long2scm(current->fd),
                             gh_str02scm(current->nick),
                             gh_str02scm(current->email),
                             gh_str02scm(current->rem_host),
                             gh_long2scm(current->read),
                             gh_long2scm(current->wrote),
                             SCM_UNDEFINED)), list);
      current = current->left;
    }
  }
  return list;
}
static SCM s_get_current_dcc_nick(win)
     SCM win;
{
  register int w;

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    return SCM_BOOL_F;
  if (winstruct[w].dccCur == winstruct[w].dccStart)
    return SCM_BOOL_F;
  else
    return gh_str02scm(winstruct[w].dccCur->nick);

}
static SCM s_get_query_nick(win)
     SCM win;
{
  register int w;

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    return SCM_BOOL_F;
  return gh_str02scm(winstruct[w].query_nick);
}
static SCM cmd_get_radio(SCM window)
{
  int w;

  if (gh_null_p(window))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(window));
  if (win_invalid(w))
    return SCM_BOOL_F;
  else
  {
    Winstruct *win = &winstruct[w];
    char *str = NULL;

    gh_defer_ints();
    if (BUTTON_IS_ON(win->chanwin->r_quote))
      str = "parse";
    else if (BUTTON_IS_ON(win->chanwin->r_query))
      str = "query";
    else if (BUTTON_IS_ON(win->chanwin->r_channel))
      str = "channel";
    else if (BUTTON_IS_ON(win->chanwin->r_dcc_chat))
      str = "chat";
    else if (BUTTON_IS_ON(win->chanwin->r_echo))
      str = "echo";
    gh_allow_ints();
    return gh_symbol2scm(str);
  }
}
static SCM cmd_set_radio(SCM which, SCM W)
{
  int w;

  if (gh_null_p(W))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(W));
  if (win_invalid(w))
    return SCM_BOOL_F;
  else
  {
    Winstruct *win = &winstruct[w];

    if (w == 0)
      return SCM_BOOL_F;
    else
    {
      SPX_OBJ(ob) = NULL;
      char *button = gh_scm2newstr(which, &w);	//gh_symbol2newstr(which,&w);
      
      if (!strcasecmp(button, "parse"))
        ob = win->chanwin->r_quote;
      else if (!strcasecmp(button, "query"))
        ob = win->chanwin->r_query;
      else if (!strcasecmp(button, "channel"))
        ob = win->chanwin->r_channel;
      else if (!strcasecmp(button, "chat"))
        ob = win->chanwin->r_dcc_chat;
      else if (!strcasecmp(button, "echo"))
        ob = win->chanwin->r_echo;
      free(button);
      gh_defer_ints();
      if (ob != NULL)
        TRIGGER_OBJECT(ob);
      gh_allow_ints();
      return gh_bool2scm(ob != NULL);
    }
  }
}
static SCM cmd_get_input_cursorpos(SCM win)
{
  int w;

#if USE_XFORMS
  int x, y;
#endif
  SPX_OBJ(ob);

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    ob = main_window->main_input;
  else
    ob = winstruct[w].chanwin->sinput;
#if USE_XFORMS
  w = fl_get_input_cursorpos(ob, &x, &y);
#else
  w = gtk_editable_get_position(GTK_EDITABLE(ob));
#endif
  return (gh_long2scm(w));
}
static SCM cmd_set_input_cursorpos(SCM pos, SCM win)
{
  int w;

  SPX_OBJ(ob);
  int x = gh_scm2long(pos);

  if (x < 0)
    return SCM_BOOL_F;
  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    ob = main_window->main_input;
  else
    ob = winstruct[w].chanwin->sinput;
#if USE_XFORMS
  fl_set_input_cursorpos(ob, x, 1);
#else
  gtk_entry_set_position(GTK_ENTRY(ob), x);
#endif
  return SCM_BOOL_T;
}
static SCM cmd_get_input(SCM win)
{
  int w;
  const char *res;
  char *str = NULL;
  SCM ret;

  SPX_OBJ(ob);

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    ob = main_window->main_input;
  else
    ob = winstruct[w].chanwin->sinput;
  gh_defer_ints();
  res = spx_get_input(ob);
  if (res)
    str = strdup(res);
  ret = gh_str02scm(str);
  free(str);
  gh_allow_ints();
  return ret;
}
static SCM cmd_set_input(SCM str, SCM win)
{
  int w;

  SPX_OBJ(ob);
  char *text = gh_scm2newstr(str, &w);

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    ob = main_window->main_input;
  else
    ob = winstruct[w].chanwin->sinput;
  gh_defer_ints();
  spx_set_input(ob, text);
  gh_allow_ints();
  if (*text)
    free(text);
  return SCM_BOOL_T;
}

static SCM fast_get_channel_info(SCM channel, SCM win)
{
  SCM res = MY_NULL;
  int w = (int) gh_scm2long(win);

  if (!win_invalid(w))
  {
    Channel *current;
    char *chan;
    int l;

    chan = gh_scm2newstr(channel, &l);
    if (l == 0)
      return MY_NULL;
    current = winstruct[w].chanStart->left;
    while (current != winstruct[w].chanStart)
    {
      if (!strcasecmp(current->name, chan))
      {
        res = gh_list(SCM_BOOL_F, SCM_BOOL_F, gh_str02scm(current->modes),	// o,v,n,t etc 
                      SCM_BOOL_F,
                      gh_str02scm(current->key), SCM_BOOL_F, SCM_UNDEFINED);
        break;
      }
      current = current->left;
    }
    free(chan);
  }
  return res;
}

static SCM get_channel_info(int w, const char *chan)
{
  if (!win_invalid(w))
  {
    Channel *current;
    Winstruct *win = &winstruct[w];

    current = win->chanStart->left;
    while (current != win->chanStart)
    {
      if (!strcasecmp(current->name, chan))
      {
        SCM list = gh_list(gh_long2scm(w),
                           gh_str02scm(current->name),
                           gh_str02scm(current->modes),	// o,v,n,t etc 
                           gh_long2scm(current->limit),
                           gh_str02scm(current->key),
                           gh_str02scm(current->topic),
                           SCM_UNDEFINED);

        return list;
      }
      current = current->left;
    }
  }
  return gh_list(SCM_UNDEFINED);
}
static SCM s_get_channel_info(args)
/*
   info about a channel (window  channel modes  topic  etc) 
 */
     SCM args;
{
  int w;
  char *ch = 0;
  SCM res;

  if (!gh_null_p(args))
  {
    w = 0;
    ch = gh_scm2newstr(gh_car(args), &w);
    if (w > 0)
      *(ch + w) = 0;
    w = -1;
    args = gh_cdr(args);
    if (!gh_null_p(args))
      w = (int) gh_scm2long(gh_car(args));
  }
  else
    w = -1;
  if (win_invalid(w))
  {
    w = get_valid_window();
    if (win_invalid(w))
      return MY_NULL;
  }
  if (winstruct[w].chanCur == winstruct[w].chanStart)
    return MY_NULL;
  if (ch == 0 || *ch == 0)
    ch = strdup(winstruct[w].chanCur->name);
  gh_defer_ints();
  res = get_channel_info(w, ch);
  free(ch);
  gh_allow_ints();
  return res;

}

static SCM get_window_message_cmd(SCM win)
{
  int w;
  char *str;

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    return SCM_BOOL_F;
#if USE_XFORMS
  str = (char *) winstruct[w].chanwin->WinStatusLine->label;
#else
  gtk_label_get(GTK_LABEL(winstruct[w].chanwin->WinStatusLine), &str);
#endif
  return gh_str02scm(str);
}
static SCM set_window_message_cmd(msg, opt)
     SCM msg;
     SCM opt;
{
  char *str;
  int l;

  str = gh_scm2newstr(msg, &l);
  if (!gh_null_p(opt))
    l = (int) gh_scm2long(gh_car(opt));
  else
    l = get_valid_window();
  if (win_invalid(l))
  {
    free(str);
    return SCM_BOOL_F;
  }
  gh_defer_ints();
  fit_object_label(winstruct[l].chanwin->WinStatusLine, str);
  spx_xflush();
  gh_allow_ints();
  free(str);
  return (SCM_BOOL_T);
}
static SCM gs_set_lag(SCM win, SCM str)
{
  int w = gh_scm2long(win);
  if(!win_invalid(w)){
     int n;
     char *label=gh_scm2newstr(str, &n);

     if(n>8)
        label[8]=0;
     fit_object_label(winstruct[w].chanwin->lag, label);
     free(label);
     return SCM_BOOL_T;
 }
  return SCM_BOOL_F;
}
static SCM get_window_title_cmd(SCM win)
{
  int w = gh_scm2long(win);

#if USE_XFORMS
  return gh_str02scm(win_invalid(w) ?
                     main_window->main_form->label :
                     winstruct[w].chanwin->chanwin->label);
#else
  return gh_str02scm(win_invalid(w) ?
                     GTK_WINDOW(main_window->main_window)->title :
                     GTK_WINDOW(winstruct[w].chanwin->chanwin)->title);
#endif
}
static SCM set_window_title_cmd(SCM title, SCM win)
{
  int w = (int) gh_scm2long(win);
  int l;
  char *str = gh_scm2newstr(title, &l);

  gh_defer_ints();
#if USE_XFORMS
  SET_WINDOW_TITLE(win_invalid(w) ?
                   main_window->main_form : winstruct[w].chanwin->chanwin,
                   str);
#else
  SET_WINDOW_TITLE(win_invalid(w) ?
                   main_window->main_window : winstruct[w].chanwin->chanwin,
                   str);
#endif
  gh_allow_ints();
  free(str);
  return SCM_BOOL_T;
}
static SCM hide_main_window_cmd(SCM how)
/*
   hide/show main window
   how==#f => show, else hide
 */
{
  if (gh_equal_p(how, SCM_BOOL_F))/* show */
  {
    /*if(! gflags&CTRL_WIN_SHOWN)*/{
#if USE_XFORMS
    fl_show_form(main_window->main_form,
                 FL_PLACE_FREE, FL_FULLBORDER,
                 sula_NAME " " sula_VERSION ": Console");
#else
    gtk_widget_show(main_window->main_window);
#endif
    gflags |= CTRL_WIN_SHOWN;
    return (SCM_BOOL_T);
    }
  }
  else if(get_valid_window() != -1/* && (gflags & CTRL_WIN_SHOWN)*/)
  {
#if USE_XFORMS
    fl_hide_form(main_window->main_form);
#else
    gtk_widget_hide(main_window->main_window);
#endif
    gflags &= ~CTRL_WIN_SHOWN;
    return (SCM_BOOL_T);
  }
  return (SCM_BOOL_F);
}
static SCM gs_console_visible(void)
{
  return gflags & CTRL_WIN_SHOWN ? SCM_BOOL_T : SCM_BOOL_F;
}
static SCM clear_main_window_cmd(void)
{
  if (main_window)
    TEXT_CLEAR(main_window->browser);
  return SCM_BOOL_T;
}
static SCM cmd_set_statline(msg, opt)
     SCM msg;
     SCM opt;
{
  char *str;
  int l;

  str = gh_scm2newstr(msg, &l);
  if (!gh_null_p(opt))
    l = (int) gh_scm2long(gh_car(opt));
  else
    l = get_valid_window();
  if (win_invalid(l))
    {free(str);return SCM_BOOL_F;}
  gh_defer_ints();
  fit_object_label(winstruct[l].chanwin->status, str);
  spx_xflush();
  free(str);
  gh_allow_ints();
  return (SCM_BOOL_T);
}
static SCM cmd_get_statline(SCM win)
{
  int w;
  char *str;

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    return SCM_BOOL_F;
#if USE_XFORMS
  str = (char *) winstruct[w].chanwin->status->label;
#else
  gtk_label_get(GTK_LABEL(winstruct[w].chanwin->status), &str);
#endif
  return gh_str02scm(str);
}

static SCM get_server_channels(const char *servername, u_short port)
{
  SCM list;
  gChannel *p;

  list = gh_list(SCM_UNDEFINED);
  p = chanStart;
  while (p != chanEnd)
  {
    if (servername == NULL || !strcasecmp(p->server, servername))
    {
      if (port == p->port || port == 0)
        list = gh_cons(gh_list(gh_str02scm(p->name),
                               gh_str02scm(p->server),
                               gh_long2scm(p->port), SCM_UNDEFINED), list);
    }
    p = p->next;
  }
  return list;

}

static SCM s_get_channel_list_by_server(args)
/*
   (gs-server->channels [server [port]]) servername=NULL --> all servers 
   port ==0 any ports port<0 -->default port (usulally 6667)  return lists of 

   (channel server  port ) 
 */
     SCM args;
{
  char *servername = NULL;
  int port = 0;
  SCM list;

  if (!gh_null_p(args))
  {
    int l = 0;

    if (gh_string_p(gh_car(args)))
    {
      servername = gh_scm2newstr(gh_car(args), &l);
      if (l < 1)
        servername = NULL;
      else
        *(servername + l) = 0;
      args = gh_cdr(args);
    }
    if (!gh_null_p(args) && gh_exact_p(args))
      port = (int) gh_scm2long(gh_car(args));
  }
  if (port < 0)
    port = DEFAULT_PORT;
  gh_defer_ints();
  list = get_server_channels(servername, (u_short) port);
  gh_allow_ints();
  if (servername)
    free(servername);
  return list;
}

static SCM fast_get_nicklist_by_channel(SCM channel, SCM win)
{
  SCM res;
  int w;

  res = gh_list(SCM_UNDEFINED);
  w = (int) gh_scm2long(win);

  if (!win_invalid(w))
  {
    Channel *current;
    char *chan;
    int l;

    chan = gh_scm2newstr(channel, &l);
    if (l < 1)
      return MY_NULL;
    current = winstruct[w].chanStart->left;

    current = winstruct[w].chanStart->left;
    while (current != winstruct[w].chanStart)
      if (!strcasecmp(chan, current->name))
      {
        User *u;

        u = current->userStart;
        while (u != current->userEnd)
        {
          res = gh_cons(gh_str02scm(u->nick), res);
          u = u->next;
        }
        break;
      }
      else
        current = current->left;
    free(chan);
  }
  return res;
}

static SCM chan_users(int w, Channel * c)
{

  SCM list = gh_list(SCM_UNDEFINED);
  User *u;

  u = c->userStart;
  while (u != c->userEnd)
  {
    list = gh_cons(gh_cons(gh_str02scm(u->nick),
                           gh_list(gh_str02scm(u->mode),
                                   gh_str02scm(u->address),
                                   gh_str02scm(u->ircname),
                                   gh_str02scm(c->name),
                                   gh_long2scm(w), SCM_UNDEFINED)), list);
    u = u->next;
  }
  return list;
}

static SCM s_get_nicklist_by_channel(args)
     SCM args;
{
  char *ch = 0;
  SCM list;

  list = gh_list(SCM_UNDEFINED);
  if (!gh_null_p(args))
  {
    int l = 0;
    SCM tmp;

    ch = gh_scm2newstr(gh_car(args), &l);
    if (l < 1)
      return list;
    tmp = gh_cdr(args);
    l = -1;
    if (!gh_null_p(tmp))
      l = (int) gh_scm2long(gh_car(tmp));
    else
      l = get_valid_window();
    if (l > -1 && l < win_count)
    {
      Channel *current;
      Winstruct *win = &winstruct[l];

      current = win->chanStart->left;
      while (current != win->chanStart)
        if (!strcasecmp(ch, current->name))
        {
          list = chan_users(l, current);
          break;
        }
        else
          current = current->left;
    }
    free(ch);
  }
  return list;
}

static SCM cmd_last_channel(SCM last_joined, SCM win)
{
  register int w;

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    return MY_NULL;
  if (gh_equal_p(last_joined, SCM_BOOL_T))
    return gh_list(gh_str02scm(winstruct[w].lastJoin->name),
                   gh_str02scm(winstruct[w].lastJoin->key), SCM_UNDEFINED);
  else
    return gh_list(gh_str02scm(winstruct[w].lastPart->name),
                   gh_str02scm(winstruct[w].lastPart->key), SCM_UNDEFINED);
}
static SCM s_get_current_by_channel(win)
     SCM win;
{
  register int w;

  if (gh_null_p(win))
    w = get_valid_window();
  else
    w = (int) gh_scm2long(gh_car(win));
  if (win_invalid(w))
    return SCM_BOOL_F;
  if (winstruct[w].chanCur == winstruct[w].chanStart)
    return SCM_BOOL_F;
  else
    return gh_str02scm(winstruct[w].chanCur->name);

}
// 
// nick groups
//
static SCM s_list_nick_groups_cmd(void)
{
// ( i . label )
  register int n;
  SCM result = gh_list(SCM_UNDEFINED);

  for (n = 0; n < iNickListCount; n++)
    if (aNickList[n].label)
      result = gh_cons(gh_list(gh_long2scm(n),
                               gh_str02scm(aNickList[n].label),
                               SCM_UNDEFINED), result);
  return result;
}
static SCM s_create_nickgroup_cmd(SCM name)
{
  int n;
  char *label = gh_scm2newstr(name, &n);

  if (n == 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  n = new_nicklist(label);
  gh_allow_ints();
  //if(n==-1)  scm_syserror("s_create_nickgroup_cmd",strerror(errno),
  free(label);
  return gh_int2scm(n);
}
static SCM s_addto_nickgroup_cmd(SCM group, SCM val)
{
  int n;
  char *nick;

  nick = gh_scm2newstr(val, &n);
  if (n == 0)
    return SCM_BOOL_F;
  n = gh_scm2int(group);
  if (n < 0)
    return SCM_BOOL_F;          // scm_wrong_type_arg("s_addto_nickgroup",1,group);

  gh_defer_ints();
  aNickList[n].current = add_nick_to_list(n, nick);
  gh_allow_ints();
  free(nick);
  return SCM_BOOL_T;
}
static SCM s_list_group_nicks_cmd(SCM group)
{
  SCM result = gh_list(SCM_UNDEFINED);
  int n = gh_scm2int(group);

  if (n < 0)
    return result;              //scm_wrong_type_arg("s_list_group_nicks",1,group);

  if (n < iNickListCount && aNickList[n].label)
  {
    Nicks *p = aNickList[n].start->left;

    while (p != aNickList[n].start)
    {
      result = gh_cons(gh_str02scm(p->nick), result);
      p = p->left;
    }
  }
  return result;
}
static Server *SCM2server(SCM server)
{
  Server *s = NULL;

  if (!gh_null_p(server) && gh_list_p(server))
  {
    int i;
    u_short port;
    char *name = gh_scm2newstr(gh_cadr(server), &i);

    if (i == 0)
      return NULL;
    port = gh_scm2ulong(gh_caddr(server));
    s = search_server_tree_byname(-1, name, port);
    free(name);
  }
  return s;
}
static SCM s_set_server_nick_group(SCM server, SCM val)
{
  Server *s;

  s = SCM2server(server);
  if (s == NULL)
    return SCM_BOOL_F;
  else
  {
    char *nick;
    int i;

    i = gh_scm2int(val);
    if (i < 0 || i > iNickListCount || aNickList[i].label == NULL)
      return SCM_BOOL_F;
    s->iNick = i;
    nick = gh_scm2newstr(gh_list_ref(server, gh_int2scm(3)), &i);
    if (i > 0)
    {
      gh_defer_ints();
      add_nick_to_list(s->iNick, nick);
      gh_allow_ints();
      free(nick);
    }
    return SCM_BOOL_T;
  }
}
// ----------EO nick ggroups
static SCM s_win_list(void)
{
  register int i;
  SCM result = gh_list(SCM_UNDEFINED);

  for (i = 0; i < win_count; i++)
    if (winstruct[i].chanwin)
      result = gh_cons(gh_long2scm(i), result);
  return result;
}
static SCM s_win_create(void)
{
  if (main_window)
  {
    gh_defer_ints();
    TRIGGER_OBJECT(main_window->newwin);
    gh_allow_ints();
  }
  return SCM_BOOL_T;
}

static SCM cmd_win_destroy(SCM win, SCM force)
{
  int w = -1;

  w = (int) gh_scm2long(win);
  if (win_invalid(w))
    return SCM_BOOL_F;
  else
  {
    gh_defer_ints();
    _destroy_window(&winstruct[w], !gh_equal_p(force, SCM_BOOL_F));
    gh_allow_ints();
  }
  return SCM_BOOL_T;

}

// ////////////////////////////////
// Named connections            //
//////////////////////////////////

static SCM s_list_nc_cmd(void)
// return a list of all NC indices
{
  register int i;
  SCM result = gh_list(SCM_UNDEFINED);

  for (i = 0; i < sp_count; i++)
    if (s_pipe[i].fd != -1)
      result = gh_cons(gh_long2scm(i), result);
  return result;
}
static SCM s_nc_cmd(SCM n)
// take an index into NC list and return inforoup);strerror(errno),n lists of
{
  int nc = (int) gh_scm2ulong(n);

  if (nc < 0 || nc > sp_count || s_pipe[nc].fd == -1)
    return MY_NULL;
  else
  {

    S_Pipe *s = &s_pipe[nc];
    SCM type, list;

    switch (s->type)
    {
       case CON_SCRIPT:
         type = gh_symbol2scm("exec");
         list = gh_list(gh_int2scm(s->to.proc->pid),
                        gh_str02scm(s->to.proc->sys_cmd), SCM_UNDEFINED);
         break;
       case CON_SERVER:
         type = gh_symbol2scm("tcp/ip");
         list = gh_list(gh_str02scm(s->to.server.name),
                        gh_int2scm(s->to.server.port),
                        gh_str02scm(inet_ntoa(s->to.server.sin_addr)),
                        gh_str02scm(s->to.server.args), SCM_UNDEFINED);
         break;
       default:
         type = gh_symbol2scm("unknown");
         list = MY_NULL;
         break;
    }
    return gh_list(gh_int2scm(nc), gh_int2scm(s->fd),	// fd 
                   gh_str02scm(s->name),	// name of connection 
                   type, list, SCM_UNDEFINED);
  }
}
static SCM s_connect_cmd(SCM name, SCM server, SCM port, SCM args)
{
  int len;
  char *bez;
  char *to, *misc = NULL;
  u_short pport;
  S_Pipe *s = NULL;

  to = gh_scm2newstr(server, &len);
  if (len == 0)
    return SCM_BOOL_F;
  bez = gh_scm2newstr(name, &len);
  if (len == 0)
    {free(to);return SCM_BOOL_F;}
  pport = gh_scm2ulong(port);
  if (gh_null_p(args))
    misc = NULL;
  else
    misc = gh_scm2newstr(gh_car(args), &len);
  gh_defer_ints();
  s = make_connection(bez, to, pport, misc);
  gh_allow_ints();
  free(to);
  free(bez);
  if (misc)
    free(misc);
  return s ? gh_int2scm(s - s_pipe) : SCM_BOOL_F;
}
static SCM s_run_program_cmd(name, args)
     SCM args;
     SCM name;
{
  int len;
  char *prog, *bez;

  prog = gh_scm2newstr(args, &len);
  if (len < 1)
    return SCM_BOOL_F;
  bez = gh_scm2newstr(name, &len);
  if (len < 1)
    {free(prog);return SCM_BOOL_F;}
  gh_defer_ints();
  launch_prog(bez, prog);
  gh_allow_ints();
  free(prog);
  free(bez);
  return SCM_BOOL_T;
}
// //////////////////////////////////////////
static SCM s_exit_cmd(SCM exit_code)
{
  int c = 0;

  if (!gh_null_p(exit_code))
    c = (int) gh_scm2long(gh_car(exit_code));
  gh_defer_ints();
#if USE_XFORMS
  fl_finish();
#endif
  exit(c);
  gh_allow_ints();
  return SCM_BOOL_T;
}
static SCM s_echo_cmd(str, args)
/*
   gs-echo msg [browser [window]]
   (gs-echo "blabla" 2) 
 */
     SCM str;
     SCM args;
{
  int w, b = 0;
  char *s;

  s = gh_scm2newstr(str, &w);
  w = -1;
  if (!gh_null_p(args))
  {
    w = (int) gh_scm2long(gh_car(args));
    args = gh_cdr(args);
    if (!gh_null_p(args))
      b = (int) gh_scm2long(gh_car(args));
  }
  gh_defer_ints();
  say(s, w, b);
  free(s);
  gh_allow_ints();
  return SCM_BOOL_T;
}

static SCM s_exec_cmd(cmd, opt)
     SCM cmd;
     SCM opt;
{
  char *scmd;
  int l;

  scmd = gh_scm2newstr(cmd, &l);
  if (l < 1)
    return SCM_BOOL_F;
  if (!gh_null_p(opt))
    l = (int) gh_scm2long(gh_car(opt));
  else
    l = get_valid_window();
  gh_defer_ints();
  l = new_process_cmd(scmd, l);
  gh_allow_ints();
  free(scmd);
  return l==0? SCM_BOOL_T:SCM_BOOL_F;
}

// H o o k s

static SCM hook_cmds(SCM hook_type, SCM preemptive, SCM icase,	/* ignore case */
                     SCM regex, /* pattern is a regular expr */
                     SCM arg1,  /* hook type */
                     SCM rest /* pattern, procedure */ )
{
  int n;
  char *pat = NULL;
  MsgNr type = INVALID;
  int numeric = 0;
  int ser_no = 0;
  SCM proc = SCM_BOOL_F;
  On w;
  unsigned char pos = 1;

  if (gh_string_p(hook_type) || gh_symbol_p(hook_type))
  {
    char *cmd;

    pos = 2;
    cmd = gh_scm2newstr(hook_type, &n);
    if (n == 0)
      return SCM_BOOL_F;
    if (*cmd == '#')
    {
      strshift(cmd, 1);
      if (*cmd == '\0')
      {
        free(cmd);
        return SCM_BOOL_F;
      }
      if (!gh_exact_p(arg1))
      {
        free(cmd);
        scm_wrong_type_arg("hook_cmds", 4, arg1);
      }
      ser_no = (int) gh_scm2long(arg1);
      pos++;
    };
    numeric = strtol(cmd, &pat, 0);
    if (*pat)
    {
      type = get_type(cmd);
      numeric = 0;
    }
    free(cmd);
  }
  else
  {
    if (!gh_exact_p(hook_type))
      scm_wrong_type_arg("hook_cmds", 3, hook_type);
    numeric = (int) gh_scm2long(hook_type);
    pos = 2;
  }
  if (type == INVALID && numeric < 1)
    scm_wrong_type_arg("hook_cmds", 3, hook_type);
  if (pos == 2)
  {
    pat = gh_scm2newstr(arg1, &n);
    if (n == 0)
      return SCM_BOOL_F;
    if (!gh_null_p(rest))
      proc = gh_car(rest);
  }
  else
  {
    if (gh_null_p(rest))
      scm_wrong_num_args(gh_str02scm("hook_cmds"));
    pat = gh_scm2newstr(gh_car(rest), &n);
    if (n == 0)
      return SCM_BOOL_F;
    rest = gh_cdr(rest);
    if (!gh_null_p(rest))
      proc = gh_car(rest);
  }
  w.flag = HK_SCM;
  if (!gh_equal_p(SCM_BOOL_F, icase))
    w.flag |= HK_ICASE;
#if REGEX_HOOKS
  if (gh_equal_p(SCM_BOOL_T, regex))
    w.flag |= HK_REGEX;
#endif
  if (gh_equal_p(SCM_BOOL_T, preemptive))
    w.flag |= HK_PREEMPTIVE;
  w.pattern = pat;
  w.cmd.scm = -1;
  w.id = -1;                    //  used by list_hooks() 

  w.ser_number = ser_no;
  gh_defer_ints();
  if (type != INVALID)
    add_hook2(type, &w, (void *) &proc);
  else
    add_numhook2(numeric, &w, (void *) &proc);
  if (script_form)
    list_hooks(1);
  gh_allow_ints();
  free(pat);
  return SCM_BOOL_T;
}
static SCM s_remove_preemptive_cmd(SCM args)
{
  int ll;
  char *str = gh_scm2newstr(args, &ll);

  if (ll == 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  delete_hook(str, 1, 1);
  free(str);
  gh_allow_ints();
  return SCM_BOOL_T;
}

// ///////////////////////
// T i m e r s          //
//////////////////////////
static SCM s_timer_list_cmd(void)
// (ref . alarmORclock?)
{
  register int i;
  SCM result = gh_list(SCM_UNDEFINED);

  for (i = 0; i < timer_count; i++)
    if (aTimer[i].type != -1)
      result = gh_cons(gh_list(gh_long2scm(aTimer[i].ref),
                               gh_bool2scm(aTimer[i].flag & TIMER_ALARM),
                               SCM_UNDEFINED), result);
  return result;
}
static int get_free_ref()
{
/* the IDs that we automatically assign to timers should be -ve.
   this makes it possible for the user to decide to assign only +ve ID
   when the explicitely specifies a timer ID.
   19Nov98 ft22 */
#define MAX_REFNUM INT_MIN      //preposterous!
  register int i, j;
  char got_it = 1;

  for (j = -1; j == MAX_REFNUM; j--, got_it = 1)
  {
    for (i = 0; i < timer_count; i++)
    {
      if (aTimer[i].type != -1)
        if (aTimer[i].ref == j)
        {
          got_it = 0;
          break;
        }
    }
    if (got_it)
      return j;
  }
  return INT_MIN;
}

static SCM s_alarm_cmd(args)
     SCM args;
{
/*
   Usage:
   xalarm 'ref <refnum> <secs> <proc>
   xalarm <secs> <proc> 
 */
  SCM arg1;
  SCM proc;
  int sec;
  Timer t;
  unsigned char pos = 1;

  if (gh_null_p(args))
    return SCM_BOOL_F;
  arg1 = gh_car(args);
  args = gh_cdr(args);
  if (gh_symbol_p(arg1))
  {
    if (gh_equal_p(arg1, gh_symbol2scm("ref")))
    {
      if (gh_null_p(args))
        return SCM_BOOL_F;
      t.ref = gh_scm2long(gh_car(args));
      pos += 2;
      args = gh_cdr(args);      // time proc

      if (gh_null_p(args))
        return SCM_BOOL_F;
      arg1 = gh_car(args);      // seconds

      args = gh_cdr(args);      // proc

    }
  }
  else
    t.ref = get_free_ref();
  sec = (int) gh_scm2long(arg1);
  if (sec < 0)
  {
    scm_wrong_type_arg("s_alarm_cmd", pos, arg1);
  }
  pos++;
  if (gh_null_p(args))
  {
    // scm_wrong_type_arg("s_alarm_cmd",pos,gh_car(args));
    return SCM_BOOL_F;
  }
  proc = gh_car(args);
  t.type = ALARM_SCM;
  t.flag = TIMER_ALARM;
  t.cmd.scm = define_func(proc);
  gh_defer_ints();
  add_timer(&t, (void *) &sec);
  gh_allow_ints();
  return gh_int2scm(t.ref);
}
static SCM s_alarm_del_cmd(sref)
     SCM sref;
{
  int ref;

  ref = (int) gh_scm2long(sref);
  gh_defer_ints();
  del_alarm(ref, ALARM_SCM);
  gh_allow_ints();
  return SCM_BOOL_T;
}
static SCM s_clock_del_cmd(sref)
     SCM sref;
{
  int ref;

  ref = (int) gh_scm2long(sref);
  gh_defer_ints();
  del_clock(ref, ALARM_SCM);
  gh_allow_ints();
  return SCM_BOOL_T;
}

static SCM s_clock_cmd(args)
     SCM args;
{
/*
   USage:
   xclock 'ref 200 '(20 15 30 20 12 1998) proc
   xclock '(20 15 30 20 12 1998) proc 
 */

  SCM arg1;
  int ref = 0;

  if (gh_null_p(args))
    goto usage;
  arg1 = gh_car(args);
  args = gh_cdr(args);
  if (gh_symbol_p(arg1))
  {
    if (gh_equal_p(arg1, gh_symbol2scm("ref")))
    {
      if (gh_null_p(args))
        goto usage;
      ref = gh_scm2long(gh_car(args));
      args = gh_cdr(args);      // procedure

      if (gh_null_p(args))
        goto usage;
      arg1 = gh_car(args);      // date

      args = gh_cdr(args);
    }
  }
  else
    ref = get_free_ref();
  if (gh_list_p(arg1) && gh_length(arg1) > 0)
  {
    Timer t;
    struct tm *when;
    time_t now;
    int year, mon;
    int i = 0;
    int *var[6];
    SCM proc;

    proc = gh_car(args);
#warning (19Nov98) No plausibilty tests performed on the date yet.
    gh_defer_ints();
    now = time(NULL);
    when = localtime(&now);
    gh_allow_ints();
    when->tm_sec = 0;
    year = 1900 + when->tm_year;
    mon = 1 + when->tm_mon;
    var[0] = &when->tm_hour;
    var[1] = &when->tm_min;
    var[2] = &when->tm_sec;
    var[3] = &when->tm_mday;
    var[4] = &mon;
    var[5] = &year;
    while (!gh_null_p(arg1) && i < 6)
    {
      *(var[i]) = gh_scm2int(gh_car(arg1));
      arg1 = gh_cdr(arg1);
      i++;
    }
    if (i > 5)
      year += 1900;
    if (year < when->tm_year + 1900)
    {
      fprintf(stderr, "clock- Invalid year given: %d\n", year);
      return SCM_BOOL_F;
    }
    when->tm_year = year - 1900;
    // this is lame
    when->tm_mon = (mon - 1) % 12;
    when->tm_mday %= 31;
    when->tm_hour %= 24;
    when->tm_sec %= 60;

    t.ref = ref;
    t.type = ALARM_SCM;
    t.flag = TIMER_CLOCK;
    gh_defer_ints();
    t.cmd.scm = define_func(proc);
    i = add_timer(&t, (void *) when);
    gh_allow_ints();
#if DEBUG >1
    fprintf(stderr, "clock %d set for %02d:%02d:%02d %02d.%02d.%d\n",
            aTimer[i].ref, aTimer[i].expire.date->tm_hour,
            aTimer[i].expire.date->tm_min, aTimer[i].expire.date->tm_sec,
            aTimer[i].expire.date->tm_mday, aTimer[i].expire.date->tm_mon + 1,
            aTimer[i].expire.date->tm_year + 1900);
#endif
    return (gh_int2scm(ref));
  }
usage:
  say
     ("Usage: (gs-clock ['ref <id>] '(<HH> <MM> <SS> <DD> <mm> <YYYY>) <proc>)",
      -1, 1);
  return SCM_BOOL_F;
}
// ////////////////////////-timers end-----//////////

static SCM s_remove_command_cmd(SCM cmd, SCM not_ctcp)
{
  int l;
  char *name;

  name = gh_scm2newstr(cmd, &l);
  if (l == 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  if (gh_equal_p(SCM_BOOL_T, not_ctcp))
    l = _remove_cmd(name, /*CMD_ALIAS| */ CMD_SCM | CMD_FD);
  else
    l = remove_ctcp_cmd(name);
  gh_allow_ints();
  free(name);
  return gh_bool2scm(l == 0);
}

static SCM s_reg_command(name, descr, func)
/*
   register a new command 
 */
     SCM name;
     SCM descr;
     SCM func;
{
  Cmd cmd;
  int l;

  cmd.name = gh_scm2newstr(name, &l);
  if (l == 0)
    return SCM_BOOL_F;
  cmd.help = gh_scm2newstr(descr, &l);
  cmd.type = CMD_SCM;
  gh_defer_ints();
  cmd.cmd.scm = define_func(func);
  insert_scm_cmd(&cmd);
  gh_allow_ints();
  return SCM_BOOL_T;
}
static SCM s_reg_ctcp_command(name, descr, func)
/*
   register a new command 
 */
     SCM name;
     SCM descr;
     SCM func;
{
  Cmd cmd;
  int l = 0;

  cmd.name = gh_scm2newstr(name, &l);
  if (l < 1)
    return SCM_BOOL_F;
  cmd.help = gh_scm2newstr(descr, &l);
  cmd.type = CMD_SCM | CMD_CTCP;
  gh_defer_ints();
  cmd.cmd.scm = define_func(func);
  insert_ctcp_cmd(&cmd);
  gh_allow_ints();
  return SCM_BOOL_T;
}

static SCM s_check_command(SCM name, SCM non_ctcp)
/*
   is name a valid command or alias? 
 */
{
  SCM result;
  char *c;
  int l;

  c = gh_scm2newstr(name, &l);
  if (l < 1)
    return SCM_UNSPECIFIED;
  if (gh_equal_p(SCM_BOOL_T, non_ctcp))
    result = gh_bool2scm(find_cmd(c, 1) != NULL);
  else
    result = gh_bool2scm(find_ctcp_cmd(c, 1) != NULL);
  free(c);
  return result;
}

/*
   notify groups
 */

static SCM list_notify_groups_cmd(void)
{
// ( i . label )
  SCM result = gh_list(SCM_UNDEFINED);
  Notify_t i;

  for (i = 0; i < NotifyListCount; i++)
    if (aNotifyList[i].label)
      result = gh_cons(gh_list(gh_long2scm(i),
                               gh_str02scm(aNotifyList[i].label),
                               SCM_UNDEFINED), result);
  return result;
}
static SCM query_notify_group_cmd(SCM group)
{
  SCM result = gh_list(SCM_UNDEFINED);
  int n = gh_scm2int(group);

  if (valid_notify_index(n))
  {
    NotifyNick *a = aNotifyList[n].start;

    while (a != aNotifyList[n].end)
    {
      result = gh_cons(gh_str02scm(a->nick), result);
      a = a->next;
    }
  }
  return result;
}
static SCM notify_id_to_name_cmd(SCM group)
{
  int n = gh_scm2int(group);

  if (valid_notify_index(n))
    return (gh_str02scm(aNotifyList[n].label));
  return SCM_BOOL_F;
}
static SCM notify_name_to_id_cmd(SCM name)
{
  int n;
  char *label = gh_scm2newstr(name, &n);

  if (n == 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  n = find_notify_list_bylabel(label);
  gh_allow_ints();
  free(label);
  return (n == -1 ? SCM_BOOL_F : gh_int2scm(n));
}
static SCM create_notify_group_cmd(SCM name)
{
  int n;
  char *label = gh_scm2newstr(name, &n);

  if (n == 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  n = new_notify_list(label);
  gh_allow_ints();
  free(label);
  return gh_int2scm(n);
}
static SCM delete_notify_group_cmd(SCM group)
{
  int n = gh_scm2int(group);

  if (!valid_notify_index(n))
    return SCM_BOOL_F;
  gh_defer_ints();
  remove_notify_list(n);
  gh_allow_ints();
  return SCM_BOOL_T;
}
static SCM edit_notify_group(SCM group, SCM val, char what)
{
  int n;
  char *nick;

  nick = gh_scm2newstr(val, &n);
  if (n == 0)
    return SCM_BOOL_F;
  n = gh_scm2int(group);
  if (n < 0)
    return SCM_BOOL_F;
  gh_defer_ints();
  if (what == 1)
    drop_from_notify(n, nick);
  else
    addto_notify(n, nick);
  gh_allow_ints();
  free(nick);
  return SCM_BOOL_T;
}

extern SCM make_server_SCM_notify_list(Server * s);
static SCM list_server_notify_groups_cmd(SCM server)
{
  Server *s;

  s = SCM2server(server);
  if (s == NULL)
    return SCM_BOOL_F;
  else
  {
    SCM ret;

    gh_defer_ints();
    ret = make_server_SCM_notify_list(s);
    gh_allow_ints();
    return ret;
  }
}
static SCM edit_server_notify_group(SCM server, SCM val, char flag)
{
  int i;
  Server *s;

  s = SCM2server(server);
  if (s == NULL)
    return SCM_BOOL_F;
  i = gh_scm2int(val);
  if (!valid_notify_index(i))
    return SCM_BOOL_F;
  gh_defer_ints();
  if (flag)
    i = detach_notify_group(s, i);
  else
    i = attach_notify_group(s, i);
  gh_allow_ints();
  return gh_bool2scm(i == 0);
}

static SCM add_to_notify_group_cmd(SCM group, SCM val)
{
  return (edit_notify_group(group, val, 0));
}
static SCM delete_from_notify_group_cmd(SCM group, SCM val)
{
  return (edit_notify_group(group, val, 1));
}

static SCM set_server_notify_group_cmd(SCM server, SCM val)
{
  return (edit_server_notify_group(server, val, 0));
}
static SCM drop_server_notify_group_cmd(SCM server, SCM val)
{
  return (edit_server_notify_group(server, val, 1));
}
//////////////////////////////////

static SCM cmd_raw_write(to, str)
/*
   quote str to a connection. <to> is the socket descriptor. 
 */
     SCM to;
     SCM str;
{
  int l, ret;
  int fd = gh_scm2long(to);
  char *txt = gh_scm2newstr(str, &l);

  if (l == 0)
    return SCM_BOOL_F;
  ret = writen(fd, txt, l);
  fd = errno;
  free(txt);
  errno = fd;
  if (ret != l)                 //throw error????????

    scm_syserror("write error");
  return SCM_BOOL_T;
}

static SCM cmd_query_set_var(SCM var)
/* query a SET variable*/
{
  int ll;
  SCM ret;
  char *name = gh_scm2newstr(var, &ll);

  if (ll == 0)
    return SCM_BOOL_F;
  ret = get_set_variable(name);
  free(name);
  return ret;
}
static SCM cmd_next_word(SCM str, SCM pos)
{
  int n;
  char *s = gh_scm2newstr(str, &n);
  char *res = NULL;
  SCM ret;

  if (n > 0)
  {
    n = gh_scm2long(pos);
    if (n > -1)
      res = nextword(s, n);
    free(s);
  }
  ret= gh_str02scm(res? res: "");
  if(res) free(res);
  return ret;
}
static SCM cmd_word_index(SCM str, SCM pos)
{
  int n;
  char *s = gh_scm2newstr(str, &n);
  int xx = 0;

  if (n > 0)
  {
    n = gh_scm2long(pos);
    if (n > -1)
      xx = find_pos(s, n);
    free(s);
  }
  return gh_int2scm(xx);
}

static SCM modify_it(char **m, SCM str, int flag)
{
   if(flag)
   {
   int n;

    if(*m)
      free(*m);
    *m= gh_scm2newstr(str, &n);
    if(n==0)
      *m=NULL;
    return SCM_BOOL_T;
   }
 return SCM_BOOL_F;
}


static SCM gs_modify_irc_output(SCM str)
{
#if MUTABLE_IRC
  SCM ret= SCM_BOOL_F;

  if(sflags&sMUTABLE_IRC){
    ret=modify_it(&Msg2, str, sflags&s_DOING_IRC_HOOK);
    if(gh_equal_p(ret, SCM_BOOL_T))
       sflags |= s_IRC_OUTPUT_MODIFIED;
  }
  return ret;
#else
  return SCM_BOOL_F;
#endif
}

static SCM gs_modify_user_input(SCM str)
{
  return modify_it(&input_line, str, sflags&s_DOING_INPUT_HOOK);
}
#define display_string(_val, _w)  do{\
  int _len_;\
  char *_res_=  gh_scm2newstr(_val, &_len_);\
   gh_defer_ints();\
    say0(_res_, _w, 1);\
    gh_allow_ints();\
    free(_res_);\
   }while(0)

static void display_other(SCM val, int w)
{
  SCM res;
  int len;
  char *buf;
 
  res= gh_call2(print_obj, val, SCM_BOOL_T);
  buf = gh_scm2newstr(res, &len);
  if (len > 0)
  {
    gh_defer_ints();
    say0(buf, w, 1);
    gh_allow_ints();
    free(buf);
  }
}

int parse_scheme(const char *str, int w)
{
/* this proc is non-reentrant*/
  SCM SCM_res;

  SCM_res = gh_eval_str_with_catch((char *) str, &exception_handler);
  if (sflags & s_VERBOSE_RETURN)
  {
    if (gh_string_p(SCM_res) || gh_symbol_p(SCM_res))
      display_string(SCM_res, w);
    else
      display_other(SCM_res, w);
  }
  else
  {
    if (gh_string_p(SCM_res) || gh_symbol_p(SCM_res))
      display_string(SCM_res, w);
    else if (gh_number_p(SCM_res))
      display_other(SCM_res, w);
  }
  return 0;
}
void scheme_setup(char **argv)
{
  int i;
  char buf[MAXPATHLEN + 255];
  static void load_scripts(const char *path);

  scm_func = NULL;
  keybinding = NULL;
  scm_func_count = bind_count = 0;
  for (i = 0; i < VAR_STEM_LEN - 1; i++)
  {
    struct timeval t;

    gettimeofday(&t, NULL);
    srandom(t.tv_usec);
    var_stem[i] = 65 + (random() % 58);
    if (var_stem[i] > 90 && var_stem[i] < 97)
      var_stem[i] = 48 + (random() % 10);
    delay(111);
  }
  var_stem[VAR_STEM_LEN - 1] = 0;
  gh_define("sula-random-string", gh_str02scm(var_stem));
  gh_define("*sula-version*",
            gh_list_to_vector(gh_list(gh_str02scm(sula_NAME),
                                      gh_str02scm(spx_MAJOR_V),
                                      gh_str02scm(spx_MINOR_V),
                                      gh_str02scm(spx_RELEASE_V),
                                      gh_str02scm(sula_VERSION),
                                      gh_ulong2scm(sula_RELEASE),
                                      SCM_UNDEFINED)));
  gh_define("*sula-infosite*",
            gh_list_to_vector(gh_list(gh_str02scm(sula_INFOSITE),
                                      gh_str02scm(sula_INFOSITE2),
                                      gh_str02scm(sula_INFOSITE3),
                                      SCM_UNDEFINED)));
  gh_define("*sula-contact*",
            gh_list_to_vector(gh_list(gh_str02scm(sula_EMAIL),
                                      SCM_UNDEFINED)));
  gh_define("*sula-home*", gh_str02scm(prog_home));
  gh_define("*sula-libdir*", gh_str02scm(sula_lib));
#if USE_XFORMS
  gh_define("*gui*", gh_str02scm("xforms"));
#elif USE_GTK
  gh_define("*gui*", gh_str02scm("gtk"));
  /* currently only implemented in the GTK version */
  initialize_skriptx();
  gh_new_procedure("gs-pup-add", gs_add_pup, 2, 0, 1);
  gh_new_procedure("gs-pup-remove", gs_remove_pup, 1, 0, 0);
  gh_new_procedure("gs-pup-clear-all", gs_clear_pup, 0, 0, 0);
  
  gh_new_procedure("gs-parse-colour", cmd_parse_colour, 1, 0, 0);
  gh_new_procedure("gs-max-colour", cmd_max_colour, 0, 0, 0);
  gh_new_procedure("gs-set-window-bg-pixmap", gs_set_window_bg_pixmap, 2, 0,
                   1);
  gh_new_procedure("gs-set-window-bg-colour", gs_set_window_bg_colour, 2, 0,
                   1);
  gh_new_procedure("gs-set-window-fg-colour", gs_set_window_fg_colour, 2, 0,
                   1);
  gh_new_procedure("gs-set-window-text-font", gs_set_window_text_font, 2, 0,
                   1);
  gh_new_procedure("gs-set-status-bg-pixmap", gs_set_status_bg_pixmap, 2, 0,
                   0);
  gh_new_procedure("gs-set-status-bg-colour", gs_set_status_bg_colour, 2, 0,
                   0);
  gh_new_procedure("gs-set-status-fg-colour", gs_set_status_fg_colour, 2, 0,
                   0);
  gh_new_procedure("gs-set-status-font", gs_set_status_font, 2, 0, 0);
  gh_new_procedure("gs-set-input-bg-colour", gs_set_input_bg_colour, 2, 0, 0);
  gh_new_procedure("gs-set-input-fg-colour", gs_set_input_fg_colour, 2, 0, 0);
  gh_new_procedure("gs-set-input-font", gs_set_input_font, 2, 0, 0);
  gh_new_procedure("gs-set-cclock-font", gs_set_console_clock_font, 1, 0, 0);
  gh_new_procedure("gs-set-cclock-fg-colour", gs_set_console_clock_fg_colour,
                   1, 0, 0);
  gh_new_procedure("gs-set-cclock-bg-colour", gs_set_console_clock_bg_colour,
                   1, 0, 0);
#endif
  gh_new_procedure("gs-modify-user-input", gs_modify_user_input, 1,0 ,0);
  gh_new_procedure("gs-modify-irc-output", gs_modify_irc_output, 1,0 ,0);

  gh_new_procedure("gs-bell", cmd_bell, 0, 0, 1);
  gh_new_procedure("gs-delay", cmd_delay, 1, 0, 0);
  gh_new_procedure("gs-exec", s_exec_cmd, 1, 0, 1);
  gh_new_procedure("gs-echo", s_echo_cmd, 1, 0, 1);
  gh_new_procedure("gs-exit", s_exit_cmd, 0, 0, 1);
  gh_new_procedure("gs-new-hook", hook_cmds, 5, 0, 1);
  gh_new_procedure("gs-remove-preempt", s_remove_preemptive_cmd, 1, 0, 0);

  gh_new_procedure("gs-clear-console", clear_main_window_cmd, 0, 0, 0);
  gh_new_procedure("gs-hide-console", hide_main_window_cmd, 1, 0, 0);
  gh_new_procedure("gs-console-visible?", gs_console_visible, 0, 0, 0);

  gh_new_procedure("gs-server-channels", s_get_channel_list_by_server, 0, 0,
                   1);
  gh_new_procedure("gs-channel-users", s_get_nicklist_by_channel, 0, 0, 1);
  gh_new_procedure("gs-channel-nicks", fast_get_nicklist_by_channel, 2, 0, 0);
  gh_new_procedure("gs-current-channel", s_get_current_by_channel, 0, 0, 1);
  gh_new_procedure("gs-message", set_window_message_cmd, 1, 0, 1);
  gh_new_procedure("gs-get-message", get_window_message_cmd, 0, 0, 1);
  gh_new_procedure("gs-window-title!", set_window_title_cmd, 2, 0, 0);
  gh_new_procedure("gs-window-title", get_window_title_cmd, 1, 0, 0);
  gh_new_procedure("gs-set-lag", gs_set_lag,2,0,0);
  gh_new_procedure("gs-set-status", cmd_set_statline, 1, 0, 1);
  gh_new_procedure("gs-get-status", cmd_get_statline, 0, 0, 1);
  gh_new_procedure("gs-window-server", s_get_server_by_win_nr, 0, 0, 1);
  gh_new_procedure("gs-window-server2", fast_window_server, 1, 0, 0);
  gh_new_procedure("gs-window-channels", s_get_channel_list_by_win, 0, 0, 1);
  gh_new_procedure("gs-window-chat", s_get_dcc_list_by_win, 0, 0, 1);
  gh_new_procedure("gs-dcc-list", gs_dcc_list,0,0,0);
  gh_new_procedure("gs-window-list", s_win_list, 0, 0, 0);
  gh_new_procedure("gs-create-window", s_win_create, 0, 0, 0);
  gh_new_procedure("gs-kill-window", cmd_win_destroy, 2, 0, 0);
  gh_new_procedure("gs-new-command", s_reg_command, 3, 0, 0);
  gh_new_procedure("gs-new-ctcp-command", s_reg_ctcp_command, 3, 0, 0);
  gh_new_procedure("gs-remove-command", s_remove_command_cmd, 2, 0, 0);
  gh_new_procedure("gs-valid-command?", s_check_command, 2, 0, 0);
  gh_new_procedure("gs-channel", s_get_channel_info, 0, 0, 1);
  gh_new_procedure("gs-channel2", fast_get_channel_info, 2, 0, 0);
  gh_new_procedure("gs-alarm", s_alarm_cmd, 0, 0, 1);
  gh_new_procedure("gs-delete-alarm", s_alarm_del_cmd, 1, 0, 0);
  gh_new_procedure("gs-clock", s_clock_cmd, 0, 0, 1);
  gh_new_procedure("gs-delete-clock", s_clock_del_cmd, 1, 0, 0);
  gh_new_procedure("gs-timer-list", s_timer_list_cmd, 0, 0, 0);
  gh_new_procedure("gs-last-channel", cmd_last_channel, 1, 0, 1);
  gh_new_procedure("gs-raw-write", cmd_raw_write, 2, 0, 0);
  gh_new_procedure("gs-set!", cmd_set_var, 2, 0, 0);
  gh_new_procedure("gs-set?", cmd_query_set_var, 1, 0, 0);
  gh_new_procedure("gs-new-variable", create_set_variable, 2, 0, 1);
  gh_new_procedure("gs-remove-variable", delete_set_variable, 1, 0, 0);
  //nickname groups
  gh_new_procedure("gs-create-nickgroup", s_create_nickgroup_cmd, 1, 0, 0);
  gh_new_procedure("gs-addto-nickgroup", s_addto_nickgroup_cmd, 2, 0, 0);	//add nick

  gh_new_procedure("gs-list-nickgroups", s_list_nick_groups_cmd, 0, 0, 0);	//return all groups

  gh_new_procedure("gs-query-nickgroup", s_list_group_nicks_cmd, 1, 0, 0);	//return this group

  gh_new_procedure("gs-set-server-nickgroup", s_set_server_nick_group, 2, 0,
                   0);

  //named connections
  gh_new_procedure("gs-run", s_run_program_cmd, 2, 0, 0);
  gh_new_procedure("gs-connect", s_connect_cmd, 3, 0, 1);
  gh_new_procedure("gs-nc-list", s_list_nc_cmd, 0, 0, 0);
  gh_new_procedure("gs-nc", s_nc_cmd, 1, 0, 0);

  gh_new_procedure("gs-ignored?", s_ignored_p, 2, 0, 0);
  gh_new_procedure("gs-current-dcc", s_get_current_dcc_nick, 0, 0, 1);
  gh_new_procedure("gs-current-query", s_get_query_nick, 0, 0, 1);

  gh_new_procedure("gs-get-input", cmd_get_input, 0, 0, 1);
  gh_new_procedure("gs-set-input", cmd_set_input, 1, 0, 1);
  gh_new_procedure("gs-get-input-cursorpos", cmd_get_input_cursorpos, 0, 0,
                   1);
  gh_new_procedure("gs-set-input-cursorpos", cmd_set_input_cursorpos, 1, 0,
                   1);
  gh_new_procedure("gs-set-radio", cmd_set_radio, 1, 0, 1);
  gh_new_procedure("gs-get-radio", cmd_get_radio, 0, 0, 1);

  gh_new_procedure("gs-bind-key", gs_bind_proc, 2, 0, 1);
  gh_new_procedure("gs-delete-key-binding", gs_delete_binding_proc, 2, 0, 0);
  gh_new_procedure("gs-query-key-bindings", gs_query_bindings_proc, 0, 0, 0);

  gh_new_procedure("gs-srandom", gs_srandom, 0, 0, 1);
  gh_new_procedure("gs-word-next", cmd_next_word, 2, 0, 0);
  gh_new_procedure("gs-word-index", cmd_word_index, 2, 0, 0);
  //notify groups
  gh_new_procedure("gs-create-notify-group", create_notify_group_cmd, 1, 0,
                   0);
  gh_new_procedure("gs-delete-notify-group", delete_notify_group_cmd, 1, 0,
                   0);
  gh_new_procedure("gs-notify-name->id", notify_name_to_id_cmd, 1, 0, 0);
  gh_new_procedure("gs-notify-id->name", notify_id_to_name_cmd, 1, 0, 0);
  gh_new_procedure("gs-add-to-notify-group", add_to_notify_group_cmd, 2, 0,
                   0);
  gh_new_procedure("gs-delete-from-notify-group",
                   delete_from_notify_group_cmd, 2, 0, 0);
  gh_new_procedure("gs-list-notify-groups", list_notify_groups_cmd, 0, 0, 0);
  gh_new_procedure("gs-query-notify-group", query_notify_group_cmd, 1, 0, 0);
  gh_new_procedure("gs-add-server-notify-group", set_server_notify_group_cmd,
                   2, 0, 0);
  gh_new_procedure("gs-drop-server-notify-group",
                   drop_server_notify_group_cmd, 2, 0, 0);
  gh_new_procedure("gs-list-server-notify-groups",
                   list_server_notify_groups_cmd, 1, 0, 0);

  gh_new_procedure("gs-add-io-watch", cmd_add_io_watch, 2, 0, 0);
  gh_new_procedure("gs-stop-io-watch", cmd_stop_io_watch, 1, 0, 0);
/*
   Do not alter this.
   Define a procedure of type SCM which will be used for printing errors when 
   an exception is caught.The procedure should return a string. Its argument
   < args > is the last argument to the exception handler SCM
   exception_handler(void *data, SCM tag, SCM args). Using
   (gs-set-print-proc(lambda blabla)) wont work. 
 */
  gh_new_procedure("gs-set-print-proc", s_sprint_command, 1, 0, 0);
  sprintf(buf,
          "(define %s-print (lambda (args flag)"
          " (call-with-output-string (lambda(n)"
          "((if flag display write) args n)))))"
          " (gs-set-print-proc %s-print)", var_stem, var_stem);
  gh_eval_str_with_catch(buf, &exception_handler);
/* *INDENT-OFF* */
  {
    char *p, *buf2;

    p = (char *) spx_get_app_resource("Sula-boot",
                                      "StringCLASS", SPX_NONE,
#ifdef LIB_DIR
                                      LIB_DIR
#else
                                      sula_lib
#endif
                                      , NULL, 0);
    sprintf(buf, "%s/sula-boot", p);
    if (access(buf, R_OK) != 0)
    {
      fprintf(stderr,
"
_________________________________________
%s: %s
The file \E[1msula-boot\E[m could not be read. Specify a valid location.
If, for example, it is located in directory /home/myhome/, use one of the
folowing methods:
 0. command line:
   \E[5m%s -bootlocation /home/myhome/\E[m
 1. an entry in an X resource file such as ~/.Xdefaults-%s or ~/.Xdefaults:
   \E[5m*.Sula-boot: /home/myhome/\E[m
  (you may have to run 'xrdb -merge ...' if ~/.Xdefaults is used.)
The file sula-boot is also obtained from " sula_GETTING ".
 _________________________________________\n\n",
 buf, strerror(errno), argv[0], host);
      exit(1);
    }
 /* *INDENT-ON* */
  gh_eval_str_with_catch("(define sula-boot-no-error #f)",
                         &exception_handler);
  if (debug)
    fprintf(stderr, "\nboot file: %s\n", buf);
  gh_eval_file_with_catch(buf, &exception_handler);
  if (gh_equal_p(gh_eval_str_with_catch("(not sula-boot-no-error)",
                                        &exception_handler), SCM_BOOL_T))
    fprintf(stderr, "\nWarning: %s is fsck!%%&d. Install a correct copy."
            "\nYou should now quit or Strange Things will happen...\n\n", buf);
  gh_eval_str_with_catch("(undefine sula-boot-no-error)", &exception_handler);
  sprintf(buf, "%s/gsularc", sula_lib);
  if (debug > 2)
    fprintf(stderr, "global: %s\n", buf);
  buf2 = strdup(buf);
  if (access(buf, R_OK) == 0)
    gh_eval_file_with_catch(buf, &exception_handler);
  else if (debug)
    perror(buf);
  sprintf(buf, "%s/.gsularc", prog_home);
  p = (char *) spx_get_app_resource("spxrc",
                                    "StringCLASS", SPX_NONE, buf, NULL, 0);
  if (access(p, R_OK) != 0)
  {
    if (debug)
      perror(p);
    if (strcmp(p, buf))
    {
      p = buf;
      if (debug)
        fprintf(stderr, "Retrying with %s\n", p);
      if (access(p, R_OK) != 0)
      {
        if (debug)
          perror(p);
        p = 0;
      }
    }
    else
      p = NULL;
  }
  if (p)
  {
    if (debug)
      fprintf(stderr, "spxrc: %s\n", p);
    if (strcmp(p, buf2))
      gh_eval_file_with_catch(p, &exception_handler);
    else if (debug)
      fprintf(stderr, "spxrc: %s is already loaded\n", p);
  }
  free(buf2);
}
sprintf(buf, "%s/scripts", sula_lib);
load_scripts(&buf[0]);
sprintf(buf, "%s/scripts", prog_home);
load_scripts(&buf[0]);
}
static void load_scripts(const char *path)
{

  glob_t gl;
  struct stat buf;
  char **ptr;

  if (debug > 2)
    fprintf(stderr, "reading %s ...\n", path);
  if (access(path, R_OK))
  {
    if (debug > 2)
      perror(path);
    return;
  }
  strcpy((char *) path + strlen(path), "/*.scm");
  errno = 0;
  gl.gl_pathc = 0;
  (void) glob(path, GLOB_NOSORT, NULL, &gl);
  if (gl.gl_pathc == 0)
  {
    if (debug > 2)
    {
      if (errno)
        perror(path);
      else
        fprintf(stderr, "no matching files found\n");
    }
    return;
  }
  ptr = gl.gl_pathv;
  while (*ptr)
  {
    if (!stat(*ptr, &buf))
    {
      if (S_ISREG(buf.st_mode))
      {
        if (debug > 2)
          fprintf(stderr, "Loading %s\n", *ptr);
        if (access(*ptr, R_OK))
        {
          if (debug > 2)
            perror(*ptr);
        }
        else
          gh_eval_file_with_catch(*ptr, &exception_handler);
      }
    }
    else if (debug > 2)
      perror(*ptr);
    ptr++;
  }
  globfree(&gl);
}
