/***************************************************
  This file is distributed as part of Sula PrimeriX
  (http://members.xoom.com/sprimerix).
****************************************************/

/* 
   servers.c - manage server entries, connect to servers etc

   Copyright (C) 1999 Tano Fotang

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free 
   Software Foundation; either version 2 of the License , or (at your option) 
   any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
   or FITNESS FOR A PARTICULAR PURPOSE. See the file LICENCE for details.

   You should have received a copy of the GNU General Public License along
   with this program; see the file COPYING.  If not, write to the Free
   Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.


*/

#include "spx.h"
#include <sys/time.h>
#include <ctype.h>
#include <assert.h>
#include <netdb.h>
#include <arpa/inet.h>
#include "config.h"
#include "nicks.h"
#include "server.h"
#include "lag.h"
#include "cmd.h"
#include "hooks.h"
#include "notify.h"

Server /**servTree = NULL, */  * servFdTree = NULL;
ServerGroup *aServerGroup = NULL;
int iServerGroupCount = 0;

gChannel *chanStart, *chanEnd;
Server_tool *server_tool = NULL;

static void new_close_connection(int group, int *fd, char *reason,

                                 int update_win);

static Server *search_server_tree_byfd(int);
static int del_server(pServer * pp, Server * s, int flag);

#define SERVERGROUP_ALLOC 5
static void group_alloc(void)
{
  int i;

  if (aServerGroup == NULL)
    aServerGroup = my_malloc(sizeof(ServerGroup) * SERVERGROUP_ALLOC);
  else
    aServerGroup = my_realloc(aServerGroup,
                              (iServerGroupCount +
                               SERVERGROUP_ALLOC) * sizeof(ServerGroup));
  for (i = iServerGroupCount; i < iServerGroupCount + SERVERGROUP_ALLOC; i++)
    aServerGroup[i].name = NULL;
  iServerGroupCount += SERVERGROUP_ALLOC;
}
static int locate_servergroup(const char *name)
{
  int k = 0;

  assert(name);

  for (; k < iServerGroupCount; k++)
    if (aServerGroup[k].name && !strcasecmp(aServerGroup[k].name, name))
      return k;
  return -1;
}
int new_group(const char *name)
{
  int i = locate_servergroup(name);

  if (i != -1)
    return i;
  if (aServerGroup == NULL)
    group_alloc();
retry:
  for (i = 0; i < iServerGroupCount; i++)
    if (!aServerGroup[i].name)
    {
      aServerGroup[i].s = NULL;
      aServerGroup[i].name = strdup(name);
      return i;
    }
  group_alloc();
  goto retry;
}

static void add_to_group(int g, const char *entry)
{
/* 
 * entry is of form  server:port:passwd:nick:email:ircname
 */
  char *tmp;                    /*password, *nick, *ircname, *email; */
  size_t ll;
  int i;
  char *ptr, *pp;
  Server *server;

  ll = strlen(entry) + 1;
  pp = (char *) entry;
  tmp = (char *) my_malloc(sizeof(char) * (ll));

  server = alloca(sizeof(Server));
  memset(server, 0, sizeof(Server));
  memset(tmp, 0, ll);
  ptr = tmp;
  while (*pp && *pp != ':')
    *ptr++ = *pp++;
  if (*pp == ':')
  {
    *ptr = '\0';
    pp++;
  }
  if (!(*tmp))
    return;
  else
    server->name = strdup(tmp);
  *tmp = 0;
  ptr = tmp;
  while (*pp && *pp != ':')
    *ptr++ = *pp++;
  if (*pp == ':')
  {
    *ptr = '\0';
    pp++;
  }
  i = atoi(tmp);
  if (i < 0)
    *tmp = '\0';
  ll = strlen(tmp);
  while ((i = atoi(tmp)) > USHRT_MAX)
    *(tmp + --ll) = '\0';
  if ((server->port = i) == 0)
    server->port = DEFAULT_PORT;
  *tmp = 0;
  ptr = tmp;
  while (*pp && *pp != ':')
    *ptr++ = *pp++;
  if (*pp == ':')
  {
    *ptr = '\0';
    pp++;
  }
  if (!(*tmp))
    server->password = NULL;
  else
    server->password = strdup(tmp);
  *tmp = 0;
  ptr = tmp;
  while (*pp && *pp != ':')
    *ptr++ = *pp++;
  if (*pp == ':')
  {
    *ptr = '\0';
    pp++;
  }
  if (!(*tmp))
  {
    char *ss = getenv("IRCNICK");

    server->nick = strdup(ss ? ss : Name);
  }
  else
    server->nick = strdup(tmp);
  *tmp = 0;
  ptr = tmp;
  while (*pp && *pp != ':')
    *ptr++ = *pp++;
  if (*pp == ':')
  {
    *ptr = '\0';
    pp++;
  }
  if (!(*tmp))
  {
    server->email = my_malloc(sizeof(char) * (strlen(Name) +

                                              strlen(host) + 3));

    sprintf(server->email, "%s@%s", Name, host);
  }
  else
    server->email = strdup(tmp);
  *tmp = 0;
  ptr = tmp;
  while ((*ptr = *pp))
  {
    ptr++;
    pp++;
  }
  if (!(*tmp))
  {
    char *ss = getenv("IRCNAME");

    if (ss)
      server->ircname = strdup(ss);
    else
    {
#warning TODO: set it to NULL
      server->ircname = malloc(sizeof(char) * (strlen(sula_NAME) + 1));

      strcpy(server->ircname, sula_NAME);
    }
  }
  else
    server->ircname = strdup(tmp);
  free(tmp);
  server->mode = NULL;
  server->flag = SERVER_FLAG_RESET;
  server->state = -1;
  server->fd = -1;
  server->g = g;
  server->w = -1;
  server->lastInvite = NULL;
  server->alias = NULL;
  server->left = server->right = NULL;

  server->notify = NULL;

  server->iNick = get_list_bylabel("default");
  aNickList[server->iNick].current =
     add_nick_to_list(server->iNick, server->nick);
  insert_server(&(aServerGroup[g].s), server, BY_NAME);
  free(server->email);
  free(server->nick);
  free(server->ircname);
  if (server->password)
    free(server->password);
  free(server->name);
  redraw_server_entry(g);
}

#define MAX_ENTRIES 64          /* max number of servers in an entry when reading from file.
                                   If you have more, split into different entries with the same label. */
int load_serverlist(const char *filename)
{
  FILE *fp;

  if ((fp = fopen(filename, "r")))
  {
    char **entries, *name;
    int count;

    entries = my_malloc(sizeof(char *) * (MAX_ENTRIES + 1));

    while ((name = get_string_entries(fp, &count, entries, MAX_ENTRIES, 0)))
    {
      int i = new_group(name);
      char **p = entries;

      while (*p)
      {
        add_to_group(i, *p);
        free(*p);
        p++;
      }
      free(name);
    }
    free(entries);
    fclose(fp);
    return 0;
  }
  return -1;
}
int init_serverlist(const char *filename)
{
  new_group(DEFAULT_GRP);
  return load_serverlist(filename);
}
static void save_server_list(FILE * fp, Server * p)
{
  if (p)
  {
    save_server_list(fp, p->left);
    fprintf(fp, "%s:%u:%s:%s:%s:%s\n", p->name, p->port,
            p->password ? p->password : "",
            p->nick, p->email, p->ircname ? p->ircname : "");
    save_server_list(fp, p->right);
  }
}

int save_servers(const char *path)
{
  char *p;
  FILE *fp;

  if (!path)
  {
    p =
       my_malloc(sizeof(char) *

                 (strlen(prog_home) + strlen(P_HOME P_SERVERS) + 8));

    sprintf(p, "%s/" P_SERVERS, prog_home);
  }
  else
    p = strdup(path);
  if (access(p, F_OK) != -1)
  {
    if (access(p, W_OK) < 0)
    {
      free(p);
      return -1;
    }
    else
    {
      char *buf = malloc(sizeof(char) * (strlen(p) + 5));

      sprintf(buf, "%s.bak", p);
      rename(p, buf);
      free(buf);
    }
  }
  if ((fp = fopen(p, "w")) != NULL)
  {
    time_t t = time(NULL);
    int i;

    fprintf(fp,
            "%c" sula_NAME " "
            sula_VERSION "\n%cServer list created by %s on %s\n",
            COMMENT, COMMENT, Name, ctime(&t));
    for (i = 0; i < iServerGroupCount; i++)
      if (aServerGroup[i].name)
      {
        fprintf(fp, "<%s>\n", aServerGroup[i].name);
        save_server_list(fp, aServerGroup[i].s);
        fprintf(fp, "</%s>\n", aServerGroup[i].name);
      }
    fflush(fp);
    fclose(fp);
    free(p);
    return 0;
  }
  free(p);
  return -1;
}

static Server *search_server_tree_byfd(int fd)
{
  Server *p = servFdTree;

  while (p)
  {
    if (fd == p->fd)
      return p;
    if (fd < p->fd)
      p = p->left;
    else
      p = p->right;
  }
  return NULL;
}

static Server *locate_server(Server * p, const char *name, u_short port)
{
  int c;

  while (p)
  {
    c = strcasecmp(name, p->name);
    if (c == 0)
    {
      if (p->port == port)
        return p;
      else if (port > p->port)
        p = p->right;
      else
        p = p->left;
    }
    else if (c < 0)
      p = p->left;
    else
      p = p->right;
  }
  return NULL;
}
Server *find_connection(const char *name, u_short port)
{
  return (locate_server(servFdTree, name, port));
}

Server *search_server_tree_byname(int i, const char *name, int port)
{
  if (VALID_GROUP(i))
    return locate_server(aServerGroup[i].s, name, port);
  else
  {
    int n;
    Server *p;

    for (n = 0; n < iServerGroupCount; n++)
      if (aServerGroup[n].name
          && (p = locate_server(aServerGroup[n].s, name, port)))
        return p;
  }
  return NULL;
}

int remove_server_from_tree(int i, const char *servname, u_short port)
{
  Server *server;

  server = search_server_tree_byname(i, servname, port);
  if (!server)
    return -1;
  else
  {
    int found;

    if (search_winlist_byservname(server->name, server->port, &found))
      return 0;
    close_server_connection(server->g, &server->fd);
    del_server(&(aServerGroup[server->g].s), server, BY_NAME);
  }
  return 1;
}

static void left_rotate(pServer * pp)
{
  pServer p = *pp, r;

  *pp = r = p->right;
  p->right = r->left;
  r->left = p;
  p->bf--;
  if (r->bf > 0)
    p->bf -= r->bf;
  r->bf--;
  if (p->bf < 0)
    r->bf += p->bf;
}

static void right_rotate(pServer * pp)
{
  pServer p = *pp, l;

  *pp = l = p->left;
  p->left = l->right;
  l->right = p;
  p->bf++;
  if (l->bf < 0)
    p->bf -= l->bf;
  l->bf++;
  if (p->bf > 0)
    l->bf += p->bf;
}

int insert_server(pServer * pp, Server * s, int flag)
{
  int dH = 0;
  pServer p = *pp;
  int tt = 0;

  assert(s && s->name && (flag == BY_FD || flag == BY_NAME));
  assert(!(flag == BY_FD && s->fd < 0));
  if (p == NULL)
  {

    *pp = p = (pServer) my_malloc(sizeof(Server));
    if (flag == BY_FD)
    {
      *p = *s;
      p->port = s->port;
      p->g = s->g;
      p->fd = s->fd;
      p->name = strdup(s->name);
      p->left = p->right = NULL;
      p->bf = 0;
      return 1;
    }
    p->g = VALID_GROUP(s->g) ? s->g : new_group(DEFAULT_GRP);
    p->name = strdup(s->name);
    p->alias = s->alias ? strdup(s->alias) : NULL;
    p->nick = s->nick ? strdup(s->nick) : strdup("spxii");
    p->iNick = s->iNick;
    if (p->iNick < 0)
    {
      Nickcount i = get_list_bylabel("default");

      if (i < 0)
        i = new_nicklist("default");
      p->iNick = i;
      aNickList[p->iNick].current = add_nick_to_list(p->iNick, p->nick);
    }
    p->email = s->email ? strdup(s->email) : strdup("spx@pope.va");	//wont  happen

    p->ircname =
       s->ircname ? strdup(s->ircname) : strdup(sula_NAME " " sula_VERSION);
    p->mode = NULL;
    p->lag = NULL;
    p->flag = 0;
    p->connecting = 0;
    p->t.tv_sec = p->t.tv_usec = 0;
    p->port = s->port;
    p->state = s->state;
    p->w = s->w;
    if (s->password)
      p->password = strdup(s->password);
    else
      p->password = NULL;
    p->fd = s->fd;
    p->lastInvite = NULL;
    p->read = 0;
    p->wrote = 0;
    p->connect_time = 0;
    p->bf = 0;
    p->left = p->right = NULL;
    // assign notify group here at the end
    if (s->notify == NULL)
    {
      p->notify = malloc(sizeof(Notify_t));
      *p->notify = NOTIFY_EOL;
    }
    else
      copy_notify(&p->notify, s->notify);

    dH = 1;
  }
  else if (
           (flag == BY_NAME && ((tt = strcasecmp(s->name, p->name)) > 0 ||
                                (tt == 0 && s->port > p->port))) ||
           (flag == BY_FD && s->fd > p->fd))
  {
    if (insert_server(&p->right, s, flag))
    {
      p->bf++;
      if (p->bf == 1)
        dH = 1;
      else if (p->bf == 2)
      {
        if (p->right->bf == -1)
          right_rotate(&p->right);
        left_rotate(pp);
      }
    }
  }
  else if ((flag == BY_NAME && (tt < 0 || (tt == 0 && s->port < p->port))) ||
           (flag == BY_FD && s->fd < p->fd))
  {
    if (insert_server(&p->left, s, flag))
    {
      p->bf--;
      if (p->bf == -1)
        dH = 1;
      else if (p->bf == -2)
      {
        if (p->left->bf == 1)
          left_rotate(&p->left);
        right_rotate(pp);
      }
    }
  }
  else
  {
    if (flag == BY_FD)
    {
      assert(s->fd != p->fd);   //two servers have same fd
    }
  }
  return dH;
}

static int del_server(pServer * pp, Server * s, int flag)
{
  pServer p = *pp, *q;
  int dH = 0;
  int tt = 0;

  if (!p)
    return 0;
  if ((flag == BY_FD && s->fd < p->fd) ||
      (flag == BY_NAME && (
                           (tt = strcasecmp(s->name, p->name)) < 0 ||
                           (tt == 0 && s->port < p->port))))
  {
    if (del_server(&p->left, s, flag))
    {
      p->bf++;
      if (p->bf == 0)
        dH = 1;
      else if (p->bf == 2)
      {
        if (p->right->bf == -1)
          right_rotate(&p->right);
        left_rotate(pp);
        if (p->bf == 0)
          dH = 1;
      }
    }
  }
  else if ((flag == BY_FD && s->fd > p->fd) ||
           (flag == BY_NAME && (tt > 0 || (tt == 0 && s->port > p->port))))
  {
    if (del_server(&p->right, s, flag))
    {
      p->bf--;
      if (p->bf == 0)
        dH = 1;
      else if (p->bf == -2)
      {
        if (p->left->bf == 1)
          left_rotate(&p->left);
        right_rotate(pp);
        if (p->bf == 0)
          dH = 1;
      }
    }
  }
  else
  {                             // found 

    if (p->right == NULL)
    {
      *pp = p->left;
      if (flag == BY_FD)
      {
        free(p->name);
        free(p);
        p = NULL;
      }
      else
      {
        free(p->name);
        if (p->alias)
          free(p->alias);
        free(p->nick);
        free(p->ircname);
        free(p->mode);
        free(p->password);
        free(p->notify);
        free(p);
        p = NULL;
      }
      return 1;
    }
    else if (p->left == NULL)
    {
      *pp = p->right;
      if (flag == BY_FD)
      {
        free(p->name);
        free(p);
        p = NULL;
      }
      else
      {
        free(p->name);
        if (p->alias)
          free(p->alias);
        free(p->nick);
        free(p->ircname);
        free(p->mode);
        free(p->password);
        free(p->notify);
        free(p);
        p = NULL;
      }
      return 1;
    }
    else
    {
      Server *foo = my_malloc(sizeof(Server));
      struct TServer *left, *right;

      q = &p->left;
      while ((*q)->right != NULL)
        q = &(*q)->right;

      memcpy((void *) foo, p, sizeof(Server));

      left = p->left;
      right = p->right;
      memcpy((void *) p, (*q), sizeof(Server));
      p->left = left;
      p->right = right;

      left = (*q)->left;
      right = (*q)->right;
      memcpy((void *) (*q), foo, sizeof(Server));
      (*q)->left = left;
      (*q)->right = right;

      if (del_server(&p->left, foo, flag))
      {
        p->bf++;
        if (p->bf == 0)
          dH = 1;
        else if (p->bf == 2)
        {
          if (p->right->bf == -1)
            right_rotate(&p->right);
          left_rotate(pp);
          dH = 1;
        }
      }
      free(foo);
    }
  }
  return dH;
}

static void RemoveChannels(Server *);
static void RemoveCommands(Server *);
static void update_connection(Server * server, int update_wins)
{
  del_server(&servFdTree, server, BY_FD);
  if(update_wins) UpdateWindows(server->fd, 0);
  ClearLag(server);
  server->fd = -1;
  server->flag &= ~SET_AWAY;
  if (server->connecting > 0)
    server->connecting = 0;     /* dont reset if <0 : ext proc. */
  RemoveCommands(server);
  RemoveChannels(server);
  remove_sent_notifies(server);
  redraw_server_entry(server->g);
}

#if USE_XFORMS
static void ircserver_cb(int fd, void *data)
#else
static void ircserver_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
{
#define SERV_RD_MAX 1024
  char line[SERV_RD_MAX + 1];
  int n;

#ifndef SA_RESTART
retry:
#endif
  if ((n = read(fd, line, SERV_RD_MAX)) < 0)
  {
    if (errno != EAGAIN)
#ifndef SA_RESTART
      if (errno == EINTR)
        return;                 // actually should be: goto retry; 

      else
#endif
      {
        Winstruct *win;

        spx_remove_io_callback(fd, SPX_READ, ircserver_cb);
        close(fd);
        win = search_winlistbyfd(fd, &n);
        assert(win != NULL);
        if (!check_msg_hook(SERVER_LOST, win - winstruct, "%s,%hu %s %s",
                            win->server->name, win->server->port,
                            win->server->nick,
                            win->server->alias ? win->server->alias : win->
                            server->name))
          say2(0, -1, 1,
               PROMPT "Read error from port %hu of server %s.Closing...",
               win->server->port, win->server->name);
        update_connection(win->server, 1);
      }
  }
  else if (n == 0)
  {
    Winstruct *win;

    spx_remove_io_callback(fd, SPX_READ, ircserver_cb);
    close(fd);
    win = search_winlistbyfd(fd, &n);
    assert(win);
    if (!check_msg_hook(SERVER_LOST, win - winstruct, "%s,%hu %s %s",
                        win->server->name,
                        win->server->port, win->server->nick,
                        win->server->alias ? win->server->alias : win->
                        server->name))
      say2(0, -1, 1, PROMPT "Connection closed from %s [%hu]",
           win->server->name, win->server->port);
    update_connection(win->server, 1);
  }
  else
  {
    line[n] = '\0';
    prepare_server_output(fd, line);	/* line is trashed after this */
  }
}
/*

   C o n n e c t   t o   a   s e r v e r           

 */

extern void connecting(int g, const char *server, u_short port, char done)
{
// to call before and after successfull connection to server
#if USE_XFORMS
  WinList *p;

  build_winlistbyservname(server, port);
  p = servWin;
  while (p)
  {

    Winstruct *win = winstruct + p->w;

    if (!done)
    {

      fl_deactivate_object(win->chanwin->close_con);
      fl_set_object_color(win->chanwin->close_con, FL_INACTIVE_COL, FL_MCOL);
    }
    else
    {
      fl_activate_object(win->chanwin->close_con);
      fl_set_object_color(win->chanwin->close_con, FL_COL1, FL_MCOL);

    }
    p = p->next;
  }
#endif
  if (done)
    redraw_server_entry(g);
}

typedef struct
{
  int g;
  char *name;                   //server name
  u_short port;
  int win;
} Hp_struct;
extern int RemoveFromWinChanList(Winstruct *, const char *);

#if USE_XFORMS
static void gethp_cb(int fd, void *data)
#else
static void gethp_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
{
#define MAXRD	255
  char line[MAXRD + 1];
  int n;
  Hp_struct *hp = data;
  Winstruct *win;
  Server *s;

  n = read(fd, line, MAXRD);
  spx_remove_io_callback(fd, SPX_READ, gethp_cb);
  close(fd);
  s = search_server_tree_byname(hp->g, hp->name, hp->port);
  win = (hp->win > -1 && winstruct[hp->win].chanwin) ?
     &winstruct[hp->win] : NULL;
  if (s == NULL)
  {                             /* server's beeen removed in the meantime!!!! */
    fprintf(stderr, "server %s (%hu) is gone?\n", hp->name, hp->port);
    free(hp->name);
    free(data);
    return;
  }
  free(hp->name);
  free(data);
  assert(s->connecting == -1);
  s->connecting = 0;
  if (win)
  {
#if USE_XFORMS
    if (win->chanwin->close_con->u_vdata)
      win->chanwin->close_con->u_vdata = NULL;
#else
    if (gtk_object_get_data(GTK_OBJECT(win->chanwin->close_con),
                            "connecting"))
      gtk_object_set_data(GTK_OBJECT(win->chanwin->close_con), "connecting",
                          0);
#endif
    else
      /* another server has connected using win */
      return;
  }
  if (n < 1 || line[0] == '_')
  {

    if (!check_msg_hook(SERVER_FAILED, win ? win - winstruct : -1,
                        "%s,%hu %s", s->name, s->port, s->nick))
    {
      char *buf;

      if (n > 0)
      {
        buf = alloca(sizeof(char) * (strlen(s->name) + n + 40));

        line[n] = 0;
      }
      else
        buf = alloca(sizeof(char) * (strlen(s->name) + 40));

      sprintf(buf, "Unable to determine host for %s: %s",
              s->name, n > 1 ? &line[1] : "");
      if (win)
        fit_object_label(win->chanwin->WinStatusLine, buf);
      else if (debug)
        error(0, "%s", buf);
      if (server_tool)
      {
        fit_object_label(server_tool->status, buf);
        redraw_server_entry(s->g);
      }
    }
  }
  else
  {
    line[n] = '\0';
    if (debug)
    {
      char *buf = alloca(sizeof(char) * (strlen(s->name) + n + 50));

      sprintf(buf, "Got hostname for %s: %s. Trying to connect...",
              s->name, line);
      fit_object_label(win->chanwin->WinStatusLine, buf);
      spx_xflush();
      if (server_tool)
        fit_object_label(server_tool->status, buf);
      error(0, "%s", buf);
    }
    if (inet_addr(line) == INADDR_NONE)
    {                           /*a bug somewhere... */
      error(0, "Bad IP numbers: %s. Stop.", line);
      line[0] = 0;
    }
    connect_to_server(s, win, line[0] ? line : NULL, 1);
  }
}
static jmp_buf env;
static void sig_alrm(int signo)
{
  longjmp(env, 1);
}

static int gethostname_for_connection(Server * s, Winstruct * win)
{
/* determine server IP number */
  int fd[2], pid;

  if (win)
  {
    fit_object_label(win->chanwin->WinStatusLine, "Resolving server name...");
    spx_xflush();
  }

  if (debug)
    if (gflags & VERBOSE_CLIENT)
      error(ERR_MSG, "Starting process for remote hostname %s", s->name);
  if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd) < 0)
  {
    error(ERR_SYS, "socketpair error");
    return -1;
  }
  if ((pid = fork()) < 0)
  {
    error(ERR_SYS, "fork error");
    return -1;
  }
  else if (pid > 0)
  {
    Hp_struct *hp = malloc(sizeof(Hp_struct));

    if (debug)
    {
      if (gflags & VERBOSE_CLIENT)
        error(ERR_MSG, "Timeout set to %d secs (/set tmout_connect)."
              " Bitte etwas Geduld...", connect_timeout);
      if (sflags & EXTERNALGETHOST)
        error(ERR_MSG, "[%s] pid %d", gethostbynameprogram, pid,
              connect_timeout);
    }
    hp->name = strdup(s->name);
    hp->port = s->port;
    hp->win = -1;
    hp->g = s->g;
    if (win)
    {
      hp->win = win - winstruct;
#if USE_XFORMS
      win->chanwin->close_con->u_vdata = (void *) &pid;
      /*anything that'll let us know that we are connecting */
#else
      gtk_object_set_data(GTK_OBJECT(win->chanwin->close_con),
                          "connecting", &win_count);
#endif
    }
    close(fd[1]);
    spx_add_io_callback(fd[0], SPX_READ, gethp_cb, hp);
    return 0;
  }
  else
  {
    int res = -1;

    close(fd[0]);
    if (fd[1] != STDOUT_FILENO)
      if (dup2(fd[1], STDOUT_FILENO) != STDOUT_FILENO)
        _exit(-1);
    if (sflags & EXTERNALGETHOST)
    {
      char buf2[12];
      char *prog = strrchr(gethostbynameprogram, '/');

      if (!prog || !(*++prog))
        prog = gethostbynameprogram;
      sprintf(buf2, "%u", connect_timeout);
      res = execlp(gethostbynameprogram, prog, "-t", buf2, s->name, NULL);
    }
    if (res == -1)
    {
      // now try again
      struct hostent *hp;

      if (sflags & EXTERNALGETHOST)
      {
        perror(gethostbynameprogram);
        if (debug)
          fprintf(stderr, "Trying again with gethostbyname.\n");
      }
      if (connect_timeout)
      {
        if (signal(SIGALRM, sig_alrm) == SIG_ERR)
        {
          printf("_signal: %s", strerror(errno));
          fflush(stdout);
          _exit(-1);
        }
        if (setjmp(env) != 0)
        {
          fprintf(stdout, "_timeout (%d secs)", connect_timeout);
          fflush(stdout);
          _exit(-1);
        }
        alarm(connect_timeout);
      }
      hp = gethostbyname(s->name);
      if (connect_timeout)
      {
        int errsave = errno;

        alarm(0);
        errno = errsave;
      }
      if (hp)
        printf("%s", inet_ntoa(*((struct in_addr *) *hp->h_addr_list)));
      else
      {
#ifdef NETDB_INTERNAL
        if (h_errno == NETDB_INTERNAL)
          printf("_gethostbyname: %s", strerror(errno));
        else
#endif
          printf("_gethostbyname: %s", hstrerror(h_errno));
      }
      fflush(stdout);
      _exit(0);
    }
    return 0;
  }
}
static void rm_cmdoncon(Server * server)
{
/* remove all commands to be done upon connecting to this server */
  CmdOnCon *p = cmdonconStart;

  while (p != cmdonconEnd)
    if ((!p->server && (p->port == server->port || p->port < 1)) ||
        (!strcasecmp(p->server, server->name) &&
         (p->port == server->port || p->port < 1)))
    {
      CmdOnCon *q = p->next;

      free(p->cmd);
      free(p->server);
      if (q == cmdonconEnd)
        cmdonconEnd = p;
      else
        *p = *q;
      free(q);
    }
    else
      p = p->next;
}
static int do_connect(Server * s, Winstruct * win, const char *ip_addr)
{
  char *win_title = 0;
  char line[512];
  int fd;
  int new_win;
  int n, err_no;
  struct linger l;
  Server *server;

  server = search_server_tree_byname(s->g, s->name, s->port);
  assert(server);
  if (debug)
    error(ERR_MSG, "Connecting to %s. Bonne chance.",
          ip_addr ? ip_addr : server->name);
  fd = tcp_open(ip_addr ? (char *) ip_addr : server->name,
                0, NULL, server->port, connect_timeout);
  if (fd < 0)
  {
    err_no = errno;
    say2(1, -1, 1,
         "@C1@v@." PROMPT "[%s %hu] Connection failed. Diagnostics:",
         server->name, server->port);
    say0(tcp_err_msg, -1, 1);
    rm_cmdoncon(server);
    errno = err_no;
    return -3;
  }
  server->fd = fd;
  if (server->alias)
    free(server->alias);
  server->alias = strdup(server->name);
  insert_server(&servFdTree, server, BY_FD);
  assert(search_server_tree_byfd(server->fd));
  if (debug)
    error(ERR_MSG, "Connected to port %hu of %s (fd=%d)", server->port,
          server->name, server->fd);
  if (fcntl(server->fd, F_SETFL, O_NONBLOCK) == -1)
  {
    err_no = errno;
    error(ERR_SYS, "F_SETFL[O_NONBLOCK]: fcntl error");
    close(server->fd);
    rm_cmdoncon(server);
    errno = err_no;
    return -4;
  }
  l.l_onoff = 0;
  l.l_linger = 0;
  if (setsockopt(server->fd, SOL_SOCKET, SO_LINGER, (void *) &l, sizeof(l)) <
      0)
    error(ERR_SYS, "(ignored) SO_LINGER- setsockopt error");
  new_win = win ? 0 : 1;
  n = 0;
  if (!win)
  {
    win = search_winlist_byservname(server->name, server->port, &n);
    if (!win)
      win = new_chan_win();
  }
  win->server = server;
  TRIGGER_OBJECT(win->chanwin->lead);
  if (win->server->state < 0)
  {
    win_title = malloc(sizeof(char) *
                       (strlen(win->server->name) +
                        strlen(sula_NAME " " sula_VERSION) + 24));

    sprintf(win_title, "[%d] " sula_NAME " " sula_VERSION ": %s %d",
            win - winstruct, win->server->name, win->server->port);
    if (new_win)
      show_chanwin(win, &win_title, 1, 0);
    else
    {
      SET_WINDOW_TITLE(win->chanwin->chanwin, win_title);
      free(win_title);
    }
    win->server->state = 1;
  }
  spx_add_io_callback(win->server->fd, SPX_READ, ircserver_cb, 0);

  sprintf(line, "NICK %s\n", win->server->nick);
  n = strlen(line);
  if (writen(win->server->fd, line, n) != n)
    goto do_error;
  if (win->server->password)
  {
    sprintf(line, "PASS %s\n", win->server->password);
    n = strlen(line);
    if (writen(win->server->fd, line, n) != n)
      goto do_error;
  }
  if (server->email)
  {
    char *name = strxstr(win->server->email, "@");
    char *hhost = strchr(win->server->email, '@');

    sprintf(line, "USER %s %s %s :%s\n", name ? name : Name,
            (hhost && *(++hhost)) ? hhost : host,
            win->server->name, win->server->ircname ?
            win->server->ircname : sula_NAME " " sula_VERSION);
    if (name)
      free(name);
  }
  else
    sprintf(line, "USER %s %s %s :%s\n", Name, host,
            server->name,
            server->ircname ? server->ircname : sula_NAME " " sula_VERSION);
  n = strlen(line);
  if ((n = writen(win->server->fd, line, n)) != n)
    goto do_error;
  fit_object_label(win->chanwin->WinStatusLine,
                   "Server contacted. Wait for confirmation");
  win->server->connecting = 1;
  win->server->connect_time = time(NULL);
  gettimeofday(&win->server->t, NULL);
  send_lag_junk(win->server, 1, 0);
  if (sflags & SUPPRESS_MOTD)
    win->server->flag |= GOT_MOTD;
  redraw_server_entry(win->server->g);
  if (new_win)
    show_chanwin(win, &win_title, 0, 1);
  return 0;

do_error:
  {
    err_no = errno;
    if (win)
      fit_object_label(win->chanwin->WinStatusLine, strerror(errno));
    perror("write error");
    close_server_connection(win->server->g, &win->server->fd);
    errno = err_no;
    return (-5);
  }
}
static int make_connection(Server * s, Winstruct *, const char *ip_addr);

extern int connect_to_server(Server * s, Winstruct * win,
                             const char *ip_addr, int have_tried)
{
  assert(s && s->name);
  if (ip_addr || have_tried)
    return make_connection(s, win, ip_addr);
  else
  {
  Server *server = find_connection(s->name, s->port);
  if (server)
  {
    say2(0, win ? win - winstruct : -1, 1,
         "already connected to %s (%hu)", server->name, server->port);
    return -1;
  }
  server = search_server_tree_byname(s->g, s->name, s->port);
  if (server)
  {
    if (server->connecting)
    {
      say2(0, win ? win - winstruct : -1, 1,
           "already resolving server name on this window:"
           "%s (%hu). Wait or use another window.", server->name,
           server->port);
      return -2;
    }
  }
  else
  {
    int g = new_group(DEFAULT_GRP);

    insert_server(&(aServerGroup[g].s), s, BY_NAME);
    server = search_server_tree_byname(g, s->name, s->port);
    assert(server);
  }
  if (inet_addr(server->name) == INADDR_NONE)
  {
    server->connecting = -1;
    if (gethostname_for_connection(server, win) == 0)
      return 0;
    else
    {
      server->connecting = 0;
      error(ERR_MSG, "Couldn't get hostname. Connecting to %s (%hu)"
            " but may block", server->name, server->port);
    }
  }
  return make_connection(server, win, ip_addr);
  }
}
static int make_connection(Server * s, Winstruct * win, const char *ip_addr)
{
  long old = 0;
  int no_of_wins = 0, old_fd = -1;
  int g = -1, ret;

  if (win)
  {
    if (win->server)
    {
      old = (long) win->server;
      g = win->server->g;
      search_winlistbyfd(win->server->fd, &no_of_wins);
      old_fd = win->server->fd;
    }
  }
  ret = do_connect(s, win, ip_addr);
  if (ret)
  {
    if (win)
      win->server = (Server *) old;
    if (!check_msg_hook(SERVER_FAILED, win ? win - winstruct : -1,
                        "%s,%hu %s", s->name, s->port, s->nick))
    {
      char *buf = malloc(sizeof(char) * (strlen(s->name) + 255));

      if (ret == 0)
        sprintf(buf, "%s contacted [%hu]", s->name, s->port);
      else
      {
        int err_save = errno;

        if (gflags & BEEP_ERR)
          spx_bell(0);
        if (ret == -1)
          sprintf(buf, "already connected to %s [%hu]", s->name, s->port);
        else if (ret == -3)
          sprintf(buf, "[%s %hu] %s", s->name, s->port, tcp_err_msg);
        else
          sprintf(buf, "[%s %hu] failed: %s", s->name, s->port,
                  strerror(err_save));
      }
      if (win && ret)
        fit_object_label(win->chanwin->WinStatusLine, buf);
      if (server_tool)
        fit_object_label(server_tool->status, buf);
      free(buf);
    }
  }
  else if (win && old)
  {
    if (no_of_wins > 1)
    {
      if (old_fd > -1)
      {
        Channel *current = win->chanStart->right;
        char line[120];
        int n;

        while (current != win->chanStart)
        {
          sprintf(line,
                  "NOTICE %s :" sula_NAME " changing a server. bye.\n",
                  current->name);
          n = strlen(line);
          if (writen(old_fd, line, n) != n)
            goto ende;
          sprintf(line, "PART %s\n", current->name);
          n = strlen(line);
          if (writen(old_fd, line, n) != n)
            goto ende;
          current = current->right;
          continue;
        ende:
          close_server_connection(g, &old_fd);
          break;
        }
      }
    }
    if ((Server *) old != win->server) 
    {
      Server *ss=(Server *) old;
      if(no_of_wins < 2){
        if(old_fd>-1)
         new_close_connection(ss->g, &ss->fd,
                           sula_NAME ": Changing servers", 0);
        ss->state = -1;
      }
      else if(old_fd>-1 && ss->w==win-winstruct)
      {
        int n;
        Winstruct *w=search_winlist_byservname(ss->name, ss->port, &n);
        assert(w);
        TRIGGER_OBJECT(w->chanwin->lead);
      }
    }
  }                             /* else if (win && old) */
  return ret;
}
static void RemoveChannels(Server * server);
static void detach_from_notify_groups(Server * s)
{
  Notify_t *p;

  for (p = s->notify; *p != NOTIFY_EOL; p++)
    if (*p != -1)
      detach_notify_group(s, *p);
}
static void new_close_connection(int g, int *fd, char *reason, int update_win)
{
  /*
     undate_win: update corresponding  windows after 
     closing connection
   */
  Server *s, *server;

  assert(*fd > -1);
  spx_remove_io_callback(*fd, SPX_READ, ircserver_cb);
  s = search_server_tree_byfd(*fd);
  assert(s && s->g == g && s->fd == *fd);
  server = search_server_tree_byname(s->g, s->name, s->port);
  assert(!strcasecmp(server->name, s->name) && server->port == s->port);
  if (reason)
    writestr(server->fd, "QUIT :%s\n", reason);
  else
  {
    char *tmp = expand_quote(quit_text, "", -1);

    writestr(server->fd, "QUIT :%s\n", tmp);
    free(tmp);
  }
  close(server->fd);
  update_connection(server, update_win);
  *fd = -1;
  detach_from_notify_groups(server);
}
void close_server_connectionbyname(int g, const char *name, u_short port)
{
  Server *server;

  server = search_server_tree_byname(g, name, port);

  if (server && server->fd > -1)
    new_close_connection(server->g, &server->fd, NULL, 1);
}

void close_server_connection(int group, int *fd)
{
  Server *s2;

  if (*fd < 0)
    return;
  s2 = search_server_tree_byfd(*fd);
  assert(s2 && s2->fd == *fd);
  new_close_connection(s2->g, fd, NULL, 1);
}
static gChannel *get_nextgchannel(const Server * s)
{
  //search global channel list for  servername after closing connection
  gChannel *p = chanStart;

  if (!s)
    return NULL;
  while (p != chanEnd)
    if (!strcasecmp(p->server, s->name) && s->port == p->port)
      return p;
    else
      p = p->next;
  return NULL;
}
static void RemoveChannels(Server * s)
{
  gChannel *p;
  short i = 0;

  assert(s);
  while ((p = get_nextgchannel(s)))
  {
    gChannel *q;

    free(p->name);
    free(p->server);
    q = p->next;
    if (q == chanEnd)
      chanEnd = p;
    else
      *p = *q;
    free(q);
    i++;
  }
}
static void RemoveCommands(Server * s)
{
/* remove all commands sent to this server */
  CmdSent *p = cmdsentStart, *q;

  while (p != cmdsentEnd)
  {
    if (winstruct[p->w].server == s)
    {
      q = p->next;
      free(p->cmd);
      free(p->arg);
      if (q == cmdsentEnd)
        cmdsentEnd = p;
      else
        *p = *q;
      free(q);
    }
    p = p->next;
  }
}

int sendto_server(Winstruct * win, const char *buf)
{
  if (!win->server || win->server->fd < 0)
  {
    fit_object_label(win->chanwin->WinStatusLine,
                     "You are not connected to a server");
    if (gflags & BEEP_ERR)
      spx_bell(0);
    return -1;
  }
  else if (!check_msg_hook(SEND_TO_SERVER, win - winstruct, "%s,%hu %s",
                           win->server->name, win->server->port, buf))
  {
    int n;

    n = strlen(buf);
    if (writen(win->server->fd, buf, n) != n)
      if (errno != EAGAIN)
      {                         // hmmm..interrupted system calls are repeated.
        // so, where's EAGAIN from? think about it

        say2(0, -1, 1, "Write error [%s %hu]: %s",
             win->server->name, win->server->port, strerror(errno));
        close_server_connection(win->server->g, &win->server->fd);
        return -1;
      }
    win->server->wrote += n;
  }
  return 0;
}

