/***************************************************
  This file is distributed as part of Sula PrimeriX
  (http://members.xoom.com/sprimerix).
****************************************************/

/*
 *  util.c  - basic programming routines
 *
 * (C) 1997-1999 Tano Fotang
 *
 * This software is distributed under the GNU General Public License,
 * either version 2, or any later version. The license is included
 * herein by reference.
 */

#include <stdlib.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/time.h>
#include <pwd.h>
#include <ctype.h>
#include <unistd.h>
#include <string.h>
#include "util.h"
struct sockaddr_in tcp_rem_addr;
char *tcp_err_msg;
Sigfunc *Signal(int signo, Sigfunc * func)
/*
   same as signal(2) but repeats interrupted system calls 
 */
{
  struct sigaction action, old_action;

  action.sa_handler = func;
  sigemptyset(&action.sa_mask);
  action.sa_flags = 0;
  if (signo == SIGALRM)
  {
#ifdef SA_INTERRUPT
    action.sa_flags |= SA_INTERRUPT;
#endif
  }
  else
  {
#ifdef SA_RESTART
    action.sa_flags |= SA_RESTART;
#endif
  }
  if (sigaction(signo, &action, &old_action) < 0)
    return (SIG_ERR);
  return (old_action.sa_handler);
}

int delay(unsigned long microsecs)
{
  struct timeval tv;

  tv.tv_sec = 0;
  tv.tv_usec = microsecs;
  return (select(0, NULL, NULL, NULL, &tv));
}
extern void new_str2args(char *buf, int *argc, char **args, size_t size)
{
  *argc = 0;
  size--;
  while (*buf != '\0' && *argc < size)
  {
    while (*buf == ' ' || *buf == '\t')
      *buf++ = '\0';
    if (*buf)
    {
      *args++ = buf;
      (*argc)++;
    }
    while (*buf != '\0' && *buf != ' ' && *buf != '\t')
      buf++;
  }
  *args = '\0';
}

char *expand_tilde(const char *file)
{
/*
 * expand ~ ina pathname. returns strdup(file) if cant exapnd.
 NULL if malloc error  */

  char *pname = (char *) file;

  if (*pname != '~')
    return strdup(pname);
  pname++;
  if (*pname == 0 || *pname == '/')
  {
    char *path;
    char *fullpath;

    if (!(path = getenv("HOME")))
      return strdup(file);
    fullpath =

       (char *) malloc(sizeof(char) * (strlen(path) + strlen(pname) + 8));

    if (!fullpath)
      return NULL;
    sprintf(fullpath, "%s%s", path, pname);
    return fullpath;
  }
  else
  {
    struct passwd *pwd;
    char *pos = strchr(pname, '/');

    if (!pos)
    {                           /*    ~tt   */
      if (!(pwd = getpwnam(pname)))
        return strdup(file);
      return strdup(pwd->pw_dir);
    }
    else
    {
      /*
         ~tt/ttt  
       */
      char *tmp = alloca(sizeof(char) * (pos - pname + 1));

      strncpy(tmp, pname, pos - pname);
      *(tmp + (pos - pname)) = 0;
      pwd = getpwnam(tmp);
      if (!pwd)
        return strdup(file);
      else
      {
        char *fullpath;
        fullpath = (char *) malloc(sizeof(char) * (strlen(pwd->pw_dir) +
                                                   strlen(pname) + 3));

        if (!fullpath)
          return NULL;
        sprintf(fullpath, "%s%s", pwd->pw_dir, pos);
        return fullpath;
      }
    }
  }
}

void unspace(const char *s)
{
/*
   remove leading spaces 
 */
  char *p = (char *) s, *q = (char *) s;

  while (isspace(*p))
    p++;
  if (p != q)
    while ((*q = *p))
    {
      q++;
      p++;
    }
}

void unescape(char *p)
{
  while (isgraph(*p))
    p++;
  *p = '\0';
}
void uncomment(char *s, int c)
{
/*
   squeeze out part of s beginning with c 
 */
  while (*s && *s != c)
    s++;
  *s = '\0';
}

void lowercase2(char *s)
{
  while ((*s = tolower(*s)))
    s++;
}

void uppercase2(char *s)
{
  while ((*s = toupper(*s)))
    s++;
}

size_t find_pos(const char *str, unsigned int n)
{
/*
   return start pos of (n)th word. words are seperated by
   blanks.
   n=0 ->1st word.
   return value:
   strlen(s)-1 if n>number of words in s 
   >=0: start pos 
 */
  register int i;
  register char *s = (char *) str;

  while (*s == ' ' || *s == '\t')
    s++;
  i = 0;                        // positioned on 1st word

  while (*s && i < n)
  {
    while (*s && (*s != ' ' && *s != '\t'))
      s++;
    if (*s)
      i++;
    while (*s == ' ' || *s == '\t')
      s++;
  }
  return (s - str);
}
char *nextword(const char *str, unsigned int n)
{
/*
 * get nth word of a string .first word: n=0 */
  register int i;
  register char *s = (char *) str, *ptr;
  char *buf;

  while (*s == ' ' || *s == '\t')
    s++;
  i = 0;                        // positioned on 1st word

  while (*s && i < n)
  {
    while (*s && (*s != ' ' && *s != '\t'))
      s++;
    if (*s)
      i++;
    while (*s == ' ' || *s == '\t')
      s++;
  }
  ptr = s;                      //start of nth word

  while (*s && (*s != ' ' && *s != '\t'))
    s++;
  i = s - ptr;
  buf = (char *) calloc(sizeof(char), (1 + i));

  if (buf == NULL)
    return NULL;
  memcpy(buf, ptr, i);
  /*(buf + i) = 0; */
  return buf;
}

void rmstr(const char *str, unsigned int n)
{
/*
   delete nth word (n>=0) from str 
 */
  if ((str && *str))
  {
    register int i;
    char *s = (char *) str, *ptr;

    i = 0;                      // positioned on str

    while (*s && i < n)
    {
      while (*s && (*s != ' ' && *s != '\t'))
        s++;
      i++;
      if (i == n)
        break;
      while (*s == ' ' || *s == '\t')
        s++;
    }
    ptr = s;
    // ptr is the start of blanks before nth word
    while (*s == ' ' || *s == '\t')
      s++;                      // remove leading blanks

    while (*s && (*s != ' ' && *s != '\t'))
      s++;                      // remove word

    if (n == 0)
      while (*s == ' ' || *s == '\t')
        s++;                    // remove leading blanks

    if (*ptr)
      while ((*ptr = *s))
      {
        ptr++;
        s++;
      }
  }
}

extern char *strdup2(const char *str)
{
  char *tmp = NULL;

  if (str == NULL || !(*str))
    return NULL;
  tmp = (char *) malloc(sizeof(char) * (strlen(str) + 1));

  if (tmp == NULL)
    return NULL;
  strcpy(tmp, str);
  return tmp;
}

extern char *strcp(const char *s, size_t n)
{
/*
 * return first n letters of s */

  if (!s)
    return NULL;
  if (n > strlen(s))
    return strdup(s);
  else
  {
    char *buf;
    buf = malloc(sizeof(char) * (n + 1));

    if (buf == NULL)
      return NULL;
    strncpy(buf, s, n);
    *(buf + n) = '\0';
    return buf;
  }
}

char *strxstr(const char *s1, const char *s2)
/*
 * return s1 up to and excluding s2 */
{
  char *t;

  if (!s1 || !(t = strstr(s1, s2)))
    return NULL;
  else
  {
    size_t len = t - s1;
    char *tmp = malloc(sizeof(char) * (len + 1));

    memcpy(tmp, s1, len);
    *(tmp + len) = 0;
    return tmp;
  }
}

char *stristr(const char *s1, const char *s2)
/*
 * return malloc'ed space containing s1 up to and including s2 */
{
  char *t;

  if (!s1 || *s1 == 0)
    return NULL;
  if ((t = strstr(s1, s2)) == NULL)
    return strdup(s1);
  else
  {
    char *tmp;

    tmp = strdup(s1);
    *(tmp + strlen(s2) + (t - s1)) = 0;
    return tmp;
  }
}

char *strsstr(const char *s1, const char *s2)
/*
 * return s1 as from and excluding s2 */
{
  char *t;

  if (!s1 || !(*s1))
    return NULL;
  t = strstr(s1, s2);
  if (t)
    return strdup2(t + strlen(s2));
  else
    return NULL;
}
#if 1
char *clean(const char *s)
{
/*
 * remove tabs n white spaces from start of s and return copy */

  if (!s || !(*s))
    return NULL;
  else
  {

    char *p = (char *) (s);

    while (*p == ' ' || *p == '\t')
      p++;
    return *p ? strdup(p) : NULL;
  }
}
#endif
char *lowercase(const char *s)
/*
 * convert string to lowercase and return a copy..   */
{
  char *t;
  t = (char *) malloc(sizeof(char) * (strlen(s) + 1));

  if (!t)
    return NULL;
  else
  {
    char *p = t;

    while ((*p = tolower(*s)))
    {
      p++;
      s++;
    }
    return t;
  }
}

char *uppercase(const char *s)
/*
 * return a copy after converting string to lowercase..   */
{
  char *t;
  t = (char *) malloc(sizeof(char) * (strlen(s) + 1));

  if (!t)
    return NULL;
  else
  {
    char *p = t;

    while ((*p = toupper(*s)))
    {
      p++;
      s++;
    }
    return t;
  }
}

char *strdchr(const char *t, int c)
{
  // remove all c from t 
  char *ptr;
  char *s;

  if (!t)
    return NULL;
  s = (char *) malloc(sizeof(char) * (strlen(t) + 1));

  if (!s)
    return NULL;
  ptr = s;
  while (*t)
  {
    if (*t != c)
      *ptr++ = *t;
    t++;
  }
  *ptr = 0;
  return s;
}
/*/ ////////////// read entries from a file ////////// */
static char *get_line(FILE * fp, char *buf, int size)
{
  while (fgets(buf, size, fp))
  {
    UNSPACE(buf);
    if (*buf == 0 || *buf == COMMENT)
      continue;
    else
    {
      register char *s = buf;

      while (*s && *s != COMMENT)
        s++;
      if (*s == 0 && *(s - 1) == '\n')
        s--;
      *s = 0;
      if (*buf == 0)
        continue;
    }
    return buf;
  }
  return NULL;
}

char *get_string_entries(FILE * fp, int *count, char **d, size_t size,
                         int multiple)
{
#define LINE_LEN  512
  char buf[LINE_LEN], label[255], label2[255];
  char got_tail = 0, first_run = 1;

  *count = 0;
  label[0] = 0;
  *d = '\0';
again:
  while (get_line(fp, buf, LINE_LEN))
    if (buf[0] == '<')
    {
      register char *p;

      if (sscanf(buf, "<%[^>]", label) < 1)
        goto again;
      sprintf(label2, "</%s>", label);
    work:
      p = strstr(buf, label2);
      if (p)
      {
        *p = '\0';
        got_tail = 1;
      }
      p = buf;
      if (first_run)
      {
        first_run = 0;
        while (*p++ != '>');
      }                         // <test>a b c,d,e 

      while (*p && *count < size)
      {
        if (multiple)           /*multiple entries per line allowed, seperated by blanks, comma or tabs */
        {
          char *pos;

          while (*p == ' ' || *p == ',' || *p == '\t' || *p == '\n'
                 || *p == '\r')
            *p++ = '\0';
          pos = p;
          while (*p && *p != ' ' && *p != ',' && *p != '\t' && *p != '\n'
                 && *p != '\r')
            p++;
          if (p == pos)
            continue;
          *d = malloc(sizeof(char) * (1 + (p - pos)));

          memcpy(*d, pos, p - pos);
          *(*d + (p - pos)) = 0;
          (*count)++;
          d++;
        }
        else
        {
          *d++ = strdup(p);
          (*count)++;
          break;
        }
      }
      if (!got_tail && *count < size && get_line(fp, buf, LINE_LEN))
        goto work;
      *d = '\0';
      return strdup(label);
    }
  return NULL;
}

/*
    - network functions -
   This file also contains modified versions of functions taken from
   W. Richard Steven's books on unix programming.

 */

struct sockaddr_in tcp_rem_addr;
char *tcp_err_msg;

static char *err_str(int flag, const char *fmt, ...)
{
  static char buf[513];
  int err = errno;
  va_list ap;

  va_start(ap, fmt);
  if (vsnprintf(buf, 512, fmt, ap) == -1)
    buf[512] = 0;
  va_end(ap);
  if ((flag == ERR_SYS) && err)
    sprintf(buf + strlen(buf), ": %s", strerror(err));
  errno = err;
  return buf;
}

int tcp_open(char *host,        /* system to connect to */
             unsigned long inetaddr,	/* used if host=NULL */
             char *service,     /* requested service; can be NULL if port>0 */
             int port,          /*
                                   =0: use port # of service 
                                   <0: bind a reserved local port
                                   >0: port # of server
                                 */
             u_int timeout)     /* -unused-timeout in seconds. 0 -> no timeout */
{
  /* return socket descriptor if ok, -1 if error */
  volatile int fd;
  int resvport;
  struct servent *sp;           /*service entry */
  int result;

  tcp_err_msg = NULL;
  memset((void *) &tcp_rem_addr, 0, sizeof(tcp_rem_addr));
  tcp_rem_addr.sin_family = AF_INET;
  if (service)
  {
    if ((sp = getservbyname(service, "tcp")) == NULL)
    {
      tcp_err_msg =
         err_str(ERR_SYS, "tcp_open: unknown service: %s/tcp", service);
      return -1;
    }
    if (port > 0)
      tcp_rem_addr.sin_port = htons(port);
    else
      tcp_rem_addr.sin_port = sp->s_port;
  }
  else
  {
    if (port <= 0)
    {
      tcp_err_msg = err_str
         (ERR_MSG, "tcp_open: must specify either service or port");
      return -1;
    }
    tcp_rem_addr.sin_port = htons(port);
  }
  if (host == NULL)
  {
    if ((tcp_rem_addr.sin_addr.s_addr = htonl(inetaddr)) <= 0)
    {
      tcp_err_msg = err_str
         (ERR_MSG, "tcp_open: must specify either host or inet_addr");
      return -1;
    }
  }
  else
  {
    tcp_rem_addr.sin_addr.s_addr = inet_addr(host);
    if (tcp_rem_addr.sin_addr.s_addr == INADDR_NONE)
    {                           /* -1 */
      struct hostent *hp = NULL;

      errno = 0;
      hp = gethostbyname(host);
      if (hp == NULL)
      {
        switch (h_errno)
        {
           case HOST_NOT_FOUND:
             tcp_err_msg =
                "tcp_open: gethostbyname error (HOST_NOT_FOUND):"
                " Unknown host";
             break;
           case TRY_AGAIN:
             tcp_err_msg = err_str(ERR_SYS,
                                   "tcp_open: gethostbyname error (TRY_AGAIN): "
                                   "Host name lookup failure.");
             break;
           case NO_RECOVERY:
             tcp_err_msg = err_str(ERR_SYS,
                                   "tcp_open: gethostbyname error (NO_RECOVERY):"
                                   " Unexpected server failure.");
             break;
           case NO_DATA:
             tcp_err_msg = err_str(ERR_SYS,
                                   "tcp_open: gethostbyname error (NO_DATA)");
             break;
#ifdef NETDB_INTERNAL
           case NETDB_INTERNAL:
             tcp_err_msg = err_str(ERR_SYS,
                                   "tcp_open: gethostbyname error (NETDB_INTERNAL)");
             break;
#endif
           default:
             tcp_err_msg = err_str(ERR_SYS, "tcp_open: gethostbyname error");
             break;
        }                       /* switch */
        return -1;
      }
      memcpy((void *) &tcp_rem_addr.sin_addr, (void *) hp->h_addr,
             hp->h_length);
    }
  }                             /* host==NULL */
  if (port >= 0)
  {
    if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
      tcp_err_msg = err_str(ERR_SYS, "tcp_open: unable to create TCP socket");
      return -1;
    }
  }
  else if (port < 0)
  {
    resvport = IPPORT_RESERVED - 1;
    if ((fd = rresvport(&resvport)) < 0)
    {
      tcp_err_msg =
         err_str(ERR_SYS, "tcp_open: unable to get reserved TCP port");
      return -1;
    }
  }

#ifndef SA_RESTART
retry:
#endif
  result =
     connect(fd, (struct sockaddr *) &tcp_rem_addr, sizeof(tcp_rem_addr));
  if (result < 0)
  {
#ifndef SA_RESTART
    if (errno == EINTR)
    {
      goto retry;
    }
    else
#endif
    {
      tcp_err_msg = err_str(ERR_SYS, "Unable to connect");
      close(fd);
      return -1;
    }
  }
  return fd;
}

extern int tcp_create_server(int backlog)
{
/* create a server on local host and rreturn fd */

  struct sockaddr_in server_addr;
  int fd;

  tcp_err_msg = NULL;
  if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
  {
    tcp_err_msg =
       err_str(ERR_SYS, "tcp_create_server: unable to create tcp socket");
    return -1;
  }
  memset(&server_addr, 0, sizeof(server_addr));
  server_addr.sin_family = AF_INET;
  server_addr.sin_addr.s_addr = INADDR_ANY;
  server_addr.sin_port = 0;
  if (bind(fd, (struct sockaddr *) &server_addr, sizeof(server_addr)) < 0)
  {
    tcp_err_msg =
       err_str(ERR_SYS, "tcp_create_server: unable to bind local addr");
    close(fd);
    return -1;
  }
  if (listen(fd, backlog) < 0)
  {
    tcp_err_msg = err_str(ERR_SYS, "tcp_create_server: listen error");
    close(fd);
    return -1;
  }
  return fd;
}

int writestr(int fd, const char *fmt, ...)
{
  char *buf;
  int count = 256;
  va_list ap;

  va_start(ap, fmt);
  buf = (char *) malloc(sizeof(char) * count);

  while (buf != NULL)
    if (vsnprintf(buf, count, fmt, ap) == -1)
    {
      count *= 2;
      buf = (char *) realloc(buf, count);
    }
    else
      break;
  va_end(ap);
  if (buf)
  {
    count = writen(fd, buf, strlen(buf));
    free(buf);
  }
  else
    count = 0;
  return count;
}

ssize_t writen(int fd, const void *vptr, size_t n)
{
  size_t nleft;
  ssize_t nwritten;
  const char *ptr;

  ptr = vptr;
  nleft = n;
  while (nleft > 0)
  {
    if ((nwritten = write(fd, ptr, nleft)) <= 0)
      return (nwritten);
    nleft -= nwritten;
    ptr += nwritten;
  }
  return (n);
}
ssize_t readn(int fd, void *vptr, size_t n)
{
  size_t nleft;
  ssize_t nread;
  char *ptr;

  ptr = vptr;
  nleft = n;
  while (nleft > 0)
  {
    if ((nread = read(fd, ptr, nleft)) < 0)
      return (nread);
    else if (nread == 0)        /*EOF */
      break;
    nleft -= nread;
    ptr += nread;
  }
  return (n - nleft);
}

static volatile sig_atomic_t sigflag;
static sigset_t newmask, oldmask, zeromask;
static void sig_usr(int signo)
{
  sigflag = 1;
}
extern int TELL_WAIT(void)
{
  if (Signal(SIGUSR1, sig_usr) == SIG_ERR)
    return -1;
  sigemptyset(&zeromask);
  sigemptyset(&newmask);
  sigaddset(&newmask, SIGUSR1);
  if (sigprocmask(SIG_BLOCK, &newmask, &oldmask) < 0)
    return -1;
  return 0;
}
extern void TELL_CHILD(pid_t pid)
{
  kill(pid, SIGUSR1);
}
extern int WAIT_PARENT(void)
{
  while (sigflag == 0)
    sigsuspend(&zeromask);
  sigflag = 0;
  if (sigprocmask(SIG_SETMASK, &oldmask, NULL) < 0)
    return -1;
  return 0;
}
