/*
   Random number module for Guile and systems with /dev/random.
   Random numbers are generated using the  kernel's  random(4)
   number generator. Longs and shorts can be generated,
   signed and unsigned.

   Last changed Wed Mar 24 04:41:22 MET 1999
     o deallocate memory if read error

   Usage:
   (gs-random[u]l [<count>])
   - generate <count> [un]signed longs using /dev/random
   (gs-urandom[u]l [<count>])
   - generate <count> [un]signed longs using /dev/urandom
   (gs-random[u]s [<count>])
   - generate <count> [un]signed shorts using /dev/random
   (gs-urandom[u]s [<count>])
   - generate <count> [un]signed shorts using /dev/urandom

   (gs-srandom)
   - return a pseudo-random number using srandom(3) and time(NULL).
   (gs-srandom <anything>)
   - return a pseudo-random number using srandom(3) and gettimeofday.tv_usec.

   <count> defaults to 2.

   NOTE:
   According to random(4), calls to /dev/random will block
   while entropy pool is empty. gs-randomXX do indeed block,
   somethings for very long periods!

   Examples:
   (use-modules (math random))
   (gs-urandomul 5)
   ==> (309740961 1108058718 969750695 3167338387 441269510)
   (gs-randoml 5)
   ==> (415662357 76787305 -523869225 -2044569091 -852185446)
   (gs-urandoms 5)
   ==> (16635 -9747 21831 19282 9873)
   (gs-randomus)
   ==> (13218 430)

   Compiling:
   gcc -Wall -fPIC -shared -o librandom.so random.c
   mkdir -p  ~/.sula/trill/math
   cp librandom.so ~/.sula/trill/math


   fotang@yahoo.com, 3/99. http://members.xoom.com/sprimerix/download.

 */

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <stdlib.h>
#include <guile/gh.h>
#include <sys/time.h>
#include <errno.h>

static SCM gs_srandom(SCM use_usec)
{
   if (!gh_null_p(use_usec))
   {
      struct timeval t;

      if (gettimeofday(&t, NULL) == -1)
	 srandom(time(NULL));
      else
	 srandom(t.tv_usec);
   }
   else
      srandom(time(NULL));
   return gh_ulong2scm(random());
}

static SCM res;
static int count;
static void *aN;

static void *do_random(SCM c, size_t size, int secure)
{
   FILE *fp;
   char *dev = secure ? "/dev/random" : "/dev/urandom";

   aN = NULL;
   res = gh_list(SCM_UNDEFINED);
   count = 2;

   if (!gh_null_p(c))
   {
      count = gh_scm2long(gh_car(c));
      if (count < 0)
	 return NULL;
   }
   if (!(fp = fopen(dev, "r")))
   {
      int err_save = errno;
      char buf[32];

      sprintf(buf, "fopen %s", dev);
      errno = err_save;
      scm_syserror(buf);
   }
   aN = malloc(size * (count));
   if (aN == NULL)
   {
      int err_save = errno;

      fclose(fp);
      errno = err_save;
      scm_syserror("malloc");
   }
   if (fread(aN, size, count, fp) != count)
   {
      int err_save = errno;

      fclose(fp);
      free(aN);
      errno = err_save;
      scm_syserror("fread");
   }
   fclose(fp);
   return aN;

}
static SCM do_ulong(void *l)
{
   if (l == NULL)
      return res;
   while (count--)
      res = gh_cons(gh_ulong2scm(((u_long *) l)[count]), res);
   free(l);
   return res;
}
static SCM do_long(void *l)
{
   if (l == NULL)
      return res;
   while (count--)
      res = gh_cons(gh_long2scm(((long *) l)[count]), res);
   free(l);
   return res;
}
static SCM gs_randomul(SCM c)
{
   return do_ulong(do_random(c, sizeof(u_long), 1));
}
static SCM gs_randoml(SCM c)
{
   return do_long(do_random(c, sizeof(long), 1));
}
static SCM gs_urandomul(SCM c)
{
   return do_ulong(do_random(c, sizeof(u_long), 0));
}
static SCM gs_urandoml(SCM c)
{
   return do_long(do_random(c, sizeof(long), 0));
}

static SCM do_ushort(void *l)
{
   if (l == NULL)
      return res;
   while (count--)
      res = gh_cons(gh_ulong2scm(((u_short *) l)[count]), res);
   free(l);
   return res;
}
static SCM do_short(void *l)
{
   if (l == NULL)
      return res;
   while (count--)
      res = gh_cons(gh_long2scm(((short *) l)[count]), res);
   free(l);
   return res;
}

static SCM gs_randomus(SCM c)
{
   return do_ushort(do_random(c, sizeof(u_short), 1));
}
static SCM gs_randoms(SCM c)
{
   return do_short(do_random(c, sizeof(short), 1));
}
static SCM gs_urandomus(SCM c)
{
   return do_ushort(do_random(c, sizeof(u_short), 0));
}
static SCM gs_urandoms(SCM c)
{
   return do_short(do_random(c, sizeof(u_short), 0));
}
void init_math_random()
{
   gh_new_procedure("gs-randoml", gs_randoml, 0, 0, 1);
   gh_new_procedure("gs-randomul", gs_randomul, 0, 0, 1);
   gh_new_procedure("gs-randoms", gs_randoms, 0, 0, 1);
   gh_new_procedure("gs-randomus", gs_randomus, 0, 0, 1);
   gh_new_procedure("gs-urandoml", gs_urandoml, 0, 0, 1);
   gh_new_procedure("gs-urandomul", gs_urandomul, 0, 0, 1);
   gh_new_procedure("gs-urandoms", gs_urandoms, 0, 0, 1);
   gh_new_procedure("gs-urandomus", gs_urandomus, 0, 0, 1);

   gh_new_procedure("gs-srandom", gs_srandom, 0, 0, 1);
}

void scm_init_math_random_module()
{
   scm_register_module_xxx("math random", init_math_random);
}
