/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __DES_HPP__
#define __DES_HPP__

#include <string.h>

#include "globals.hpp"
#include "block.hpp"

class SCL_API CDESKey : public CBlockKey
{
	public:
		enum { KEYSIZE = 8 };

	public:
		CDESKey();
		CDESKey(const CDESKey &);
		CDESKey(const Byte *, Word = KEYSIZE);
		CDESKey(Dword);
		virtual ~CDESKey();

	public:
		Dword GetMaster() const { return dwMaster; };
		Dword GetKey(Word i) const { return dwKey[i]; };

	public:
		virtual Word GetKeySize() { return KEYSIZE; };

	private:
		void MakeKeys();

	private:
		Dword dwMaster;
		Dword dwKey[16];
};

class SCL_API C3DESKey : public CBlockKey
{
	public:
		enum { KEYSIZE = 24 };

	public:
		C3DESKey() { };
		C3DESKey(Dword, Dword, Dword);
		C3DESKey(const Byte *, Word = KEYSIZE);
		virtual ~C3DESKey();
	
	public:
		const CDESKey & GetKey(Word i) const { return cDESKey[i]; };

	public:
		virtual Word GetKeySize() { return KEYSIZE; };

	private:
		CDESKey cDESKey[3];
};

class SCL_API CDESBlock : public CBlockObject
{
	public:
		enum { BLOCKSIZE = 8 };

	public:
		CDESBlock();
		CDESBlock(const CDESBlock &);
		CDESBlock(Dword);
		CDESBlock(const Byte *, Word wLength = BLOCKSIZE);
		virtual ~CDESBlock();

	public:
		virtual void Encrypt(const CBlockKey &cKey) { Encrypt((const CDESKey &)cKey); };
		virtual void Decrypt(const CBlockKey &cKey) { Decrypt((const CDESKey &)cKey); };
		virtual void SetData(const Byte *, Word = BLOCKSIZE);
		virtual Byte *GetData();
		virtual Word GetBlockSize() { return BLOCKSIZE; };

	public:
		void Encrypt(const CDESKey &);
		void Decrypt(const CDESKey &);
		void SetValue(Dword);
		Dword GetValue();
	
	protected:
		void Crypt(const CDESKey &, bool);

	private:
		Dword dwData, dwResult;
};

class SCL_API C3DESBlock : public CDESBlock
{
	public:
		C3DESBlock();
		C3DESBlock(Dword);
		C3DESBlock(const Byte *, Word = BLOCKSIZE);
		virtual ~C3DESBlock() { /* dummy destructor */ };

	public:
		virtual void Encrypt(const CBlockKey &cKey) { Encrypt((const C3DESKey &)cKey); };
		virtual void Decrypt(const CBlockKey &cKey) { Decrypt((const C3DESKey &)cKey); };

	public:
		void Encrypt(const C3DESKey &);
		void Decrypt(const C3DESKey &);
};

#endif // __DES_HPP__
