/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __OBN_HPP__
#define __OBN_HPP__

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "config.h"
#include "globals.hpp"
#include "errors.hpp"
#include "settings.hpp"

#define BLOCK					 128

class SCL_API CBigNumber
{
	public:										// Construction and destruction
		CBigNumber();
		CBigNumber(Word);
		CBigNumber(char *);
		CBigNumber(const CBigNumber &);
		CBigNumber(void *, Word);
		CBigNumber(Word *, Word);
		virtual ~CBigNumber();

	public:
		void Swap(CBigNumber &);
		void SetWords(Word);
		void SetRandom(Word, bool);
		Word GetWords() const { return wWords; }
		Word *GetData() const { return pwBigNumber; }

	public:										// Predefined operators
		CBigNumber operator = (const CBigNumber &);
		CBigNumber operator + (const CBigNumber &) const;
		CBigNumber operator - (const CBigNumber &) const;
		CBigNumber operator * (const CBigNumber &) const;
		CBigNumber operator / (const CBigNumber &) const;
		CBigNumber operator % (const CBigNumber &) const;
		CBigNumber & operator += (const CBigNumber &);
		CBigNumber & operator -= (const CBigNumber &);
		CBigNumber operator / (Word) const;
		CBigNumber operator * (Word) const;
		Word operator % (Word) const;
		CBigNumber operator << (Word) const;
		CBigNumber operator >> (Word) const;
		CBigNumber & operator <<= (Word);
		CBigNumber & operator >>= (Word);
		CBigNumber & operator += (Word);
		CBigNumber & operator -= (Word);
	
		bool operator < (const CBigNumber &) const;
		bool operator <= (const CBigNumber &) const;
		bool operator > (const CBigNumber &) const;
		bool operator >= (const CBigNumber &) const;
		bool operator == (const CBigNumber &) const;
		bool operator != (const CBigNumber &) const;
		bool operator < (Word);
		bool operator <= (Word);
		bool operator > (Word);
		bool operator >= (Word);
		bool operator == (Word);
		bool operator != (Word);

	public:										// Methods
		inline bool IsZero() const { return (wWords == 0) || ((wWords == 1) && (pwBigNumber[0] == 0)); };
		inline bool IsOne() const { return (wWords == 1) && (pwBigNumber[0] == 1); };
		inline bool IsNegative() const { return fgSign; }
		inline bool IsPositive() const { return !fgSign; }
		inline bool IsEven() const { return pwBigNumber[0] & 1 == 1; };
		inline bool IsOdd() const { return pwBigNumber[0] & 1 == 0; };

		void Add(const CBigNumber &);	  	// Standard addition
		void Sub(const CBigNumber &);		// Standard subtraction
		void Div(CBigNumber);				// Division for numbers which are at least from two words
		void Div(Word);
		void Mod(CBigNumber);				// Modulus for numbers which are at least from two words
		Word Mod(Word) const;
		Word Lg() const;
		void Sqr();
	
		void NormalSub(const CBigNumber &);
													// Assert that the subtrahend is < minuend
		void Shr();								// Logical shift right with one bit
		void Shr(Word);
		void Shl();								// Logical shift left with one bit
		void Shl(Word);
		void Neg();								// Negate
	
		Word GetSum();
		void SetCheckSum();
		bool CheckSum();

		int Cmp(const CBigNumber &, const CBigNumber &) const;

		inline void Abs() { fgSign = false; };
		void ModExpWord(Word, const CBigNumber &);
	
		static CBigNumber ModExp(const CBigNumber &, const CBigNumber &, const CBigNumber &);
		static CBigNumber GCD(const CBigNumber &, const CBigNumber &);
		static CBigNumber GCD(CBigNumber, CBigNumber, CBigNumber &, CBigNumber &);
		static CBigNumber LCM(const CBigNumber &, const CBigNumber &);
		static CBigNumber ModInv(const CBigNumber &, const CBigNumber &);
		static int Jacobi(CBigNumber, CBigNumber);

		void Dump() const;
	
	public:
		inline void SetBit(Word wBit) { pwBigNumber[wBit / BITSINWORD] |= (1 << (wBit % BITSINWORD)); };
		inline void ClearBit(Word wBit) { pwBigNumber[wBit / BITSINWORD] &= ((~((Word)1)) << (wBit % BITSINWORD)); };
		inline Word GetBit(Word wBit) { return (pwBigNumber[wBit / BITSINWORD] >> (wBit % BITSINWORD)) & 1; };

	protected:
		void StripLeadingZeroes();

	protected:
		Word *pwBigNumber;
		Word wWords;
		Word wAllocated;
		bool fgSign;
};

#endif // __OBN_HPP__
