/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __RABIN_HPP__
#define __RABIN_HPP__

#include "globals.hpp"
#include "obn.hpp"
#include "primes.hpp"
#include "der.hpp"
#include "public.hpp"

#define RABIN_PRIVATE_KEY_VERSION			0

#define BEGIN_RABIN_PRIVATE_KEY			"-----BEGIN RABIN PRIVATE KEY-----"
#define END_RABIN_PRIVATE_KEY				"-----END RABIN PRIVATE KEY-----"
#define BEGIN_RABIN_PUBLIC_KEY			"-----BEGIN RABIN PUBLIC KEY-----"
#define END_RABIN_PUBLIC_KEY				"-----END RABIN PUBLIC KEY-----"

class SCL_API CRabinKey : public CAsymmetricKey
{
	public:
		CRabinKey();
		CRabinKey(Word);
		CRabinKey(const CRabinKey &);
		CRabinKey(const CBigNumber &);
		CRabinKey(const CBigNumber &, const CProbablePrime &, const CProbablePrime &);
	
	public:
		CBigNumber & GetModulus() { return N; };
		CProbablePrime & GetP() { return P; };
		CProbablePrime & GetQ() { return Q; };
	
	public:
		bool EncryptOnlyFlag() { return fgEncryptOnly; };
		bool HoldKeyFlag() { return fgHoldKey; };
		virtual void WritePrivateKey(int, bool = true);
		virtual void WritePublicKey(int, bool = true);
		virtual void ReadPrivateKey(int, bool = true);
		virtual void ReadPublicKey(int, bool = true);
		void Dump();
		Word Check();
	
	private:
		void GenerateKeys();
	
	private:
		CProbablePrime P, Q;
		CBigNumber N;
		Word wModulusSize;
		bool fgEncryptOnly;
		bool fgHoldKey;
};

class SCL_API CRabinBlock : public CAsymmetricBlock
{
	public:
		CRabinBlock(const CRabinKey &);
		CRabinBlock(const CRabinKey &, void *, Word);
		CRabinBlock(const CRabinKey &, const CBigNumber &);
	
	public:
		virtual void Encrypt();
		virtual void Decrypt();
		virtual void Write(int);							// ciphertext
		virtual void Read(int);								// ciphertext
		virtual void SetData(Byte *, Word);				// plaintext
		virtual Byte *GetData();							// plaintext
		virtual Word GetDataSize();						// plaintext

	public:
		void Dump();

	private:
		CRabinKey cKey;
		CBigNumber cData;
};

#endif // __RABIN_HPP__
