/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __SQUARE_HPP__
#define __SQUARE_HPP__

#include <string.h>

#include "globals.hpp"
#include "block.hpp"

#define SQUAREROUNDS 8

class SCL_API CSquareKey : public CBlockKey
{
	public:
		enum { KEYSIZE = 16 };

	public:
		CSquareKey();
		CSquareKey(const CSquareKey &);
		CSquareKey(const Word *);
		CSquareKey(const Byte *, Word = KEYSIZE);
		virtual ~CSquareKey();

	public:
		Word GetMaster(Word i) const { return wMaster[i]; };
		Word *GetEncryptionKeys() const { return (Word *)wEncryptionKey[0]; };
		Word *GetDecryptionKeys() const { return (Word *)wDecryptionKey[0]; };
		Word GetEncryptionKey(Word wRound, Word wKey) const { return wEncryptionKey[wRound][wKey]; };
		Word GetDecryptionKey(Word wRound, Word wKey) const { return wDecryptionKey[wRound][wKey]; };

	public:
		virtual Word GetKeySize() { return KEYSIZE; };

	private:
		void MakeKeys();

	private:
		Word wMaster[4];
		Word wEncryptionKey[SQUAREROUNDS + 1][4];
		Word wDecryptionKey[SQUAREROUNDS + 1][4];
};

class SCL_API CSquareBlock : public CBlockObject
{
	public:
		enum { BLOCKSIZE = 16 };

	public:
		CSquareBlock();
		CSquareBlock(const CSquareBlock &);
		CSquareBlock(const Word *);
		CSquareBlock(const Byte *, Word = 16);
		virtual ~CSquareBlock();
	
	public:
		void Encrypt(const CSquareKey &);
		void Decrypt(const CSquareKey &);
		void SetData(Word *);
		Word GetData(Word);

	public:
		virtual void Encrypt(const CBlockKey &cKey) { Encrypt((const CSquareKey &)cKey); };
		virtual void Decrypt(const CBlockKey &cKey) { Decrypt((const CSquareKey &)cKey); };
		virtual void SetData(const Byte *, Word = BLOCKSIZE);
		virtual Byte *GetData();
		virtual Word GetBlockSize() { return BLOCKSIZE; };

	private:
		Word wData[4];
		Word wResult[4];
};

#endif // __SQUARE_HPP__
