/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "bg.hpp"

CBGKey::CBGKey()
{
}

CBGKey::CBGKey(Word wBits)
{
	P.SetRandom(wBits / 2, true);
	Q.SetRandom(wBits / 2, true);
	S.SetRandom(wBits, true);

	MakePrimes();
	Initialize();
	CalcBits();
	CBigNumber::GCD(P, Q, A, B);
	
	fgEncryptOnly = false;
	fgHoldKey = true;
}

CBGKey::CBGKey(const CProbablePrime &P,
					const CProbablePrime &Q,
					const CBigNumber &A,
					const CBigNumber &B)
{
	CBGKey::P = P;
	CBGKey::Q = Q;
	CBGKey::A = A;
	CBGKey::B = B;
	
	N = P * Q;
	
	S.SetRandom(N.GetWords() * BITSINWORD, true);

	Initialize();
	CalcBits();

	fgEncryptOnly = false;
	fgHoldKey = true;
}

CBGKey::CBGKey(const CBigNumber &N)
{
	CBGKey::N = N;

	S.SetRandom(N.GetWords() * BITSINWORD, true);

	Initialize();
	CalcBits();

	fgEncryptOnly = true;
	fgHoldKey = true;
}

CBGKey::CBGKey(const CBGKey &cBGKey) : CBlumBlumShubPRNG((const CBlumBlumShubPRNG &)cBGKey)
{
	A = cBGKey.A;
	B = cBGKey.B;

	fgEncryptOnly = false;
	fgHoldKey = true;
}

void CBGKey::Next()
{
	S.Sqr();
	S.Mod(N);
	T += 1;
}

void CBGKey::WritePrivateKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cVersion((Word)BG_PRIVATE_KEY_VERSION);
	CDEREncodedBigNumber cModulus(N);
	CDEREncodedBigNumber cPrime1(P);
	CDEREncodedBigNumber cPrime2(Q);
	CDEREncodedBigNumber cPrivate1(A);
	CDEREncodedBigNumber cPrivate2(B);

	CDEREncodedSequence cSequence;
	
	cSequence.AddPrimitive(cVersion);
	cSequence.AddPrimitive(cModulus);
	cSequence.AddPrimitive(cPrime1);
	cSequence.AddPrimitive(cPrime2);
	cSequence.AddPrimitive(cPrivate1);
	cSequence.AddPrimitive(cPrivate2);
	
	if (true == fgBase64) {
		write_string(iOut, BEGIN_BG_PRIVATE_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_BG_PRIVATE_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CBGKey::WritePublicKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cModulus(N);

	CDEREncodedSequence cSequence;
	
	cSequence.AddPrimitive(cModulus);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_BG_PUBLIC_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_BG_PUBLIC_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CBGKey::ReadPrivateKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_BG_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_BG_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cVersion(cSequence);
	CDEREncodedBigNumber cModulus(cSequence);
	CDEREncodedBigNumber cPrime1(cSequence);
	CDEREncodedBigNumber cPrime2(cSequence);
	CDEREncodedBigNumber cPrivate1(cSequence);
	CDEREncodedBigNumber cPrivate2(cSequence);

	N = cModulus;
	P = cPrime1;
	Q = cPrime2;
	A = cPrivate1;
	B = cPrivate2;

	S.SetRandom(N.GetWords() * BITSINWORD, true);

	Initialize();
	CalcBits();

	fgEncryptOnly = false;
	fgHoldKey = true;
}

void CBGKey::ReadPublicKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_BG_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_BG_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cModulus(cSequence);

	N = cModulus;

	S.SetRandom(N.GetWords() * BITSINWORD, true);

	Initialize();
	CalcBits();

	fgEncryptOnly = true;
	fgHoldKey = true;
}

void CBGKey::Dump()
{
	printf("P = "); P.Dump();
	printf("Q = "); Q.Dump();
	printf("N = "); N.Dump();
	printf("S = "); S.Dump();
	printf("A = "); A.Dump();
	printf("B = "); B.Dump();
}

Word CBGKey::Check()
{
	if (!fgHoldKey || fgEncryptOnly)
		throw(BAD_BG_OPERATION);
	
	Word wResult = BG_OK;
	
	if (!P.IsPrime())
		wResult |= BG_PNOTPRIME;
	if (!Q.IsPrime())
		wResult |= BG_QNOTPRIME;
	if (P % 4 != 3)
		wResult |= BG_PNOTCON;
	if (Q % 4 != 3)
		wResult |= BG_QNOTCON;
	if (N != P * Q)
		wResult |= BG_BADN;
	
	CBigNumber T, S;
	CBigNumber::GCD(P, Q, T, S);

//	if ((T != A) || (S != B))
//		wResult |= BG_BADSEC;
	
	return wResult;
}

CBGBlock::CBGBlock(const CBGKey &cBGKey, void *pvData, Word wDataLength, const CBigNumber &cTemp)
{
	cKey = cBGKey;
	cKey.SetS(cTemp);
	bData = (Byte *)pvData;
	wLength = wDataLength;
}

CBGBlock::CBGBlock(const CBGKey &cBGKey, void *pvData, Word wDataLength)
{
	cKey = cBGKey;
	bData = (Byte *)pvData;
	wLength = wDataLength;
}

CBGBlock::CBGBlock(const CBGKey &cBGKey, const CBigNumber &cBGData, const CBigNumber &cTemp)
{
	cKey = cBGKey;
	cKey.SetS(cTemp);
	cData = cBGData;
	bData = (Byte *)(cData.GetData() + 1);
	wLength = cData.GetData()[0];
}

CBGBlock::CBGBlock(const CBGKey &cBGKey, const CBigNumber &cBGData)
{
	cKey = cBGKey;
	cData = cBGData;
	bData = (Byte *)(cData.GetData() + 1);
	wLength = cData.GetData()[0];
}

CBGBlock::CBGBlock(const CBGKey &cBGKey)
{
	cKey = cBGKey;
}

void CBGBlock::Encrypt()
{
	if (!cKey.HoldKeyFlag())
		throw(BAD_RW_OPERATION);

	for (Word w = 0; w < wLength; w++) {
		if (0 == w % BLOCK)
			cKey.GetRandomData(bBuffer, Min(BLOCK, wLength - w));
		bData[w] ^= bBuffer[w % BLOCK];
	}
	cKey.Next();
}

void CBGBlock::Decrypt()
{
	if (!cKey.HoldKeyFlag() || cKey.EncryptOnlyFlag())
		throw(BAD_RW_OPERATION);
	
	CBigNumber T = (wLength * BITSINBYTE + cKey.GetBits() - 1) / cKey.GetBits() + 1;
	CBigNumber f = cKey.GetP() + 1; f.Shr(2);
	CBigNumber g = cKey.GetQ() + 1; g.Shr(2);
	CBigNumber u = CBigNumber::ModExp(cKey.GetS(), CBigNumber::ModExp(f, T, cKey.GetP() - 1), cKey.GetP());
	CBigNumber v = CBigNumber::ModExp(cKey.GetS(), CBigNumber::ModExp(g, T, cKey.GetQ() - 1), cKey.GetQ());
	cKey.SetS((v * cKey.GetA() * cKey.GetP() + u * cKey.GetB() * cKey.GetQ()) % cKey.GetN());
	for (Word w = 0; w < wLength; w++) {
		if (0 == w % BLOCK)
			cKey.GetRandomData(bBuffer, Min(BLOCK, wLength - w));
		bData[w] ^= bBuffer[w % BLOCK];
	}
}

void CBGBlock::Dump()
{
	for (Word w = 0; w < wLength; w++)
		printf("%02x ", bData[w]);
	printf("\n");
}

void CBGBlock::Write(int iOut)
{
	CBigNumber cData(bData, wLength);
	CDEREncodedBigNumber cData1(cData);
	CDEREncodedBigNumber cData2(cKey.GetS());
	cData1.Write(iOut);
	cData2.Write(iOut);
}

void CBGBlock::Read(int iIn)
{
	CDEREncodedBigNumber cData1;
	CDEREncodedBigNumber cData2;

	cData1.Read(iIn);
	cData2.Read(iIn);

	cData = cData1;
	cKey.SetS(cData2);

	bData = (Byte *)(cData.GetData() + 1);
	wLength = cData.GetData()[0];
}


void CBGBlock::SetData(Byte *pbData, Word wDataLength)
{
	bData = pbData;
	wLength = wDataLength;
}

Byte *CBGBlock::GetData()
{
	return bData;
}

Word CBGBlock::GetDataSize()
{
	return wLength;
}
