/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "dsa.hpp"

CDSAKey::CDSAKey() : fgHoldParams(false), fgHoldKey(false)
{
}

CDSAKey::CDSAKey(Word wPrimeSize) : fgHoldParams(false)
{
	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_DSA_PRIMESIZE);
	CDSAKey::wPrimeSize = wPrimeSize;
	GenerateKeys();
}

CDSAKey::CDSAKey(const CProbablePrime &cPrime,
					  const CProbablePrime &cSubPrime,
					  const CProbablePrime &cGenerator,
					  const CBigNumber &cPublic)
{
	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_DSA_PRIMESIZE);

	CDSAKey::cPrime = cPrime;
	CDSAKey::cSubPrime = cSubPrime;
	CDSAKey::cGenerator = cGenerator;
	CDSAKey::cPublic = cPublic;

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = true;
}

CDSAKey::CDSAKey(const CProbablePrime &cPrime,
					  const CProbablePrime &cSubPrime,
					  const CProbablePrime &cGenerator,
					  const CBigNumber &cPublic,
					  const CBigNumber &cPrivate)
{
	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_DSA_PRIMESIZE);

	CDSAKey::cPrime = cPrime;
	CDSAKey::cSubPrime = cSubPrime;
	CDSAKey::cGenerator = cGenerator;
	CDSAKey::cPublic = cPublic;
	CDSAKey::cPrivate = cPrivate;

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = false;
}

CDSAKey::CDSAKey(const CDSAKey &cDSAKey)
{
	wPrimeSize = cDSAKey.wPrimeSize;
	wSubPrimeSize = cDSAKey.wSubPrimeSize;
	cPrime = cDSAKey.cPrime;
	cSubPrime = cDSAKey.cSubPrime;
	cGenerator = cDSAKey.cGenerator;
	cPublic = cDSAKey.cPublic;
	cPrivate = cDSAKey.cPrivate;

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = false;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_DSA_PRIMESIZE);
}

void CDSAKey::GenerateKeys()
{
	if (false == fgHoldParams) {
		wSubPrimeSize = 160;

		do {
			do {
				cSubPrime.SetRandom(wSubPrimeSize, true);
			} while (!cSubPrime.IsPrime());
			cPrime.SetRandom(wPrimeSize - wSubPrimeSize, false);
			cPrime.ClearBit(0);
			cPrime = cPrime * cSubPrime + 1;
		} while (!cPrime.IsPrime());

		do {
			CBigNumber cTemp;
			cTemp.SetRandom(wPrimeSize, false);

			cGenerator = CBigNumber::ModExp(cTemp, (cPrime - 1) / cSubPrime, cPrime);
		} while (cGenerator <= 1);

		fgHoldParams = true;
	}

	cPrivate.SetRandom(wSubPrimeSize, false);
	cPublic = CBigNumber::ModExp(cGenerator, cPrivate, cPrime);

	fgHoldKey = true;
	fgEncryptOnly = false;
}

void CDSAKey::Dump()
{
	printf("prime = "); cPrime.Dump();
	printf("sub prime = "); cSubPrime.Dump();
	printf("generator = "); cGenerator.Dump();
	printf("public = "); cPublic.Dump();
	printf("private = "); cPrivate.Dump();
}

void CDSAKey::WritePrivateKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cVersion((Word)DSA_PRIVATE_KEY_VERSION);
	CDEREncodedBigNumber cPrime(CDSAKey::cPrime);
	CDEREncodedBigNumber cSubPrime(CDSAKey::cSubPrime);
	CDEREncodedBigNumber cGenerator(CDSAKey::cGenerator);
	CDEREncodedBigNumber cPublic(CDSAKey::cPublic);
	CDEREncodedBigNumber cPrivate(CDSAKey::cPrivate);

	CDEREncodedSequence cSequence;

	cSequence.AddPrimitive(cVersion);
	cSequence.AddPrimitive(cPrime);
	cSequence.AddPrimitive(cSubPrime);
	cSequence.AddPrimitive(cGenerator);
	cSequence.AddPrimitive(cPublic);
	cSequence.AddPrimitive(cPrivate);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_DSA_PRIVATE_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_DSA_PRIVATE_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CDSAKey::WritePublicKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cPrime(CDSAKey::cPrime);
	CDEREncodedBigNumber cSubPrime(CDSAKey::cSubPrime);
	CDEREncodedBigNumber cGenerator(CDSAKey::cGenerator);
	CDEREncodedBigNumber cPublic(CDSAKey::cPublic);

	CDEREncodedSequence cSequence;
	cSequence.AddPrimitive(cPrime);
	cSequence.AddPrimitive(cSubPrime);
	cSequence.AddPrimitive(cGenerator);
	cSequence.AddPrimitive(cPublic);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_DSA_PUBLIC_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_DSA_PUBLIC_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CDSAKey::WriteParams(char *pszFile, bool fgBase64)
{
	int iFile = 1;
	if (NULL != pszFile)
		if (-1 == (iFile = open(pszFile, O_RDWR | O_CREAT, 0644)))
		  throw OPEN_ERROR;
	WriteParams(iFile, fgBase64);
	if (NULL != pszFile)
		close(iFile);
}

void CDSAKey::WriteParams(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cPrime(CDSAKey::cPrime);
	CDEREncodedBigNumber cSubPrime(CDSAKey::cSubPrime);
	CDEREncodedBigNumber cGenerator(CDSAKey::cGenerator);

	CDEREncodedSequence cSequence;
	cSequence.AddPrimitive(cPrime);
	cSequence.AddPrimitive(cSubPrime);
	cSequence.AddPrimitive(cGenerator);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_DSA_PARAMS "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_DSA_PARAMS "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CDSAKey::ReadPrivateKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_DSA_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_DSA_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cVersion(cSequence);
	CDEREncodedBigNumber cPrime(cSequence);
	CDEREncodedBigNumber cSubPrime(cSequence);
	CDEREncodedBigNumber cGenerator(cSequence);
	CDEREncodedBigNumber cPublic(cSequence);
	CDEREncodedBigNumber cPrivate(cSequence);

	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_DSA_PRIMESIZE);

	CDSAKey::cPrime = cPrime;
	CDSAKey::cSubPrime = cSubPrime;
	CDSAKey::cGenerator = cGenerator;
	CDSAKey::cPublic = cPublic;
	CDSAKey::cPrivate = cPrivate;

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = false;
}

void CDSAKey::ReadPublicKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_DSA_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_DSA_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cPrime(cSequence);
	CDEREncodedBigNumber cSubPrime(cSequence);
	CDEREncodedBigNumber cGenerator(cSequence);
	CDEREncodedBigNumber cPublic(cSequence);

	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_DSA_PRIMESIZE);

	CDSAKey::cPrime = cPrime;
	CDSAKey::cSubPrime = cSubPrime;
	CDSAKey::cGenerator = cGenerator;
	CDSAKey::cPublic = cPublic;

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = true;
}

void CDSAKey::ReadParams(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_DSA_PARAMS "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_DSA_PARAMS "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cPrime(cSequence);
	CDEREncodedBigNumber cSubPrime(cSequence);
	CDEREncodedBigNumber cGenerator(cSequence);

	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_DSA_PRIMESIZE);

	CDSAKey::cPrime = cPrime;
	CDSAKey::cSubPrime = cSubPrime;
	CDSAKey::cGenerator = cGenerator;

	fgHoldParams = true;
}

void CDSAKey::ReadParams(char *pszFile, bool fgBase64)
{
	int iFile = 0;
	if (NULL != pszFile)
		if (-1 == (iFile = open(pszFile, O_RDONLY)))
		  throw OPEN_ERROR;
	ReadParams(iFile, fgBase64);
	if (NULL != pszFile)
		close(iFile);
}

Word CDSAKey::Check()
{
	if (!fgHoldKey || fgEncryptOnly)
		throw(BAD_DSA_OPERATION);

	Word wResult = DSA_OK;

	if (!cPrime.IsPrime())
		wResult |= DSA_PNOTPRIME;
	if (!cSubPrime.IsPrime())
		wResult |= DSA_QNOTPRIME;
	if ((cPrime - 1) % cSubPrime != 0)
		wResult |= DSA_QNOTSUBPRIME;
	if (cPublic != CBigNumber::ModExp(cGenerator, cPrivate, cPrime))
		wResult |= DSA_BADPUBLIC;

	return wResult;
}

CDSABlock::CDSABlock(const CDSAKey &cDSAKey, void *pvData, Word wData)
{
	cKey = cDSAKey;
	cData = CBigNumber(pvData, wData);
}

CDSABlock::CDSABlock(const CDSAKey &cDSAKey, const CBigNumber &cDSAData)
{
	cKey = cDSAKey;
	cData = cDSAData;
}

void CDSABlock::Verify()
{
	if (!cKey.HoldKeyFlag())
		throw(BAD_DSA_OPERATION);
	CBigNumber cTemp = CBigNumber::ModInv(cCipher2, cKey.GetSubPrime());
	cCipher2 = ((CBigNumber::ModExp(cKey.GetGenerator(), (cTemp * cData) % cKey.GetSubPrime(), cKey.GetPrime()) * CBigNumber::ModExp(cKey.GetPublic(), (cCipher1 * cTemp) % cKey.GetSubPrime(), cKey.GetPrime())) % cKey.GetPrime()) % cKey.GetSubPrime();
}

void CDSABlock::Sign()
{
	if (!cKey.HoldKeyFlag() || cKey.EncryptOnlyFlag())
		throw(BAD_DSA_OPERATION);
	CBigNumber cTempPrivate;
	cTempPrivate.SetRandom(cKey.GetSubPrimeSize(), false);
	cCipher1 = CBigNumber::ModExp(cKey.GetGenerator(), cTempPrivate, cKey.GetPrime()) % cKey.GetSubPrime();
	cCipher2 = (CBigNumber::ModInv(cTempPrivate, cKey.GetSubPrime()) * (cCipher1 * cKey.GetPrivate() + cData)) % cKey.GetSubPrime();
}

void CDSABlock::Dump()
{
	printf("d = "); cData.Dump();
	printf("r = "); cCipher1.Dump();
	printf("s = "); cCipher2.Dump();
}
