/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "nr.hpp"

CNRKey::CNRKey()
{
}

CNRKey::CNRKey(Word wPrimeSize)
{
	CNRKey::wPrimeSize = wPrimeSize;
	GenerateKeys();
}

CNRKey::CNRKey(const CProbablePrime &cPrime,
					const CProbablePrime &cSubPrime,
					const CBigNumber &cGenerator,
					const CBigNumber &cPublic)
{
	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	CNRKey::cPrime = cPrime;
	CNRKey::cSubPrime = cSubPrime;
	CNRKey::cGenerator = cGenerator;
	CNRKey::cPublic = cPublic;
}

CNRKey::CNRKey(const CProbablePrime &cPrime,
					const CProbablePrime &cSubPrime,
					const CBigNumber &cGenerator,
					const CBigNumber &cPublic,
					const CBigNumber &cPrivate)
{
	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	CNRKey::cPrime = cPrime;
	CNRKey::cSubPrime = cSubPrime;
	CNRKey::cGenerator = cGenerator;
	CNRKey::cPublic = cPublic;
	CNRKey::cPrivate = cPrivate;
}

CNRKey::CNRKey(const CNRKey &cNRKey)
{
	wPrimeSize = cNRKey.wPrimeSize;
	wSubPrimeSize = cNRKey.wSubPrimeSize;
	cPrime = cNRKey.cPrime;
	cSubPrime = cNRKey.cSubPrime;
	cGenerator = cNRKey.cGenerator;
	cPublic = cNRKey.cPublic;
	cPrivate = cNRKey.cPrivate;
}

void CNRKey::GenerateKeys()
{
	wSubPrimeSize = 160;

	do {
		do {
			cSubPrime.SetRandom(wSubPrimeSize, true);
		} while (!cSubPrime.IsPrime());
		cPrime.SetRandom(wPrimeSize - wSubPrimeSize, false);
		cPrime.ClearBit(0);
		cPrime = cPrime * cSubPrime + 1;
	} while (!cPrime.IsPrime());

	do {
		CBigNumber cTemp;
		cTemp.SetRandom(wPrimeSize, false);

		cGenerator = CBigNumber::ModExp(cTemp, (cPrime - 1) / cSubPrime, cPrime);
	} while (cGenerator <= 1);

	cPrivate.SetRandom(wSubPrimeSize, false);
	cPublic = CBigNumber::ModExp(cGenerator, cPrivate, cPrime);
}

void CNRKey::Dump()
{
	printf("prime = "); cPrime.Dump();
	printf("sub prime = "); cSubPrime.Dump();
	printf("generator = "); cGenerator.Dump();
	printf("private = "); cPrivate.Dump();
	printf("public = "); cPublic.Dump();
}

void CNRKey::WritePrivateKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cVersion((Word)NR_PRIVATE_KEY_VERSION);
	CDEREncodedBigNumber cPrime(CNRKey::cPrime);
	CDEREncodedBigNumber cSubPrime(CNRKey::cSubPrime);
	CDEREncodedBigNumber cGenerator(CNRKey::cGenerator);
	CDEREncodedBigNumber cPublic(CNRKey::cPublic);
	CDEREncodedBigNumber cPrivate(CNRKey::cPrivate);

	CDEREncodedSequence cSequence;

	cSequence.AddPrimitive(cVersion);
	cSequence.AddPrimitive(cPrime);
	cSequence.AddPrimitive(cSubPrime);
	cSequence.AddPrimitive(cGenerator);
	cSequence.AddPrimitive(cPublic);
	cSequence.AddPrimitive(cPrivate);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_NR_PRIVATE_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_NR_PRIVATE_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CNRKey::WritePublicKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cPrime(CNRKey::cPrime);
	CDEREncodedBigNumber cSubPrime(CNRKey::cSubPrime);
	CDEREncodedBigNumber cGenerator(CNRKey::cGenerator);
	CDEREncodedBigNumber cPublic(CNRKey::cPublic);

	CDEREncodedSequence cSequence;
	cSequence.AddPrimitive(cPrime);
	cSequence.AddPrimitive(cSubPrime);
	cSequence.AddPrimitive(cGenerator);
	cSequence.AddPrimitive(cPublic);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_NR_PUBLIC_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_NR_PUBLIC_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CNRKey::ReadPrivateKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_NR_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_NR_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cVersion(cSequence);
	CDEREncodedBigNumber cPrime(cSequence);
	CDEREncodedBigNumber cSubPrime(cSequence);
	CDEREncodedBigNumber cGenerator(cSequence);
	CDEREncodedBigNumber cPublic(cSequence);
	CDEREncodedBigNumber cPrivate(cSequence);

	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	CNRKey::cPrime = cPrime;
	CNRKey::cSubPrime = cSubPrime;
	CNRKey::cGenerator = cGenerator;
	CNRKey::cPublic = cPublic;
	CNRKey::cPrivate = cPrivate;
}

void CNRKey::ReadPublicKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_NR_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_NR_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cPrime(cSequence);
	CDEREncodedBigNumber cSubPrime(cSequence);
	CDEREncodedBigNumber cGenerator(cSequence);
	CDEREncodedBigNumber cPublic(cSequence);

	wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	CNRKey::cPrime = cPrime;
	CNRKey::cSubPrime = cSubPrime;
	CNRKey::cGenerator = cGenerator;
	CNRKey::cPublic = cPublic;
}

CNRBlock::CNRBlock(const CNRKey &cNRKey, void *pvData, Word wData)
{
	cKey = cNRKey;
	cData = CBigNumber(pvData, wData);
}

CNRBlock::CNRBlock(const CNRKey &cNRKey, const CBigNumber &cNRData)
{
	cKey = cNRKey;
	cData = cNRData;
}

void CNRBlock::Verify()
{
	CBigNumber j = CBigNumber::ModExp(cKey.GetGenerator(), cCipher, cKey.GetPrime()) * CBigNumber::ModInv(CBigNumber::ModExp(cKey.GetPublic(), cData, cKey.GetPrime()), cKey.GetPrime()) % cKey.GetPrime();
	cData = (cData * j) % cKey.GetPrime();
	cCipher = (Word)0;
}

void CNRBlock::Sign()
{
	CBigNumber cTempPrivate;
	cTempPrivate.SetRandom(cKey.GetSubPrimeSize(), false);
	CBigNumber cTempPublic = CBigNumber::ModInv(CBigNumber::ModExp(cKey.GetGenerator(), cTempPrivate, cKey.GetPrime()), cKey.GetPrime());
	
	cData = (cTempPublic * cData) % cKey.GetPrime();
	cCipher = (cTempPrivate + cKey.GetPrivate() * cData) % cKey.GetSubPrime();
}

void CNRBlock::Dump()
{
	printf("d = "); cData.Dump();
	printf("e = "); cCipher.Dump();
}
