/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "rw.hpp"

CRWKey::CRWKey()
{
	fgHoldKey = false;
}

CRWKey::CRWKey(Word wModulusSize)
{
	if ((wModulusSize < 512) || (wModulusSize > 3072))
		throw(BAD_RW_MODULUSSIZE);
	CRWKey::wModulusSize = wModulusSize;
	GenerateKeys();
	fgHoldKey = true;
	fgEncryptOnly = false;
}

CRWKey::CRWKey(const CBigNumber &cModulus)
{
	wModulusSize = cModulus.GetWords() * BITSINWORD;

	N = cModulus;

	fgHoldKey = true;
	fgEncryptOnly = true;
}

CRWKey::CRWKey(const CBigNumber &cModulus,
					const CBigNumber &cPrivateKey,
					const CProbablePrime &cFirstPrime,
					const CProbablePrime &cSecondPrime)
{
	wModulusSize = cModulus.GetWords() * BITSINWORD;

	N = cModulus;
	K = cPrivateKey;
	P = cFirstPrime;
	Q = cSecondPrime;

	if ((wModulusSize < 512) || (wModulusSize > 3072))
		throw(BAD_RW_MODULUSSIZE);
	fgHoldKey = true;
	fgEncryptOnly = false;
}

CRWKey::CRWKey(const CBigNumber &cModulus, const CBigNumber &cPrivateKey)
{
	wModulusSize = cModulus.GetWords() * BITSINWORD;

	N = cModulus;
	K = cPrivateKey;

	if ((wModulusSize < 512) || (wModulusSize > 3072))
		throw(BAD_RW_MODULUSSIZE);
	fgHoldKey = true;
	fgEncryptOnly = false;
}

CRWKey::CRWKey(const CRWKey &cRWKey)
{
	wModulusSize = cRWKey.wModulusSize;
	N = cRWKey.N;
	K = cRWKey.K;
	P = cRWKey.P;
	Q = cRWKey.Q;

	if ((wModulusSize < 512) || (wModulusSize > 3072))
		throw(BAD_RW_MODULUSSIZE);
	fgHoldKey = true;
	fgEncryptOnly = false;
}

void CRWKey::GenerateKeys()
{
	do {
		P.SetRandom(wModulusSize / 2, true);
	
		while (P % 8 != 3)
			P += 1;
	
		while (!P.IsPrime())
			P += 8;
	} while (!CBigNumber::GCD(3, P - 1).IsOne());

	do {
		Q.SetRandom(wModulusSize / 2, true);
		Q.SetBit(0);
		Q.SetBit(wModulusSize / 2 - 1);
	
		while (Q % 8 != 7)
			Q += 1;

		while (!Q.IsPrime())
			Q += 8;
	} while (!CBigNumber::GCD(3, Q - 1).IsOne());
	
	N = P * Q;

	K = CBigNumber::ModInv(3, (P - 1) * (Q - 1));
}

void CRWKey::WritePrivateKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cVersion((Word)RW_PRIVATE_KEY_VERSION);
	CDEREncodedBigNumber cModulus(N);
	CDEREncodedBigNumber cPrime1(P);
	CDEREncodedBigNumber cPrime2(Q);
	CDEREncodedBigNumber cPrivate(K);

	CDEREncodedSequence cSequence;
	
	cSequence.AddPrimitive(cVersion);
	cSequence.AddPrimitive(cModulus);
	cSequence.AddPrimitive(cPrime1);
	cSequence.AddPrimitive(cPrime2);
	cSequence.AddPrimitive(cPrivate);
	
	if (true == fgBase64) {
		write_string(iOut, BEGIN_RW_PRIVATE_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_RW_PRIVATE_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CRWKey::WritePublicKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cModulus(N);

	CDEREncodedSequence cSequence;
	
	cSequence.AddPrimitive(cModulus);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_RW_PUBLIC_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_RW_PUBLIC_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CRWKey::ReadPrivateKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_RW_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_RW_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cVersion(cSequence);
	CDEREncodedBigNumber cModulus(cSequence);
	CDEREncodedBigNumber cPrime1(cSequence);
	CDEREncodedBigNumber cPrime2(cSequence);
	CDEREncodedBigNumber cPrivate(cSequence);

	N = cModulus;
	P = cPrime1;
	Q = cPrime2;
	K = cPrivate;

	wModulusSize = cModulus.GetWords() * BITSINWORD;

	if ((wModulusSize < 512) || (wModulusSize > 3072))
		throw(BAD_RW_MODULUSSIZE);
	fgHoldKey = true;
	fgEncryptOnly = false;
}

void CRWKey::ReadPublicKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_RW_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_RW_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cModulus(cSequence);

	N = cModulus;

	wModulusSize = cModulus.GetWords() * BITSINWORD;

	if ((wModulusSize < 512) || (wModulusSize > 3072))
		throw(BAD_RW_MODULUSSIZE);
	fgHoldKey = true;
	fgEncryptOnly = true;
}

void CRWKey::Dump()
{
	printf("p [%d bits] = ", P.GetWords() * BITSINWORD); P.Dump();
	printf("q [%d bits] = ", Q.GetWords() * BITSINWORD); Q.Dump();
	printf("n [%d bits] = ", N.GetWords() * BITSINWORD); N.Dump();
	printf("k [%d bits] = ", K.GetWords() * BITSINWORD); K.Dump();
}

Word CRWKey::Check()
{
	if (!fgHoldKey || fgEncryptOnly)
		throw(BAD_RW_OPERATION);
	
	Word wResult = RW_OK;
	
	if (!P.IsPrime())
		wResult |= RW_PNOTPRIME;
	if (!Q.IsPrime())
		wResult |= RW_QNOTPRIME;
	if (P % 8 != 3)
		wResult |= RW_PNOTCON;
	if (Q % 8 != 7)
		wResult |= RW_QNOTCON;
	if (N != P * Q)
		wResult |= RW_BADN;
	if (K != CBigNumber::ModInv(3, (P - 1) * (Q - 1)))
		wResult |= RW_BADSEC;
	
	return wResult;
}

CRWBlock::CRWBlock(const CRWKey &cRWKey, void *pvData, Word wData)
{
	cKey = cRWKey;
	cData = CBigNumber(pvData, wData);
}

CRWBlock::CRWBlock(const CRWKey &cRWKey, const CBigNumber &cRWData)
{
	cKey = cRWKey;
	cData = cRWData;
}

void CRWBlock::Verify()
{
	if (!cKey.HoldKeyFlag())
		throw(BAD_RW_OPERATION);

	cData = CBigNumber::ModExp(cData, 3, cKey.GetModulus());
	if (cData % 8 == 6) {
		cData.Shl();
	} else {
		cData = cKey.GetModulus() - cData;
		if (cData % 8 == 6)
			cData.Shl();
	}
	cData -= 12;
	cData.Shr(4);
}

void CRWBlock::Sign()
{
	if (!cKey.HoldKeyFlag() || cKey.EncryptOnlyFlag())
		throw(BAD_RW_OPERATION);

	cData.Shl(4);
	cData += 12;
	if (CBigNumber::Jacobi(cData, cKey.GetModulus()) *
		 CBigNumber::Jacobi(cData, cKey.GetModulus()) != 1)
		cData.Shr();
	cData = CBigNumber::ModExp(cData, cKey.GetPrivate(), cKey.GetModulus());
	if (cKey.GetModulus() - cData < cData)
		cData = cKey.GetModulus() - cData;
}

void CRWBlock::Dump()
{
	printf("d = "); cData.Dump();
}