/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "mqv.hpp"

CMQVKeyExchange::CMQVKeyExchange(Word wPrimeSize)
{
	CMQVKeyExchange::wPrimeSize = wPrimeSize;
	wSubPrimeSize = 160;

	do {
		do {
			cSubPrime.SetRandom(wSubPrimeSize, true);
		} while (!cSubPrime.IsPrime());
		cPrime.SetRandom(wPrimeSize - wSubPrimeSize, false);
		cPrime.ClearBit(0);
		cPrime = cPrime * cSubPrime + 1;
	} while (!cPrime.IsPrime());

	do {
		CBigNumber cTemp;
		cTemp.SetRandom(wPrimeSize, false);

		cGenerator = CBigNumber::ModExp(cTemp, (cPrime - 1) / cSubPrime, cPrime);
	} while (cGenerator <= 1);

	cPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);
	cTempPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);

	MakeKeys();
}

CMQVKeyExchange::CMQVKeyExchange(const CProbablePrime &cPrime, const CProbablePrime &cSubPrime, const CBigNumber &cGenerator, const CBigNumber &cPrivate)
{
	CMQVKeyExchange::wPrimeSize = cPrime.GetWords() * BITSINWORD;
	wSubPrimeSize = 160;

	CMQVKeyExchange::cPrime = cPrime;
	CMQVKeyExchange::cSubPrime = cSubPrime;
	CMQVKeyExchange::cGenerator = cGenerator;
	CMQVKeyExchange::cPrivate = cPrivate;
	
	MakeKeys();
}

CMQVKeyExchange::CMQVKeyExchange(const CProbablePrime &cPrime, const CProbablePrime &cSubPrime, const CBigNumber &cGenerator)
{
	CMQVKeyExchange::wPrimeSize = cPrime.GetWords() * BITSINWORD;
	CMQVKeyExchange::wSubPrimeSize = (Word)(2.4 * pow(wPrimeSize, 1.0 / 3.0) * pow(log((double)wPrimeSize), 2.0 / 3.0) - 5);

	CMQVKeyExchange::cPrime = cPrime;
	CMQVKeyExchange::cSubPrime = cSubPrime;
	CMQVKeyExchange::cGenerator = cGenerator;

	cPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);
	cTempPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);
	
	MakeKeys();
}

CMQVKeyExchange::CMQVKeyExchange(const CMQVKeyExchange &/*cMQVKeyExchange*/)
{
}

void CMQVKeyExchange::MakeKeys()
{
	cPublic = CBigNumber::ModExp(cGenerator, cPrivate, cPrime);
	cTempPublic = CBigNumber::ModExp(cGenerator, cTempPrivate, cPrime);
}

void CMQVKeyExchange::Dump()
{
	printf("p = "); cPrime.Dump();
	printf("r = "); cSubPrime.Dump();
	printf("g = "); cGenerator.Dump();
	printf("pri = "); cPrivate.Dump();
	printf("pub = "); cPublic.Dump();
	printf("temp pri = "); cTempPrivate.Dump();
	printf("temp pub = "); cTempPublic.Dump();
	printf("key = "); cKey.Dump();
}

void CMQVKeyExchange::SetOtherPublic(const CBigNumber &cSecret, const CBigNumber &cTempSecret)
{
	cOtherPublic = cSecret;
	cOtherTempPublic = cTempSecret;
}

void CMQVKeyExchange::Agree()
{
	CBigNumber cTmp = 1;
	cTmp.Shl((wSubPrimeSize + 1) / 2);
	CBigNumber t1 = cTempPublic % cTmp + cTmp;
	CBigNumber t2 = cOtherTempPublic % cTmp + cTmp;
	cKey = CBigNumber::ModExp((cOtherTempPublic * CBigNumber::ModExp(cOtherPublic, t2, cPrime)), (t1 * cPrivate + cTempPrivate) % cSubPrime, cPrime);
}

void CMQVKeyExchange::WritePublic(int iOut)
{
	CDEREncodedBigNumber cData1(cPublic);
	CDEREncodedBigNumber cData2(cTempPublic);
	cData1.Write(iOut);
	cData2.Write(iOut);
}

void CMQVKeyExchange::ReadOtherPublic(int iIn)
{
	CDEREncodedBigNumber cData1;
	CDEREncodedBigNumber cData2;

	cData1.Read(iIn);
	cData2.Read(iIn);

	cOtherPublic = cData1;
	cOtherTempPublic = cData2;
}

Byte *CMQVKeyExchange::GetKey()
{
	return (Byte *)cKey.GetData();
}

Word CMQVKeyExchange::GetKeySize()
{
	return cKey.GetWords() * BYTESINWORD;
}
