/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "base64.hpp"

#include <stdio.h>

CBase64Coder::CBase64Coder() : pszEncode(NULL), wLastBlock(0), wLinePos(0), wDecodePos(0)
{
}

void CBase64Coder::Reset()
{
	pszEncode = NULL;
	wLastBlock = 0;
	wLinePos = 0;
	wDecodePos = 0;
}

Word CBase64Coder::EncodeData(const Byte *pbSource, Word wLength, char *pszTarget, bool fgPad)
{
	Word wOutput = 0;
	pszEncode = ((pszTarget == NULL) ? pszEncode : pszTarget);
	for (Word i = 0; i < wLength; i++, wLastBlock++) {
		if (3 == wLastBlock) {
			memcpy(pszEncode + wOutput, Encode(bLastBlock, 3), 4);
			wOutput += 4;
			wLastBlock = 0;
			wLinePos += 4;
			if (BASE64_MAXLINESIZE <= wLinePos) {
				pszEncode[wOutput++] = '\n';
				wLinePos = 0;
			}
		}
		bLastBlock[wLastBlock] = pbSource[i];
	}
	if (true == fgPad) {
		memcpy(pszEncode + wOutput, Encode(bLastBlock, wLastBlock), 4);
		wOutput += 4;
		wLastBlock = 0;
		wLinePos += 4;
		wLinePos = 0;
	}
	pszEncode[wOutput] = '\0';
	pszEncode += wOutput;

	return wOutput;
}

Word CBase64Coder::DecodeData(const char *pszSource, Word wLength, Byte *pbTarget)
{
	Word wOutput = 0;
	Word i, j;

	for (i = 0; i < wLength; i += 4) {
		if (pszSource[i] == '\n')
			i += 1;
		j = Decode(pszSource + i, bDecode);
		memcpy(pbTarget + wOutput, bDecode, j);
		wOutput += j;
	}

	return wOutput;
}

Word CBase64Coder::DecodeData(Byte *pbDest, Word wLength, int iIn)
{
	for (Word i = 0; i < wLength; i++) {
		if (0 == wDecodePos) {
			for (Word j = 0; j < 4; j++)
				do {
					if (1 != scl_read(iIn, szEncode + j, 1))
						break;
				} while (('\n' == szEncode[j]) || ('\r' == szEncode[j]));
			Decode(szEncode, bDecode);
		}
		pbDest[i] = bDecode[wDecodePos++];
		if (3 == wDecodePos)
			wDecodePos = 0;
	}
	return wLength;
}

char *CBase64Coder::Encode(const Byte *pbData, Word wLength)
{
	Byte bData[3] = { 0, 0, 0 };
	memcpy(bData, pbData, wLength);
	szEncode[0] = EncodeChar(bData[0] >> 2);
	szEncode[1] = EncodeChar(((bData[0] & 3) << 4) | (bData[1] >> 4));
	szEncode[2] = ((wLength < 2) ? BASE64_PADCHAR : EncodeChar(((bData[1] & 15) << 2) | (bData[2] >> 6)));
	szEncode[3] = ((wLength < 3) ? BASE64_PADCHAR : EncodeChar(bData[2] & 63));
	return szEncode;
}

Word CBase64Coder::Decode(const char *pszSource, Byte *pbTarget)
{
	Byte a, b, c, d;
	a = DecodeChar(pszSource[0]);
	b = DecodeChar(pszSource[1]);
	pbTarget[0] = a << 2 | b >> 4;
	if (BASE64_PADCHAR == pszSource[2])
		return 1;
	c = DecodeChar(pszSource[2]);
	pbTarget[1] = b << 4 | c >> 2;
	if (BASE64_PADCHAR == pszSource[3])
		return 2;
	d = DecodeChar(pszSource[3]);
	pbTarget[2] = c << 6 | d;
	return 3;
}

char CBase64Coder::EncodeChar(Byte i)
{
	return BASE64_VECTOR[i];
}

Byte CBase64Coder::DecodeChar(char c)
{
	if (c == '/')
		return 63;
	if (c == '+')
		return 62;
	if (c >= '0' && c <= '9')
		return c - '0' + 52;
	if (c >= 'a' && c <= 'z')
		return c - 'a' + 26;
	if (c >= 'A' && c <= 'Z')
		return c - 'A';
	throw(BASE64DECODING_ERROR);
}
