/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "conf.hpp"

#ifdef WIN32
#define snprintf _snprintf
#define strcasecmp _stricmp
#endif

CConfigurationFile::CConfigurationFile(char *pszName, char **pszError)
{
	pConfig = NULL;
	
	FILE *fpFile;
	static char szBuf[256];
	static char szTmp[256];
	static char szSection[256];
	static char szResult[256];
	
	if (NULL == (fpFile = fopen(pszName, "r"))) {
		snprintf(szResult, sizeof(szResult), "Error opening configuration file '%s'...", pszName);
		if (NULL != pszError)
			*pszError = szResult;
		return;
	}
	
	int iLineNumber = 0;	
	while (NULL != fgets(szBuf, sizeof(szBuf), fpFile)) {
		iLineNumber += 1;
		// Strip leading white spaces and tabs
		int i;
		for (i = 0; '\0' != szBuf[i]; i++)
			if ((' ' != szBuf[i]) && ('\t' != szBuf[i]))
				break;
		strcpy(szTmp, &szBuf[i]);
		// Strip trailing CRs, LFs, spaces and tabs
		for (i = strlen(szTmp) - 1; i >= 0; i--)
			if ((' ' != szTmp[i]) && 
				 ('\t' != szTmp[i]) && 
				 ('\x0a' != szTmp[i]) &&
				 ('\x0d' != szTmp[i]))
				break;
		szTmp[i + 1] = '\0';		
		// Skip empty lines
		if ('\0' == szTmp[0])
			continue;
		// Skip comments
		if (('#' == szTmp[0]) || ('%' == szTmp[0]))
			continue;
		// First line must be always section entry
		if (1 == iLineNumber)
			if ('[' != szTmp[0]) {
				snprintf(szResult, sizeof(szResult), "Parsing error in line %d of '%s'...", iLineNumber, pszName);				
				if (NULL != pszError)
					*pszError = szResult;
				return;
			}
		// Check section entry
		if ('[' == szTmp[0]) {
			if (']' != szTmp[strlen(szTmp) - 1]) {
				snprintf(szResult, sizeof(szResult), "Parsing error in line %d of '%s'...", iLineNumber, pszName);				
				if (NULL != pszError)
					*pszError = szResult;
				return;
			}
			strncpy(szSection, szTmp + 1, strlen(szTmp) - 2);
			szSection[strlen(szTmp) - 2] = '\0';
			continue;
		}
		// If there is no delimter in the entry it is invalid
		if (NULL == strchr(szTmp, '=')) {
			snprintf(szResult, sizeof(szResult), "Parsing error in line %d of '%s'...", iLineNumber, pszName);
			if (NULL != pszError)
				*pszError = szResult;
			return;
		}
		char *p = strtok(szTmp, "=");
		char *q = strtok(NULL, "");
		if (false == AddItem(p, q, szSection)) {
			if (NULL != pszError)
				*pszError = "Error allocating memory...";
			return;
		}
	}
	
	fclose(fpFile);
	
	if (NULL != pszError)
		*pszError = NULL;
}

CConfigurationFile::~CConfigurationFile()
{
	while(DeleteLastItem());
}

bool CConfigurationFile::AddItem(char *pszKey, char *pszData, char *pszSection)
{
	SItem *pLast = pConfig;
	
	if (NULL == pConfig) {
		pConfig = (SItem *)malloc(sizeof(SItem));
		if (NULL == pConfig)
			return false;
		pLast = pConfig;
	} else {
		while (NULL != pLast->pNext)
			pLast = (SItem *)pLast->pNext;
		pLast->pNext = (SItem *)malloc(sizeof(SItem));
		if (NULL == pLast->pNext)
			return false;
		pLast = (SItem *)pLast->pNext;
	}
	
	pLast->pszKey = NULL;
	pLast->pszData = NULL;
	pLast->pszSection = NULL;
	if (NULL != pszKey)
		pLast->pszKey = strdup(pszKey);
	if (NULL != pszData)
		pLast->pszData = strdup(pszData);
	if (NULL != pszSection)
		pLast->pszSection = strdup(pszSection);
	pLast->pNext = NULL;
	
	return true;
}

char *CConfigurationFile::FindItem(char *pszSection, char *pszKey)
{
	SItem *pLast = pConfig;
	
	while (NULL != pLast) {
		if ((0 == strcmp(pszSection, pLast->pszSection)) && 
			 (0 == strcmp(pszKey, pLast->pszKey)))
			return pLast->pszData;
		pLast = (SItem *)pLast->pNext;
	}
	return NULL;
}

bool CConfigurationFile::DeleteLastItem()
{
	SItem *pLast = pConfig;
	
	if (NULL == pLast)
		return false;
	
	if (NULL != pLast->pNext) {
		while (NULL != ((SItem *)(pLast->pNext))->pNext)
			pLast = (SItem *)pLast->pNext;
		free(((SItem *)(pLast->pNext))->pszKey);
		free(((SItem *)(pLast->pNext))->pszData);
		free(((SItem *)(pLast->pNext))->pszSection);
		free(pLast->pNext);
		pLast->pNext = NULL;
		return true;
	} else {
		free(pConfig->pszKey);
		free(pConfig->pszData);
		free(pConfig->pszSection);
		free(pConfig);
		pConfig = NULL;
		return false;
	}
}

char *CConfigurationFile::GetString(char *pszSection, char *pszEntry, char *pszDefault)
{
	char *pszString = FindItem(pszSection, pszEntry);
	return pszString == NULL ? pszDefault : pszString;
}

long CConfigurationFile::GetInteger(char *pszSection, char *pszEntry, long lDefault)
{
	char *p = FindItem(pszSection, pszEntry);
	char *q;
	long l;
	
	if (NULL == p)
		return lDefault;
	
	l = strtol(p, &q, 10);
	
	if (('\0' == *q) && ('\0' != *p))
		return l;
	else
		return lDefault;
}

bool CConfigurationFile::GetBoolean(char *pszSection, char *pszEntry, bool fgDefault)
{
	char *p = FindItem(pszSection, pszEntry);
	if (NULL == p)
		return fgDefault;
	if (0 == strcmp(p, "0") ||
		 0 == strcasecmp(p, "false") ||
		 0 == strcasecmp(p, "no") ||
		 0 == strcasecmp(p, "n") ||
		 0 == strcasecmp(p, "nak"))
		return false;
	if (0 == strcmp(p, "1") ||
		 0 == strcasecmp(p, "true") ||
		 0 == strcasecmp(p, "yes") ||
		 0 == strcasecmp(p, "y") ||
		 0 == strcasecmp(p, "ack"))
		return true;
	return fgDefault;
}

void CConfigurationFile::EnumSection(char *pszSection, void (*pEnumerator)(char *, char *))
{
	SItem *pLast = pConfig;

	while (NULL != pLast) {
		if (0 == strcmp(pszSection, pLast->pszSection))
			(*pEnumerator)(pLast->pszKey, pLast->pszData);
		pLast = (SItem *)pLast->pNext;
	}
}
