/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "blumblumshub.hpp"

CBlumBlumShubPRNG::CBlumBlumShubPRNG()
{
}

CBlumBlumShubPRNG::CBlumBlumShubPRNG(Word wModulus)
{
	P.SetRandom(wModulus / 2, true);
	Q.SetRandom(wModulus / 2, true);
	S.SetRandom(wModulus, true);
	
	MakePrimes();
	CalcBits();
	Initialize();
}

CBlumBlumShubPRNG::CBlumBlumShubPRNG(const CProbablePrime &P,
												 const CProbablePrime &Q,
												 const CBigNumber &S)
{
	CBlumBlumShubPRNG::P = P;
	CBlumBlumShubPRNG::Q = Q;
	CBlumBlumShubPRNG::S = S;
	
	MakePrimes();
	CalcBits();
	Initialize();
}

CBlumBlumShubPRNG::CBlumBlumShubPRNG(const CBigNumber &N, const CBigNumber &S)
{
	CBlumBlumShubPRNG::N = N;
	CBlumBlumShubPRNG::S = S;
	CalcBits();
	Initialize();
}

CBlumBlumShubPRNG::CBlumBlumShubPRNG(const CBlumBlumShubPRNG &cBlumBlumShubPRNG)
{
	P = cBlumBlumShubPRNG.P;
	Q = cBlumBlumShubPRNG.Q;
	N = cBlumBlumShubPRNG.N;
	S = cBlumBlumShubPRNG.S;
	wBits = cBlumBlumShubPRNG.wBits;
}

void CBlumBlumShubPRNG::MakePrimes()
{
	while (3 != P % 4)
		P += 1;
	while (!P.IsPrime())
		P += 4;
	while (3 != Q % 4)
		Q += 1;
	while (!Q.IsPrime())
		Q += 4;
	N = P * Q;
}

void CBlumBlumShubPRNG::CalcBits()
{
	wBits = 0;
	Dword dwTemp = N.Lg();
	Dword dwTmp = 1;
	while (dwTmp <= dwTemp) {
		wBits += 1;
		dwTmp <<= 1;
	}
	wBits -= 1;
}

void CBlumBlumShubPRNG::Initialize()
{
	while (!CBigNumber::GCD(S, N).IsOne())
		S += 1;

	S.Sqr();
	S.Mod(N);
}

void CBlumBlumShubPRNG::GetRandomData(void *pvRandom, int iLength)
{
	for (int i = 0; i < iLength; i++) {
		((Byte *)pvRandom)[i] = 0;
		for (int j = 0; j < BITSINBYTE; j++) {
			if (0 == (i * BITSINBYTE + j) % wBits) {
				S.Sqr();
				S.Mod(N);
				T += 1;
			}
			((Byte *)pvRandom)[i] <<= 1;
			((Byte *)pvRandom)[i] |= S.GetBit(wBits - 1 - ((i * BITSINBYTE + j) % wBits));
		}
	}
}
