/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "initrnd.hpp"
#include "timers.hpp"

#define KB_GET_STATE(fd, s) tcgetattr(fd, s)
#define KB_ALTER_STATE(s) ((s)->c_cc[VMIN] = 1, (s)->c_cc[VTIME] = 0, (s)->c_lflag &= ~(ECHO | ICANON))
#define KB_SET_STATE(fd, s) tcsetattr(fd, TCSAFLUSH, s)


CPRNG &GetDefPRNG()
{
	return cPRNG;
}

bool InitDefPRNG(Word wBytes)
{
	CPGPRandomPool cPGPRandomPool;

	if (false == InitDefPRNGFromFile(wBytes, RANDOM_FILE))
		if (false == InitDefPRNGFromRandDev(wBytes))
#ifdef WIN32
			InitDefPRNGFromWindowsTTY(wBytes);
#else
			InitDefPRNGFromUnixTTY(wBytes);
#endif
	char szRandomFile[256] = RANDOM_FILE;
#ifndef WIN32
	snprintf(szRandomFile, sizeof(szRandomFile), "%s/%s", getenv("HOME"), RANDOM_FILE);
#endif
	CreateRandomFile(wBytes, szRandomFile);
	return true;
}

bool InitDefPRNGFromRandDev(Word wBytes)
{
	return InitDefPRNGFromFile(wBytes, "/dev/urandom");
}

bool CreateRandomFile(Word wBytes, char *pszFile)
{
	CPGPRandomPool cPGPRandomPool;

	int iRandFile = open(pszFile, O_RDWR | O_CREAT
#ifdef WIN32
								| O_BINARY, 0755);
#else
								, 0600);
#endif
	if (-1 == iRandFile)
		return false;
	Byte *pbBytes = new Byte[wBytes];
	if (NULL == pbBytes)
		return false;
	cPRNG.GetRandomData(pbBytes, wBytes);
	if (wBytes != (Word)write(iRandFile, pbBytes, wBytes)) {
		delete pbBytes;
		return false;
	}
	close(iRandFile);
	delete pbBytes;

	return true;
}

bool InitDefPRNGFromFile(Word wBytes, char *pszFile)
{
	CPGPRandomPool cPGPRandomPool;

	FILE *fpRandDev = fopen(pszFile, "rb");
	if (NULL == fpRandDev)
		return false;
//		throw(OPEN_RAND_DEV_ERROR);
	Byte *pbBytes = new Byte[wBytes];
	if (NULL == pbBytes)
		throw(MEMORYALLOCATION_ERROR);
	if (wBytes != fread(pbBytes, sizeof(Byte), wBytes, fpRandDev)) {
		delete pbBytes;
		throw(READ_ERROR);
	}
	cPGPRandomPool.PutData(pbBytes, wBytes);
	fclose(fpRandDev);
	delete pbBytes;

	Byte *pbSeed = new Byte[cPRNG.GetMaxSeedSize()];
	if (NULL == pbSeed)
		throw MEMORYALLOCATION_ERROR;
	cPGPRandomPool.GetData(pbSeed, cPRNG.GetMaxSeedSize());
	cPRNG.Initialize(pbSeed, cPRNG.GetMaxSeedSize());
	delete pbSeed;

	return true;
}

#ifdef WIN32
bool InitDefPRNGFromWindowsTTY(Word wBytes)
{
	CPGPRandomPool cPGPRandomPool;

	puts("Please write something (meaningful): ");

	while (true) {
		char c;
		c = getch();
		if (RandPoolKeystroke(cPGPRandomPool, (Byte)c, wBytes))
			break;
	}

	puts("Thanks.");

	Byte *pbSeed = new Byte[cPRNG.GetMaxSeedSize()];
	if (NULL == pbSeed)
		throw MEMORYALLOCATION_ERROR;
	cPGPRandomPool.GetData(pbSeed, cPRNG.GetMaxSeedSize());
	cPRNG.Initialize(pbSeed, cPRNG.GetMaxSeedSize());
	delete pbSeed;

	return true;
}
#else
bool InitDefPRNGFromUnixTTY(Word wBytes)
{
	CPGPRandomPool cPGPRandomPool;

	struct termios kbState0, kbState1;
	KB_GET_STATE(0, &kbState0);
	kbState1 = kbState0;
	KB_ALTER_STATE(&kbState1);
	KB_SET_STATE(0, &kbState1);

	puts("Please write something (meaningful): ");

	while (true) {
		int c;
		read(0, &c, 1);
		if (RandPoolKeystroke(cPGPRandomPool, (Byte)c, wBytes))
			break;
	}
	KB_SET_STATE(0, &kbState0);

	puts("Thanks.");

	Byte *pbSeed = new Byte[cPRNG.GetMaxSeedSize()];
	if (NULL == pbSeed)
		throw MEMORYALLOCATION_ERROR;
	cPGPRandomPool.GetData(pbSeed, cPRNG.GetMaxSeedSize());
	cPRNG.Initialize(pbSeed, cPRNG.GetMaxSeedSize());
	delete pbSeed;

	return true;
}
#endif

bool RandPoolKeystroke(CPGPRandomPool &cPGPRandomPool, Byte c, Word wBytes)
{
	static Byte d;
	static Word wEntered;
//	static struct timeb t;
//	static struct timeval u;

	if (d == c)
		return false;
	d = c;

	cPGPRandomPool.PutByte(c);
	wEntered += 1;
	if (wEntered == wBytes) {
		wEntered = 0;
		return true;
	}

	static timetype t;
	static timetype u;
	gettime(&u);
	cPGPRandomPool.PutByte((Byte)(tickdiff(u, t)));
	wEntered += 1;
	if (wEntered == wBytes) {
		wEntered = 0;
		return true;
	}
	t = u;

	return false;
}
