/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "pgprp.hpp"

CPGPRandomPool::CPGPRandomPool(Word wPoolSize) : wGetPos(wPoolSize), wPutPos(0)
{
	CPGPRandomPool::wPoolSize = wPoolSize;
	CPGPRandomPool::wKeySize = cPGPRPDigest.GetBlockSize();
	pbPool = (Byte *)calloc(wPoolSize, sizeof(Byte));
	if (NULL == pbPool)
		throw(MEMORYALLOCATION_ERROR);
	pbKey = (Byte *)calloc(wKeySize, sizeof(Byte));
	if (NULL == pbKey)
		throw(MEMORYALLOCATION_ERROR);
}

CPGPRandomPool::~CPGPRandomPool()
{
	Stir();
	if (NULL != pbPool)
		free(pbPool);
	if (NULL != pbKey)
		free(pbKey);
}

void CPGPRandomPool::Stir()
{
	for (int i = 0; i < 2; i++) {
		CDigestCipherKey cDigestCipherKey(cPGPRPDigest, pbKey);
		CDigestCipherBlock cDigestCipherBlock(cPGPRPDigest);
		CCFBCryptographer cCFBCryptographer(cDigestCipherBlock, cDigestCipherKey, pbPool + wPoolSize - cDigestCipherBlock.GetBlockSize());
		cCFBCryptographer.EncryptData(pbPool, wPoolSize);
		memcpy(pbKey, pbPool, wKeySize);
	}
	wPutPos = 0;
	wGetPos = wKeySize;
}

Byte CPGPRandomPool::GetByte()
{
	if (wGetPos == wPoolSize)
		Stir();
	return pbPool[wGetPos++];
}

void CPGPRandomPool::PutByte(Byte bByte)
{
	if (wPutPos == wPoolSize)
		Stir();
	
	pbPool[wPutPos++] ^= bByte;
	wGetPos = wPoolSize;						// Force stir on get
}

void CPGPRandomPool::GetData(void *pbData, int iBytes)
{
	for (int i = 0; i < iBytes; i++)
		((Byte *)pbData)[i] = GetByte();
}

void CPGPRandomPool::PutData(void *pbData, int iBytes)
{
	for (int i = 0; i < iBytes; i++)
		PutByte(((Byte *)pbData)[i]);
}
