/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "blowfish.hpp"
#include "blowfishsboxes.hpp"

#define f(a, wS) (((wS[WBYTE0(a)] + wS[WBYTE1(a) + 256]) ^ wS[WBYTE2(a) + 512]) + wS[WBYTE3(a) + 768])
#define encrypt(a, wP, wS)				\
{												\
	Word l = HiWORD(a);					\
	Word r = LoWORD(a);					\
	l ^= wP[ 0]; r ^= f(l, wS);		\
	r ^= wP[ 1]; l ^= f(r, wS);		\
	l ^= wP[ 2]; r ^= f(l, wS);		\
	r ^= wP[ 3]; l ^= f(r, wS);		\
	l ^= wP[ 4]; r ^= f(l, wS);		\
	r ^= wP[ 5]; l ^= f(r, wS);		\
	l ^= wP[ 6]; r ^= f(l, wS);		\
	r ^= wP[ 7]; l ^= f(r, wS);		\
	l ^= wP[ 8]; r ^= f(l, wS);		\
	r ^= wP[ 9]; l ^= f(r, wS);		\
	l ^= wP[10]; r ^= f(l, wS);		\
	r ^= wP[11]; l ^= f(r, wS);		\
	l ^= wP[12]; r ^= f(l, wS);		\
	r ^= wP[13]; l ^= f(r, wS);		\
	l ^= wP[14]; r ^= f(l, wS);		\
	r ^= wP[15]; l ^= f(r, wS);		\
	l ^= wP[16];							\
	r ^= wP[17];							\
	a = ((Dword)r << 32) | l;			\
}
#define decrypt(a, wP, wS)				\
{												\
	Word l = HiWORD(a);					\
	Word r = LoWORD(a);					\
	l ^= wP[17]; r ^= f(l, wS);		\
	r ^= wP[16]; l ^= f(r, wS);		\
	l ^= wP[15]; r ^= f(l, wS);		\
	r ^= wP[14]; l ^= f(r, wS);		\
	l ^= wP[13]; r ^= f(l, wS);		\
	r ^= wP[12]; l ^= f(r, wS);		\
	l ^= wP[11]; r ^= f(l, wS);		\
	r ^= wP[10]; l ^= f(r, wS);		\
	l ^= wP[ 9]; r ^= f(l, wS);		\
	r ^= wP[ 8]; l ^= f(r, wS);		\
	l ^= wP[ 7]; r ^= f(l, wS);		\
	r ^= wP[ 6]; l ^= f(r, wS);		\
	l ^= wP[ 5]; r ^= f(l, wS);		\
	r ^= wP[ 4]; l ^= f(r, wS);		\
	l ^= wP[ 3]; r ^= f(l, wS);		\
	r ^= wP[ 2]; l ^= f(r, wS);		\
	l ^= wP[ 1];							\
	r ^= wP[ 0];							\
	a = ((Dword)r << 32) | l;			\
}

CBlowfishKey::CBlowfishKey()
{
}

CBlowfishKey::CBlowfishKey(const CBlowfishKey &cBlowfishKey)
{
	wKeyLength = cBlowfishKey.wKeyLength;
	for (Word w = 0; w < wKeyLength; w++)
		wMaster[w] = cBlowfishKey.wMaster[w];
	MakeKeys();
}

CBlowfishKey::CBlowfishKey(const Word *pwBlowfishKey, Word wKeyLength)
{
	CBlowfishKey::wKeyLength = wKeyLength;
	for (Word w = 0; w < wKeyLength; w++)
		wMaster[w] = REVERSEWORD(pwBlowfishKey[w]);
	MakeKeys();
}

CBlowfishKey::CBlowfishKey(const Byte *pbBlowfishKey, Word wKeyLength)
{
	wKeyLength /= BYTESINWORD;
	CBlowfishKey::wKeyLength = wKeyLength;
	for (Word w = 0; w < wKeyLength; w++)
		wMaster[w] = REVERSEWORD(((Word *)pbBlowfishKey)[w]);
	MakeKeys();
}

CBlowfishKey::~CBlowfishKey()
{
	int i;
	for (i = 0; i < 16; i++)				// Cleanup
		wMaster[i] = 0;
	for (i = 0; i < 14; i++)				// Cleanup
		wP[i] = 0;
	for (i = 0; i < 1024; i++)				// Cleanup
		wS[i] = 0;
}

void CBlowfishKey::MakeKeys()
{
	int i;
	for (i = 0; i < 18; i++)
		CBlowfishKey::wP[i] = ::wP[i];
	for (i = 0; i < 1024; i++)
		CBlowfishKey::wS[i] = ::wS[i];
	for (i = 0; i < 18; i++)
		CBlowfishKey::wP[i] ^= wMaster[i % wKeyLength];
	Dword d = 0;
	for (i = 0; i < 9; i++) {
		encrypt(d, CBlowfishKey::wP, CBlowfishKey::wS);
		CBlowfishKey::wP[i * 2] = HiWORD(d);
		CBlowfishKey::wP[i * 2 + 1] = LoWORD(d);
	}
	for (i = 0; i < 512; i++) {
		encrypt(d, CBlowfishKey::wP, CBlowfishKey::wS);
		CBlowfishKey::wS[i * 2] = HiWORD(d);
		CBlowfishKey::wS[i * 2 + 1] = LoWORD(d);
	}
}

CBlowfishBlock::CBlowfishBlock()
{
	dwData = 0;									// This is here for definiteness
}

CBlowfishBlock::CBlowfishBlock(const CBlowfishBlock &cBlowfishBlock)
{
	dwData = cBlowfishBlock.dwData;
}

CBlowfishBlock::CBlowfishBlock(Dword dwValue)
{
	dwData = dwValue;
}

CBlowfishBlock::CBlowfishBlock(const Byte *pbData, Word wLength)
{
	SetData(pbData, wLength);
}

CBlowfishBlock::~CBlowfishBlock()
{
	dwData = 0;									// Cleanup
}

void CBlowfishBlock::Encrypt(const CBlowfishKey &cBlowfishKey)
{
	encrypt(dwData, cBlowfishKey.GetWp(), cBlowfishKey.GetSp());
}

void CBlowfishBlock::Decrypt(const CBlowfishKey &cBlowfishKey)
{
	decrypt(dwData, cBlowfishKey.GetWp(), cBlowfishKey.GetSp());
}

void CBlowfishBlock::SetValue(Dword dwValue)
{
	dwData = dwValue;
}

Dword CBlowfishBlock::GetValue()
{
	return dwData;
}

void CBlowfishBlock::SetData(const Byte *pbData, Word wLength)
{
	dwData = 0;
	memcpy(&dwData, pbData, wLength);
	dwData = REVERSEDWORD(dwData);
}

Byte *CBlowfishBlock::GetData()
{
	dwResult = REVERSEDWORD(dwData);
	return (Byte *)&dwResult;
}
