/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "des.hpp"
#include "dessboxes.hpp"

Word wMask[] = { 1, 1, 3, 3, 3, 3, 3, 3,
					  1, 3, 3, 3, 3, 3, 3, 1 };

Byte bShift[] = { 1, 1, 2, 2, 2, 2, 2, 2,
						1, 2, 2, 2, 2, 2, 2, 1 };

// Table for key permutation
Byte bPC1[] = { 56, 48, 40, 32, 24, 16,  8,
					  0, 57, 49, 41, 33, 25, 17,
					  9,  1, 58, 50, 42, 34, 26,
					 18, 10,  2, 59, 51, 43, 35,
					 62, 54, 46, 38, 30, 22, 14,
					  6, 61, 53, 45, 37, 29, 21,
					 13,  5, 60, 52, 44, 36, 28,
					 20, 12,  4, 27, 19, 11,  3 };

// Table for compression permutation
Byte bPC2[] = { 13, 16, 10, 23,  0,  4,
					  2, 27, 14,  5, 20,  9,
					 22, 18, 11,  3, 25,  7,
					 15,  6, 26, 19, 12,  1,
					 40, 51, 30, 36, 46, 54,
					 29, 39, 50, 44, 32, 47,
					 43, 48, 38, 55, 33, 52,
					 45, 41, 49, 35, 28, 31 };

// Table for initial permutation IP
Byte bIP1[] = { 57, 49, 41, 33, 25, 17,  9,  1,
					 59, 51, 43, 35, 27, 19, 11,  3,
			  		 61, 53, 45, 37, 29, 21, 13,  5,
					 63, 55, 47, 39, 31, 23, 15,  7,
					 56, 48, 40, 32, 24, 16,  8,  0,
					 58, 50, 42, 34, 26, 18, 10,  2,
					 60, 52, 44, 36, 28, 20, 12,  4,
					 62, 54, 46, 38, 30, 22, 14,  6 };

//                               -1
// Table for final permutation IP
Byte bIP2[] = { 39,  7, 47, 15, 55, 23, 63, 31,
					 38,  6, 46, 14, 54, 22, 62, 30,
					 37,  5, 45, 13, 53, 21, 61, 29,
					 36,  4, 44, 12, 52, 20, 60, 28,
					 35,  3, 43, 11, 51, 19, 59, 27,
					 34,  2, 42, 10, 50, 18, 58, 26,
					 33,  1, 41,  9, 49, 17, 57, 25,
					 32,  0, 40,  8, 48, 16, 56, 24 };

// Table for expansion permutation
Byte bE[] = { 31,  0,  1,  2,  3,  4,
				   3,  4,  5,  6,  7,  8,
				   7,  8,  9, 10, 11, 12,
	  			  11, 12, 13, 14, 15, 16,
				  15, 16, 17, 18, 19, 20,
				  19, 20, 21, 22, 23, 24,
				  23, 24, 25, 26, 27, 28,
				  27, 28, 29, 30, 31,  0 };

// Table for P-Box permutation
Byte bP[] = { 15,  6, 19, 20, 28, 11, 27, 16,
				   0, 14, 22, 25,  4, 17, 30,  9,
					1,  7, 23, 13, 31, 26,  2,  8,
				  18, 12, 29,  5, 21, 10,  3, 24 };

CDESKey::CDESKey()
{
}

CDESKey::CDESKey(const CDESKey &cDESKey)
{
	dwMaster = cDESKey.dwMaster;
	MakeKeys();
}

CDESKey::CDESKey(const Byte *pbMaster, Word wKeySize)
{
	Dword dwDESKey = DWORDCONST(0);
	memcpy(&dwDESKey, pbMaster, wKeySize);
	dwDESKey = REVERSEDWORD(dwDESKey);
	MIRROR(dwDESKey, 64, dwMaster);
	MakeKeys();
	dwDESKey = 0;
}

CDESKey::CDESKey(Dword dwDESKey)
{
	MIRROR(dwDESKey, 64, dwMaster);
	MakeKeys();
	dwDESKey = 0;								// Cleanup
}

CDESKey::~CDESKey()
{
	for (int i = 0; i < 16; i++)			// Cleanup
		dwKey[i] = 0;
	dwMaster = 0;
}

void CDESKey::MakeKeys()
{
	Dword c = 0;
	Dword d = 0;
	for (int k = 0; k < 28; k++) {
		PUTBIT(c, k, GETBIT(dwMaster, bPC1[k]));
		PUTBIT(d, k, GETBIT(dwMaster, bPC1[k + 28]));
	}
	for (int q = 0; q < 16; q++) {
		c = (c >> bShift[q]) | ((c & wMask[q]) << (28 - bShift[q]));
		d = (d >> bShift[q]) | ((d & wMask[q]) << (28 - bShift[q]));
		Dword t = ((Dword)d << 28) | c;
		STIRBITS(t, 48, bPC2, dwKey[q]);
	}
	c = d = 0;									// Cleanup
}

C3DESKey::C3DESKey(Dword a, Dword b, Dword c)
{
	cDESKey[0] = CDESKey(a);
	cDESKey[1] = CDESKey(b);
	cDESKey[2] = CDESKey(c);
	a = b = c = 0;								// Cleanup
}

C3DESKey::C3DESKey(const Byte *pbData, Word/* wKeySize*/)
{
	cDESKey[0] = CDESKey(pbData + 0 * CDESKey::KEYSIZE, CDESKey::KEYSIZE);
	cDESKey[1] = CDESKey(pbData + 1 * CDESKey::KEYSIZE, CDESKey::KEYSIZE);
	cDESKey[2] = CDESKey(pbData + 2 * CDESKey::KEYSIZE, CDESKey::KEYSIZE);
}

C3DESKey::~C3DESKey()
{
}

CDESBlock::CDESBlock()
{
	dwData = 0;									// This is here for definiteness
}

CDESBlock::CDESBlock(const CDESBlock &cDESBlock)
{
	dwData = cDESBlock.dwData;
}

CDESBlock::CDESBlock(Dword dwValue)
{
	MIRROR(dwValue, 64, dwData);
}

CDESBlock::CDESBlock(const Byte *pbData, Word wLength)
{
	SetData(pbData, wLength);
}

CDESBlock::~CDESBlock()
{
	dwData = 0;									// Cleanup
}

void CDESBlock::Encrypt(const CDESKey &cDESKey)
{
	Crypt(cDESKey, false);
}

void CDESBlock::Decrypt(const CDESKey &cDESKey)
{
	Crypt(cDESKey, true);
}

// fgDirection: true = decrypt
//              false = encrypt
void CDESBlock::Crypt(const CDESKey &cDESKey, bool fgDirection)
{
	Dword a;
	Dword b;

	STIRBITS(dwData, 64, bIP1, a);
	Word r = HiWORD(a);
	Word l = LoWORD(a);
	Word s;
	for (int i = 0; i < 16; i++) {
		STIRBITS(r, 48, bE, a);
		if (fgDirection)
			b = a ^ cDESKey.GetKey(15 - i);
		else
			b = a ^ cDESKey.GetKey(i);
		a = 0;
		for (int j = 0; j < 8; j++) {
			a <<= 4;
			a |= bS[j][(b >> ((7 - j) * 6)) & 0x3f];
		}
		STIRBITS(a, 32, bP, b);
		
		s = r;
		r = l ^ LoWORD(b);
		l = s;
	}
	a = ((Dword)l << 32) | r;
	STIRBITS(a, 64, bIP2, dwData);
	a = b = l = r = 0;						// Cleanup
}

void CDESBlock::SetValue(Dword dwValue)
{
	MIRROR(dwValue, 64, dwData);
}

Dword CDESBlock::GetValue()
{
	MIRROR(dwData, 64, dwResult);
	return dwResult;							// It is normal to invoke this method
													// only for encrypted data, so we do
													// not need to clean the buffer variable.
}

void CDESBlock::SetData(const Byte *pbData, Word/* wLength*/)
{
	Dword dwValue = 0;
	memcpy(&dwValue, pbData, BLOCKSIZE);
	dwValue = REVERSEDWORD(dwValue);
	MIRROR(dwValue, 64, dwData);
	dwValue = 0;								// Cleanup
}

Byte *CDESBlock::GetData()
{
	MIRROR(dwData, 64, dwResult);
	dwResult = REVERSEDWORD(dwResult);
	return (Byte *)&dwResult;
}

C3DESBlock::C3DESBlock() : CDESBlock()
{
}

C3DESBlock::C3DESBlock(Dword dwValue) : CDESBlock(dwValue)
{
}

C3DESBlock::C3DESBlock(const Byte *pbData, Word wLength) : CDESBlock(pbData, wLength)
{
}

void C3DESBlock::Encrypt(const C3DESKey &c3DESKey)
{
	CDESBlock::Encrypt(c3DESKey.GetKey(0));
	CDESBlock::Decrypt(c3DESKey.GetKey(1));
	CDESBlock::Encrypt(c3DESKey.GetKey(2));
}

void C3DESBlock::Decrypt(const C3DESKey &c3DESKey)
{
	CDESBlock::Decrypt(c3DESKey.GetKey(2));
	CDESBlock::Encrypt(c3DESKey.GetKey(1));
	CDESBlock::Decrypt(c3DESKey.GetKey(0));
}
