/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "idea.hpp"

CIDEAKey::CIDEAKey()
{
}

CIDEAKey::CIDEAKey(const CIDEAKey &cIDEAKey)
{
	dwMaster[0] = cIDEAKey.dwMaster[0];
	dwMaster[1] = cIDEAKey.dwMaster[1];
	MakeKeys();
}

CIDEAKey::CIDEAKey(const Byte *pbMaster, Word wKeyLength)
{
	memcpy((Byte *)dwMaster, pbMaster, wKeyLength);
	memset((Byte *)dwMaster + wKeyLength, 0, sizeof(dwMaster) - wKeyLength);
	dwMaster[0] = REVERSEDWORD(dwMaster[0]);
	dwMaster[1] = REVERSEDWORD(dwMaster[1]);
	MakeKeys();
}

CIDEAKey::CIDEAKey(Dword dwLeft, Dword dwRight)
{
	dwMaster[0] = dwLeft;
	dwMaster[1] = dwRight;
	MakeKeys();
	dwLeft = 0;									// Cleanup
	dwRight = 0;
}

CIDEAKey::~CIDEAKey()
{
	for (int i = 0; i < 104; i++)
		sKey[i] = 0;
	dwMaster[0] = 0;							// Cleanup
	dwMaster[1] = 0;
}

void CIDEAKey::MakeKeys()
{
	int i;
	for (i = 0; i < 6; i++) {
		sKey[i * 8    ] = DWSHORT0(dwMaster[0]);
		sKey[i * 8 + 1] = DWSHORT1(dwMaster[0]);
		sKey[i * 8 + 2] = DWSHORT2(dwMaster[0]);
		sKey[i * 8 + 3] = DWSHORT3(dwMaster[0]);
		sKey[i * 8 + 4] = DWSHORT0(dwMaster[1]);
		sKey[i * 8 + 5] = DWSHORT1(dwMaster[1]);
		sKey[i * 8 + 6] = DWSHORT2(dwMaster[1]);
		sKey[i * 8 + 7] = DWSHORT3(dwMaster[1]);
		Dword l = (dwMaster[0] << 25) | ((dwMaster[1] >> 39) & 0x1ffffff);
		Dword r = (dwMaster[1] << 25) | ((dwMaster[0] >> 39) & 0x1ffffff);
		dwMaster[0] = l;
		dwMaster[1] = r;
	}
	sKey[48] = DWSHORT0(dwMaster[0]);
	sKey[49] = DWSHORT1(dwMaster[0]);
	sKey[50] = DWSHORT2(dwMaster[0]);
	sKey[51] = DWSHORT3(dwMaster[0]);
	
	sKey[52] = ModInv(sKey[48], 0x10001);
	sKey[53] = -sKey[49];
	sKey[54] = -sKey[50];
	sKey[55] = ModInv(sKey[51], 0x10001);
	sKey[56] = sKey[46];
	sKey[57] = sKey[47];
	for (i = 1; i < 8; i++) {
		sKey[52 + i * 6] = ModInv(sKey[48 - i * 6], 0x10001);
		sKey[53 + i * 6] = -sKey[50 - i * 6];
		sKey[54 + i * 6] = -sKey[49 - i * 6];
		sKey[55 + i * 6] = ModInv(sKey[51 - i * 6], 0x10001);
		sKey[56 + i * 6] = sKey[46 - i * 6];
		sKey[57 + i * 6] = sKey[47 - i * 6];
	}
	sKey[100] = ModInv(sKey[0], 0x10001);
	sKey[101] = -sKey[1];
	sKey[102] = -sKey[2];
	sKey[103] = ModInv(sKey[3], 0x10001);
}

Short CIDEAKey::ModInv(Word a, Word m)
{
	Word j = 1, i = 0, b = m, c = a, x, y;
	if (a == 0)
		c = 0x10000;

	while (c != 0) {
		x = b / c;
		y = b - x * c;
		b = c;
		c = y;
		y = j;
		j = i - j * x;
		i = y;
	} 
	if (i & 0x80000000)
		i += m;
	return (Short)i;
}

CIDEABlock::CIDEABlock()
{
	dwData = 0;									// This is here for definiteness
}

CIDEABlock::CIDEABlock(const CIDEABlock &cIDEABlock)
{
	dwData = cIDEABlock.dwData;
}

CIDEABlock::CIDEABlock(Dword dwValue)
{
	dwData = dwValue;
}

CIDEABlock::CIDEABlock(const Byte *pbData, Word wLength)
{
	SetData(pbData, wLength);
}

CIDEABlock::~CIDEABlock()
{
	dwData = 0;									// Cleanup
}

void CIDEABlock::Encrypt(const CIDEAKey &cIDEAKey)
{
	Crypt(cIDEAKey, false);
}

Short CIDEABlock::Mul(Short a, Short b)
{
	Word p = (Word)a * (Word)b;
	if (p) {
		p = LOSHORT(p) - HISHORT(p);
		a = (Short)(p - (Word)HISHORT(p));
	} else {
		a = 1 - a - b;
	}
	return a;
}

void CIDEABlock::Decrypt(const CIDEAKey &cIDEAKey)
{
	Crypt(cIDEAKey, true);
}

// fgDirection: true = decrypt
//              false = encrypt
void CIDEABlock::Crypt(const CIDEAKey &cIDEAKey, bool fgDirection)
{
	Short x1 = DWSHORT0(dwData);
	Short x2 = DWSHORT1(dwData);
	Short x3 = DWSHORT2(dwData);
	Short x4 = DWSHORT3(dwData);
	Short t0, t1, t2, a, b = 0;
	
	if (fgDirection)
		b = 52;

	for (int i = 0; i < 8; i++) {
		x1 = Mul(x1, cIDEAKey.GetKey(b + i * 6));
		x4 = Mul(x4, cIDEAKey.GetKey(b + i * 6 + 3));
		x2 = x2 + cIDEAKey.GetKey(b + i * 6 + 1);
		x3 = x3 + cIDEAKey.GetKey(b + i * 6 + 2);
		t0 = Mul(cIDEAKey.GetKey(b + i * 6 + 4), x1 ^ x3);
		t1 = Mul(cIDEAKey.GetKey(b + i * 6 + 5), t0 + (x2 ^ x4));
		t2 = t0 + t1;
		x1 ^= t1;
		x4 ^= t2;
		a = x2 ^ t2;
		x2 = x3 ^ t1;
		x3 = a;
	}
	dwData = MAKEDWORD((Short)(Mul(x1, cIDEAKey.GetKey(b + 48))),
							 (Short)(x3 + cIDEAKey.GetKey(b + 49)),
							 (Short)(x2 + cIDEAKey.GetKey(b + 50)),
							 (Short)(Mul(x4, cIDEAKey.GetKey(b + 51))));
}

void CIDEABlock::SetValue(Dword dwValue)
{
	dwData = dwValue;
}

Dword CIDEABlock::GetValue()
{
	return dwData;
}

void CIDEABlock::SetData(const Byte *pbData, Word wLength)
{
	dwData = 0;
	memcpy(&dwData, pbData, wLength);
	dwData = REVERSEDWORD(dwData);
}

Byte *CIDEABlock::GetData()
{
	dwResult = REVERSEDWORD(dwData);
	return (Byte *)&dwResult;
}
