/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "mars.hpp"
#include "marssboxes.hpp"

#define rol(p, q) (((p) << (q)) | ((p) >> (32 - (q))))
#define ror(p, q) (((p) >> (q)) | ((p) << (32 - (q))))

#define f_mix(a, b, c, d)			\
	r = ror(a, 8);						\
	b ^= wS[a & 255];					\
	b += wS[(r & 255) + 256];		\
	r = ror(a, 16);					\
	a  = ror(a, 24);					\
	c += wS[r & 255];					\
	d ^= wS[(a & 255) + 256]

#define b_mix(a, b, c, d)			\
	r = rol(a, 8);						\
	b ^= wS[(a & 255) + 256];		\
	c -= wS[r & 255];					\
	r = rol(a, 16);					\
	a = rol(a, 24);					\
	d -= wS[(r & 255) + 256];		\
	d ^= wS[a & 255]

#define f_ktr(a, b, c, d, i, w)	\
	m = a + w[i];						\
	a = rol(a, 13);					\
	r = a * w[i + 1];					\
	l = wS[m & 511];					\
	r = rol(r, 5);						\
	c += rol(m, r);					\
	l ^= r;								\
	r = rol(r, 5);						\
	l ^= r;								\
	d ^= r;								\
	b += rol(l, r)

#define r_ktr(a, b, c, d, i, w)	\
	r = a * w[i + 1];					\
	a = ror(a, 13);					\
	m = a + w[i];						\
	l = wS[m & 511];					\
	r = rol(r, 5);						\
	l ^= r;								\
	c -= rol(m, r);					\
	r = rol(r, 5);						\
	l ^= r;								\
	d ^= r;								\
	b -= rol(l, r)

CMARSKey::CMARSKey()
{
}

CMARSKey::CMARSKey(const CMARSKey &cMARSKey)
{
	for (int i = 0; i < 14; i++)
		wMaster[i] = cMARSKey.wMaster[i];
	wKeyLength = cMARSKey.wKeyLength;
	MakeKeys();
}

CMARSKey::CMARSKey(const Byte *pbMasterKey, Word wKeyLength)
{
	memcpy((Byte *)wMaster, pbMasterKey, wKeyLength);
	memset((Byte *)wMaster + wKeyLength, 0, sizeof(wMaster) - wKeyLength);
	CMARSKey::wKeyLength = wKeyLength / BYTESINWORD;
	MakeKeys();
}

CMARSKey::CMARSKey(const Word *pwMasterKey, Word wKeyLength)
{
	for (Word i = 0; i < wKeyLength; i++)
		wMaster[i] = pwMasterKey[i];
	CMARSKey::wKeyLength = wKeyLength;
	MakeKeys();
}

CMARSKey::CMARSKey(Dword l, Dword r)
{
	wMaster[0] = HiWORD(l);
	wMaster[1] = LoWORD(l);
	wMaster[2] = HiWORD(r);
	wMaster[3] = LoWORD(r);
	wKeyLength = 4;
	MakeKeys();
}

CMARSKey::~CMARSKey()
{
	int i;
	for (i = 0; i < 14; i++)				// Cleanup
		wMaster[i] = 0;
	for (i = 0; i < 40; i++)
		wKey[i] = 0;
}

void CMARSKey::MakeKeys()
{
	int i, j;
	Word T[47], v, w;

	for (i = 0; i < 7; i++)
		T[i] = wS[i];

	for (i = 0; i < 39; i++)
		T[i + 7] = rol(T[i] ^ T[i + 5], 3) ^ wMaster[i % wKeyLength] ^ i;
	T[46] = wKeyLength;
	
	for (j = 0; j < 7; j++) {
		for(i = 1; i < 40; i++)
			T[i + 7] = rol(T[i + 7] + wS[T[i + 6] & 0x1ff], 9);
		T[7] = rol(T[7] + wS[T[46] & 0x1ff], 9);
	}

	for (i = 0; i < 40; i++)
		wKey[(i * 7) % 40] = T[i + 7];

	for (i = 5; i < 37; i += 2) {
		v = wKey[i] | 3;
		w = (~v ^ (v >> 1)) & 0x7fffffff;
		w &= (w >> 1) & (w >> 2);
		w &= (w >> 3) & (w >> 6);
		if (w) {
			w <<= 1;
			w |= (w << 1);
			w |= (w << 2);
			w |= (w << 4);
			w |= (w << 1) & ~v & 0x80000000;
			w &= 0xfffffffc;
			v ^= (rol(wS[265 + (wKey[i] & 3)], wKey[i + 3] & 31) & w);
		}
		wKey[i] = v;
	}
}

CMARSBlock::CMARSBlock()
{
	for (int i = 0; i < 4; i++)			// This is here for definiteness
		wData[i] = 0;
}

CMARSBlock::CMARSBlock(const CMARSBlock &cMARSBlock)
{
	for (int i = 0; i < 4; i++)
		wData[i] = cMARSBlock.wData[i];
}

CMARSBlock::CMARSBlock(Dword l, Dword r)
{
	wData[0] = HiWORD(l);
	wData[1] = LoWORD(l);
	wData[2] = HiWORD(r);
	wData[3] = LoWORD(r);
}

CMARSBlock::CMARSBlock(const Byte *pbData, Word wLength)
{
	SetData(pbData, wLength);
}

CMARSBlock::~CMARSBlock()
{
	for (int i = 0; i < 4; i++)			// Cleanup
		wData[i] = 0;
}

void CMARSBlock::Encrypt(const CMARSKey &cMARSKey)
{
	const Word *w = cMARSKey.GetKeys();
	Word a = wData[0] + w[0];
	Word b = wData[1] + w[1];
	Word c = wData[2] + w[2];
	Word d = wData[3] + w[3];
	Word m, l, r;

	f_mix(a, b, c, d); a += d;
	f_mix(b, c, d, a); b += c;
	f_mix(c, d, a, b);
	f_mix(d, a, b, c);
	f_mix(a, b, c, d); a += d;
	f_mix(b, c, d, a); b += c;
	f_mix(c, d, a, b);
	f_mix(d, a, b, c);
	
	f_ktr(a, b, c, d,  4, w);
	f_ktr(b, c, d, a,  6, w);
	f_ktr(c, d, a, b,  8, w);
	f_ktr(d, a, b, c, 10, w);
	f_ktr(a, b, c, d, 12, w);
	f_ktr(b, c, d, a, 14, w);
	f_ktr(c, d, a, b, 16, w);
	f_ktr(d, a, b, c, 18, w);
	f_ktr(a, d, c, b, 20, w);
	f_ktr(b, a, d, c, 22, w);
	f_ktr(c, b, a, d, 24, w);
	f_ktr(d, c, b, a, 26, w);
	f_ktr(a, d, c, b, 28, w);
	f_ktr(b, a, d, c, 30, w);
	f_ktr(c, b, a, d, 32, w);
	f_ktr(d, c, b, a, 34, w);

	b_mix(a, b, c, d);
	b_mix(b, c, d, a); c -= b;
	b_mix(c, d, a, b); d -= a;
	b_mix(d, a, b, c);
	b_mix(a, b, c, d);
	b_mix(b, c, d, a); c -= b;
	b_mix(c, d, a, b); d -= a;
	b_mix(d, a, b, c);

	wData[0] = a - w[36];
	wData[1] = b - w[37];
	wData[2] = c - w[38];
	wData[3] = d - w[39];
}

void CMARSBlock::Decrypt(const CMARSKey &cMARSKey)
{
	const Word *w = cMARSKey.GetKeys();
	Word d = wData[0] + w[36];
	Word c = wData[1] + w[37];
	Word b = wData[2] + w[38];
	Word a = wData[3] + w[39];
	Word m, l, r;

	f_mix(a, b, c, d); a += d;
	f_mix(b, c, d, a); b += c;
	f_mix(c, d, a, b);
	f_mix(d, a, b, c);
	f_mix(a, b, c, d); a += d;
	f_mix(b, c, d, a); b += c;
	f_mix(c, d, a, b);
	f_mix(d, a, b, c);
	
	r_ktr(a, b, c, d, 34, w);
	r_ktr(b, c, d, a, 32, w);
	r_ktr(c, d, a, b, 30, w);
	r_ktr(d, a, b, c, 28, w);
	r_ktr(a, b, c, d, 26, w);
	r_ktr(b, c, d, a, 24, w);
	r_ktr(c, d, a, b, 22, w);
	r_ktr(d, a, b, c, 20, w);
	r_ktr(a, d, c, b, 18, w);
	r_ktr(b, a, d, c, 16, w);
	r_ktr(c, b, a, d, 14, w);
	r_ktr(d, c, b, a, 12, w);
	r_ktr(a, d, c, b, 10, w);
	r_ktr(b, a, d, c,  8, w);
	r_ktr(c, b, a, d,  6, w);
	r_ktr(d, c, b, a,  4, w);

	b_mix(a, b, c, d);
	b_mix(b, c, d, a); c -= b;
	b_mix(c, d, a, b); d -= a;
	b_mix(d, a, b, c);
	b_mix(a, b, c, d);
	b_mix(b, c, d, a); c -= b;
	b_mix(c, d, a, b); d -= a;
	b_mix(d, a, b, c);

	wData[0] = d - w[0];
	wData[1] = c - w[1];
	wData[2] = b - w[2];
	wData[3] = a - w[3];
}

void CMARSBlock::SetData(const Byte *pbData, Word wLength)
{
	memcpy((Byte *)wData, pbData, wLength);
	memset((Byte *)wData + wLength, 0, sizeof(wData) - wLength);
}

void CMARSBlock::SetData(Word a, Word b, Word c, Word d)
{
	wData[0] = a;
	wData[1] = b;
	wData[2] = c;
	wData[3] = d;
}

Word CMARSBlock::GetData(Word i)
{
	return wData[i];
}

Byte *CMARSBlock::GetData()
{
	return (Byte *)wData;
}
