/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "rc2.hpp"
#include "rc2sboxes.hpp"

#define rol(p, q) (((p) << (q)) | ((p) >> (16 - (q))))
#define ror(p, q) (((p) >> (q)) | ((p) << (16 - (q))))

#define f(a, b, c, d, j) a = rol((Short)(a + (b & ~d) + (c & d) + S[j    ]), 1); \
								 b = rol((Short)(b + (c & ~a) + (d & a) + S[j + 1]), 2); \
								 c = rol((Short)(c + (d & ~b) + (a & b) + S[j + 2]), 3); \
								 d = rol((Short)(d + (a & ~c) + (b & c) + S[j + 3]), 5)

#define g(a, b, c, d, j) d = ror(d, 5) - (a & ~c) - (b & c) - S[j + 3]; \
								 c = ror(c, 3) - (d & ~b) - (a & b) - S[j + 2]; \
								 b = ror(b, 2) - (c & ~a) - (d & a) - S[j + 1]; \
								 a = ror(a, 1) - (b & ~d) - (c & d) - S[j    ];

#define q(a, b, c, d) a += S[d & 63]; \
							 b += S[a & 63]; \
							 c += S[b & 63]; \
							 d += S[c & 63]

#define r(a, b, c, d) d -= S[c & 63]; \
							 c -= S[b & 63]; \
							 b -= S[a & 63]; \
							 a -= S[d & 63]

CRC2Key::CRC2Key()
{
}

CRC2Key::CRC2Key(const CRC2Key &cRC2Key)
{
	for (int i = 0; i < 128; i++)
		bMaster[i] = cRC2Key.bMaster[i];
	wKeyLength = cRC2Key.wKeyLength;
	MakeKeys();
}

CRC2Key::CRC2Key(const Byte *pbMasterKey, Word wKeyLength)
{
	for (Word i = 0; i < wKeyLength; i++)
		bMaster[i] = pbMasterKey[i];
	CRC2Key::wKeyLength = wKeyLength;
	MakeKeys();
}

CRC2Key::~CRC2Key()
{
	int i;
	for (i = 0; i < 128; i++)				// Cleanup
		bMaster[i] = 0;
	for (i = 0; i < 64; i++)
		sKey[i] = 0;
}

void CRC2Key::MakeKeys()
{
	Byte *s = (Byte *)sKey;
	Word w;
	
	for (w = 0; w < wKeyLength; w++)
		s[w] = bMaster[w];
	for ( ; w < 128; w++)
		s[w] = bRC2S[(s[w - wKeyLength] + s[w - 1]) % 0x100];
	s[0] = bRC2S[s[0]];
}

CRC2Block::CRC2Block()
{
	dwData = 0;									// This is here for definiteness
}

CRC2Block::CRC2Block(const CRC2Block &cRC2Block)
{
	dwData = cRC2Block.dwData;
}

CRC2Block::CRC2Block(Dword dwValue)
{
	dwData = dwValue;
}

CRC2Block::CRC2Block(const Byte *pbData, Word wLength)
{
	dwData = 0;
	memcpy(&dwData, pbData, wLength);
}

CRC2Block::~CRC2Block()
{
	dwData = 0;									// Cleanup
}

void CRC2Block::Encrypt(const CRC2Key &cRC2Key)
{
	Short a = DWSHORT3(dwData);
	Short b = DWSHORT2(dwData);
	Short c = DWSHORT1(dwData);
	Short d = DWSHORT0(dwData);
	const Short *S = cRC2Key.GetKeys();
	
	f(a, b, c, d,  0);
	f(a, b, c, d,  4);
	f(a, b, c, d,  8);
	f(a, b, c, d, 12);
	f(a, b, c, d, 16);
	q(a, b, c, d);
	f(a, b, c, d, 20);
	f(a, b, c, d, 24);
	f(a, b, c, d, 28);
	f(a, b, c, d, 32);
	f(a, b, c, d, 36);
	f(a, b, c, d, 40);
	q(a, b, c, d);
	f(a, b, c, d, 44);
	f(a, b, c, d, 48);
	f(a, b, c, d, 52);
	f(a, b, c, d, 56);
	f(a, b, c, d, 60);
	dwData = MAKEDWORD(d, c, b, a);
}

void CRC2Block::Decrypt(const CRC2Key &cRC2Key)
{
	Short a = DWSHORT3(dwData);
	Short b = DWSHORT2(dwData);
	Short c = DWSHORT1(dwData);
	Short d = DWSHORT0(dwData);
	const Short *S = cRC2Key.GetKeys();

	g(a, b, c, d, 60);
	g(a, b, c, d, 56);
	g(a, b, c, d, 52);
	g(a, b, c, d, 48);
	g(a, b, c, d, 44);
	r(a, b, c, d);
	g(a, b, c, d, 40);
	g(a, b, c, d, 36);
	g(a, b, c, d, 32);
	g(a, b, c, d, 28);
	g(a, b, c, d, 24);
	g(a, b, c, d, 20);
	r(a, b, c, d);
	g(a, b, c, d, 16);
	g(a, b, c, d, 12);
	g(a, b, c, d,  8);
	g(a, b, c, d,  4);
	g(a, b, c, d,  0);
	dwData = MAKEDWORD(d, c, b, a);
}

void CRC2Block::SetValue(Dword dwValue)
{
	dwData = dwValue;
}

Dword CRC2Block::GetValue()
{
	return dwData;
}

void CRC2Block::SetData(const Byte *pbData, Word wLength)
{
	dwData = 0;
	memcpy(&dwData, pbData, wLength);
}

Byte *CRC2Block::GetData()
{
	return (Byte *)&dwData;
}
