/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "safer.hpp"
#include "safersboxes.hpp"

#ifdef WORDS_BIGENDIAN
#define byte(p, q)				(*((Byte *)&p + q))
#else
#define byte(p, q)				(*((Byte *)&p + (7 - q)))
#endif

#define rol(p, q)					(((p) << (q)) | ((p) >> (8 - (q))))
#define f(a, b, c, d, e, f)	\
{										\
	Byte l = byte(d, e);			\
	Byte r = byte(d, f);			\
	byte(a, b) = l + l + r;		\
	byte(a, c) = l + r;			\
}

#define g(a, b, c, d, e, f)	\
{										\
	Byte l = byte(d, e);			\
	Byte r = byte(d, f);			\
	byte(a, b) = l - r;			\
	byte(a, c) = r + r - l;		\
}

CSAFERKey::CSAFERKey()
{
}

CSAFERKey::CSAFERKey(const CSAFERKey &cSAFERKey)
{
	dwMaster = cSAFERKey.dwMaster;
	MakeKeys();
}

CSAFERKey::CSAFERKey(Dword dwValue)
{
	dwMaster = dwValue;
	MakeKeys();
	dwValue = 0;								// Cleanup
}

CSAFERKey::CSAFERKey(Byte *pbMaster, Word wLength)
{
	dwMaster = 0;
	memcpy(&dwMaster, pbMaster, wLength);
	dwMaster = REVERSEDWORD(dwMaster);
	MakeKeys();
}

CSAFERKey::~CSAFERKey()
{
	dwMaster = 0;								// Cleanup
	for (int i = 0; i < (2 * SAFERROUNDS + 1); i++)
		dwKey[i] = 0;
}

void CSAFERKey::MakeKeys()
{
	dwKey[0] = dwMaster;
	Byte R[] = {
					 byte(dwMaster, 0),
					 byte(dwMaster, 1),
					 byte(dwMaster, 2),
					 byte(dwMaster, 3),
					 byte(dwMaster, 4),
					 byte(dwMaster, 5),
					 byte(dwMaster, 6),
					 byte(dwMaster, 7),
#ifdef SK
					 0,
#endif
				  };
	int i, j;
#ifdef SK
	R[8] = R[0] ^ R[1] ^ R[2] ^ R[3] ^ R[4] ^ R[5] ^ R[6] ^ R[7];
#endif
	for (i = 1; i < (2 * SAFERROUNDS + 1); i++) {
#ifdef SK
		for (j = 0; j < 9; j++)
			R[j] = rol(R[j], 3);
		for (j = 0; j < 8; j++)
			byte(dwKey[i], j) = R[(i + j) % 9] + B[i - 1][j];
#else
		for (j = 0; j < 8; j++) {
			R[j] = rol(R[j], 3);
			byte(dwKey[i], j) = R[j] + B[i - 1][j];
		}
#endif
	}
}

CSAFERBlock::CSAFERBlock()
{
	dwData = 0;									// This is here for definiteness
}

CSAFERBlock::CSAFERBlock(const CSAFERBlock &cSAFERBlock)
{
	dwData = cSAFERBlock.dwData;
}

CSAFERBlock::CSAFERBlock(Dword dwValue)
{
	dwData = dwValue;
}

CSAFERBlock::CSAFERBlock(const Byte *pbData, Word wLength)
{
	SetData(pbData, wLength);
}

CSAFERBlock::~CSAFERBlock()
{
	dwData = 0;									// Cleanup
}

void CSAFERBlock::Encrypt(const CSAFERKey &cSAFERKey)
{
	Encrypt(cSAFERKey, cSAFERKey);
}

void CSAFERBlock::Encrypt(const CSAFERKey &cSAFERKey, const CSAFERKey &cSAFERKeyExt)
{
	Dword k, y;
	for (int i = 0; i < SAFERROUNDS; i++) {
		k = cSAFERKeyExt.GetKey(2 * i);

		byte(dwData, 0) ^= byte(k, 0);
		byte(dwData, 1) += byte(k, 1);
		byte(dwData, 2) += byte(k, 2);
		byte(dwData, 3) ^= byte(k, 3);
		byte(dwData, 4) ^= byte(k, 4);
		byte(dwData, 5) += byte(k, 5);
		byte(dwData, 6) += byte(k, 6);
		byte(dwData, 7) ^= byte(k, 7);				// (a)

		byte(dwData, 0) = S[byte(dwData, 0)];
		byte(dwData, 1) = T[byte(dwData, 1)];
		byte(dwData, 2) = T[byte(dwData, 2)];
		byte(dwData, 3) = S[byte(dwData, 3)];
		byte(dwData, 4) = S[byte(dwData, 4)];
		byte(dwData, 5) = T[byte(dwData, 5)];
		byte(dwData, 6) = T[byte(dwData, 6)];
		byte(dwData, 7) = S[byte(dwData, 7)];		// (b)
		
		k = cSAFERKey.GetKey(2 * i + 1);

		byte(dwData, 0) += byte(k, 0);
		byte(dwData, 1) ^= byte(k, 1);
		byte(dwData, 2) ^= byte(k, 2);
		byte(dwData, 3) += byte(k, 3);
		byte(dwData, 4) += byte(k, 4);
		byte(dwData, 5) ^= byte(k, 5);
		byte(dwData, 6) ^= byte(k, 6);
		byte(dwData, 7) += byte(k, 7);				// (c)
		
		f(dwData, 0, 1, dwData, 0, 1);
		f(dwData, 2, 3, dwData, 2, 3);
		f(dwData, 4, 5, dwData, 4, 5);
		f(dwData, 6, 7, dwData, 6, 7);				// (d)
		
		f(y, 0, 1, dwData, 0, 2);
		f(y, 2, 3, dwData, 4, 6);
		f(y, 4, 5, dwData, 1, 3);
		f(y, 6, 7, dwData, 5, 7);
		dwData = y;											// (e)

		f(y, 0, 1, dwData, 0, 2);
		f(y, 2, 3, dwData, 4, 6);
		f(y, 4, 5, dwData, 1, 3);
		f(y, 6, 7, dwData, 5, 7);
		dwData = y;											// (f)
	}
	k = cSAFERKeyExt.GetKey(2 * SAFERROUNDS);
	byte(dwData, 0) ^= byte(k, 0);
	byte(dwData, 1) += byte(k, 1);
	byte(dwData, 2) += byte(k, 2);
	byte(dwData, 3) ^= byte(k, 3);
	byte(dwData, 4) ^= byte(k, 4);
	byte(dwData, 5) += byte(k, 5);
	byte(dwData, 6) += byte(k, 6);
	byte(dwData, 7) ^= byte(k, 7);
	
	k = 0;													// Cleanup
	y = 0;
}

void CSAFERBlock::Decrypt(const CSAFERKey &cSAFERKey)
{
	Decrypt(cSAFERKey, cSAFERKey);
}

void CSAFERBlock::Decrypt(const CSAFERKey &cSAFERKey, const CSAFERKey &cSAFERKeyExt)
{
	Dword k, y;

	k = cSAFERKeyExt.GetKey(2 * SAFERROUNDS);
	byte(dwData, 0) ^= byte(k, 0);
	byte(dwData, 1) -= byte(k, 1);
	byte(dwData, 2) -= byte(k, 2);
	byte(dwData, 3) ^= byte(k, 3);
	byte(dwData, 4) ^= byte(k, 4);
	byte(dwData, 5) -= byte(k, 5);
	byte(dwData, 6) -= byte(k, 6);
	byte(dwData, 7) ^= byte(k, 7);
	
	for (int i = SAFERROUNDS - 1; i >= 0; i--) {
		g(dwData, 0, 1, dwData, 0, 1);
		g(dwData, 2, 3, dwData, 2, 3);
		g(dwData, 4, 5, dwData, 4, 5);
		g(dwData, 6, 7, dwData, 6, 7);				// (d)
		
		g(y, 0, 1, dwData, 0, 4);
		g(y, 2, 3, dwData, 1, 5);
		g(y, 4, 5, dwData, 2, 6);
		g(y, 6, 7, dwData, 3, 7);
		dwData = y;											// (e)

		g(y, 0, 1, dwData, 0, 4);
		g(y, 2, 3, dwData, 1, 5);
		g(y, 4, 5, dwData, 2, 6);
		g(y, 6, 7, dwData, 3, 7);
		dwData = y;											// (f)

		k = cSAFERKey.GetKey(2 * i + 1);

		byte(dwData, 0) -= byte(k, 0);
		byte(dwData, 1) ^= byte(k, 1);
		byte(dwData, 2) ^= byte(k, 2);
		byte(dwData, 3) -= byte(k, 3);
		byte(dwData, 4) -= byte(k, 4);
		byte(dwData, 5) ^= byte(k, 5);
		byte(dwData, 6) ^= byte(k, 6);
		byte(dwData, 7) -= byte(k, 7);				// (c)

		byte(dwData, 0) = T[byte(dwData, 0)];
		byte(dwData, 1) = S[byte(dwData, 1)];
		byte(dwData, 2) = S[byte(dwData, 2)];
		byte(dwData, 3) = T[byte(dwData, 3)];
		byte(dwData, 4) = T[byte(dwData, 4)];
		byte(dwData, 5) = S[byte(dwData, 5)];
		byte(dwData, 6) = S[byte(dwData, 6)];
		byte(dwData, 7) = T[byte(dwData, 7)];		// (b)

		k = cSAFERKeyExt.GetKey(2 * i);

		byte(dwData, 0) ^= byte(k, 0);
		byte(dwData, 1) -= byte(k, 1);
		byte(dwData, 2) -= byte(k, 2);
		byte(dwData, 3) ^= byte(k, 3);
		byte(dwData, 4) ^= byte(k, 4);
		byte(dwData, 5) -= byte(k, 5);
		byte(dwData, 6) -= byte(k, 6);
		byte(dwData, 7) ^= byte(k, 7);				// (a)
	}
	
	k = 0;													// Cleanup
	y = 0;
}

void CSAFERBlock::SetValue(Dword dwValue)
{
	dwData = dwValue;
}

Dword CSAFERBlock::GetValue()
{
	return dwData;
}

void CSAFERBlock::SetData(const Byte *pbData, Word wLength)
{
	dwData = 0;
	memcpy(&dwData, pbData, wLength);
	dwData = REVERSEDWORD(dwData);
}

Byte *CSAFERBlock::GetData()
{
	dwResult = REVERSEDWORD(dwData);
	return (Byte *)&dwResult;
}
