/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "agree.hpp"

bool ServerAgree(CConfigurationFile *pConfigFile, int iRemote, AGREEMENT_STR *pstrAgree)
{
	Byte bServerSymmetric[64];
	Byte bServerDigest[64];
	Byte bServerAsymmetric[64];

	Byte bClientSymmetric[64];
	Byte bClientDigest[64];
	Byte bClientAsymmetric[64];

	Byte bCommonSymmetric[64];
	Byte bCommonDigest[64];
	Byte bCommonAsymmetric[64];

	Byte i = 0, j = 0, k = 0;
	Byte e = 0, f = 0, g = 0;
	Byte a, b, c, y, z;

	bzero(bCommonSymmetric, sizeof(bCommonSymmetric));
	bzero(bCommonDigest, sizeof(bCommonDigest));
	bzero(bCommonAsymmetric, sizeof(bCommonAsymmetric));
	
	char *pszTemp;

	CTokenizedString cSymmetric(pConfigFile->GetString("algorithms", "symmetric", "serpent"), ",;:.");
	CTokenizedString cDigest(pConfigFile->GetString("algorithms", "digest", "tiger"), ",;:.");
	CTokenizedString cAsymmetric(pConfigFile->GetString("algorithms", "keyexchange", "rsa"), ",;:.");

	if ((pszTemp = cSymmetric.GetFirstString()))
		do {
			z = FindID(TrimBoth(pszTemp, " \t"), SYMMETRIC, DIGEST);
			if (z)
				bServerSymmetric[i++] = z;
		} while ((pszTemp = cSymmetric.GetNextString()));

	if ((pszTemp = cDigest.GetFirstString()))
		do {
			z = FindID(TrimBoth(pszTemp, " \t"), DIGEST, ASYMMETRIC);
			if (z)
				bServerDigest[j++] = z;
		} while ((pszTemp = cDigest.GetNextString()));

	if ((pszTemp = cAsymmetric.GetFirstString()))
		do {
			z = FindID(TrimBoth(pszTemp, " \t"), ASYMMETRIC, LAST);
			if (z)
				bServerAsymmetric[k++] = z;
		} while ((pszTemp = cAsymmetric.GetNextString()));
	
	if (i == 0 || j == 0 || k == 0)
		return false;

	if (sizeof(Byte) != scl_read(iRemote, &a, sizeof(Byte)) ||
		 sizeof(Byte) != scl_read(iRemote, &b, sizeof(Byte)) ||
		 sizeof(Byte) != scl_read(iRemote, &c, sizeof(Byte)))
		return false;
	if (a > sizeof(bClientSymmetric) ||
		 b > sizeof(bClientDigest) ||
		 c > sizeof(bClientAsymmetric) ||
		 a == 0 ||
		 b == 0 ||
		 c == 0)
		return false;
	
	if (a != scl_read(iRemote, bClientSymmetric, a) ||
		 b != scl_read(iRemote, bClientDigest, b) ||
		 c != scl_read(iRemote, bClientAsymmetric, c))
		return false;

	Byte p, q;
	for (p = 0; p < i; p++)
		for (q = 0; q < a; q++)
			if (bServerSymmetric[p] == bClientSymmetric[q]) {
				bCommonSymmetric[e++] = bServerSymmetric[p];
				break;
			}
	for (p = 0; p < j; p++)
		for (q = 0; q < b; q++)
			if (bServerDigest[p] == bClientDigest[q]) {
				bCommonDigest[f++] = bServerDigest[p];
				break;
			}
	for (p = 0; p < k; p++)
		for (q = 0; q < c; q++)
			if (bServerAsymmetric[p] == bClientAsymmetric[q]) {
				bCommonAsymmetric[g++] = bServerAsymmetric[p];
				break;
			}

	y = 0;
	do {
		pstrAgree->bSymmetric = bCommonSymmetric[y++];
		if (sizeof(Byte) != scl_write(iRemote, &pstrAgree->bSymmetric, sizeof(Byte)))
			return false;
		Word wBlockSize = FindParam(pstrAgree->bSymmetric, 0);
		Word wKeySize = FindParam(pstrAgree->bSymmetric, 1);
		pstrAgree->wBlockSize = wBlockSize;
		pstrAgree->wKeySize = wKeySize;
		wBlockSize = htonl(wBlockSize);
		wKeySize = htonl(wKeySize);
		if (sizeof(Word) != scl_write(iRemote, &wBlockSize, sizeof(Word)))
			return false;
		if (sizeof(Word) != scl_write(iRemote, &wKeySize, sizeof(Word)))
			return false;
		if (sizeof(Byte) != scl_read(iRemote, &z, sizeof(Byte)))
			return false;
	} while (AGREE != z);

	y = 0;
	do {
		pstrAgree->bDigest = bCommonDigest[y++];
		if (sizeof(Byte) != scl_write(iRemote, &pstrAgree->bDigest, sizeof(Byte)))
			return false;
		Word wDigestSize = FindParam(pstrAgree->bDigest, 0);
		pstrAgree->wDigestSize = wDigestSize;
		wDigestSize = htonl(wDigestSize);
		if (sizeof(Word) != scl_write(iRemote, &wDigestSize, sizeof(Word)))
			return false;
		if (sizeof(Byte) != scl_read(iRemote, &z, sizeof(Byte)))
			return false;
	} while (AGREE != z);

	y = 0;
	do {
		pstrAgree->bAsymmetric = bCommonAsymmetric[y++];
		if (sizeof(Byte) != scl_write(iRemote, &pstrAgree->bAsymmetric, sizeof(Byte)))
			return false;
		if (sizeof(Byte) != scl_read(iRemote, &z, sizeof(Byte)))
			return false;
	} while (AGREE != z);
	
	pstrAgree->fgKeyExchange = ((pstrAgree->bAsymmetric == DH) || (pstrAgree->bAsymmetric == MQV));

	return true;
}

bool ClientAgree(CConfigurationFile *pConfigFile, int iRemote, AGREEMENT_STR *pstrAgree)
{
	Byte bClientSymmetric[64];
	Byte bClientDigest[64];
	Byte bClientAsymmetric[64];
	Byte i = 0, j = 0, k = 0;
	Byte z;
	
	char *pszTemp;

	CTokenizedString cSymmetric(pConfigFile->GetString("algorithms", "symmetric", "serpent"), ",;:.");
	CTokenizedString cDigest(pConfigFile->GetString("algorithms", "digest", "tiger"), ",;:.");
	CTokenizedString cAsymmetric(pConfigFile->GetString("algorithms", "keyexchange", "rsa"), ",;:.");

	if ((pszTemp = cSymmetric.GetFirstString()))
		do {
			z = FindID(TrimBoth(pszTemp, " \t"), SYMMETRIC, DIGEST);
			if (z)
				bClientSymmetric[i++] = z;
		} while ((pszTemp = cSymmetric.GetNextString()));

	if ((pszTemp = cDigest.GetFirstString()))
		do {
			z = FindID(TrimBoth(pszTemp, " \t"), DIGEST, ASYMMETRIC);
			if (z)
				bClientDigest[j++] = z;
		} while ((pszTemp = cDigest.GetNextString()));

	if ((pszTemp = cAsymmetric.GetFirstString()))
		do {
			z = FindID(TrimBoth(pszTemp, " \t"), ASYMMETRIC, LAST);
			if (z)
				bClientAsymmetric[k++] = z;
		} while ((pszTemp = cAsymmetric.GetNextString()));

	if (i == 0 || j == 0 || k == 0)
		return false;

	if (sizeof(Byte) != scl_write(iRemote, &i, sizeof(Byte)) ||
		 sizeof(Byte) != scl_write(iRemote, &j, sizeof(Byte)) ||
		 sizeof(Byte) != scl_write(iRemote, &k, sizeof(Byte)) ||
		 i != scl_write(iRemote, bClientSymmetric, i) ||
		 j != scl_write(iRemote, bClientDigest, j) ||
		 k != scl_write(iRemote, bClientAsymmetric, k))
		return false;

	Word wBlockSize;
	Word wKeySize;
	Word wDigestSize;

	do {
		if (sizeof(Byte) != scl_read(iRemote, &pstrAgree->bSymmetric, sizeof(Byte)))
			return false;
		if (0 == pstrAgree->bSymmetric)
			return false;
		if (sizeof(Word) != scl_read(iRemote, &wBlockSize, sizeof(Word)))
			return false;
		if (sizeof(Word) != scl_read(iRemote, &wKeySize, sizeof(Word)))
			return false;
		pstrAgree->wBlockSize = ntohl(wBlockSize);
		pstrAgree->wKeySize = ntohl(wKeySize);
		z = AGREE;
		if (pstrAgree->wBlockSize == FindParam(pstrAgree->bSymmetric, 0) &&
			 pstrAgree->wKeySize == FindParam(pstrAgree->bSymmetric, 1))
			z = AGREE;
		else
			z = REFUSE;
		if (sizeof(Byte) != scl_write(iRemote, &z, sizeof(Byte)))
			return false;
	} while (AGREE != z);

	do {
		if (sizeof(Byte) != scl_read(iRemote, &pstrAgree->bDigest, sizeof(Byte)))
			return false;
		if (0 == pstrAgree->bDigest)
			return false;
		if (sizeof(Word) != scl_read(iRemote, &wDigestSize, sizeof(Word)))
			return false;
		pstrAgree->wDigestSize = ntohl(wDigestSize);
		if (pstrAgree->wDigestSize == FindParam(pstrAgree->bDigest, 0))
			z = AGREE;
		else
			z = REFUSE;
		if (sizeof(Byte) != scl_write(iRemote, &z, sizeof(Byte)))
			return false;
	} while (AGREE != z);

	do {
		if (sizeof(Byte) != scl_read(iRemote, &pstrAgree->bAsymmetric, sizeof(Byte)))
			return false;
		if (0 == pstrAgree->bAsymmetric)
			return false;
		z = AGREE;
		if (sizeof(Byte) != scl_write(iRemote, &z, sizeof(Byte)))
			return false;
	} while (AGREE != z);
	
	pstrAgree->fgKeyExchange = ((pstrAgree->bAsymmetric == DH) || (pstrAgree->bAsymmetric == MQV));

	return true;
}
