/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>

#ifndef WIN32
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#endif

#include "config.h"
#include "sssl.hpp"
#include "initrnd.hpp"

char *write_buffer;
char *read_buffer;

// Function prototypes
int Client();
int Server();

#ifdef WIN32
DWORD WINAPI ServerThread(LPVOID);
#endif

struct sockaddr_in client;
struct sockaddr_in child;
struct sockaddr_in serv;
int listenfd;
int connectfd;

int main(int/* argc*/, char ** /*argv*/)
{
#ifdef WIN32
	WSADATA WSAData;
	int iStatus;
	if (0 != (iStatus = WSAStartup(MAKEWORD(2, 2), &WSAData))) {
		printf("Socket init error: %s\n%s.\n", WSAData.szDescription, WSAData.szSystemStatus);
		return 1;
	}
#endif

	InitDefPRNG();

// Create a socket for the server process	
	if ((listenfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		printf("Failed.\nInvalid socket call!\n");
		return 1;
	}
// Create a socket for the client socket
	if ((connectfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		printf("Failed.\nInvalid socket call!\n");
		SSSLclose(listenfd);
		return 1;
	}

	memset(&serv, 0, sizeof(serv));
	serv.sin_family = AF_INET;
	serv.sin_addr.s_addr = htonl(INADDR_ANY);

	if (bind(listenfd, (struct sockaddr *)&serv, sizeof(serv)) < 0) {
		printf("Failed %s.\nInvalid bind call!\n", strerror(errno));
		SSSLclose(listenfd);
		return 1;
	}

// Declare that the server socket will listen for incoming connections	
	if (listen(listenfd, SOMAXCONN) < 0) {
		printf("Failed: %s.\nInvalid listen call!\n", strerror(errno));
		SSSLclose(listenfd);
		SSSLclose(connectfd);
		return 1;
	}
// Get the name of the server socket. We need this info give it to the
// SSSLconnect call.
	socklen_t servlen = (socklen_t)sizeof(struct sockaddr_in);
	bzero(&serv, sizeof(struct sockaddr_in));
	if (getsockname(listenfd, (struct sockaddr *)&serv, &servlen) == -1) {
		printf("Failed.\nInvalid getsockname call!\n");
		SSSLclose(listenfd);
		SSSLclose(connectfd);
		return 4;
	}
	
	printf("Allocating buffer, one megabyte in size: ......................\n");
	write_buffer = new char[1048576];
	read_buffer = new char[1048576];
	if (NULL == write_buffer ||
		 NULL == read_buffer) {
		printf("Failed.\nError allocating memory!\n");
		if (NULL != write_buffer)
			delete write_buffer;
		SSSLclose(listenfd);
		SSSLclose(connectfd);
		return 3;
	}

	printf("Filling the newly allocated buffer with pseudo random data: ...\n");
	GetDefPRNG().GetRandomData(write_buffer, sizeof(write_buffer));

	int iResult = 0;
	
#ifndef WIN32
	pid_t child_process;
	if ((child_process = fork()) == -1) {
		printf("Failed.\nFork error!\n");
		iResult = 8;
	} else if (child_process) {
		iResult = Server();
	} else {
		iResult = Client();
	}
#else
	unsigned long lThreadID;
	CreateThread(NULL, 0, ServerThread, 0L, 0, &lThreadID);
	iResult = Server();
#endif
	
	delete write_buffer;
	delete read_buffer;
	
	SSSLclose(listenfd);
	SSSLclose(connectfd);

#ifdef WIN32
	WSACleanup();
#endif

	return iResult;
}

#ifdef WIN32
DWORD WINAPI ServerThread(LPVOID)
{
#ifdef WIN32
	Sleep(1000);
#else
	sleep(1);
#endif
	Client();
	return 0L;
}
#endif

int Server()
{
	int sockfd;
	bzero(&child, sizeof(struct sockaddr_in));
	socklen_t childlen = (socklen_t)sizeof(struct sockaddr_in);
	printf("Ready to accept a test connection: ............................\n");
	if ((sockfd = SSSLaccept(listenfd, (struct sockaddr *)&child, &childlen)) < 0) {
		printf("Failed.\nInvalid SSSLaccept call: %s!\n", SSSLerror());
		return 2;
	}
	printf("Reading 1 byte of test data: ..................................\n");
	if (1 != SSSLread(sockfd, read_buffer, 1)) {
		printf("Failed.\nInvalid SSSLread call: %s!\n", SSSLerror());
		return 7;
	}
	if (read_buffer[0] != write_buffer[0])
		printf("Failed.\nRead/write mismatch!\n");
	printf("Writing 4 bytes of test data: .................................\n");
	if (4 != SSSLwrite(sockfd, write_buffer, 4)) {
		printf("Failed.\nInvalid SSSLwrite call: %s!\n", SSSLerror());
		return 6;
	}
	printf("Reading 1 kilobyte of test data: ..............................\n");
	if (1024 != SSSLread(sockfd, read_buffer, 1024)) {
		printf("Failed.\nInvalid SSSLread call: %s!\n", SSSLerror());
		return 7;
	}
	Word i;
	for (i = 0; i < 1024; i++)
		if (read_buffer[i] != write_buffer[i]) {
			printf("Failed.\nRead/write mismatch!\n");
			break;
		}
	printf("Reading 1 megabyte of test data: ..............................\n");
	if (1048576 != SSSLread(sockfd, read_buffer, 1048576)) {
		printf("Failed.\nInvalid SSSLread call %s!\n", SSSLerror());
		return 7;
	}
	for (i = 0; i < 1048576; i++)
		if (read_buffer[i] != write_buffer[i]) {
			printf("Failed.\nRead/write mismatch!\n");
			break;
		}
	SSSLclose(sockfd);
	printf("............................................................... Ok.\n");
// Finish gracefully		
//	waitpid(child_process, NULL, 0);
	return 0;
}

int Client()
{
	printf(" Creating a client process. Connecting to the server: .........\n");
	bzero(&client, sizeof(struct sockaddr_in));
	client.sin_family = AF_INET;
#ifdef WIN32
	client.sin_addr.s_addr = inet_addr("127.0.0.1");
#endif
	client.sin_port = serv.sin_port;
#ifndef WIN32
	inet_aton("127.0.0.1", &client.sin_addr);
#endif
	if (SSSLconnect(connectfd, (sockaddr *)&client, sizeof(struct sockaddr_in)) == -1) {
		printf("Failed.\n Invalid SSSLconnect call: %s!\n", SSSLerror());
		return 3;
	}
	printf(" Writing 1 byte of test data: .................................\n");
	if (1 != SSSLwrite(connectfd, write_buffer, 1)) {
		printf("Failed.\n Invalid SSSLwrite call: %s!\n", SSSLerror());
		return 6;
	}
	printf(" Reading 4 bytes of test data: ................................\n");
	if (4 != SSSLread(connectfd, read_buffer, 4)) {
		printf("Failed.\n Invalid SSSLread call: %s!\n", SSSLerror());
		return 7;
	}
	Word i;
	for (i = 0; i < 4; i++)
		if (read_buffer[i] != write_buffer[i]) {
			printf("Failed.\n Read/write mismatch!\n");
			break;
		}
	printf(" Writing 1 kilobyte of test data: .............................\n");
	if (1024 != SSSLwrite(connectfd, write_buffer, 1024)) {
		printf("Failed.\n Invalid SSSLwrite call: %s!\n", SSSLerror());
		return 6;
	}
	printf(" Writing 1 megabyte of test data: .............................\n");
	if (1048576 != SSSLwrite(connectfd, write_buffer, 1048576)) {
		printf("Failed.\n Invalid SSSLwrite call: %s!\n", SSSLerror());
		return 6;
	}
	printf(" .............................................................. Ok.\n");
	return 0;
}
