/*
 * @(#)ResultSet.java	1.8 98/07/01
 *
 * Copyright 1995-1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 * 
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package java.sql;

import java.math.BigDecimal;

/**
 * <P>A ResultSet provides access to a table of data generated by
 * executing a Statement. The table rows are retrieved in
 * sequence. Within a row its column values can be accessed in any
 * order.
 * 
 * <P>A ResultSet maintains a cursor pointing to its current row of
 * data.  Initially the cursor is positioned before the first row.
 * The 'next' method moves the cursor to the next row.
 *
 * <P>The getXXX methods retrieve column values for the current
 * row.  You can retrieve values either using the index number of the
 * column, or by using the name of the column.  In general using the 
 * column index will be more efficient.  Columns are numbered from 1.
 *
 * <P>For maximum portability, ResultSet columns within each row should be
 * read in left-to-right order and each column should be read only once.
 *
 * <P>For the getXXX methods, the JDBC driver attempts to convert the
 * underlying data to the specified Java type and returns a suitable
 * Java value.  See the JDBC specification for allowable mappings
 * from SQL types to Java types with the ResultSet.getXXX methods.
 *
 * <P>Column names used as input to getXXX methods are case
 * insensitive.  When performing a getXXX using a column name, if
 * several columns have the same name, then the value of the first
 * matching column will be returned. The column name option is
 * designed to be used when column names are used in the SQL
 * query. For columns that are NOT explicitly named in the query, it
 * is best to use column numbers. If column names were used there is
 * no way for the programmer to guarantee that they actually refer to
 * the intended columns.
 *
 * <P>A ResultSet is automatically closed by the Statement that
 * generated it when that Statement is closed, re-executed, or is used
 * to retrieve the next result from a sequence of multiple results.
 * 
 * <P>The number, types and properties of a ResultSet's columns are
 * provided by the ResulSetMetaData object returned by the getMetaData
 * method.
 *
 * @see Statement#executeQuery 
 * @see Statement#getResultSet 
 * @see ResultSetMetaData 
 */

public interface ResultSet {

    /**
     * A ResultSet is initially positioned before its first row; the
     * first call to next makes the first row the current row; the
     * second call makes the second row the current row, etc. 
     *
     * <P>If an input stream from the previous row is open, it is
     * implicitly closed. The ResultSet's warning chain is cleared
     * when a new row is read.
     *
     * @return true if the new current row is valid; false if there
     * are no more rows 
     * @exception SQLException if a database-access error occurs.
     */
    boolean next() throws SQLException;


    /**
     * In some cases, it is desirable to immediately release a
     * ResultSet's database and JDBC resources instead of waiting for
     * this to happen when it is automatically closed; the close
     * method provides this immediate release.
     *
     * <P><B>Note:</B> A ResultSet is automatically closed by the
     * Statement that generated it when that Statement is closed,
     * re-executed, or is used to retrieve the next result from a
     * sequence of multiple results. A ResultSet is also automatically
     * closed when it is garbage collected.  
     *
     * @exception SQLException if a database-access error occurs.
     */
    void close() throws SQLException;

    /**
     * A column may have the value of SQL NULL; wasNull reports whether
     * the last column read had this special value.
     * Note that you must first call getXXX on a column to try to read
     * its value and then call wasNull() to find if the value was
     * the SQL NULL.
     *
     * @return true if last column read was SQL NULL
     * @exception SQLException if a database-access error occurs.
     */
    boolean wasNull() throws SQLException;
    
    //======================================================================
    // Methods for accessing results by column index
    //======================================================================

    /**
     * Get the value of a column in the current row as a Java String.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    String getString(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java boolean.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is false
     * @exception SQLException if a database-access error occurs.
     */
    boolean getBoolean(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java byte.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    byte getByte(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java short.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    short getShort(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java int.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    int getInt(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java long.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    long getLong(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java float.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    float getFloat(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java double.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    double getDouble(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a java.lang.BigDecimal object.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param scale the number of digits to the right of the decimal
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    BigDecimal getBigDecimal(int columnIndex, int scale) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java byte array.
     * The bytes represent the raw values returned by the driver.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    byte[] getBytes(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a java.sql.Date object.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    java.sql.Date getDate(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a java.sql.Time object.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    java.sql.Time getTime(int columnIndex) throws SQLException;

    /**
     * Get the value of a column in the current row as a java.sql.Timestamp object.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    java.sql.Timestamp getTimestamp(int columnIndex) throws SQLException;

    /**
     * A column value can be retrieved as a stream of ASCII characters 
     * and then read in chunks from the stream.  This method is particularly
     * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
     * do any necessary conversion from the database format into ASCII.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream. . Also, a
     * stream may return 0 for available() whether there is data
     * available or not.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     * as a stream of one byte ASCII characters.  If the value is SQL NULL
     * then the result is null.  
     * @exception SQLException if a database-access error occurs.
     */
    java.io.InputStream getAsciiStream(int columnIndex) throws SQLException;

    /**
     * A column value can be retrieved as a stream of Unicode characters 
     * and then read in chunks from the stream.  This method is particularly
     * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
     * do any necessary conversion from the database format into Unicode.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream. . Also, a
     * stream may return 0 for available() whether there is data
     * available or not.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     * as a stream of two byte Unicode characters.  If the value is SQL NULL
     * then the result is null.  
     * @exception SQLException if a database-access error occurs.
     */
    java.io.InputStream getUnicodeStream(int columnIndex) throws SQLException;

    /**
     * A column value can be retrieved as a stream of uninterpreted bytes 
     * and then read in chunks from the stream.  This method is particularly
     * suitable for retrieving large LONGVARBINARY values.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream. Also, a
     * stream may return 0 for available() whether there is data
     * available or not.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     * as a stream of uninterpreted bytes.  If the value is SQL NULL
     * then the result is null.  
     * @exception SQLException if a database-access error occurs.
     */
    java.io.InputStream getBinaryStream(int columnIndex)
        throws SQLException;


    //======================================================================
    // Methods for accessing results by column name
    //======================================================================

    /**
     * Get the value of a column in the current row as a Java String.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    String getString(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java boolean.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is false
     * @exception SQLException if a database-access error occurs.
     */
    boolean getBoolean(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java byte.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    byte getByte(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java short.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    short getShort(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java int.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    int getInt(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java long.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    long getLong(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java float.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    float getFloat(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java double.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database-access error occurs.
     */
    double getDouble(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a java.lang.BigDecimal object.
     *
     * @param columnName is the SQL name of the column
     * @param scale the number of digits to the right of the decimal
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    BigDecimal getBigDecimal(String columnName, int scale) throws SQLException;

    /**
     * Get the value of a column in the current row as a Java byte array.
     * The bytes represent the raw values returned by the driver.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    byte[] getBytes(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a java.sql.Date object.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    java.sql.Date getDate(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a java.sql.Time object.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    java.sql.Time getTime(String columnName) throws SQLException;

    /**
     * Get the value of a column in the current row as a java.sql.Timestamp object.
     *
     * @param columnName is the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database-access error occurs.
     */
    java.sql.Timestamp getTimestamp(String columnName) throws SQLException;

    /**
     * A column value can be retrieved as a stream of ASCII characters 
     * and then read in chunks from the stream.  This method is particularly
     * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
     * do any necessary conversion from the database format into ASCII.
     *
     * <P><B>Note:</B> All the data in the returned stream must
     * be read prior to getting the value of any other column. The
     * next call to a get method implicitly closes the stream.
     *
     * @param columnName is the SQL name of the column
     * @return a Java input stream that delivers the database column value
     * as a stream of one byte ASCII characters.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database-access error occurs.
     */
    java.io.InputStream getAsciiStream(String columnName) throws SQLException;

    /**
     * A column value can be retrieved as a stream of Unicode characters 
     * and then read in chunks from the stream.  This method is particularly
     * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
     * do any necessary conversion from the database format into Unicode.
     *
     * <P><B>Note:</B> All the data in the returned stream must
     * be read prior to getting the value of any other column. The
     * next call to a get method implicitly closes the stream.
     *
     * @param columnName is the SQL name of the column
     * @return a Java input stream that delivers the database column value
     * as a stream of two byte Unicode characters.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database-access error occurs.
     */
    java.io.InputStream getUnicodeStream(String columnName) throws SQLException;

    /**
     * A column value can be retrieved as a stream of uninterpreted bytes 
     * and then read in chunks from the stream.  This method is particularly
     * suitable for retrieving large LONGVARBINARY values.
     *
     * <P><B>Note:</B> All the data in the returned stream must
     * be read prior to getting the value of any other column. The
     * next call to a get method implicitly closes the stream.
     *
     * @param columnName is the SQL name of the column
     * @return a Java input stream that delivers the database column value
     * as a stream of uninterpreted bytes.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database-access error occurs.
     */
    java.io.InputStream getBinaryStream(String columnName)
        throws SQLException;


    //=====================================================================
    // Advanced features:
    //=====================================================================

    /**
     * <p>The first warning reported by calls on this ResultSet is
     * returned. Subsequent ResultSet warnings will be chained to this
     * SQLWarning.
     *
     * <P>The warning chain is automatically cleared each time a new
     * row is read.
     *
     * <P><B>Note:</B> This warning chain only covers warnings caused
     * by ResultSet methods.  Any warning caused by statement methods
     * (such as reading OUT parameters) will be chained on the
     * Statement object. 
     *
     * @return the first SQLWarning or null 
     * @exception SQLException if a database-access error occurs.
     */
    SQLWarning getWarnings() throws SQLException;

    /**
     * After this call getWarnings returns null until a new warning is
     * reported for this ResultSet.  
     *
     * @exception SQLException if a database-access error occurs.
     */
    void clearWarnings() throws SQLException;

    /**
     * Get the name of the SQL cursor used by this ResultSet.
     *
     * <P>In SQL, a result table is retrieved through a cursor that is
     * named. The current row of a result can be updated or deleted
     * using a positioned update/delete statement that references the
     * cursor name. 
     * 
     * <P>JDBC supports this SQL feature by providing the name of the
     * SQL cursor used by a ResultSet. The current row of a ResultSet
     * is also the current row of this SQL cursor.
     *
     * <P><B>Note:</B> If positioned update is not supported a
     * SQLException is thrown
     *
     * @return the ResultSet's SQL cursor name
     * @exception SQLException if a database-access error occurs.
     */
    String getCursorName() throws SQLException;

    /**
     * The number, types and properties of a ResultSet's columns
     * are provided by the getMetaData method.
     *
     * @return the description of a ResultSet's columns
     * @exception SQLException if a database-access error occurs.
     */
    ResultSetMetaData getMetaData() throws SQLException;

    /**
     * <p>Get the value of a column in the current row as a Java object.
     *
     * <p>This method will return the value of the given column as a
     * Java object.  The type of the Java object will be the default
     * Java Object type corresponding to the column's SQL type,
     * following the mapping specified in the JDBC spec.
     *
     * <p>This method may also be used to read datatabase specific abstract
     * data types.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return A java.lang.Object holding the column value.  
     * @exception SQLException if a database-access error occurs.
     */
    Object getObject(int columnIndex) throws SQLException;

    /**
     * <p>Get the value of a column in the current row as a Java object.
     *
     * <p>This method will return the value of the given column as a
     * Java object.  The type of the Java object will be the default
     * Java Object type corresponding to the column's SQL type,
     * following the mapping specified in the JDBC spec.
     *
     * <p>This method may also be used to read datatabase specific abstract
     * data types.
     *
     * @param columnName is the SQL name of the column
     * @return A java.lang.Object holding the column value.  
     * @exception SQLException if a database-access error occurs.
     */
    Object getObject(String columnName) throws SQLException;

    //----------------------------------------------------------------

    /**
     * Map a Resultset column name to a ResultSet column index.
     *
     * @param columnName the name of the column
     * @return the column index
     * @exception SQLException if a database-access error occurs.
     */
    int findColumn(String columnName) throws SQLException;
}
