/* -*- Mode:C; tab-width: 4 -*-
 * remote.c (navhelp.c) --- remote control of Netscape Navigator for Unix.
 * version 1.1.0, for Netscape Navigator 1.1 and newer.
 *
 * Copyright  1995 Netscape Communications Corporation, all rights reserved.
 * Created: Jamie Zawinski <jwz@netscape.com>, 24-Dec-94.
 *
 * Borrowed from xfe for use as a help mechanism.  
 * CCM Tue Mar 19 23:06:04 PST 1996 <mcafee@netscape.com>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 *
 * To compile:
 *
 *    Solaris:
 *           cc -DSTANDALONE -I. -I/usr/openwin/include -I/usr/dt/include 
 *           navhelp.c -L/usr/dt/lib -lXm -L/usr/openwin/lib -lXt -lX11 
 *           -lXmu -lnsl -L/usr/ucblib -lucb -o navhelp
 *           
 *
 * To use:
 *
 *    netscape-help -help
 *
 * Documentation for the protocol which this code implements may be found at:
 *
 *    http://home.netscape.com/newsref/std/x-remote.html
 *
 * Bugs and commentary to x_cbug@netscape.com.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xmu/WinUtil.h>	/* for XmuClientWindow() */

/* vroot.h is a header file which lets a client get along with `virtual root'
   window managers like swm, tvtwm, olvwm, etc.  If you don't have this header
   file, you can find it at "http://home.netscape.com/newsref/std/vroot.h".
   If you don't care about supporting virtual root window managers, you can
   comment this line out.
 */
 
/* #include "vroot.h" */

typedef enum {eDefaultHelp = 0, 
              eNamedWindowHelp, 
              eCrossPlatformHelp} tHelpLevel;

static char        *progname = "netscape";
static tHelpLevel  helpLevel = eDefaultHelp;

Display        *dpy;
unsigned long  remote_window = 0;

#define MOZILLA_VERSION_PROP   "_MOZILLA_VERSION"
#define MOZILLA_LOCK_PROP      "_MOZILLA_LOCK"
#define MOZILLA_COMMAND_PROP   "_MOZILLA_COMMAND"
#define MOZILLA_RESPONSE_PROP  "_MOZILLA_RESPONSE"
static Atom XA_MOZILLA_VERSION  = 0;
static Atom XA_MOZILLA_LOCK     = 0;
static Atom XA_MOZILLA_COMMAND  = 0;
static Atom XA_MOZILLA_RESPONSE = 0;

static void
mozilla_remote_init_atoms (Display *dpy)
{
    if (! XA_MOZILLA_VERSION)
        XA_MOZILLA_VERSION = XInternAtom (dpy, MOZILLA_VERSION_PROP, False);
    if (! XA_MOZILLA_LOCK)
        XA_MOZILLA_LOCK = XInternAtom (dpy, MOZILLA_LOCK_PROP, False);
    if (! XA_MOZILLA_COMMAND)
        XA_MOZILLA_COMMAND = XInternAtom (dpy, MOZILLA_COMMAND_PROP, False);
    if (! XA_MOZILLA_RESPONSE)
        XA_MOZILLA_RESPONSE = XInternAtom (dpy, MOZILLA_RESPONSE_PROP, False);
}

/* Return window if we find it, otherwise return NULL. */
static Window
mozilla_remote_find_window (Display *dpy)
{
    int i;
    Window root = RootWindowOfScreen (DefaultScreenOfDisplay (dpy));
    Window root2, parent, *kids;
    unsigned int nkids;
    Window tenative = 0;
    unsigned char *tenative_version = 0;

    if (! XQueryTree (dpy, root, &root2, &parent, &kids, &nkids))
        abort ();
    if (root != root2)
        abort ();
    if (parent)
        abort ();
    if (! (kids && nkids))
        abort ();
    for (i = 0; i < nkids; i++)
    {
        Atom type;
        int format;
        unsigned long nitems, bytesafter;
        unsigned char *version = 0;
        Window w = XmuClientWindow (dpy, kids[i]);
        int status = XGetWindowProperty (dpy, w, XA_MOZILLA_VERSION,
                                         0, (65536 / sizeof (long)),
                                         False, XA_STRING,
                                         &type, &format, &nitems, &bytesafter,
                                         &version);
        if (! version)
            continue;
        if(!tenative)
        {
            tenative = w;
            tenative_version = version;
            continue;
        }
        XFree (version);
    }

    /* result   = expected version was found.   */
    /* tenative = unexpected version was found. */

    /* If any version was found, figure out which */
    /* helpLevel we should use.                   */
    if(tenative_version != NULL) {
        /* Expecting a string like "3.0b4" */
        printf("Navigator version = %s\n", (char*)tenative_version);
        
        /*  Are we at 3.0? */
        if(tenative_version[0] >= '3') {
            /* Are we a beta? */
            if(strstr((char*)tenative_version, "b") != NULL) {
                if(tenative_version[3] <= '2') {
                    /* At 3.2 or earlier.  ? I think        */
                    /* you just get default help here. CCM  */
                    helpLevel = eDefaultHelp;
                } else if(tenative_version[3] == '3') {
                    /* Ok we're 3.0b3 or earlier.  Only try named windows. */
                    printf("Using named-window help\n");
                    helpLevel = eNamedWindowHelp;
                } else {
                    /* Ok we're at 3.0b4 or higher.  Try cross-platform help. */
                    printf("Using cross-platform help\n");
                    helpLevel = eCrossPlatformHelp;
                }
            } else {
                /* We're at 3.0 or greater.  Fabulous. Give them  */
                /* the cross-platform help.                       */
                printf("Using cross-platform help\n");
                helpLevel = eCrossPlatformHelp;
            }

        } else {
            /* Use default help, ie. just open an URL. */
            printf("Using default help\n");
            helpLevel = eDefaultHelp;
        }
    }
    
    /* Clean-up and return. */
    if (tenative)
    {
        XFree (tenative_version);
        return tenative;
    }
    else
    {
        static int firstTime = True;

        fprintf (stderr, "%s: not running on display %s\n", progname,
                 DisplayString (dpy));
        
        /* Try launching navigator, only try once. */
        if(firstTime == True) {
            firstTime = False;
            
            /* Launch navigator. */
            fprintf (stderr, "Attempting to launch navigator...");
            system("netscape&");
            
            /* Delay, microseconds. */
            usleep(12*1000000);

            /* Try again. */
            return mozilla_remote_find_window(dpy);
        }

        return NULL;
    }
}


/*  Return 0 if Ok, -1 otherwise. CCM  */
static int mozilla_remote_check_window (Display *dpy, Window window)
{
    Atom type;
    int format;
    unsigned long nitems, bytesafter;
    unsigned char *version = 0;
    int status = XGetWindowProperty (dpy, window, XA_MOZILLA_VERSION,
                                     0, (65536 / sizeof (long)),
                                     False, XA_STRING,
                                     &type, &format, &nitems, &bytesafter,
                                     &version);
    if (status != Success || !version)
    {
        fprintf (stderr, "%s: window 0x%x is not a Netscape window.\n",
                 progname, (unsigned int) window);
        return -1;
    }
    XFree (version);
  
    return 0;
}

/* Hmm, had to add this to get it to link.  This is BSD! CCM */
extern int gethostname(char *name, int namelen);

static char *lock_data = 0;

static int mozilla_remote_obtain_lock (Display *dpy, Window window)
{
    Bool locked = False;
    Bool waited = False;

    if (! lock_data)
    {
        lock_data = (char *) malloc (255);
        sprintf (lock_data, "pid%d@", getpid ());
        if (gethostname (lock_data + strlen (lock_data), 100))
        {
            perror ("gethostname");
            return -1;
        }
    }

    do
    {
        int result;
        Atom actual_type;
        int actual_format;
        unsigned long nitems, bytes_after;
        unsigned char *data = 0;

        XGrabServer (dpy);              /* ################################# DANGER! */

        result = XGetWindowProperty (dpy, window, XA_MOZILLA_LOCK,
                                     0, (65536 / sizeof (long)),
                                     False, /* don't delete */
                                     XA_STRING,
                                     &actual_type, &actual_format,
                                     &nitems, &bytes_after,
                                     &data);
        if (result != Success || actual_type == None)
        {
            /* It's not now locked - lock it. */
#ifdef DEBUG_PROPS
            fprintf (stderr, "%s: (writing " MOZILLA_LOCK_PROP
                     " \"%s\" to 0x%x)\n",
                     progname, lock_data, (unsigned int) window);
#endif
            XChangeProperty (dpy, window, XA_MOZILLA_LOCK, XA_STRING, 8,
                             PropModeReplace, (unsigned char *) lock_data, 
                             strlen (lock_data));
            locked = True;
        }

        XUngrabServer (dpy); /* ############################# danger over */
        XSync (dpy, False);

        if (! locked)
        {
            /* We tried to grab the lock this time, and failed because someone
               else is holding it already.  So, wait for a PropertyDelete event
               to come in, and try again. */

            fprintf (stderr, "%s: window 0x%x is locked by %s; waiting...\n",
                     progname, (unsigned int) window, data);
            waited = True;

            while (1)
            {
                XEvent event;
                XNextEvent (dpy, &event);
                if (event.xany.type == DestroyNotify &&
                    event.xdestroywindow.window == window)
                {
                    fprintf (stderr, 
                             "%s: window 0x%x unexpectedly destroyed.\n",
                             progname, (unsigned int) window);
		  
                    /* Why do we want to exit-6 here? CCM  */
                        /* exit (6); */
                    return -1;
                }
                else if (event.xany.type == PropertyNotify &&
                         event.xproperty.state == PropertyDelete &&
                         event.xproperty.window == window &&
                         event.xproperty.atom == XA_MOZILLA_LOCK)
                {
                    /* Ok!  Someone deleted their lock, so now we can try
                       again. */
#ifdef DEBUG_PROPS
                    fprintf (stderr, "%s: (0x%x unlocked, trying again...)\n",
                             progname, (unsigned int) window);
#endif
                    break;
                }
            }
        }
        if (data)
            XFree (data);
    }
    while (! locked);

    if (waited)
        fprintf (stderr, "%s: obtained lock.\n", progname);
  
    return 0;
}


static int mozilla_remote_free_lock (Display *dpy, Window window)
{
    int result;
    Atom actual_type;
    int actual_format;
    unsigned long nitems, bytes_after;
    unsigned char *data = 0;

#ifdef DEBUG_PROPS
    fprintf (stderr, "%s: (deleting " MOZILLA_LOCK_PROP
             " \"%s\" from 0x%x)\n",
             progname, lock_data, (unsigned int) window);
#endif

    result = XGetWindowProperty (dpy, window, XA_MOZILLA_LOCK,
                                 0, (65536 / sizeof (long)),
                                 True,  /* atomic delete after */
                                 XA_STRING,
                                 &actual_type, &actual_format,
                                 &nitems, &bytes_after,
                                 &data);
    if (result != Success)
    {
        fprintf (stderr, "%s: unable to read and delete " MOZILLA_LOCK_PROP
                 " property\n",
                 progname);
        return -1;
    }
    else if (!data || !*data)
    {
        fprintf (stderr, "%s: invalid data on " MOZILLA_LOCK_PROP
                 " of window 0x%x.\n",
                 progname, (unsigned int) window);
        return -1;
    }
    else if (strcmp ((const char *)data, lock_data))
    {
        fprintf (stderr, "%s: " MOZILLA_LOCK_PROP
                 " was stolen!  Expected \"%s\", saw \"%s\"!\n",
                 progname, lock_data, data);
        return -1;
    }

    if (data)
        XFree (data);
  
    return 0;
}


static int mozilla_remote_command (Display *dpy, Window window, 
                                   const char *command, Bool raise_p)
{
    int result;
    Bool done = False;
    char *new_command = 0;

    /* The -noraise option is implemented by passing a "noraise" argument
       to each command to which it should apply.
       */
    if (! raise_p)
    {
        char *close;
        new_command = (char *) malloc (strlen (command) + 20);
        strcpy (new_command, command);
        close = strrchr (new_command, ')');
        if (close)
            strcpy (close, ", noraise)");
        else
            strcat (new_command, "(noraise)");
        command = new_command;
    }

#ifdef DEBUG_PROPS
    fprintf (stderr, "%s: (writing " MOZILLA_COMMAND_PROP " \"%s\" to 0x%x)\n",
             progname, command, (unsigned int) window);
#endif

    XChangeProperty (dpy, window, XA_MOZILLA_COMMAND, XA_STRING, 8,
                     PropModeReplace, (unsigned char *) command, strlen (command));

    while (!done)
    {
        XEvent event;
        XNextEvent (dpy, &event);
        if (event.xany.type == DestroyNotify &&
            event.xdestroywindow.window == window)
        {
            fprintf (stderr, "%s: window 0x%x unexpectedly destroyed.\n",
                     progname, (unsigned int) window);
            result = 6;
            goto DONE;
        }
        else if (event.xany.type == PropertyNotify &&
                 event.xproperty.state == PropertyNewValue &&
                 event.xproperty.window == window &&
                 event.xproperty.atom == XA_MOZILLA_RESPONSE)
        {
            Atom actual_type;
            int actual_format;
            unsigned long nitems, bytes_after;
            unsigned char *data = 0;

            result = XGetWindowProperty (dpy, window, XA_MOZILLA_RESPONSE,
                                         0, (65536 / sizeof (long)),
                                         True, /* atomic delete after */
                                         XA_STRING,
                                         &actual_type, &actual_format,
                                         &nitems, &bytes_after,
                                         &data);
#ifdef DEBUG_PROPS
            if (result == Success && data && *data)
            {
                fprintf (stderr, "%s: (server sent " MOZILLA_RESPONSE_PROP
                         " \"%s\" to 0x%x.)\n",
                         progname, data, (unsigned int) window);
            }
#endif

            if (result != Success)
            {
                fprintf (stderr, "%s: failed reading " MOZILLA_RESPONSE_PROP
                         " from window 0x%0x.\n",
                         progname, (unsigned int) window);
                result = 6;
                done = True;
            }
            else if (!data || !*data)
            {
                fprintf (stderr, "%s: invalid data on " MOZILLA_RESPONSE_PROP
                         " property of window 0x%0x.\n",
                         progname, (unsigned int) window);
                result = 6;
                done = True;
            }
            else if (*data == '1')      /* positive preliminary reply */
            {
                fprintf (stderr, "%s: %s\n", progname, data + 4);
                /* keep going */
                done = False;
            }
#if 1
            else if (!strncmp ((const char*)data, "200", 3)) /* positive completion */
            {
                result = 0;
                done = True;
            }
#endif
            else if (*data == '2')		/* positive completion */
            {
                fprintf (stderr, "%s: %s\n", progname, data + 4);
                result = 0;
                done = True;
            }
            else if (*data == '3')      /* positive intermediate reply */
            {
                fprintf (stderr, "%s: internal error: "
                         "server wants more information?  (%s)\n",
                         progname, data);
                result = 3;
                done = True;
            }
            else if (*data == '4' ||	/* transient negative completion */
                     *data == '5')      /* permanent negative completion */
            {
                fprintf (stderr, "%s: %s\n", progname, data + 4);
                result = (*data - '0');
                done = True;
            }
            else
            {
                fprintf (stderr,
                         "%s: unrecognised " MOZILLA_RESPONSE_PROP
                         " from window 0x%x: %s\n",
                         progname, (unsigned int) window, data);
                result = 6;
                done = True;
            }

            if (data)
                XFree (data);
        }
#ifdef DEBUG_PROPS
        else if (event.xany.type == PropertyNotify &&
                 event.xproperty.window == window &&
                 event.xproperty.state == PropertyDelete &&
                 event.xproperty.atom == XA_MOZILLA_COMMAND)
        {
            fprintf (stderr, "%s: (server 0x%x has accepted "
                     MOZILLA_COMMAND_PROP ".)\n",
                     progname, (unsigned int) window);
        }
#endif /* DEBUG_PROPS */
    }

 DONE:

    if (new_command)
        free (new_command);

    return result;
}

int mozilla_remote_commands (Display *dpy, Window window, 
                             char **commands, int numCommands)
{
    Bool raise_p = True;
    int status = 0;
    int i;

    mozilla_remote_init_atoms (dpy);

    if (window == 0) {
        window = mozilla_remote_find_window (dpy);
        if(window == NULL) {
            return -1;
        }
    }
    else
        mozilla_remote_check_window (dpy, window);

    XSelectInput (dpy, window, (PropertyChangeMask|StructureNotifyMask));

    mozilla_remote_obtain_lock (dpy, window);

    for(i=0;i<numCommands;i++) {
        if (!strcmp (commands[i], "-raise"))
            raise_p = True;
        else if (!strcmp (commands[i], "-noraise"))
            raise_p = False;
        else
            status = mozilla_remote_command (dpy, window, commands[i], raise_p);

        if (status != 0)
            break;
    }

    mozilla_remote_free_lock (dpy, window);

    return status;
}


#ifdef STANDALONE

#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/PushB.h>
#include <Xm/Separator.h>
#include <Xm/TextF.h>

Widget         fileTextField;
Widget         idTextField;

void QuitCB(Widget w, 
            XtPointer clientData,
            XtPointer callData)
{
    exit(0);
}


void ButtonCB(Widget w, 
              XtPointer clientData,  /* Passed textfield in.  */
              XtPointer callData
              )
{
    char    command[1000];
    int     status;
    int     selectedItem;
    char    *id = NULL;
    Widget  textfield;

    /* Id file is stored in fileTextField. */
    

    /* Build up our command string to send to the Navigator. */
    sprintf(command,
            "htmlHelp(%s, %s, searchTextNotUsed)", 
            XmTextFieldGetString(fileTextField), 
            XmTextFieldGetString(idTextField)
            );
    
    printf("Command = %s\n", command);

    /* Obtain lock for navigator window. */
    mozilla_remote_obtain_lock(dpy, remote_window);

    /* Send the command to the Navigator. */
    status = 
        mozilla_remote_command(dpy,                    /* Display. */
                               (Window) remote_window, /* Window.  */
                               command,                /* Remote command.   */
                               False);                 /* Raise the window? */

    /* Clean up. */
    mozilla_remote_free_lock (dpy, remote_window);
}



/* Create Motif list to hold help options. */
void CreateInputArea(Widget toplevel)
{
    Widget form,
             fileLabel,
             fileButton,
             idLabel,
             separator,
             testButton,
             quitButton;

    char    cwd[100];
    char    url[1000];
    

    /* Form. */
    form = XmCreateForm(toplevel, "form", NULL, 0);
    XtManageChild(form);

    /* File label. */
    fileLabel = XmCreateLabel(form, "ID File:", NULL, 0);
    XtVaSetValues(fileLabel,
                  XmNtopAttachment,  XmATTACH_FORM,
                  XmNtopOffset,      12,
                  XmNleftAttachment, XmATTACH_FORM,
                  XmNleftOffset,     5,
                  XmNalignment,      XmALIGNMENT_END,
                  XmNwidth,          60,
                  NULL);
    XtManageChild(fileLabel);

    /* File TextField. */
    getcwd(cwd, 100);
    sprintf(url, "file:%s/%s", cwd, "smhelp.h"); 

    fileTextField = XmCreateTextField(form, "fileTextField", NULL, 0);
    XtVaSetValues(fileTextField,
                  XmNtopAttachment,    XmATTACH_FORM,
                  XmNtopOffset,        5,
                  XmNleftAttachment,   XmATTACH_WIDGET,
                  XmNleftWidget,       fileLabel,
                  XmNleftOffset,       1,                  
                  XmNrightAttachment,  XmATTACH_WIDGET,
                  XmNrightOffset,      8,                  
                  XmNwidth,            500,
                  NULL);
    XmTextFieldSetString(fileTextField, url);
    XtManageChild(fileTextField);

    /* ID Label */
    idLabel = XmCreateLabel(form, "Help ID:", NULL, 0);
    XtVaSetValues(idLabel,
                  XmNtopAttachment,  XmATTACH_WIDGET,
                  XmNtopWidget,      fileTextField,
                  XmNtopOffset,      8,
                  XmNleftAttachment, XmATTACH_FORM,
                  XmNleftOffset,     5,
                  XmNalignment,      XmALIGNMENT_END,
                  XmNwidth,          60,
                  NULL);
    XtManageChild(idLabel);

    /* ID TextField. */
    idTextField = XmCreateTextField(form, "idTextField", NULL, 0);
    XtVaSetValues(idTextField,
                  XmNtopAttachment,    XmATTACH_WIDGET,
                  XmNtopWidget,        fileTextField,
                  XmNtopOffset,        1,
                  XmNleftAttachment,   XmATTACH_WIDGET,
                  XmNleftWidget,       fileLabel,
                  XmNleftOffset,       1,                  
                  XmNrightAttachment,  XmATTACH_WIDGET,
                  XmNrightOffset,      8,                  
                  XmNwidth,            500,
                  NULL);
    XtManageChild(idTextField);
    
    /* Separator */
    separator = XmCreateSeparator(form, "separator", NULL, 0);
	XtVaSetValues(separator,
				  XmNleftAttachment,   XmATTACH_FORM,
				  XmNleftOffset,       1,
				  XmNrightAttachment,  XmATTACH_FORM,
				  XmNrightOffset,      1,
				  XmNtopAttachment,    XmATTACH_WIDGET,
				  XmNtopWidget,        idTextField,
				  XmNtopOffset,        4,
				  NULL);
	XtManageChild(separator);

    /* Test Button. */
    testButton = XmCreatePushButton(form, "Test", NULL, 0);
    XtVaSetValues(testButton,
                  XmNtopAttachment,    XmATTACH_WIDGET,
                  XmNtopWidget,        separator,
                  XmNtopOffset,        10,
                  XmNleftAttachment,   XmATTACH_FORM,
                  XmNleftOffset,       40,
                  XmNbottomAttachment, XmATTACH_FORM,
                  XmNbottomOffset,     10,
                  XmNheight,           30,
                  XmNwidth,            60,
                  NULL);
    XtAddCallback(testButton, XmNactivateCallback, ButtonCB, NULL);
    XtManageChild(testButton);

    /* Quit Button. */
    quitButton = XmCreatePushButton(form, "Done", NULL, 0);
    XtVaSetValues(quitButton,
                  XmNtopAttachment,    XmATTACH_WIDGET,
                  XmNtopWidget,        separator,
                  XmNtopOffset,        10,
                  XmNrightAttachment,  XmATTACH_FORM,
                  XmNrightOffset,      40,
                  XmNbottomAttachment, XmATTACH_FORM,
                  XmNbottomOffset,     10,
                  XmNheight,           30,
                  XmNwidth,            60,
                  NULL);
    XtAddCallback(quitButton, XmNactivateCallback, QuitCB, idTextField);
    XtManageChild(quitButton);
}


void
main(int argc, char **argv)
{
    char           *dpy_string = 0;
    Bool           sync_p = False;
    int            i;
    Widget         toplevel;
    XtAppContext   app;

    /* Find the Navigator window, and determine help level.*/

    dpy = XOpenDisplay (dpy_string);
    if (! dpy)
        exit (-1);

    if (sync_p)
        XSynchronize (dpy, True);

    mozilla_remote_init_atoms (dpy);

    if (remote_window == 0) {
        remote_window = mozilla_remote_find_window (dpy);
        if(remote_window == NULL) {
            exit(-1);
        }
    }
    else
        mozilla_remote_check_window (dpy, remote_window);

    XSelectInput (dpy, remote_window, 
                  (PropertyChangeMask|StructureNotifyMask));


    /* Put up an XmList to ask the user which help topic to display. */
    XtSetLanguageProc (NULL, NULL, NULL);
    toplevel = XtVaAppInitialize (&app, "", NULL, 0,
                                  &argc, argv, NULL, NULL);

    
    /* Create our user interface. */
    CreateInputArea(toplevel);

    /* Enter event loop. */
    XtRealizeWidget (toplevel);
    XtAppMainLoop (app);

}

#endif /* STANDALONE */
