/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.commons.exporter.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;

import javax.faces.component.UIColumn;
import javax.faces.component.UIComponent;
import javax.faces.component.ValueHolder;
import javax.faces.component.html.HtmlDataTable;
import javax.faces.context.FacesContext;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;

import com.lowagie.text.Document;
import com.lowagie.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfWriter;

/**
 * This class is a utility class for serving PDF exporting.
 */
public class PDFExporterUtil
{

    /*
     * This method is used for setting the response headers of the pdf.
     */
    private static void setPDFResponseHeaders(HttpServletResponse response,
            ByteArrayOutputStream byteArrayStream, String fileName)
            throws IOException
    {

        // setting response headers.
        response.setHeader("Expires", "0");
        response.setHeader("Cache-Control",
                "must-revalidate, post-check=0, pre-check=0");
        response.setHeader("Pragma", "public");
        response.setHeader("Content-disposition", "attachment;filename="
                + fileName + ".pdf");

        // setting the content type.
        response.setContentType("application/pdf");

        // the contentlength is needed for MSIE.
        response.setContentLength(byteArrayStream.size());

        // write ByteArrayOutputStream to the ServletOutputStream.
        ServletOutputStream outputStream = response.getOutputStream();

        byteArrayStream.writeTo(outputStream);
        outputStream.flush();
    }

    /*
     * This method is used for adding the columns headers to the pdfTable.
     */
    private static void generateTableHeader(PdfPTable pdfTable, List columns)
    {
        for (int i = 0; i < columns.size(); i++)
        {
            UIColumn column = (UIColumn) columns.get(i);
            UIComponent columnHeaderCell = column.getHeader();
            if (columnHeaderCell instanceof ValueHolder)
            {
                String cellValue = ComponentUtils.getStringValue(FacesContext
                        .getCurrentInstance(), columnHeaderCell);
                pdfTable.addCell(cellValue);
            }
        }
    }

    /*
     * This method is used for adding the columns values to the pdfTable.
     */
    private static void generateTableContent(FacesContext facesContext,
            PdfPTable pdfTable, List columns, HtmlDataTable dataTable)
    {
        int numberOfColumns = columns.size();
        int numberOfRows = dataTable.getRowCount();
        int startFrom = 0;
        int endAt = numberOfRows;

        /* fill the table with the data. */
        for (int i = startFrom; i < endAt; ++i)
        {
            dataTable.setRowIndex(i);
            for (int j = 0; j < numberOfColumns; ++j)
            {
                String cellValue = "";
                UIColumn currentColumn = (UIColumn) columns.get(j);

                for (int k = 0; k < currentColumn.getChildren().size(); ++k)
                {
                    if (currentColumn.getChildren().get(k) instanceof ValueHolder)
                    {
                        cellValue += ((ValueHolder) currentColumn.getChildren()
                                .get(k)).getValue();
                    }
                }

                pdfTable.addCell(cellValue);
            }
        }
    }

    /*
     * This method is used for creating the PDFTable model.
     */
    public static PdfPTable generatePDFTableModel(FacesContext facesContext,
            HtmlDataTable dataTable)
    {
        int numberOfColumns;
        List columns = null;
        PdfPTable pdfTable = null;

        /* getting the HTMLDataTable Columns */
        columns = ComponentUtils.getHTMLDataTableColumns(dataTable);

        if (columns.size() == 0)
        {
            return null;
        }
        else
        {
            numberOfColumns = columns.size();
        }

        /* creating the PDF Table */
        pdfTable = new PdfPTable(numberOfColumns);

        generateTableHeader(pdfTable, columns);

        generateTableContent(facesContext, pdfTable, columns, dataTable);

        return pdfTable;
    }

    /**
     * This method is responsible for writing the PDF to the response stream.
     * 
     * @param facesContext
     * @param response
     * @param fileName
     * @param dataTable
     */
    public static void generatePDF(FacesContext facesContext,
            HttpServletResponse response, String fileName,
            HtmlDataTable dataTable) throws Exception
    {
        int currentRowIndex;
        Document document = new Document();
        ByteArrayOutputStream byteArrayStream = new ByteArrayOutputStream();
        PdfWriter.getInstance(document, byteArrayStream);
        PdfPTable pdfTable = null;

        /*
         * By default if the fileName is not specified, then use the table id.
         */
        if (fileName == null)
        {
            fileName = dataTable.getId();
        }

        currentRowIndex = dataTable.getRowIndex();

        // generate the PDF table model.
        pdfTable = generatePDFTableModel(facesContext, dataTable);

        // open the document and write the generated PDF.
        document.open();
        document.add(pdfTable);
        document.close();

        // write the response headers.
        setPDFResponseHeaders(response, byteArrayStream, fileName);

        dataTable.setRowIndex(currentRowIndex);

    }
}
