/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.Query;

import org.eclipse.ebam.model.entities.Rule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Handle transactions of class Rule
 * 
 * @author gavardi
 *
 */


public class RulesTransactions extends EMF {

	/**
	 * Logger.
	 */
	private static Logger logger = 
		LoggerFactory.getLogger(RulesTransactions.class);


	/** Get all EbamRules.
	 * 
	 * @return list of EbamRules
	 * @throws Exception
	 */

	public final List getEbamRulesList() throws Exception {
		List rules = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(Rule.class);	
			Query query = em.createQuery("select r from "
					+ "Rule r");
			//query.setParameter("code", code);
			rules = query.getResultList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return rules;
	}

	/** get Rule by id
	 * 
	 * @param id
	 * @return one Rule
	 * @throws Exception
	 */

	public final Rule getRuleById(Integer id) 
	throws Exception {
		Rule rule = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(Rule.class);	
			Query query = em.createQuery("select r from "
					+ "Rule r where r.cepRuleId = :id");
			query.setParameter("id", id);
			rule = (Rule) query.getSingleResult();        

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return rule;
	}

	/**Method that persists <code>Rule</code> on database.
	 * @param j <code>Rule</code> to save
	 * @return operation result
	 * @throws Exception error on saving 
	 */
	public final boolean createNewRule(final Rule rule) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.persist(rule);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}


	/** merge an existng rule
	 * 
	 * @param rule
	 * @return
	 * @throws Exception
	 */

	public final boolean mergeRule(final Rule rule) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			//take all contacts

			em.merge(rule);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	/** Delete a rule
	 * 
	 * @param rule
	 * @return
	 * @throws Exception
	 */
	public final boolean deleteRule(final Rule rule) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			em.setFlushMode(FlushModeType.COMMIT);

			Rule toRemove = em.find(Rule.class, rule.getCepRuleId());
			//toRemove.setMessages(null);
			em.merge(toRemove);
			em.remove(toRemove);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}


}
