/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.Query;

import org.eclipse.ebam.model.entities.SbiAlarmContact;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



public class SbiAlarmContactsTransactions extends EMF {

	/**
	 * Logger.
	 */
	private static Logger logger = 
		LoggerFactory.getLogger(SbiAlarmContactsTransactions.class);


	/**Method that persists <code>SbiAlarmContact</code> on database.
	 * @param j <code>SbiAlarm</code> to save
	 * @return operation result
	 * @throws Exception error on saving 
	 */
	public final boolean createNewSbiAlarmContact(final SbiAlarmContact sbiAlarmContact) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.persist(sbiAlarmContact);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}


	/** Get all alarms contacts.
	 * 
	 * @return list of alarms
	 * @throws Exception
	 */

	public final List getSbiAlarmContactList() throws Exception {
		List alContacts = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiAlarmContact.class);	
			Query query = em.createQuery("select a from "
					+ "SbiAlarmContact a");
			//query.setParameter("code", code);
			alContacts = query.getResultList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return alContacts;
	}

	/** get AlarmContact by id
	 * 
	 * @param id
	 * @return one AlarmContact
	 * @throws Exception
	 */

	public final SbiAlarmContact getSbiAlarmContactById(Integer id) 
	throws Exception {
		SbiAlarmContact alarmContact = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiAlarmContact.class);	
			Query query = em.createQuery("select s from "
					+ "SbiAlarmContact s where s.alarmContactId = :id");
			query.setParameter("id", id);
			alarmContact = (SbiAlarmContact) query.getSingleResult();        

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return alarmContact;
	}	


	/** get AlarmContact by name
	 * 
	 * @param id
	 * @return one AlarmContact
	 * @throws Exception
	 */

	public final SbiAlarmContact getSbiAlarmContactByName(String name) 
	throws Exception {
		SbiAlarmContact alarmContact = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiAlarmContact.class);	
			Query query = em.createQuery("select s from "
					+ "SbiAlarmContact s where s.name = :name");
			query.setParameter("name", name);
			alarmContact = (SbiAlarmContact) query.getSingleResult();        

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return alarmContact;
	}	




	public final boolean mergeSbiAlarmContact(final SbiAlarmContact sbiAlarmContact) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			//take all contacts

			em.merge(sbiAlarmContact);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	public final boolean deleteSbiAlarmContact(final SbiAlarmContact sbiAlarmContact) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			em.setFlushMode(FlushModeType.COMMIT);

			SbiAlarmContact toRemove = em.find(SbiAlarmContact.class, sbiAlarmContact.getAlarmContactId());
			em.merge(toRemove);
			em.remove(toRemove);

			//em.remove(sbiAlarmContact);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

}
