/*
** Copyright (C) 1999, 2000, 2002 Lorenzo Bettini <bettini@gnu.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*/

// abstract factory for generators

#ifndef GENERATORFACTORY_H
#define GENERATORFACTORY_H

#include <string>

using std::string;

class DocGenerator;
class ColorMap;
class Tag;
class GeneratorMap;
class CharTranslator;
class TextGenerator;

class GeneratorFactory
{
 private:
  CharTranslator *char_translator; // translator of special characters, singleton

 protected:
  bool line_number_option; // whether line number option is given
  bool gen_line_ref; // whether line numbers are added an anchor
  string title;
  string start_tag_buffer;
  string close_tag_buffer;

  virtual ColorMap *createColorMap() = 0;
  virtual TextGenerator *createGenerator(const string &key);
  virtual TextGenerator *createGenerator(const Tag *tag);
  virtual TextGenerator *createDefaultGenerator();
  virtual DocGenerator *createDocGenerator();

  virtual CharTranslator *createCharTranslator() = 0;

  virtual GeneratorMap *createGeneratorMap();

  void resetTagBuffers() { start_tag_buffer = close_tag_buffer = ""; }
  void addStartTag(const string &tag) { start_tag_buffer = tag + start_tag_buffer; }
  void addCloseTag(const string &tag) { close_tag_buffer += tag; }
  int startTagLen() const { return start_tag_buffer.size(); }
  int closeTagLen() const { return close_tag_buffer.size(); }
  virtual const string getPreTag() { return start_tag_buffer; }
  virtual const string getPostTag() { return close_tag_buffer; }
  virtual void addBold() {}
  virtual void addItalic() {}
  virtual void addUnderline() {}
  virtual void addColor(const string &color) {}

 public:
  GeneratorFactory(const string &t = "");
  virtual ~GeneratorFactory();

  virtual void createGenerators();
  CharTranslator *getCharTranslator();
};

#endif // GENERATORFACTORY_H
