/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>

#include "cmp.h"
#include "kalc.h"


void doHelp(void)
{
  /*
   * This command displays calls the showHelp function with the appropriate
   * argument in order to show help about what the user requested.
   *
   * All directives have `d_' prepended to their names. This function deals
   * with that.
   */

  char *cmd;

  cmd = getNextToken();
  if (cmd == NULL)
    showHelp("index");
  else if (!strcasecmp(cmd, "set") || !strcasecmp(cmd, "show")) {
    char topic[25] = "d_";
    cmd = getNextToken();
    if (cmd == NULL)
      showHelp("set");
    else {
      strcat(topic, cmd);
      showHelp(topic);
    }
  } else
    showHelp(cmd);
}


void doSet(void)
{
  /*
   * This function gets the next two tokens. If there is no more tokens,
   * displays help for "set". If there is only one more token, displays
   * the current value and help for that directive. If there are two
   * more tokens, sets the value of that directive.
   */

  char *cmd, *opt;

  cmd = getNextToken();
  if (cmd == NULL)
    showHelp("set");
  else {
    char *dir = strdup(cmd);

    free(cmd);
    cmd = getNextToken();

    if (!cmd) {
      char helpString[25] = "d_";
      strcat(helpString, dir);
      showDirective(dir);
      showHelp(helpString);
      return;
    }

    /* Skip quote if necessary */
    if (*cmd == '\"')
      opt = cmd + 1;
    else
      opt = cmd;
    
    processDirective(dir, opt);
    free(dir);
  }
}



void doShow(void)
{
  /*
   * This funciton gets the next token, and displays the current value of
   * that directive, or help for "set" if there is not next token.
   */

  char *cmd;
 
  cmd = getNextToken();
  if (cmd == NULL)
    showHelp("set");
  else
    showDirective(cmd);
}


void processDirective(char *dir, char *value)
{
  /*
   * Processes a directive found in the configuration file, or later set
   * by user.
   */

  /* Angle mode */
  if (!strcasecmp(dir, "anglemode")) {
    if (!strcasecmp(value, "deg"))
      userOptions.angleMode = ANGLE_DEG;
    else if (!strcasecmp(value, "rad"))
      userOptions.angleMode = ANGLE_RAD;
    else
      printf("set: invalid option %s for %s\n", value, dir);
  }

  /* Coordinate mode */
  else if (!strcasecmp(dir, "coordinate")) {
    if (!strcasecmp(value, "rect"))
      userOptions.coordinate = COORD_RECT;
    else if (!strcasecmp(value, "cylin"))
      userOptions.coordinate = COORD_CYLIN;
    else
      printf("set: invalid option %s for %s\n", value, dir);
  }

  /* Number mode */
  else if (!strcasecmp(dir, "number-format")) {
    if (!strcasecmp(value, "std"))
      userOptions.numberFormat = FMT_STD;
    else if (!strcasecmp(value, "fix"))
      userOptions.numberFormat = FMT_FIX;
    else if (!strcasecmp(value, "sci"))
      userOptions.numberFormat = FMT_SCI;
    else if (!strcasecmp(value, "eng"))
      userOptions.numberFormat = FMT_ENG;
    else
      printf("set: invalid option %s for %s\n", value, dir);
  }

  /* Decimal places */
  else if (!strcasecmp(dir, "precision")) {
    int precision = atoi(value);
    if (precision < 0)
      precision = 0;
    else if (precision > 15)
      precision = 15;
    userOptions.decimalPlaces = precision;
  }

  /* Base */
  else if (!strcasecmp(dir, "base")) {
    int base;
    if (isdigit(*value)) {
      base = atoi(value);
      if (base < 2)
	base = 2;
      else if (base > 36)
	base = 36;
    } else {
      if (!strcasecmp(value, "bin"))
	base = 2;
      else if (!strcasecmp(value, "oct"))
	base = 8;
      else if (!strcasecmp(value, "dec"))
	base = 10;
      else if (!strcasecmp(value, "hex"))
	base = 16;
      else {
	printf("set: invalid option %s for %s", value, dir);
	return;
      }
    }
    userOptions.base = base;
  }   

  /* Word size */
  else if (!strcasecmp(dir, "wordsize")) {
    int ws = atoi(value);
    int maxSize = MAXWORDSIZE;
    if (ws < 1)
      ws = 1;
    else if (ws > maxSize)
      ws = maxSize;
    userOptions.wordSize = ws;
  } 

  /* Lines To Display */
  else if (!strcasecmp(dir, "lines")) {
    userOptions.nLines = atoi(value);
    linesToDisplay = userOptions.nLines;
  }

  /* Screen width */
  else if (!strcasecmp(dir, "width")) {
    int width = atoi(value);
    if (width >= 26)
      userOptions.width = width;
  }

  /* Prompt */
  else if (!strcasecmp(dir, "prompt")) {
    strncpy(userOptions.prompt, value, MAXPROMPTSIZE + 1);
    *(userOptions.prompt + MAXPROMPTSIZE) = '\0';
  }

  /* Expert mode */
  else if (!strcasecmp(dir, "expert")) {
    if (!strcasecmp(value, "on"))
      userOptions.expert = 1;
    else if (!strcasecmp(value, "off"))
      userOptions.expert = 0;
    else
      printf("set: invalid option %s for %s\n", value, dir);
  }

  /* Invalid */
  else
    printf("set: no such directive: %s\n", dir);
}


void showDirective(char *dir)
{
  /*
   * This function displays the current value of the dir directive in an
   * human-readable format.
   */

  char val[100];

  /* Angle mode */
  if (!strcasecmp(dir, "anglemode")) {
    if (userOptions.angleMode == ANGLE_RAD)
      strcpy(val, "rad");
    else if (userOptions.angleMode == ANGLE_DEG)
      strcpy(val, "deg");
  }

  /* Coordinate mode */
  else if (!strcasecmp(dir, "coordinate")) {
    if (userOptions.angleMode == COORD_RECT)
      strcpy(val, "rect");
    else if (userOptions.angleMode == COORD_CYLIN)
      strcpy(val, "cylin");
  }

  /* Number format */
  else if (!strcasecmp(dir, "number-format")) {
    if (userOptions.numberFormat == FMT_STD)
      strcpy(val, "std");
    else if (userOptions.numberFormat == FMT_FIX)
      strcpy(val, "fix");
    else if (userOptions.numberFormat == FMT_SCI)
      strcpy(val, "sci");
    else if (userOptions.numberFormat == FMT_ENG)
      strcpy(val, "eng");
  }

  /* Decimal places */
  else if (!strcasecmp(dir, "precision"))
    sprintf(val, "%d", userOptions.decimalPlaces);

  /* Base */
  else if (!strcasecmp(dir, "base"))
    sprintf(val, "%d", userOptions.base);

  /* Word size */
  else if (!strcasecmp(dir, "wordsize"))
    sprintf(val, "%u", userOptions.wordSize);
  
  /* Prompt */
  else if (!strcasecmp(dir, "prompt"))
    strcpy(val, userOptions.prompt);

  /* Lines to display */
  else if (!strcasecmp(dir, "lines"))
    sprintf(val, "%d", userOptions.nLines);

  /* Width */
  else if (!strcasecmp(dir, "width"))
    sprintf(val, "%d", userOptions.width);

  else if (!strcasecmp(dir, "expert"))
    strcpy(val, userOptions.expert ? "on" : "off");

  /* Current status file */
  else if(!strcasecmp(dir, "statusfile"))
    strcpy(val, currStatusFile);

  /* Invalid */
  else {
    printf("show: no such directive: %s\n", dir);
    return;
  }

  /* Show it */
  printf("%s is set to %s\n", dir, val);
}


void setDefaults(void)
{
  /*
   * Sets default values for all options.
   */

  userOptions.angleMode = ANGLE_RAD;
  userOptions.coordinate = COORD_RECT;
  userOptions.numberFormat = FMT_STD;
  userOptions.decimalPlaces = 15;
  userOptions.expert = 0;
  userOptions.base = 10;
  userOptions.wordSize = MAXWORDSIZE;
  userOptions.nLines = 4;
  userOptions.width = 78;
  strcpy(userOptions.prompt, "> ");
}


void showHelp(char *topic)
{
  /*
   * This function displays help for the given topic.
   *
   * The help file is searched in SHAREPATH/kalc.hlp on UNIX systems, and
   * in the current directory for MS-DOS systems. SHAREPATH is
   * INSTALLPREFIX/share/kalc-2.2.0, where INSTALLPREFIX can be set while
   * building, and defaults to /usr/local .
   */

  char l[81];
  char title[25] = "[";
#ifdef MSDOS
  char *helpFilePath = "kalc.hlp";
#else
  char helpFilePath[PATH_MAX] = SHAREPATH "/kalc.hlp";
#endif
  FILE *f;

  if ((f = fopen(helpFilePath, "r")) == NULL) {
    printf("Help file unavailable\n");
    return;
  }

  strcat(title, topic);
  strcat(title, "]");
  while(1) {
    fgets(l, 80, f);
    /* Get rid of new line character */
    *(l + strlen(l) - 1) = '\0';
    if (!strcasecmp(l, title))
      break;
    if (feof(f)) {
      printf("help: topic %s unavailable\n", topic);
      return;
    }
  }

  while (1) {
    fgets(l, 80, f);
    if ((*l == '[') || feof(f))
      break;
    printf("%s", l);
  }

  fclose(f);

  /* Do not show stack */
  linesToDisplay = 0;
  longjmp(jmpError, 999);
}


void f_rad(void)
{
  /*
   * This function sets radians angle mode.
   */

  processDirective("anglemode", "rad");
}


void f_deg(void)
{
  /*
   * This function sets degrees angle mode.
   */
  
  processDirective("anglemode", "deg");
}


void f_rect(void)
{
  /*
   * This function sets rectangular coordinate mode.
   */
  
  processDirective("coordinate", "rect");
}


void f_cylin(void)
{
  /*
   * This command sets polar (cylindrical) coordinate mode.
   */

  processDirective("coordinate", "cylin");
}

void f_std(void)
{
  /*
   * This function sets standard number-format.
   */
  
  processDirective("number-format", "std");
}


void f_fix(void)
{
  /*
   * This funciton calls the _f_fix function through the wrapper.
   */
  
  run1_0_Function(_f_fix, "fix");
}


int _f_fix(Object n)
{
  /*
   * This function sets fixed precision number format with n decimal places.
   */

  if (type(n) == TYPE_REAL) {
    char str[7];
    sprintf(str, "%d", (int) n.value.real);

    processDirective("number-format", "fix");
    processDirective("precision", str);
    return ERR_NOERR;
  } else
    return ERR_BADARGUMENTTYPE;
}


void f_sci(void)
{
  /*
   * This function calls the _f_sci function through the wrapper.
   */
  
  run1_0_Function(_f_sci, "sci");
}


int _f_sci(Object n)
{
  /*
   * This function sets scientific notation number format with n decimal
   * places.
   */
  
  if (type(n) == TYPE_REAL) {
    char str[7];
    sprintf(str, "%d", (int) n.value.real);

    processDirective("number-format", "sci");
    processDirective("precision", str);
    return ERR_NOERR;
  } else
    return ERR_BADARGUMENTTYPE;
}


void f_eng(void)
{
  /*
   * This function calls the _f_eng function through the wrapper.
   */
  
  run1_0_Function(_f_eng, "eng");
}


int _f_eng(Object n)
{
  /*
   * This function sets engineering notation number format with n
   * decimal places.
   */

  if (type(n) == TYPE_REAL) {
    char str[7];
    sprintf(str, "%d", (int) n.value.real);

    processDirective("number-format", "eng");
    processDirective("precision", str);
    return ERR_NOERR;
  } else
    return ERR_BADARGUMENTTYPE;
}


void f_stws(void) 
{
  /*
   * This function calls the _f_stws function through the wrapper.
   */

  run1_0_Function(_f_stws, "stws");
}


int _f_stws(Object n)
{
  /*
   * This function sets the wordsize to the given value.
   */

  if (type(n) == TYPE_REAL) {
    char str[7];
    sprintf(str, "%d", (int) n.value.real);

    processDirective("wordsize", str);
    return ERR_NOERR;
  } else if (type(n) == TYPE_HXS) {
    char str[7];
    sprintf(str, "%d", (int) n.value.h);

    processDirective("wordsize", str);
    return ERR_NOERR;
  } else
    return ERR_BADARGUMENTTYPE;
}
  

void f_rcws(void)
{
  /*
   * This function returns the current wordsize to the stack.
   */

  insertReal((double)userOptions.wordSize);
}


void f_bin(void) 
{
  /*
   * This function sets the base to 2 (binary).
   */

  processDirective("base", "2");
}


void f_oct(void)
{
  /*
   * This function sets the base to 8 (octal).
   */

  processDirective("base", "8");
}


void f_dec(void)
{
  /*
   * This function sets the base to 10 (decimal).
   */

  processDirective("base", "10");
}


void f_hex(void)
{
  /*
   * This function sets the base to 16 (hexadecimal).
   */

  processDirective("base", "16");
}
