/*

  authc-kerberos-tgt.c

  Author: Tatu Ylonen <ylo@ssh.com>

  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Kerberos authentication, client side.

  This code assumes the presence of Kerberos libraries.
  
*/

#include "ssh2includes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "sshclient.h"
#include "sshconfig.h"
#include "authc-kerberos-tgt.h"

#ifdef KERBEROS

#undef ctime
#undef free

/* The #%@!$(#$ krb5.h header redefines these.  ARRRGH! -ylo */
#undef SIZEOF_INT
#undef SIZEOF_LONG
#undef SIZEOF_SHORT
#undef HAVE_STDARG_H
#undef HAVE_SYS_TYPES_H

#include <krb5.h>

#define SSH_DEBUG_MODULE "Ssh2AuthKerberosTgtClient"

/* Kerberos authentication, client-side. */

void ssh_client_auth_kerberos_tgt(SshAuthClientOperation op,
                                  const char *user,
                                  unsigned int packet_type,
                                  SshBuffer packet_in,
                                  const unsigned char *session_id,
                                  size_t session_id_len,
                                  void **state_placeholder,
                                  SshAuthClientCompletionProc completion,
                                  void *completion_context,
                                  void *method_context)
{
  SshConfig clientconf = ((SshClient)method_context)->config;
  SshBuffer b;
  krb5_error_code kerr;
  krb5_context kcontext;
  krb5_auth_context kauth;
  krb5_principal server;
  krb5_principal client;
  krb5_ccache ccache;
  krb5_data auth_data;
  char server_name[512];

  switch (op)
    {
    case SSH_AUTH_CLIENT_OP_START:
      /* XXX should really swap this code and START_NONINTERACTIVE code, but
         apparently there is some bug in sshauthc.c that causes NONINTERACTIVE
         to be called even when a method is not allowed. */

      /* Initialize data that we test in cleanup. */
      client = NULL;
      server = NULL;
      memset(&kcontext, 0, sizeof(kcontext));
      memset(&ccache, 0, sizeof(ccache));
      memset(&auth_data, 0, sizeof(auth_data));

      /* Initialize the Kerberos application context. */
      kerr = krb5_init_context(&kcontext);
      if (kerr)
        goto error;
      
      /* Get the name of the default credentials cache (KRB5CCACHE or
         perhaps some fallback). */
      kerr = krb5_cc_default(kcontext, &ccache);
      if (kerr)
        goto error;
      
      /* Initialize the authentication context. */
      kerr = krb5_auth_con_init(kcontext, &kauth);
      if (kerr)
        goto error;

      /* Enable saving timestamps to output structure. */
      krb5_auth_con_setflags(kcontext, kauth,
                             KRB5_AUTH_CONTEXT_RET_TIME);

      /* This hack is from SSH1 (where I think it originates to Dug
         Song and/or Glenn Machin).  To quite: "We need to get the TGT
         for the clients realm.  However if remotehost is in another
         realm krb5_fwd_tgt_creds will try to go to that realm to get
         the TGT, which will fail.  So we create the server principal
         and point it to clients realm.  This way we pass over a TGT
         of the clients realm." */
      kerr = krb5_cc_get_principal(kcontext, ccache, &client);
      if (kerr)
        goto error;
      snprintf(server_name, sizeof(server_name), "host/%s@%.*s",
               clientconf->host_to_connect,
               client->realm.length, client->realm.data);
      kerr = krb5_parse_name(kcontext, server_name, &server);
      if (kerr)
        goto error;
      server->type = KRB5_NT_SRV_HST;

      /* Get a ticket granting ticket for use at the remote host. */
      kerr = krb5_fwd_tgt_creds(kcontext, kauth, NULL, client, server,
                                ccache, 1, &auth_data);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_fwd_tgt_creds %d", kerr));
          goto error;
        }

      /* Construct the method-specific part of the authentication
         request. */
      b = ssh_buffer_allocate();
      ssh_encode_buffer(b,
                        SSH_FORMAT_UINT32_STR,
                          auth_data.data, auth_data.length,
                        SSH_FORMAT_END);
      
      /* Send the authentication request (and complete this operation). */
      (*completion)(SSH_AUTH_CLIENT_SEND, user, b, completion_context);
      
      /* Free the buffer. */
      ssh_buffer_free(b);

    cleanup:
      /* Clean up allocated Kerberos data. */
      if (auth_data.data)
        free(auth_data.data);
      if (client)
        krb5_free_principal(kcontext, client);
      if (server)
        krb5_free_principal(kcontext, server);
      krb5_cc_close(kcontext, ccache);
      if (kauth)
        krb5_auth_con_free(kcontext, kauth);
      if (kcontext)
        krb5_free_context(kcontext);
      break;
      
    error:
      ssh_debug("Kerberos5 TGT forwarding failed: %s", error_message(kerr));
      /* Indicate that we cannot use this authentication method. */
      (*completion)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD, user, NULL,
                    completion_context);
      goto cleanup;
      
    case SSH_AUTH_CLIENT_OP_START_NONINTERACTIVE:
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
      
    case SSH_AUTH_CLIENT_OP_CONTINUE:
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
      
    case SSH_AUTH_CLIENT_OP_ABORT:
      *state_placeholder = NULL;
      break;
      
    default:
      ssh_debug("ssh_client_auth_kerberos: unknown op %d", (int)op);
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
    }
}

#endif /* KERBEROS */
