/*

auths-securid.c

  Author: Graeme Ahokas <gahokas@ssh.com>

  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  SecurID authentication, server-side. This calls the SecurID library
  functions to authenticate the user.
*/

#include "sshincludes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshserver.h"
#include "sshconfig.h"
#include "auths-common.h"
#include "sshfsm.h"

#ifdef SSH_SERVER_WITH_SECURID

#include "sdi_athd.h"
#include "sdconf.h"
#include "sdi_defs.h"
#include "sdacmvls.h"


#define SSH_DEBUG_MODULE "Ssh2AuthSecurIDServer"

union config_record configure;

typedef struct SshServerSecurIDAuthRec
{
  SshFSM fsm;
  SshFSMThread main_thread;
  SshUser uc;

  SshBuffer client_packet;
  SshBuffer response_packet;
  Boolean pin_change_success;
  SshConfig config;

  struct SD_CLIENT *sd_data; /*handle for the ace server */

  void **state_placeholder;
  SshAuthServerCompletionProc completion_proc;
  void *completion_context;
} *SshServerSecurIDAuth;


/* initializes communications with the SecurID server.
   returns 0 if no errors, non zero otherwise */
int ssh_server_securid_init( struct SD_CLIENT *sd_data )
{
  int ret_val;

  if (sd_data == NULL)
    return( 1 );

  if (creadcfg() )
    {
      SSH_DEBUG(2,("Error reading sdconf.rec"));
      return (1);
    }

  memset(sd_data, 0, sizeof(*sd_data));

  ret_val = sd_init( sd_data );

  if (ret_val != 0)
    {
      SSH_DEBUG(2,("Error initializing SecurID server communications."));
      return( 1 );
    }

  return( 0 );

}

/* begins server communcations with the securid server and
   asks the client for the user's passphrase */
SSH_FSM_STEP(server_securid_start)
{
  SSH_FSM_GDATA(SshServerSecurIDAuth);
  char *passcode;
  int  length;
  int ret_code;

  if ( ssh_server_securid_init( gdata->sd_data ) )
    {
      ssh_buffer_free(gdata->client_packet); 
      SSH_DEBUG(2,("Error connecting to SecurID server."));
      SSH_FSM_SET_NEXT("server_securid_reject_and_disable");
      return SSH_FSM_CONTINUE;
    }

  /* communications with securid server established
     so check passcode */

  SSH_DEBUG(2,("Communications with SecurID server established."));

  if (ssh_decode_buffer(gdata->client_packet,
                        SSH_FORMAT_BOOLEAN,
                        NULL,
                        SSH_FORMAT_UINT32_STR,
                        &passcode, &length,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_free(gdata->client_packet);
      SSH_DEBUG(4,("Client packet decode failed.\r\n"));
      SSH_FSM_SET_NEXT("server_securid_reject_and_disable");
      return SSH_FSM_CONTINUE; 
    }

  /* XXX could check now for valid length of passcode */

  ret_code = sd_check(passcode, ssh_user_name(gdata->uc), gdata->sd_data);

  ssh_xfree(passcode);
  ssh_buffer_free(gdata->client_packet);

  switch(ret_code)
    {
      case ACM_OK:

        SSH_DEBUG(2,("User AUTHENTICATED.\r\n"));
        SSH_FSM_SET_NEXT("server_securid_auth_success");
        return SSH_FSM_CONTINUE; 

      case ACM_ACCESS_DENIED:

        gdata->config->securid_guesses--;
        if (gdata->config->securid_guesses <= 0)
          {
            SSH_FSM_SET_NEXT("server_securid_reject_and_disable");
            return SSH_FSM_CONTINUE;
          }

        SSH_DEBUG(2,("Access denied from SecurID server"));
        SSH_FSM_SET_NEXT("server_securid_reject");
        return SSH_FSM_CONTINUE;

      case ACM_NEXT_CODE_REQUIRED:

        SSH_DEBUG(2,("Next code required"));
        SSH_FSM_SET_NEXT("server_securid_get_next_code");
        return SSH_FSM_CONTINUE;

      case ACM_NEW_PIN_REQUIRED:

        SSH_DEBUG(2,("New PIN required."));
        SSH_FSM_SET_NEXT("server_securid_get_new_pin");
        return SSH_FSM_CONTINUE;

      default:
        SSH_DEBUG(4,("Unknown return code from SecurID server sd_check"));
        SSH_FSM_SET_NEXT("server_securid_reject_and_disable");
        return SSH_FSM_CONTINUE;
    }

  /* not reached */
}

/* Successful authentication, send ACCEPTED msg and quit */
SSH_FSM_STEP(server_securid_auth_success)
{

  SSH_FSM_GDATA(SshServerSecurIDAuth);

  *gdata->state_placeholder = NULL;

  (*gdata->completion_proc)(SSH_AUTH_SERVER_ACCEPTED,
                            NULL,
                            gdata->completion_context);

  SSH_FSM_SET_NEXT("server_securid_finish");
  return SSH_FSM_CONTINUE;
}

/* Sends a CHALLENGE msg to client for next token */
SSH_FSM_STEP(server_securid_get_next_code)
{

  SSH_FSM_GDATA(SshServerSecurIDAuth);

  gdata->response_packet = ssh_buffer_allocate();

  ssh_encode_buffer(gdata->response_packet,
                    SSH_FORMAT_CHAR,
                    (unsigned int)SSH_MSG_USERAUTH_SECURID_CHALLENGE,
                    SSH_FORMAT_BOOLEAN,
                    TRUE,
                    SSH_FORMAT_END);

  (*gdata->completion_proc)(SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK,
                            gdata->response_packet,
                            gdata->completion_context);

  SSH_FSM_SET_NEXT("server_securid_process_next_code");
  return SSH_FSM_SUSPENDED;
}


/* Finishes the fsm */
SSH_FSM_STEP(server_securid_finish)
{
  SSH_FSM_GDATA(SshServerSecurIDAuth);

  /* close communications with securid server */
  sd_close();

  ssh_fsm_destroy(gdata->fsm);
  gdata->fsm = NULL;

  *gdata->state_placeholder = NULL;

  return SSH_FSM_FINISH;
}

/* receives challenge response from client and tests*/
SSH_FSM_STEP(server_securid_process_next_code)
{
  SSH_FSM_GDATA(SshServerSecurIDAuth);
  char *passcode;
  int length;
  int ret_code;

  if (ssh_decode_buffer(gdata->client_packet,
                        SSH_FORMAT_BOOLEAN,
                        NULL,
                        SSH_FORMAT_UINT32_STR,
                        &passcode, &length,
                        SSH_FORMAT_END) == 0)
    {
      ssh_xfree(gdata->client_packet);
      SSH_DEBUG(4,("Client packet decode failed."));
      SSH_FSM_SET_NEXT("server_securid_reject_and_disable");
      return SSH_FSM_CONTINUE; 
    }

  /* we're verifying the challenge from the server for the next token 
     or making user authenticate with new pin */

  /* XXX could check now for valid length of passcode */

  /* if we have changed the pin, we must call sd_check, NOT sd_next, even
     though we actually want the next token. unintuitive, i know... */
  if (gdata->pin_change_success)
    ret_code = sd_check(passcode, ssh_user_name(gdata->uc), gdata->sd_data);
  else
    ret_code = sd_next(passcode, gdata->sd_data);

  ssh_xfree(passcode);
  ssh_buffer_free(gdata->client_packet);

  switch(ret_code)
    {
      case ACM_OK:

        SSH_DEBUG(2,("User AUTHENTICATED by challenge."));
        SSH_FSM_SET_NEXT("server_securid_auth_success");
        return SSH_FSM_CONTINUE;

      case ACM_ACCESS_DENIED:

        SSH_DEBUG(2,("User access denied from challenge."));
        SSH_FSM_SET_NEXT("server_securid_reject");
        return SSH_FSM_CONTINUE;

      case ACM_NEXT_CODE_REQUIRED:

        /* this probably isn't necessary, as we likely wouldn't get
           a next code req msg after a successful authentication with
           a new pin, but just to be safe */
        SSH_DEBUG(2,("Next code required"));
        SSH_FSM_SET_NEXT("server_securid_get_next_code");
        return SSH_FSM_CONTINUE;

      default:

        SSH_DEBUG(4,("Unknown return code from securid server sd_next"));
        SSH_FSM_SET_NEXT("server_securid_reject_and_disable");
        return SSH_FSM_CONTINUE;
    }
}

/* Sends a reject authentication msg to client and stops fsm */
SSH_FSM_STEP(server_securid_reject)
{

  SSH_FSM_GDATA(SshServerSecurIDAuth);
  SSH_PRECOND(gdata);

  *gdata->state_placeholder = NULL;

  (*gdata->completion_proc)(SSH_AUTH_SERVER_REJECTED,
                            NULL,
                            gdata->completion_context);

  SSH_FSM_SET_NEXT("server_securid_finish");

  return SSH_FSM_CONTINUE;
}

/* sends reject and method disable msg and stops fsm */
SSH_FSM_STEP(server_securid_reject_and_disable)
{
  SSH_FSM_GDATA(SshServerSecurIDAuth);
  SSH_PRECOND(gdata);

  *gdata->state_placeholder = NULL;

  (*gdata->completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                            NULL,
                            gdata->completion_context);

  SSH_FSM_SET_NEXT("server_securid_finish");
  return SSH_FSM_CONTINUE;
}


/* sends the client a new pin request, complete with information
   on the limitations of the pin, and a server assigned pin if
   the user is not permitted to choose */
SSH_FSM_STEP(server_securid_get_new_pin)
{
  SSH_FSM_GDATA(SshServerSecurIDAuth);
  unsigned int user_selectable;
  Boolean alphanumeric;
  char *system_pin;


  gdata->response_packet = ssh_buffer_allocate();

  /* assign the user_selectable field to inform
     the user about their limitations in pin selection*/

  if (gdata->sd_data->user_selectable == CANNOT_CHOOSE_PIN)
    user_selectable = SSH_SECURID_CANNOT_CHOOSE_PIN;
  else if (gdata->sd_data->user_selectable == MUST_CHOOSE_PIN)
    user_selectable = SSH_SECURID_MUST_CHOOSE_PIN;
  else
    user_selectable = SSH_SECURID_USER_SELECTABLE_PIN;

  if (gdata->sd_data->alphanumeric == 0)
    alphanumeric = FALSE;
  else
    alphanumeric = TRUE;

  if (user_selectable == SSH_SECURID_MUST_CHOOSE_PIN)
    system_pin = ssh_xstrdup("");
  else
    system_pin = ssh_xstrdup(gdata->sd_data->system_pin);

  ssh_encode_buffer( gdata->response_packet,
                     SSH_FORMAT_CHAR,
                     SSH_MSG_USERAUTH_SECURID_NEW_PIN_REQD,
                     SSH_FORMAT_UINT32,
                     gdata->sd_data->min_pin_len,
                     SSH_FORMAT_UINT32,
                     gdata->sd_data->max_pin_len,
                     SSH_FORMAT_CHAR,
                     (unsigned int)user_selectable,
                     SSH_FORMAT_BOOLEAN,
                     alphanumeric,
                     SSH_FORMAT_UINT32_STR,
                     system_pin,
                     strlen(system_pin),
                     SSH_FORMAT_END);

  /* send packet to notify client of the message, and wait for reply */
  (*gdata->completion_proc)(SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK,
                            gdata->response_packet,
                            gdata->completion_context);

  ssh_xfree(system_pin);

  SSH_FSM_SET_NEXT("server_securid_process_new_pin");
  return SSH_FSM_SUSPENDED;
}


/* takes the pin from the client and informs the securid server */
SSH_FSM_STEP(server_securid_process_new_pin)
{
  SSH_FSM_GDATA(SshServerSecurIDAuth);

  int ret_code;
  char *new_pin;
  int new_pin_len;
  char cancel_new_pin;


  /* decode the packet from the client */
  if (ssh_decode_buffer(gdata->client_packet,
                        SSH_FORMAT_BOOLEAN, NULL,
                        SSH_FORMAT_UINT32_STR, &new_pin, &new_pin_len,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_free(gdata->client_packet);
      SSH_FSM_SET_NEXT("server_securid_reject_and_disable");
      return SSH_FSM_CONTINUE;
    }

  ssh_buffer_free(gdata->client_packet);

  if (strlen(new_pin) == 0)
    cancel_new_pin = 1;
  else
    cancel_new_pin = 0;
 
  /* verify the new pin with the securid server */
  ret_code = sd_pin(new_pin, cancel_new_pin, gdata->sd_data);

  ssh_xfree(new_pin);

  if (ret_code == ACM_NEW_PIN_ACCEPTED)
    {
      SSH_DEBUG(2,("New PIN operation successful"));
      gdata->pin_change_success = TRUE;
      /* new pin is ok so send a passphrase next token msg to client */
      SSH_FSM_SET_NEXT("server_securid_get_next_code");
      return SSH_FSM_CONTINUE;
    }

  SSH_DEBUG(2,("New PIN operation failed."));
  /* new pin operation failed */
  SSH_FSM_SET_NEXT("server_securid_reject");
  return SSH_FSM_CONTINUE;
}


SshFSMStateMapItemStruct server_securid_states[] =
{
  { "server_securid_start", "Starting securid authentication",
     server_securid_start },
  { "server_securid_finish", "Finished authentication",
     server_securid_finish },
  { "server_securid_reject", "Rejects authentication",
     server_securid_reject },
  { "server_securid_reject_and_disable", "Rejects and disables authentication",
     server_securid_reject_and_disable },
  { "server_securid_get_next_code", "Asks client for next passcode",
     server_securid_get_next_code },
  { "server_securid_process_next_code", "Checks next code with server",
     server_securid_process_next_code },
  { "server_securid_get_new_pin", "Asks or gives the user a new pin",
     server_securid_get_new_pin },
  { "server_securid_process_new_pin", "Register new pin with server",
     server_securid_process_new_pin },
  { "server_securid_auth_success", "Sends successful auth msg to client",
     server_securid_auth_success }
};


void server_securid_destructor( void *gdata_ctx )
{
  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) gdata_ctx;
  SSH_PRECOND(gdata);

  if ( gdata->fsm )
    ssh_fsm_destroy( gdata->fsm );
  gdata->fsm = NULL;
  *gdata->state_placeholder = NULL;

  if (gdata->sd_data)
    ssh_xfree(gdata->sd_data);
}


/* SecurID authentication, server side. */

void ssh_server_auth_securid(SshAuthServerOperation op,
                            const char *user,
                            SshUser uc,
                            SshBuffer packet,
                            const unsigned char *session_id,
                            size_t session_id_len,
                            void **state_placeholder,
                            void **longtime_placeholder,
                            SshAuthServerCompletionProc completion_proc,
                            void *completion_context,
                            void *method_context)
{
  SshServer server = (SshServer)method_context;
  SshConfig config = server->config;
  SshServerSecurIDAuth state = *state_placeholder; 
  int packet_type;
  SshFSM fsm;

  SSH_TRACE(6, ("securid auth."));

  SSH_DEBUG(6, ("op = %d  user = %s", op, user));

  switch (op)
    {
    case SSH_AUTH_SERVER_OP_START:

      /* receives the start securid authentication request
         from the client. will send AUTH_SUCCESS or request
         new pin or challenge */

      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_SECURID))
        {
          (*completion_proc) (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                              packet,
                              completion_context);
          return;
        }


      if(ssh_user_uid(uc) == SSH_UID_ROOT &&
         (config->permit_root_login == SSH_ROOTLOGIN_FALSE ||
          config->permit_root_login == SSH_ROOTLOGIN_NOPWD))
        {
          /* XXX Add client addresses etc. */
          SSH_DEBUG(2, ("root logins are not permitted."));
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "root login denied for user %s.",
                        ssh_user_name(uc));

          (*completion_proc)
             (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED, packet,
             completion_context);

          return;
        }

      /* start up the fsm */

      SSH_ASSERT(*state_placeholder == NULL);
      fsm = ssh_fsm_allocate(sizeof(*state),
                             server_securid_states,
                             SSH_FSM_NUM_STATES(server_securid_states),
                             server_securid_destructor );

      state = ssh_fsm_get_gdata_fsm(fsm);
      memset(state, 0, sizeof(*state));
      state->fsm = fsm;
 
      state->uc = uc;
      state->completion_proc = completion_proc;
      state->completion_context = completion_context;
      state->state_placeholder = state_placeholder;
      state->client_packet = packet;
      state->response_packet = NULL;
      state->sd_data = ssh_xcalloc(1, (sizeof(*state->sd_data)));
      state->pin_change_success = FALSE;
      state->config = config;
      *state_placeholder = state;
 
      state->main_thread = ssh_fsm_spawn(state->fsm,
                                         0, "server_securid_start",
                                         NULL, NULL);

      return;

    case SSH_AUTH_SERVER_OP_CONTINUE:

      /* pass the packet to the fsm for processing */
      SSH_ASSERT(state && state->main_thread);
      state->client_packet = packet;
      state->completion_proc = completion_proc;
      state->completion_context = completion_context;

      ssh_fsm_continue(state->main_thread);
      return;

    case SSH_AUTH_SERVER_OP_ABORT:

      SSH_ASSERT(state && state->main_thread);
      ssh_fsm_set_next(state->main_thread, "server_securid_finish");
      ssh_fsm_continue(state->main_thread);
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet, completion_context);
      return;

    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      /* Clean up state. */
      if (state)
        if (state->main_thread)
          {
            ssh_fsm_set_next(state->main_thread,
                             "server_securid_finish");
            ssh_fsm_continue(state->main_thread);
          }

      *state_placeholder = NULL;
      (*completion_proc) (SSH_AUTH_SERVER_REJECTED, packet,
                          completion_context);
      return;

    default:
      ssh_fatal("ssh_server_auth_securid: unknown op %d", (int)op);
    }
 
  SSH_NOTREACHED;
}

#endif /* SSH_SERVER_WITH_SECURID */
