/*

  sshadt.h

  Author: Antti Huima <huima@ssh.fi>

  Copyright (c) 1999, 2000 SSH Communications Security, Finland
  All rights reserved.

  Created Wed Sep  8 17:01:04 1999.

  */

#ifndef SSHADT_H_INCLUDED
#define SSHADT_H_INCLUDED

/* The fast macro interface is used unless DEBUG_LIGHT is defined. */

#ifndef DEBUG_LIGHT
#define SSH_ADT_WITH_MACRO_INTERFACE
#endif

#define SSHADT_INSIDE_SSHADT_H

/* Generic header type. */
typedef struct {
  void *ptr[4];
} SshADTHeaderStruct;

typedef struct SshADTContainerRec *SshADTContainer;

/* Handles for doing iteration, inserting objs etc. */
typedef void *SshADTHandle;

/* Function type to compare two user-supplied data items. */
typedef int (* SshADTCompareFunc)(const void *obj1, const void *obj2,
                                  void *context);

/* Function type to duplicate a user-supplied data item when the
   void pointer semantics are used.

   `obj' is not constant because it is possible that the object is
   duplicated by e.g. increasing its reference count and returning
   itself. */
typedef void *(* SshADTDuplicateFunc)(void *obj, void *context);

/* Function type to initialize an object from an another --- to do copy ---
   when internally allocated objects are used. */
typedef void (* SshADTCopyFunc)(void *dst, size_t dst_size,
                                const void *src, void *context);

/* Function type to destroy a user-supplied data item. */
typedef void (* SshADTDestroyFunc)(void *obj, void *context);

/* Function type to initialize a user-supplied data item when it is
   allocated by the library. */
typedef void (* SshADTInitFunc)(void *obj, size_t size, void *context);

/* Function type to calculate the hash of a user-supplied data item. */
typedef SshUInt32 (* SshADTHashFunc)(const void *obj, void *context);

typedef enum {
  /* Argument type identifier           C type of the next argument
     ========================           =========================== */

  /* Set the user-supplied context. Defaults to NULL when not set. */
  SSH_ADT_CONTEXT,                      /* void * */

  /* Set the compare func. */
  SSH_ADT_COMPARE,                      /* SshADTCompareFunc */

  /* Set the duplicate func. */
  SSH_ADT_DUPLICATE,                    /* SshADTDuplicateFunc */

  /* Set the copy func. */
  SSH_ADT_COPY,                         /* SshADTCopyFunc */

  /* Set the destroy func. */
  SSH_ADT_DESTROY,                      /* SshADTDestroyFunc */

  /* Set the hash func. */
  SSH_ADT_HASH,                         /* SshADTHashFunc */

  /* Set the init func. */
  SSH_ADT_INIT,                         /* SshADTInitFunc */

  /* Tell the ADT library to use a header inside the objects instead
     of allocating auxiliary data outside the objects. The next
     argument must be the offset (size_t) to an SshADTHeader field. */
  SSH_ADT_HEADER,                       /* SSH_ADT_OFFSET_OF(...) */

  /* Tell the ADT library the default size of the objects. This turns
     automatic object allocation on. */
  SSH_ADT_SIZE,                         /* size_t */

  /* Tell the ADT library to allocate objects by using the
     sshfastalloc utility. Two restrictions: 1) SSH_ADT_SIZE must be
     set also. 2) If you use SSH_ADT_FASTALLOC you cannot allocate
     objects that are not of the default size. Thus, ssh_adt_realloc
     is not available and ssh_adt_alloc does not work with non-default
     size. */
  SSH_ADT_FASTALLOC,                    /* int */

  /* Give a hint about the maximum number of objects that will be inserted
     to the container. This can be used e.g. to guess the initial size
     of a hash table. */
  SSH_ADT_SIZE_HINT,                    /* long */

  /* end of args marker. */
  SSH_ADT_ARGS_END                      /* None */
} SshADTArgumentType;

#define SSH_ADT_OFFSET_OF(type,field) \
((unsigned char *)(&((type *)0)->field) - (unsigned char *)0)

/* Container types. */

typedef void *SshADTContainerType;

/* Invalid handle. */

#define SSH_ADT_INVALID   NULL

/******************************************** Handling containers as a whole */

/* Generic container allocation. This can return NULL if memory
   allocation fails in those cases where memory allocation can fail
   (e.g. in kernel code). */
SshADTContainer ssh_adt_create_generic(SshADTContainerType type,
                                       ...);

/* Destroy a container and all objects contained. */
void ssh_adt_destroy(SshADTContainer container);

/* Make the container empty: destroy the contained objects but not the
   container itself. This returns the container basically to the state
   just after create. */
void ssh_adt_clear(SshADTContainer container);

/****************************************** Absolute and relative locations. */

typedef long SshADTAbsoluteLocation;

/* The negative integers denote `special' values. */
#define SSH_ADT_BEGINNING -1
#define SSH_ADT_END       -2
#define SSH_ADT_DEFAULT   -3

/* Integers >= 0 denote positions in a total order (e.g. dynamic array
   slots). */
#define SSH_ADT_INDEX(n)    (n)
#define SSH_ADT_UNINDEX(n)  (n)

typedef enum {
  SSH_ADT_BEFORE,
  SSH_ADT_AFTER
} SshADTRelativeLocation;

/******************************************* Creating and inserting objects. */

/* 1. Void pointer semantics. */

/* Insert an object to container. This interface works only 
   when the void pointers semantics is used. Returns a handle to the
   inserted object. */
SshADTHandle ssh_adt_insert_at(SshADTContainer container,
                               SshADTRelativeLocation location,
                               SshADTHandle handle,
                               void *object);

SshADTHandle ssh_adt_insert_to(SshADTContainer container,
                               SshADTAbsoluteLocation location,
                               void *object);

SshADTHandle ssh_adt_insert(SshADTContainer container,
                            void *object);

SshADTHandle ssh_adt_duplicate_at(SshADTContainer container,
                               SshADTRelativeLocation location,
                               SshADTHandle handle,
                               void *object);

SshADTHandle ssh_adt_duplicate_to(SshADTContainer container,
                               SshADTAbsoluteLocation location,
                               void *object);

SshADTHandle ssh_adt_duplicate(SshADTContainer container,
                               void *object);

/* 2. Internally allocated objects. */

SshADTHandle ssh_adt_alloc_n_at(SshADTContainer container,
                                SshADTRelativeLocation location,
                                SshADTHandle handle,
                                size_t size);

SshADTHandle ssh_adt_alloc_n_to(SshADTContainer container,
                                SshADTAbsoluteLocation location,
                                size_t size);

SshADTHandle ssh_adt_alloc_at(SshADTContainer container,
                              SshADTRelativeLocation location,
                              SshADTHandle handle);

SshADTHandle ssh_adt_alloc_to(SshADTContainer container,
                              SshADTAbsoluteLocation location);

SshADTHandle ssh_adt_alloc_n(SshADTContainer container,
                             size_t size);

SshADTHandle ssh_adt_alloc(SshADTContainer container);

/* These perform copy operation from the object *obj instead of new
   initialization. Cannot use a separate duplication function as above
   is used because internally allocated objects are initialized using
   the initialization `callback' and now we want to avoid the
   initialization when using copy constructor. */
SshADTHandle ssh_adt_put_n_at(SshADTContainer container,
                              SshADTRelativeLocation location,
                              SshADTHandle handle,
                              size_t size,
                              void *obj);

SshADTHandle ssh_adt_put_n_to(SshADTContainer container,
                              SshADTAbsoluteLocation location,
                              size_t size,
                              void *obj);

SshADTHandle ssh_adt_put_at(SshADTContainer container,
                            SshADTRelativeLocation location,
                            SshADTHandle handle,
                            void *obj);

SshADTHandle ssh_adt_put_to(SshADTContainer container,
                            SshADTAbsoluteLocation location,
                            void *obj);

SshADTHandle ssh_adt_put_n(SshADTContainer container,
                           size_t size,
                           void *obj);

SshADTHandle ssh_adt_put(SshADTContainer container,
                         void *obj);

/********************************************************* Accessing objects */

/* Get the object at the handle. */
void *ssh_adt_get(SshADTContainer container, SshADTHandle handle);

/* Get the number of objects inside a container. */
size_t ssh_adt_num_objects(SshADTContainer container);

/*********************************************************** Setting handles */

/* Get a handle to the object. The pointers must match exactly. This
   can be a slow operation if the void pointer semantics is used. */
SshADTHandle ssh_adt_get_handle_to(SshADTContainer container,
                                   void *object);

/* Get a handle to some object that is equal to `object'. */
SshADTHandle ssh_adt_get_handle_to_equal(SshADTContainer container,
                                         void *object);

/* Moving handles in rudimentary was for containers that support some
   kind of ordering (the most assumedly do). */

/* Move the handle to the next object. */
SshADTHandle ssh_adt_next(SshADTContainer container, SshADTHandle handle);

/* Move the handle to the previous object. */
SshADTHandle ssh_adt_previous(SshADTContainer container, SshADTHandle handle);

/* get the handle / object from an absolute location */
SshADTHandle ssh_adt_get_handle_to_location(SshADTContainer container,
                                            SshADTAbsoluteLocation location);

/* get the handle / object from an absolute location */
void *ssh_adt_get_object_from_location(SshADTContainer container,
                                       SshADTAbsoluteLocation location);

/***************************************** Removing objects from a container */

/* Detach the object that ssh_adt_get(container, handle) would return
   from the container.  After this the handle is invalidated.  Handle
   must be valid initial to the method call.  */
void *ssh_adt_detach(SshADTContainer container,              
                     SshADTHandle handle);

/* Detach object from a valid location.  */
void *ssh_adt_detach_from(SshADTContainer container,
                          SshADTAbsoluteLocation location);

/* Detach object from container.  It is an error for object not been
   found in the container. */
void *ssh_adt_detach_object(SshADTContainer container, void *object);

/*********************************** Deleting objects when inside container */

/* Destroy the object that ssh_adt_get(container, handle) would
   return.  Invalidates the handle.  Handle must be valid initial to
   the method call.  */
void ssh_adt_delete(SshADTContainer container,
                    SshADTHandle handle);

/* Destroy object from a valid location.  */
void ssh_adt_delete_from(SshADTContainer container,
                         SshADTAbsoluteLocation location);

/* Delete object from container.  It is an error for object not been
   found in the container. */
void ssh_adt_delete_object(SshADTContainer container,
                           void *object);

/************************************* Resizing internally allocated objects */

/* Reallocate a object inside a container. */
void *ssh_adt_realloc(SshADTContainer container, void *object,
                      size_t new_size);

/* Return the default object size. */
size_t ssh_adt_default_size(SshADTContainer container);

/******************************************************* Duplicating objects */

/* Duplicate a void pointer object. */
void *ssh_adt_duplicate_object(SshADTContainer container, void *object);

/********************************************* Generic enumeration functions */

/* Start enumerating a container. */
SshADTHandle ssh_adt_enumerate_start(SshADTContainer container);

/* Continue enumeration. Returns `SSH_ADT_INVALID' when all objects have been
   enumerated. */
SshADTHandle ssh_adt_enumerate_next(SshADTContainer container,
                                    SshADTHandle handle);

/************************************************ Other container operations */

/* Create a clone of container.  This is not a proper class method,
   but an external client function.  Whether this is a good idea or
   not will turn out some day.  */
SshADTContainer ssh_adt_duplicate_container(SshADTContainer container);

/************************ If the macro interface is active, include the rest */

#ifdef SSH_ADT_WITH_MACRO_INTERFACE
#include "sshadt_structs.h"
#include "sshadt_impls.h"
#include "sshadt_shortcuts.h"
#endif

#undef SSHADT_INSIDE_SSHADT_H

#endif /* SSHADT_H_INCLUDED */
