/*

  sshadt_conv.c

  Author: Antti Huima <huima@ssh.fi>

  Copyright (c) 2000 SSH Communications Security, Finland
  All rights reserved.

  Created Wed Sep 20 05:00:29 2000.

  */

#include "sshincludes.h"
#include "sshdebug.h"
#include "sshadt_i.h"
#include "sshadt_map.h"
#include "sshadt_strmap.h"
#include "sshadt_intmap.h"

#define SSH_DEBUG_MODULE "SshADTConv"



/**************************************** Callbacks that are frequently used */

void ssh_adt_callback_destroy_free(void *obj, void *context)
{
  SSH_MEM_FREE(obj);
}

void ssh_adt_callback_destroy_free_null(void *obj, void *context)
{
  if (obj == NULL) return;
  SSH_MEM_FREE(obj);
}

int ssh_adt_callback_compare_str(const void *obj1, const void *obj2, void *ctx)
{
  const char *s1 = obj1, *s2 = obj2;
  return strcmp(s1, s2);
}

void *ssh_adt_callback_duplicate_str(const void *obj, void *ctx)
{
  const char *s = obj;
  return SSH_MEM_STRDUP(s);
}

unsigned long ssh_adt_callback_hash_str(const void *obj, void *ctx)
{
  unsigned long hash = 0;
  const char *c = obj;
  while (*c)
    {
      hash = (hash << 17) + (hash >> 15) + *c;
      c++; /* I dislike c++. */
    }
  return hash;
}

int ssh_adt_callback_compare_int(const void *obj1, const void *obj2,
                                 void *context)
{
  if (*(SshInt32*)obj1 < *(SshInt32*)obj2) return -1;
  if (*(SshInt32*)obj1 > *(SshInt32*)obj2) return 1;
  return 0;
}

unsigned long ssh_adt_callback_hash_int(const void *obj, void *ctx)
{
  return *(unsigned long *)obj;
}


/******************************************** Maps from strings to something */

SshADTContainer ssh_adt_create_strmap(void)
{
  return ssh_adt_create_generic(SSH_ADT_MAP,
                                SSH_ADT_HASH, ssh_adt_callback_hash_str,
                                SSH_ADT_DUPLICATE, ssh_adt_callback_duplicate_str,
                                SSH_ADT_DESTROY, ssh_adt_callback_destroy_free,
                                SSH_ADT_COMPARE, ssh_adt_callback_compare_str,
                                SSH_ADT_ARGS_END);
}

SshADTHandle ssh_adt_strmap_add(SshADTContainer c, const char *key,
                                void *value)
{
  SshADTHandle h;
  SSH_ASSERT(!(ssh_adt_strmap_exists(c, key)));
  h = ssh_adt_duplicate(c, (void *)key);
  SSH_ASSERT(ssh_adt_strmap_exists(c, key));
  ssh_adt_map_set_map(c, h, value);
  return h;
}

void ssh_adt_strmap_remove(SshADTContainer c, const char *key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, (void *)key);
  if (h != SSH_ADT_INVALID)
    {
      ssh_adt_delete(c, h);
    }
}

void ssh_adt_strmap_set(SshADTContainer c, const char *key, void *value)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, (void *)key);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  ssh_adt_map_set_map(c, h, value);  
}

void *ssh_adt_strmap_get(SshADTContainer c, const char *key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, (void *)key);
  if (h == SSH_ADT_INVALID)
    return NULL;
  return ssh_adt_map_map(c, h);  
}

Boolean ssh_adt_strmap_exists(SshADTContainer c, const char *key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, (void *)key);
  return (h != SSH_ADT_INVALID);
}

/******************************************* Maps from integers to something */

SshADTContainer ssh_adt_create_intmap(void)
{
  return ssh_adt_create_generic(SSH_ADT_MAP,
                                SSH_ADT_HASH, ssh_adt_callback_hash_int,
                                SSH_ADT_COMPARE, ssh_adt_callback_compare_int,
                                SSH_ADT_SIZE, sizeof(SshUInt32),
                                SSH_ADT_ARGS_END);
}

SshADTHandle ssh_adt_intmap_add(SshADTContainer c, SshUInt32 key,
                                void *value)
{
  SshADTHandle h;
  SSH_ASSERT(!(ssh_adt_intmap_exists(c, key)));
  h = ssh_adt_put(c, &key);
  SSH_ASSERT(ssh_adt_intmap_exists(c, key));
  ssh_adt_map_set_map(c, h, value);
  return h;
}

void ssh_adt_intmap_remove(SshADTContainer c, SshUInt32 key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, &key);
  if (h != SSH_ADT_INVALID)
    {
      ssh_adt_delete(c, h);
    }
}

void ssh_adt_intmap_set(SshADTContainer c, SshUInt32 key, void *value)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, &key);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  ssh_adt_map_set_map(c, h, value);  
}

void *ssh_adt_intmap_get(SshADTContainer c, SshUInt32 key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, &key);
  if (h == SSH_ADT_INVALID)
    return NULL;
  return ssh_adt_map_map(c, h);  
}

Boolean ssh_adt_intmap_exists(SshADTContainer c, SshUInt32 key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, &key);
  return (h != SSH_ADT_INVALID);
}
