/*

  t-adt-list-torture.c

  Author: Matthias Fischmann <fis@ssh.fi>

  Copyright (c) 2000 SSH Communications Security, Finland
  All rights reserved.

  Created Mon Jul 31 16:29:39 2000.

  This is a mutated clone of "./t-adt.c".  The name `list-torture' is
  still there for historical reasons.  Actually, this file will be
  about all the test that I come up with until I get confused and need
  to open a new file.

  */

#include "sshincludes.h"
#include "sshadt_i.h"           /* Needed for testing the internals */
#include "sshadt_assoc.h"
#include "sshadt_list.h"
#include "sshadt_map.h"
#include "sshadt_array.h"
#include "sshadt_priority_queue.h"
#include "sshadt_strmap.h"
#include "sshdebug.h"
#include "sshregression.h"
#include "sshbuffer.h"

#define SSH_DEBUG_MODULE "SshADTTestListTorture"

#define T SSH_REGRESSION_TEST_TIME
#define TI SSH_REGRESSION_TEST_WITH_INIT
#define TN(d,f) T(d,f,())

#define NUM_ITEMS 20


/*************************************************************** auxiliaries */

void DUMP_EL(SshBuffer b,
             SshADTContainer list,
             const int debug_level,
             const char *komma,
             SshADTHandle h)
{
  void *o;
  char s[1000];

  o = ssh_adt_get(list, h);

  if (o == NULL)
    {
      snprintf(s, 1000, "%s", "NULL");
    }
  else
    {
      if (debug_level < 5)
        snprintf(s, 1000, "%i", *((int *) o));
      else
        snprintf(s, 1000, "%i{%p,%p}", *((int *)o), o, h);
    }

  ssh_buffer_append_cstrs(b, komma, s, NULL);
}                                                     



static void int_container_print(SshADTContainer list,
                                const char *name,
                                const int debug_level)
{
  /* prints all elements in ML list style.  depends on enumeration methods */

  SshBuffer b;
  char s[1000];
  SshADTHandle h;

  b = ssh_buffer_allocate();
  snprintf(s, 1000, "%s=[", name);
  ssh_buffer_append_cstrs(b, s, NULL);

  if ((h = ssh_adt_enumerate_start(list)) != SSH_ADT_INVALID)
    {
      DUMP_EL(b, list, debug_level, "", h);

      while ((h = ssh_adt_enumerate_next(list, h)) != SSH_ADT_INVALID)
        DUMP_EL(b, list, debug_level, ", ", h);
    }

  ssh_buffer_append(b, "]\0", 2);
  SSH_DEBUG(debug_level, ("%s", ssh_buffer_ptr(b)));
  ssh_buffer_free(b);
}



static void memdump(void *p, size_t len)
{
  SshBuffer b = ssh_buffer_allocate();
  ssh_buffer_append(b, p, len);
  buffer_dump(b);
  ssh_buffer_free(b);
}


static void container_dump(SshADTContainer c)
{
  /* dumps some of the internal data structure of a container.  could
     probably be extended significantly, but I don't need it now.  */

  SSH_DEBUG(0, ("======================================================================\n"));

  /* static data */
  
  SSH_DEBUG(0, ("[static_data:]\n  methods = %p\n  internal_header_size = %i\n  flags = %li\n",
           &(c->static_data->methods),
           c->static_data->internal_header_size,
           c->static_data->flags));

  /* container specific */

  SSH_DEBUG(0, ("[container_specific:]\n  %p\n",
           c->container_specific));

  memdump(&c->container_specific, sizeof(c->container_specific));

  /* hooks */


  /* instance flags */

  SSH_DEBUG(0, ("[flags:]\n  %li\n",
           c->flags));

  /* standard fields */

  SSH_DEBUG(0, ("[standard_fields:]\n  callbacks = %p\n  osize = %i\n  hoffset = %li\n",
           &(c->f.app_methods),
           c->f.default_object_size,
           c->f.header_offset));

  memdump(&c->f.app_methods, sizeof(c->f.app_methods));

  /* num objects */

  SSH_DEBUG(0, ("[aux:]\n  num_objects = %i\n  fast_alloc = %p\n",
           c->aux.num_objects,
           c->aux.fast_alloc));
    
  SSH_DEBUG(0, ("======================================================================\n"));
}


int ssh_adt_container_compare(SshADTContainer c1, SshADTContainer c2, int order_matters)
{
  /* Returns non-zero if c1 contains another set of elements than c2.
     The order of elements as generated by enumerate matters iff
     order_matters is non-zero.  */

  SshADTHandle h;

  if (order_matters)
    {
      SSH_NOTREACHED;
    }
  else
    {
#define CMP(c1, c2, code)                                                  \
      if ((h = ssh_adt_enumerate_start(c1)) != SSH_ADT_INVALID)            \
        {                                                                  \
          do                                                               \
            {                                                              \
              if (ssh_adt_get_handle_to_equal(c2, ssh_adt_get(c1, h)) ==   \
                  SSH_ADT_INVALID)                                         \
                return code;                                               \
            }                                                              \
          while ((h = ssh_adt_enumerate_next(c1, h)) != SSH_ADT_INVALID);  \
        }

  CMP(c1, c2, 1);
  CMP(c2, c1, -1);

#undef CMP
    }

  return 0;
}


/***************************************************************** callbacks */

typedef struct {
  int i;
  SshADTHeaderStruct header;
} IntRecord;

static void myalloc_destructor(void *ptr, void *ctx)
{
  SSH_DEBUG(8, ("Myalloc destructor"));
  SSH_ASSERT(ptr);
  ssh_xfree(ptr);
}

static void liballoc_destructor(void *ptr, void *ctx)
{
  SSH_DEBUG(8, ("Liballoc destructor"));
}

static unsigned long int_hash(const void *ptr, void *ctx)
{
  return *((int *)ptr);
}

static int int_cmp(const void *ptr1, const void *ptr2, void *ctx)
{
  int a, b;

  /* if pointers are NULL we don't care. */
  if (ptr1 == NULL) return 0;
  if (ptr2 == NULL) return 0;

  a = *((int *)ptr1);
  b = *((int *)ptr2);

  return a - b;
}

static void *int_dupl(const void *o1, void *ctx)
{
  int *o2 = ssh_xmalloc(sizeof(int));
  *o2 = *((int *)o1);
  return ((void *)o2);
}

static void *int_dupl_with_header(const void *o1, void *ctx)
{
  IntRecord *o2 = ssh_xmalloc(sizeof(IntRecord));
  o2->i = ((IntRecord *)o1)->i;
  return ((void *)o2);
}

static void insert_to_voidptr_with_header(SshADTContainer c,
                                          SshADTAbsoluteLocation l,
                                          int i)
{
  IntRecord *ptr = ssh_xmalloc(sizeof(IntRecord));
  ptr->i = i;
  ssh_adt_insert_to(c, l, ptr);
}


/**************************************************** creation and insertion */

SshADTContainer create_list_voidptr(void)
{
  return ssh_adt_create_generic(SSH_ADT_LIST,
                                SSH_ADT_COMPARE, int_cmp,
                                SSH_ADT_DUPLICATE, int_dupl,
                                SSH_ADT_DESTROY, myalloc_destructor,
                                SSH_ADT_ARGS_END);
}

SshADTContainer create_list_voidptr_with_header(void)
{
  return ssh_adt_create_generic(SSH_ADT_LIST,
                                SSH_ADT_COMPARE, int_cmp,
                                SSH_ADT_DUPLICATE, int_dupl_with_header,
                                SSH_ADT_DESTROY, myalloc_destructor,
                                SSH_ADT_HEADER,
                                SSH_ADT_OFFSET_OF(IntRecord, header),
                                SSH_ADT_ARGS_END);
}

SshADTContainer create_list_liballoc(void)
{
  return ssh_adt_create_generic(SSH_ADT_LIST,
                                SSH_ADT_COMPARE, int_cmp,
                                SSH_ADT_DESTROY, liballoc_destructor,
                                SSH_ADT_SIZE, sizeof(int),
                                SSH_ADT_ARGS_END);
}

SshADTContainer create_list_liballoc_with_header(void)
{
  return ssh_adt_create_generic(SSH_ADT_LIST,
                                SSH_ADT_COMPARE, int_cmp,
                                SSH_ADT_DESTROY, liballoc_destructor,
                                SSH_ADT_SIZE, sizeof(IntRecord),
                                SSH_ADT_HEADER,
                                SSH_ADT_OFFSET_OF(IntRecord, header),
                                SSH_ADT_ARGS_END);
}

SshADTContainer create_map_voidptr(void)
{
  return ssh_adt_create_generic(SSH_ADT_MAP,
                                SSH_ADT_COMPARE, int_cmp,
                                SSH_ADT_DUPLICATE, int_dupl,
                                SSH_ADT_DESTROY, myalloc_destructor,
                                SSH_ADT_HASH, int_hash,
                                SSH_ADT_ARGS_END);
}

SshADTContainer create_map_voidptr_with_header(void)
{
  return ssh_adt_create_generic(SSH_ADT_MAP,
                                SSH_ADT_COMPARE, int_cmp,
                                SSH_ADT_DUPLICATE, int_dupl_with_header,
                                SSH_ADT_DESTROY, myalloc_destructor,
                                SSH_ADT_HEADER,
                                SSH_ADT_OFFSET_OF(IntRecord, header),
                                SSH_ADT_HASH, int_hash,
                                SSH_ADT_ARGS_END);
}

SshADTContainer create_map_liballoc(void)
{
  return ssh_adt_create_generic(SSH_ADT_MAP,
                                SSH_ADT_COMPARE, int_cmp,
                                SSH_ADT_DESTROY, liballoc_destructor,
                                SSH_ADT_SIZE, sizeof(int),
                                SSH_ADT_HASH, int_hash,
                                SSH_ADT_ARGS_END);
}

static void add_voidptr(SshADTContainer c, int i)
{
  int *ptr = ssh_xmalloc(sizeof(i));
  memcpy(ptr, &i, sizeof(i));
  ssh_adt_insert(c, ptr);
}

static void add_voidptr_with_header(SshADTContainer c, int i)
{
  IntRecord *ptr = ssh_xmalloc(sizeof(IntRecord));
  ptr->i = i;
  ssh_adt_insert(c, ptr);
}

static void add_liballoc(SshADTContainer c, int i)
{
  ssh_adt_put(c, &i);
}


/***************************************************** CHECK: list insertion */

static Boolean list_check_nullobj()
{
  SshADTContainer c;
  SshADTHandle h;
  int i;
  void *o;

  c = create_list_voidptr();
  SSH_DEBUG(4, ("list created."));

  for (i = 0; i < NUM_ITEMS; i++)
    ssh_adt_insert(c, (random() % 4) ? ((void *)i) : NULL);

  if ((h = ssh_adt_enumerate_start(c)) != SSH_ADT_INVALID)
    {
#define LAMBDA                        \
do                                    \
  {                                   \
    o = ssh_adt_get(c, h);            \
    if (o == NULL)                    \
      SSH_DEBUG(3, ("NULL."));        \
    else                              \
      SSH_DEBUG(3, ("%i.", (int)o));  \
  }                                   \
 while(0)

      LAMBDA;

      while ((h = ssh_adt_enumerate_next(c, h)) != SSH_ADT_INVALID)
        LAMBDA;

#undef LAMBDA
    }

  /* ?! does not work: ssh_adt_list_sort(c); */

  return TRUE;
}


/***************************************************** CHECK: list insertion */

static Boolean list_check_insert_to(SshADTContainer (* create)(void),
                                    void (* adder)(SshADTContainer, int),
                                    void (* inserter)(SshADTContainer, 
                                                      SshADTAbsoluteLocation,
                                                      int))
{
  SshADTContainer c;
  int i;

  c = (*(create))();
  SSH_DEBUG(4, ("list created."));

  for (i = 0; i < NUM_ITEMS; i++) (*(adder))(c, i);

  int_container_print(c, "list", 4);

#define GET(c, l)                                                           \
  (*((int *)                                                                \
     ssh_adt_get((c),                                                       \
                 ssh_adt_get_handle_to_location((c), SSH_ADT_INDEX(l)))))

  SSH_DEBUG(80, ("%i/%i/%i.", GET(c, 3), GET(c, 5), GET(c, 13)));
#undef GET

  for (i = 0; i < NUM_ITEMS * 2; i += 2) (*(inserter))(c, SSH_ADT_INDEX(i), 0);
  int_container_print(c, "list", 4);

  ssh_adt_list_sort(c);
  int_container_print(c, "list", 4);

  return TRUE;
}


/*************************************************** CHECK: list duplication */

static Boolean list_check_dupl_pass(SshADTContainer list1, SshADTContainer list2)
{
  /* reads two lists and checks their contents against what is expected */
  SshADTHandle h;

#define GET_H(c, h, i)  if (!(h = ssh_adt_get_handle_to_location(c, i))) return FALSE
#define CHECK_O(c, h, o)  if (*((int *)ssh_adt_get(c, h)) != o)  return FALSE

  GET_H(list1, h, 0); CHECK_O(list1, h, 0);
  GET_H(list1, h, 1); CHECK_O(list1, h, 1);
  GET_H(list1, h, 2); CHECK_O(list1, h, 2);
  GET_H(list1, h, 3); CHECK_O(list1, h, 3);
  GET_H(list1, h, 4); CHECK_O(list1, h, 4);
 
  GET_H(list2, h, 0); CHECK_O(list2, h, 0);
  GET_H(list2, h, 1); CHECK_O(list2, h, 1);
  GET_H(list2, h, 2); CHECK_O(list2, h, 2);
  GET_H(list2, h, 3); CHECK_O(list2, h, 3);
  GET_H(list2, h, 4); CHECK_O(list2, h, 199);
 
#undef GET_H
#undef CHECK_O

  return TRUE;
}

Boolean list_check_dupl_1(SshADTContainer (* create)(void),
                          void (* adder)(SshADTContainer, int))
{
  SshADTContainer list1,  list2;
  int i;

  list1 = (*(create))();
  SSH_ASSERT(list1);

  for (i = 1; i < NUM_ITEMS; i++) (*(adder))(list1, i);
  SSH_DEBUG(4, ("list1 initialized."));
  int_container_print(list1, "list1", 4);

  list2 = ssh_adt_duplicate_container(list1);
  SSH_ASSERT(list2);

  SSH_DEBUG(4, ("list1 copied into newly created list2."));
  int_container_print(list1, "list1", 4);
  int_container_print(list2, "list2", 4);

  SSH_DEBUG(4, ("freeing..."));
  ssh_adt_destroy(list1);
  SSH_DEBUG(4, ("list1 destroyed."));
  ssh_adt_destroy(list2);
  SSH_DEBUG(4, ("list2 destroyed."));

  return TRUE;
}

Boolean list_check_dupl_2(SshADTContainer (* create)(void),
                          void (* adder)(SshADTContainer, int))
{
  SshADTContainer list1, list2;
  SshADTHandle h;
  int i, *pi;

  SSH_DEBUG(4, ("."));

  list1 = (*(create))();
  SSH_ASSERT(list1);
  
  for (i = 0; i < 5; i++)
    (*(adder))(list1, i);

  list2 = ssh_adt_duplicate_container(list1);
  SSH_ASSERT(list2);

  h = ssh_adt_get_handle_to_location(list2, SSH_ADT_DEFAULT);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  pi = (int *)ssh_adt_get(list2, h);
  *pi = 199;

  int_container_print(list1, "list1", 4);
  int_container_print(list2, "list2", 4);

  if (!list_check_dupl_pass(list1, list2))
    {
      SSH_DEBUG(1, ("list_check_dupl_pass failed."));
      return FALSE;
    }

  /* punch list2 around a little and hope it makes the overall
     structure break */

  for (i = 0; i < NUM_ITEMS; i++)
    (*(adder))(list2, rand() % NUM_ITEMS);

  int_container_print(list2, "list2", 4);
  ssh_adt_list_sort(list2);
  int_container_print(list2, "list2", 4);

  /* free all memroy */

  ssh_adt_destroy(list1);
  SSH_DEBUG(4, ("list1 destroyed."));
  ssh_adt_destroy(list2);
  SSH_DEBUG(4, ("list2 destroyed."));

  return TRUE;
}


/***************************************************** CHECK: duplicate maps */

static Boolean map_check_dupl(SshADTContainer (* create)(void),
                              void (* adder)(SshADTContainer, int))
{
  SshADTContainer map1,  map2;
  SshADTHandle h;
  int i, k;

  map1 = (*(create))();
  SSH_ASSERT(map1);

  for (i = 1; i < NUM_ITEMS; i++)
    {
      (*(adder))(map1, i);
      k = i;
      h = ssh_adt_get_handle_to_equal(map1, &k);
      if (h == SSH_ADT_INVALID)
        {
          SSH_DEBUG(0, ("maps are bad."));
          SSH_DEBUG(0, ("tried to find k=%i", k));
          int_container_print(map1, "map1", 4);
          return FALSE;
        }
      ssh_adt_map_set_map(map1, h, (void *)i);
    }

  SSH_DEBUG(4, ("map1 initialized."));
  int_container_print(map1, "map1", 4);

  map2 = ssh_adt_duplicate_container(map1);
  SSH_ASSERT(map2);

  SSH_DEBUG(4, ("map2 copied."));
  int_container_print(map2, "map2", 4);

  ssh_adt_destroy(map1);
  SSH_DEBUG(9, ("map1 destroyed."));
  ssh_adt_destroy(map2);
  SSH_DEBUG(9, ("map2 destroyed."));
  return TRUE;
}


/************************************************************* RUN THE TESTS */

static void run_tests(void)
{
  ssh_regression_section("ListTorture");

  /* NULL as valid object pointer */

  T("List with NULL objects (only for void pointers / liballoked header)",
    list_check_nullobj, ());
  /* insertion */

  T("List, insert_to (2 = void pointers / inlined header)",
    list_check_insert_to, (create_list_voidptr_with_header,
                           add_voidptr_with_header,
                           insert_to_voidptr_with_header));
  /* list duplication */

  T("List, duplication (1 = liballoc)",
    list_check_dupl_1, (create_list_liballoc, add_liballoc));

  T("List, duplication (2 = voidptr_hidden_header)",
    list_check_dupl_1, (create_list_voidptr_with_header, add_voidptr_with_header));

  T("List, duplication (3 = voidptr)",
    list_check_dupl_1, (create_list_voidptr, add_voidptr));

  T("List, duplication II (1 = liballoc)",
    list_check_dupl_2, (create_list_liballoc, add_liballoc));

  T("List, duplication II (2 = voidptr_hidden_header)",
    list_check_dupl_2, (create_list_voidptr_with_header, add_voidptr_with_header));

  T("List, duplication II (3 = voidptr)",
    list_check_dupl_2, (create_list_voidptr, add_voidptr));

  /* map duplication */

  T("Map, duplication (1 = liballoc)",
    map_check_dupl, (create_map_liballoc, add_liballoc));

  T("Map, duplication (2 = voidptr_with_header)",
    map_check_dupl, (create_map_voidptr_with_header, add_voidptr_with_header));

  T("Map, duplication (3 = voidptr)",
    map_check_dupl, (create_map_voidptr, add_voidptr));
}

static void scratch_pad(void)
{
  SSH_DEBUG(0, ("[entering]"));

  do
    {
      SshADTContainer c;
      SshADTHandle handle;
      int i;

      c = ssh_adt_create_strmap();

      i = 3;
      ssh_adt_strmap_add(c, "foo", &i);
      SSH_DEBUG(0, ("foo -> %d.", *((int *)ssh_adt_strmap_get(c, "foo"))));

      i = 18;
      ssh_adt_strmap_set(c, "foo", &i);
      SSH_DEBUG(0, ("foo -> %d.", *((int *)ssh_adt_strmap_get(c, "foo"))));
      SSH_DEBUG(0, ("size = %i.", ssh_adt_num_objects(c)));

      handle = ssh_adt_get_handle_to_equal(c, "foo");
      SSH_ASSERT(handle != SSH_ADT_INVALID);
      SSH_DEBUG(0, ("key handle \"foo\" found."));

      i = *((int *) ssh_adt_map_map(c, handle));
      SSH_ASSERT(i == 18);
      SSH_DEBUG(0, ("value attached to foo is 18.  good"));

      handle = ssh_adt_enumerate_start(c);
      while (handle != SSH_ADT_INVALID)
        {
          i = *((int *)ssh_adt_map_map(c, handle));
          printf("enum is at %d\n", i);
          handle = ssh_adt_enumerate_next(c, handle);
        }
    }
  while(0);

  SSH_DEBUG(0, ("[exiting]"));
  exit(0);
}

int main(int argc, char **argv)
{
  ssh_regression_init(&argc, &argv, "ADT Library (more tests)",
                      "fis@ssh.fi");

  ssh_debug_set_level_string("*=1");

  /* scratch_pad(); */  /* for making quick checks */

  run_tests();
  ssh_regression_finish();

  SSH_NOTREACHED;
  return 1;
}
