/*

  t-randspeed.c

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (c) 1999 SSH Communications Security Corp
  All rights reserved.

  Created Fri Oct 20 22:07:14 2000.

  */

/* Simple test for testing random number generation speed, and
   producing at the side a large quantity or pseudo-random numbers. */

#include "sshincludes.h"
#include "sshrand.h"
#include "sshtimemeasure.h"


void tstart(SshTimeMeasure tmit, char *fmt, ...)
{
  va_list ap;
  char buffer[1024];

  va_start(ap, fmt);
  vsnprintf(buffer, 1024, fmt, ap);
  va_end(ap);

  printf("Timing start: %s\n", buffer);
  
  ssh_time_measure_reset(tmit);
  ssh_time_measure_start(tmit);
}

void tstop(SshTimeMeasure tmit, char *fmt, ...)
{
  va_list ap;
  char buffer[1024];
  ssh_time_measure_stop(tmit);

  va_start(ap, fmt);
  vsnprintf(buffer, 1024, fmt, ap);
  va_end(ap);

  printf("Timing stop @ %f sec : %s\n",
         (double) ssh_time_measure_get(tmit,
                                       SSH_TIME_GRANULARITY_MILLISECOND) /
         1000.0, 
         buffer);
}


void tstartn(SshTimeMeasure tmit, int total, char *fmt, ...)
{
  va_list ap;
  char buffer[1024];

  if (total > 0)
    {
      ssh_time_measure_reset(tmit);
      ssh_time_measure_start(tmit);
      return;
    }
  
  va_start(ap, fmt);
  vsnprintf(buffer, 1024, fmt, ap);
  va_end(ap);

  printf("Timing start: %s\n", buffer);
  
  ssh_time_measure_reset(tmit);
  ssh_time_measure_start(tmit);
}

int tstopn(SshTimeMeasure tmit, int total, char *fmt, ...)
{
  va_list ap;
  char buffer[1024];
  ssh_time_measure_stop(tmit);

  /* Just check that the operation takes at least some time. */
  if (ssh_time_measure_get(tmit, SSH_TIME_GRANULARITY_MILLISECOND) < 100.0)
    {
      return 1;
    }

  va_start(ap, fmt);
  vsnprintf(buffer, 1024, fmt, ap);
  va_end(ap);

  printf("Timing stop @ %6.3f sec / %u ops = %6.4f sec / op: \n  %s\n",
         (double) ssh_time_measure_get(tmit,
                                       SSH_TIME_GRANULARITY_MILLISECOND) /
         1000.0, total,
         ((double) ssh_time_measure_get(tmit,
                                       SSH_TIME_GRANULARITY_MILLISECOND) /
         1000.0)/(double)total,
         buffer);
  return 0;
}

int main(int ac, char *av[])
{
  size_t i, t, f, prnd;
  struct SshTimeMeasureRec tmit = SSH_TIME_MEASURE_INITIALIZER;

  /* Generating a megawords of randomness. */
  prnd = 1024 * 1024;
  f = 0;

#define OK_CRAND   0x01d80000
#define OK_GNURAND 0x3e17b13d
#define OK_SSHRAND 0xf5ee4fa9
  
  tstart(&tmit, "Initialization of C rand.");
  srand(1001);
  tstop(&tmit, "C rand initialized.");
  
  tstart(&tmit, "Generating %u bytes of prandom data with C rand.",
         prnd);
  
  for (i = 0, t = 0; i < prnd; i++)
    {
      t += rand();
    }

  tstop(&tmit, "C rand stopped with checksum %s (%08x)",
        (t == OK_CRAND) ? "ok" : "failed",
        t);

  if (t != OK_CRAND) f++;
  
  tstart(&tmit, "Initialization of GNU random.");
  srandom(1001);
  tstop(&tmit, "GNU random initialized.");
    
  tstart(&tmit, "Generating %u bytes of prandom data with GNU random.",
         prnd);
  
  for (i = 0, t = 0; i < prnd; i++)
    {
      t += random();
    }
  
  tstop(&tmit, "GNU random stopped with checksum %s (%08x).",
        (t == OK_GNURAND) ? "ok" : "failed",
        t);

  if (t != OK_GNURAND) f++;
  
  tstart(&tmit, "Initialization of SSH rand.");
  ssh_rand_seed(1001);
  tstop(&tmit, "SSH random initialized.");
    
  tstart(&tmit, "Generating %u bytes of prandom data with SSH rand.",
         prnd);
  
  for (i = 0, t = 0; i < prnd; i++)
    {
      t += ssh_rand();
    }

  tstop(&tmit, "SSH rand stopped with checksum %s (%08x).",
        (t == OK_SSHRAND) ? "ok" : "failed",
        t);

  if (t != OK_SSHRAND) f++;

  printf("%u failures.\n", f);
  
  return f;
}
