/*

  authc-pam.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000, 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  PAM (Pluggable Authentication Modules) authentication, client side.

*/
/*
  Todo:

  - Handle SSH_PAM_BINARY_PROMPT and SSH_PAM_BINARY_MSG
    styles. (ie. implement agent support).

 */
#include "sshincludes.h"
#include "ssholdfsm.h"
#include "sshclient.h"
#include "sshencode.h"
#include "sshmsgs.h"
#include "readpass.h"
#include "sshappcommon.h"





#define SSH_DEBUG_MODULE "Ssh2AuthPAMClient"

#ifdef CLIENT_WITH_PAM

typedef struct SshClientPAMAuthRec
{
  SshOldFSM fsm;
  SshOldFSMThread main_thread;
  SshClient client;
  SshBuffer server_packet;
  SshBuffer response_packet;

  int num_msgs;
  int cur_msg;
  char *user;

  char *response;
  size_t response_len;
  /* Placeholder for future PAM extensions; currently always 0. */
  SshUInt32 response_retcode;

  void **state_placeholder;
  SshAuthClientCompletionProc completion_callback;
  void *completion_context;
} *SshClientPAMAuth, SshClientPAMAuthStruct;

SSH_OLDFSM_STEP(ssh_pam_suspend)
{
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_process_packet)
{
  SshUInt32 num_msgs;

  SSH_OLDFSM_GDATA(SshClientPAMAuth);
  SSH_PRECOND(gdata);

  if (!ssh_decode_buffer(gdata->server_packet,
                         SSH_FORMAT_UINT32, &num_msgs,
                         SSH_FORMAT_END))
    {
      SSH_OLDFSM_SET_NEXT("ssh_pam_error");
      return SSH_OLDFSM_CONTINUE;
    }

  gdata->num_msgs = (int) num_msgs;
  SSH_TRACE(4, ("Received %d messages.", num_msgs));

  ssh_encode_buffer(gdata->response_packet,
                    SSH_FORMAT_BOOLEAN, TRUE,
                    SSH_FORMAT_UINT32, gdata->num_msgs,
                    SSH_FORMAT_END);

  SSH_OLDFSM_SET_NEXT("ssh_pam_process_next_msg");

  return SSH_OLDFSM_CONTINUE;
}

void client_pam_rl_cb(const char *line, void *context)
{
  SshClientPAMAuth gdata;
  SshOldFSMThread thread = (SshOldFSMThread)context;
  SSH_PRECOND(thread);
  gdata = (SshClientPAMAuth) ssh_oldfsm_get_gdata(thread);

  if (line)
    {
      gdata->response = ssh_xstrdup(line);
      gdata->response_len = strlen(line);
    }
  else
    {
      gdata->response = NULL;
      gdata->response_len = 0;
    }

  SSH_OLDFSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_OLDFSM_STEP(ssh_pam_process_next_msg)
{
  SshUInt32 msg_style;
  char *msg;
  size_t msg_len;





  SSH_OLDFSM_GDATA(SshClientPAMAuth);
  SSH_PRECOND(gdata);

  if (gdata->cur_msg >= gdata->num_msgs)
    {
      /* All messages processed. Send response packet to
         userauth-layer. */
      SSH_OLDFSM_SET_NEXT("ssh_pam_send_response_packet");
      return SSH_OLDFSM_CONTINUE;
    }

  gdata->cur_msg++;

  if (!ssh_decode_buffer(gdata->server_packet,
                         SSH_FORMAT_CHAR, &msg_style,
                         SSH_FORMAT_UINT32_STR, &msg, &msg_len,
                         SSH_FORMAT_END))
    {
      SSH_DEBUG_HEXDUMP(6, ("Rest of received conversation packet:"),
                        ssh_buffer_ptr(gdata->server_packet),
                        ssh_buffer_len(gdata->server_packet));
      goto error_decoding;
    }

  SSH_OLDFSM_SET_NEXT("ssh_pam_append_response");

  switch (msg_style)
    {
    case SSH_PAM_PROMPT_ECHO_OFF:

      if (gdata->client->config->batch_mode)
        {
          SSH_TRACE(2, ("In Batchmode, so we're not asking the "
                        "user anything. (prompt: %s)", msg));
          break;
        }
      /* XXX async, please! */
      gdata->response = ssh_read_passphrase(msg, FALSE);






      if (gdata->response)
        gdata->response_len = strlen(gdata->response);
      else
        gdata->response_len = 0;

      return SSH_OLDFSM_CONTINUE;
    case SSH_PAM_PROMPT_ECHO_ON:

      if (gdata->client->config->batch_mode)
        {
          SSH_TRACE(2, ("In Batchmode, so we're not asking the "
                        "user anything. (prompt: %s)", msg));
          break;
        }

      SSH_OLDFSM_ASYNC_CALL(ssh_readline_eloop(msg, "", gdata->client->rl,
                                            client_pam_rl_cb,
                                            (void *)thread));







      break;
    case SSH_PAM_ERROR_MSG:
      ssh_informational("pam-error: %s", msg);
      break;
    case SSH_PAM_TEXT_INFO:
      ssh_informational("pam-info: %s", msg);
      break;
      /* XXX binary prompts (messages to pam-agents) are not yet
         handled. */
    case SSH_PAM_BINARY_PROMPT:
    case SSH_PAM_BINARY_MSG:
      SSH_TRACE(2, ("Received binary prompt or message. (msg_style: %d) "
                    "Currently, these are not handled.", msg_style));
      SSH_DEBUG_HEXDUMP(4, ("Hexdump of received packet:"), msg, msg_len);
      break;
    default:
      SSH_TRACE(2, ("Invalid msg_style %d.", msg_style));
      goto error_decoding;
    }

  /* Append empty (NULL) response. */
  gdata->response = NULL;
  gdata->response_len = 0;

  return SSH_OLDFSM_CONTINUE;

 error_decoding:
  /* Send protocol error to userauth. */
  SSH_TRACE(0, ("Invalid packet."));
  SSH_OLDFSM_SET_NEXT("ssh_pam_error");
  return SSH_OLDFSM_CONTINUE;
}

SSH_OLDFSM_STEP(ssh_pam_append_response)
{
  SSH_OLDFSM_GDATA(SshClientPAMAuth);

  SSH_OLDFSM_SET_NEXT("ssh_pam_process_next_msg");

  /* This might be used by PAM in the future. Currently, always 0. */
  gdata->response_retcode = SSH_PAM_DEFAULT_RESP_RETCODE;

  ssh_encode_buffer(gdata->response_packet,
                    SSH_FORMAT_CHAR, gdata->response_retcode,
                    SSH_FORMAT_UINT32_STR,
                    gdata->response_len ? gdata->response : "",
                    gdata->response_len,
                    SSH_FORMAT_END);
  ssh_xfree(gdata->response);
  gdata->response = NULL;
  gdata->response_len = 0;

  return SSH_OLDFSM_CONTINUE;
}

SSH_OLDFSM_STEP(ssh_pam_send_response_packet)
{
  SSH_OLDFSM_GDATA(SshClientPAMAuth);

  SSH_OLDFSM_SET_NEXT("ssh_pam_finish");

  SSH_TRACE(2, ("Sending response packet."));

  /* Cleared here, because if we get back to this authentication
     method before ssh_pam_finish state, it will lead to an
     SSH_ASSERT. */
  *gdata->state_placeholder = NULL;

  (*gdata->completion_callback)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE,
                                gdata->user, gdata->response_packet,
                                gdata->completion_context);
  ssh_buffer_clear(gdata->response_packet);

  return SSH_OLDFSM_CONTINUE;
}

SSH_OLDFSM_STEP(ssh_pam_error)
{
  SSH_OLDFSM_GDATA(SshClientPAMAuth);

  (*gdata->completion_callback)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD,
                                gdata->user, NULL, gdata->completion_context);

  SSH_OLDFSM_SET_NEXT("ssh_pam_finish");

  return SSH_OLDFSM_CONTINUE;
}

SSH_OLDFSM_STEP(ssh_pam_finish)
{
  SSH_OLDFSM_GDATA(SshClientPAMAuth);
  ssh_oldfsm_destroy(gdata->fsm);
  gdata->fsm = NULL;

  *gdata->state_placeholder = NULL;
  return SSH_OLDFSM_FINISH;
}

SshOldFSMStateMapItemStruct client_pam_states[] =
{
  { "ssh_pam_suspend", "Suspend thread", ssh_pam_suspend},

  { "ssh_pam_process_packet", "Process received packet.",
    ssh_pam_process_packet},
  { "ssh_pam_process_next_msg", "Process next message in packet",
    ssh_pam_process_next_msg },
  { "ssh_pam_append_response", "Append a response to return packet",
    ssh_pam_append_response },
  { "ssh_pam_send_response_packet", "Send response packet to userauth-layer",
    ssh_pam_send_response_packet },

  { "ssh_pam_error", "Send disable method packet to userauth",
    ssh_pam_error },
  { "ssh_pam_finish", "Destroy FSM.", ssh_pam_finish }
};

void client_pam_destructor(void *gdata)
{
  /* XXX Something else? */
  SshClientPAMAuth state = (SshClientPAMAuth) gdata;

  SSH_PRECOND(gdata);

  ssh_buffer_free(state->server_packet);
  ssh_buffer_free(state->response_packet);

  ssh_xfree(state->user);

  ssh_xfree(state->response);

  *state->state_placeholder = NULL;
}

void ssh_client_auth_pam(SshAuthClientOperation op,
                         const char *user,
                         unsigned int packet_type,
                         SshBuffer packet_in,
                         const unsigned char *session_id,
                         size_t session_id_len,
                         void **state_placeholder,
                         SshAuthClientCompletionProc completion,
                         void *completion_context,
                         void *method_context)
{
  SshBuffer buffer = NULL;
  SshClient client = (SshClient) method_context;
  SshClientPAMAuth state;
  SshOldFSM fsm;

  SSH_PRECOND(client);

  state = (SshClientPAMAuth)*state_placeholder;

  switch (op)
    {
    case SSH_AUTH_CLIENT_OP_START:
      buffer = ssh_xbuffer_allocate();
      ssh_encode_buffer(buffer,
                        SSH_FORMAT_BOOLEAN, FALSE,
                        SSH_FORMAT_END);
      (*completion)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE,
                    user, buffer, completion_context);
      ssh_buffer_free(buffer);
      break;

    case SSH_AUTH_CLIENT_OP_START_NONINTERACTIVE:
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;

    case SSH_AUTH_CLIENT_OP_CONTINUE:
      if (packet_type != SSH_MSG_USERAUTH_PAM_MSG)
        {
          SSH_TRACE(0, ("unknown packet type %d for continuation packet",
                        (int)op));
          (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
          break;
        }
      SSH_ASSERT(state == NULL);

      /* Allocate state. */
      fsm = ssh_oldfsm_allocate(sizeof(*state),
                             client_pam_states,
                             SSH_OLDFSM_NUM_STATES(client_pam_states),
                             client_pam_destructor);
      state = ssh_oldfsm_get_gdata_fsm(fsm);
      memset(state, 0, sizeof(*state));

      state->fsm = fsm;
      state->completion_callback = completion;
      state->completion_context = completion_context;
      state->server_packet = ssh_xbuffer_allocate();
      state->response_packet = ssh_xbuffer_allocate();
      state->user = ssh_xstrdup(user);
      state->state_placeholder = state_placeholder;
      state->client = client;

      /* Copy packet_in (it will be freed after this function
         returns). */
      ssh_xbuffer_append(state->server_packet, ssh_buffer_ptr(packet_in),
                        ssh_buffer_len(packet_in));

      /* Start thread. */
      state->main_thread = ssh_oldfsm_spawn(fsm, 0,
                                         "ssh_pam_suspend",
                                         NULL, NULL);

      /* wake up thread to process packet. */
      ssh_oldfsm_set_next(state->main_thread, "ssh_pam_process_packet");
      ssh_oldfsm_continue(state->main_thread);

      /* Rest is done in callbacks. */
      break;

    case SSH_AUTH_CLIENT_OP_ABORT:
      /* Clean up state. */
      if (state)
        ssh_oldfsm_set_next(state->main_thread, "ssh_fsm_finish");

      *state_placeholder = NULL;
      break;

    default:
      SSH_TRACE(0, ("unknown op %d", (int)op));
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
    }
}

#endif /* CLIENT_WITH_PAM */
