/*

  auths-hostbased.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Hostbased authentication, server-side.

*/

#include "ssh2includes.h"
#include "sshauth.h"
#include "sshuser.h"
#include "sshkeyfile.h"
#include "sshserver.h"
#include "sshencode.h"
#include "sshmsgs.h"
#include "sshcipherlist.h"
#include "sshuserfiles.h"
#include "sshuserfile.h"
#include "auths-common.h"
#include "auths-hostbased.h"
#include "sshdsprintf.h"
#include "ssh2compat.h"
#include "ssh2pubkeyencode.h"





#define SSH_DEBUG_MODULE "Ssh2AuthHostBasedServer"

typedef struct HostbasedAuthsCtxRec
{
  char *client_pubkey_alg;
  unsigned char *certs;
  size_t certs_len;
  char *client_hostname;
  char *client_hostname_fqdn;
  char *client_user_name;
  unsigned char *sig;
  size_t sig_len;
  SshServer server;
  SshUser uc;
  const unsigned char *session_id;
  size_t session_id_len;
  SshAuthServerCompletionProc completion_proc;
  void *completion_context;
  SshBuffer packet;
} HostbasedAuthsCtxStruct, *HostbasedAuthsCtx;

void hostbased_destroy_ctx(HostbasedAuthsCtx ctx)
{
  if (ctx == NULL)
    return;

  ssh_xfree(ctx->client_pubkey_alg);
  ssh_xfree(ctx->certs);
  ssh_xfree(ctx->client_hostname);
  ssh_xfree(ctx->client_hostname_fqdn);
  ssh_xfree(ctx->client_user_name);
  ssh_xfree(ctx->sig);
}

/* Forward declaration. */
void hostbased_verify_sig(SshPublicKey pubkey, HostbasedAuthsCtx ctx);






































































































void hostbased_verify_sig(SshPublicKey pubkey, HostbasedAuthsCtx ctx)
{
  SshBuffer buf = NULL;
  char *service;
  Boolean sig_ok = FALSE;

  /* Verify the signature.*/
  /* Contruct a throw-away SSH_MSG_USERAUTH_REQUEST packet.*/

  buf = ssh_xbuffer_allocate();

  if (




      !(*ctx->server->common->compat_flags->
        hostbased_service_name_draft_incompatibility)

      )
    service = SSH_CONNECTION_SERVICE;
  else
    service = SSH_USERAUTH_SERVICE;

  ssh_encode_buffer(buf,
                    SSH_FORMAT_UINT32_STR, ctx->session_id,
                    ctx->session_id_len,
                    /* byte SSH_MSG_USERAUTH_REQUEST */
                    SSH_FORMAT_CHAR,
                    (unsigned int) SSH_MSG_USERAUTH_REQUEST,
                    /* string User name*/
                    SSH_FORMAT_UINT32_STR, ssh_user_name(ctx->uc),
                    strlen(ssh_user_name(ctx->uc)),
                    /* string service_name */
                    SSH_FORMAT_UINT32_STR, service, strlen(service),
                    /* string "hostbased" */
                    SSH_FORMAT_UINT32_STR, SSH_AUTH_HOSTBASED,
                    strlen(SSH_AUTH_HOSTBASED),
                    /* string public key algorithm */
                    SSH_FORMAT_UINT32_STR, ctx->client_pubkey_alg,
                    strlen(ctx->client_pubkey_alg),
                    /* string client's public host key and len. */
                    SSH_FORMAT_UINT32_STR, ctx->certs, ctx->certs_len,
                    /* string client hostname */
                    SSH_FORMAT_UINT32_STR, ctx->client_hostname_fqdn,
                    strlen(ctx->client_hostname_fqdn),
                    /* string user's user name at client host */
                    SSH_FORMAT_UINT32_STR, ctx->client_user_name,
                    strlen(ctx->client_user_name),
                    SSH_FORMAT_END);

  SSH_DEBUG_HEXDUMP(99, ("Verifying following data"),
                    ssh_buffer_ptr(buf), ssh_buffer_len(buf));
  SSH_DEBUG_HEXDUMP(99, ("Signature"), ctx->sig, ctx->sig_len);

  {
    size_t decoded_len, real_sig_len;
    char *recv_pubkeytype, *pubkeytype;
    unsigned char *real_sig;

    if (




        !*(ctx->server->common->compat_flags->
           malformed_signatures_draft_incompatibility)

        )
      {



          {
            pubkeytype =
              ssh_pubkeyblob_type(ctx->certs, ctx->certs_len);

            if (!pubkeytype)
              {
                ssh_warning("Received malformed pubkey during hostbased "
                            "authentication.");
                goto error;
              }
          }







        decoded_len =
          ssh_decode_array(ctx->sig, ctx->sig_len,
                           SSH_FORMAT_UINT32_STR,
                           &recv_pubkeytype, NULL,
                           SSH_FORMAT_UINT32_STR,
                           &real_sig, &real_sig_len,
                           SSH_FORMAT_END);

        if (decoded_len == 0 || decoded_len != ctx->sig_len)
          {
            ssh_xfree(pubkeytype);
            SSH_DEBUG(2, ("decoded_len: %ld, sig_len: %ld",
                          decoded_len, ctx->sig_len));
            ssh_warning("Received malformed signature during hostbased "
                        "authentication.");
            goto error;
          }

        if (strcmp(recv_pubkeytype, pubkeytype) != 0)
          {
            ssh_warning("Received malformed signature during hostbased "
                        "authentication. (public key type doesn't "
                        "match the one in signature.)");
            ssh_xfree(recv_pubkeytype);
            ssh_xfree(pubkeytype);
            memset(real_sig, 'F', real_sig_len);
            ssh_xfree(real_sig);
            goto error;
          }
        ssh_xfree(recv_pubkeytype);
        ssh_xfree(pubkeytype);
      }
    else
      {
        real_sig = ctx->sig;
        real_sig_len = ctx->sig_len;
        ctx->sig = NULL;
      }

    if (!strncmp(ctx->client_pubkey_alg, SSH_SSH_RSA, strlen(SSH_SSH_RSA)) &&
        !ssh_compat_rsa_public_key_change_scheme
        (pubkey,




         !*(ctx->server->common->compat_flags->
                    rsa_hash_scheme_draft_incompatibility)

         ))
      {
        ssh_public_key_free(pubkey);
        ssh_buffer_free(buf);
        ssh_log_event(ctx->server->config->log_facility,
                      SSH_LOG_WARNING,
                      "Hostbased: Crypto operation failed for public "
                      "key (for user %s).",
                      ssh_user_name(ctx->uc));
        goto error;
      }

    /* Do the actual verifying. */
    sig_ok = ssh_public_key_verify_signature(pubkey,
                                             real_sig, real_sig_len,
                                             ssh_buffer_ptr(buf),
                                             ssh_buffer_len(buf));

    if (




        !*(ctx->server->common->compat_flags->
           malformed_signatures_draft_incompatibility)

        )
      {
        memset(real_sig, 'F', real_sig_len);
        ssh_xfree(real_sig);
      }
  }

  ssh_public_key_free(pubkey);
  ssh_buffer_free(buf);

  if (!sig_ok)
    {
      ssh_log_event(ctx->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "Hostbased authentication failed for %s.",
                    ssh_user_name(ctx->uc));
      goto error;
    }

  ssh_log_event(ctx->server->config->log_facility,
                SSH_LOG_NOTICE,
                "Hostbased authentication for user %s accepted.",
                ssh_user_name(ctx->uc));

  ctx->server->authenticated_client_user_name = ctx->client_user_name;
  ctx->client_user_name = NULL;
  /* XXX should this be the name client gave us, or the name we
     got from DNS? */
  ctx->server->authenticated_client_host = ssh_xstrdup(ctx->client_hostname);
  /* XXX Free memory, pliis. */
  ssh_userfile_uninit();
  (*ctx->completion_proc)(SSH_AUTH_SERVER_ACCEPTED, ctx->packet,
                          ctx->completion_context);
  hostbased_destroy_ctx(ctx);
  return;
 error:
  /* An error occurred, either an internal, or an protocol
     error. Either way, authentication isn't successful. */
  ssh_userfile_uninit();
  if (



      !(*ctx->server->common->compat_flags->hostbased_loop_bug)

      )
    (*ctx->completion_proc)(SSH_AUTH_SERVER_REJECTED,
                            ctx->packet, ctx->completion_context);
  else
    (*ctx->completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                            ctx->packet, ctx->completion_context);
  hostbased_destroy_ctx(ctx);
}

void ssh_server_auth_hostbased(SshAuthServerOperation op,
                               const char *user,
                               SshUser uc,
                               SshBuffer packet,
                               const unsigned char *session_id,
                               size_t session_id_len,
                               void **state_placeholder,
                               void **longtime_placeholder,
                               SshAuthServerCompletionProc completion_proc,
                               void *completion_context,
                               void *method_context)
{
  unsigned char *certs, *sig;
  size_t certs_len, sig_len, hostname_len;
  char *client_user_name, *client_hostname, *client_pubkey_alg, *hostname;
  char *client_hostname_copy;
  SshServer server = (SshServer) method_context;
  SshConfig config = server->config;
  SshPublicKey pubkey;
  HostbasedAuthsCtx ctx = NULL;

  SSH_PRECOND(server);
  SSH_PRECOND(server->config);

  SSH_DEBUG(6, ("auth_hostbased op = %d  user = %s", op, user));

  switch(op)
    {
    case SSH_AUTH_SERVER_OP_START:
      /* This is actually the only thing we do in this authentication
         method. */
      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_HOSTBASED))
        goto error_no_uninit;

      if(ssh_user_uid(uc) == UID_ROOT &&
         config->permit_root_login == SSH_ROOTLOGIN_FALSE)
        {
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "Root logins are not permitted.");
          SSH_DEBUG(2, ("ssh_server_auth_passwd: root logins are "
                        "not permitted."));
          goto error_no_uninit;
        }

      if (!ssh_decode_buffer(packet,
                             SSH_FORMAT_UINT32_STR, &client_pubkey_alg, NULL,
                             SSH_FORMAT_UINT32_STR, &certs, &certs_len,
                             SSH_FORMAT_UINT32_STR, &client_hostname, NULL,
                             SSH_FORMAT_UINT32_STR, &client_user_name, NULL,
                             SSH_FORMAT_UINT32_STR, &sig, &sig_len,
                             SSH_FORMAT_END))
        {
          /* Error during decoding. */
          SSH_TRACE(1, ("Error decoding packet."));
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "Error decoding \"hostbased\" packet.");
          goto error_no_uninit;
        }

      ctx = ssh_xcalloc(1, sizeof(*ctx));

      ctx->client_pubkey_alg = client_pubkey_alg;
      ctx->certs = certs;
      ctx->certs_len = certs_len;
      ctx->client_hostname_fqdn = client_hostname;
      ctx->client_user_name = client_user_name;
      ctx->sig = sig;
      ctx->sig_len = sig_len;
      ctx->server = server;
      ctx->uc = uc;
      ctx->session_id = session_id;
      ctx->session_id_len = session_id_len;
      ctx->completion_proc = completion_proc;
      ctx->completion_context = completion_context;
      ctx->packet = packet;

      /* Initialize SshUserFile, if not already so. */
      ssh_userfile_init(ssh_user_name(uc), ssh_user_uid(uc),
                        ssh_user_gid(uc), NULL, NULL);

      /* Check that client host name matches with the name it gave us.*/
      hostname = ssh_xstrdup(server->common->remote_host);
      hostname_len = strlen(hostname);
      hostname = ssh_xrealloc(hostname, hostname_len  + 2);
      hostname[hostname_len] = '.';
      hostname[hostname_len + 1] = '\0';
      hostname_len++;

      if (strcasecmp(client_hostname, hostname))
        {
          char *s = "Client gave us a hostname ('%s') which "
            "doesn't match the one we got from DNS ('%s')%s";

          if (config->hostbased_force_client_hostname_dns_match)
            {
              ssh_log_event(config->log_facility,
                            SSH_LOG_WARNING,
                            s, client_hostname, hostname,
                            " (sending failure)");
              goto error;
            }
          else
            {
              ssh_log_event(config->log_facility,
                            SSH_LOG_WARNING,
                            s, client_hostname, hostname,
                            " (trusting that client is valid, if "
                            "signature verification succeeds)");
            }
        }

      ssh_xfree(hostname);

      client_hostname_copy = ssh_xstrdup(client_hostname);

      if (client_hostname_copy[strlen(client_hostname_copy) - 1] == '.')
        client_hostname_copy[strlen(client_hostname_copy) - 1] = '\0';

      {
        /* Check that user in client end is authorized to log in as
           the user requested by checking the /etc/{sh,h}osts.equiv and
           $HOME/.[rs]hosts.*/
        Boolean ret = ssh_server_auth_hostbased_rhosts(uc, client_user_name,
                                                       client_hostname_copy,
                                                       server);


        if (!ret)
          {
            /* Didn't succeed, so this is an error. */
            /* We don't log this here, as failure is already logged in
               ssh_server_auth_hostbased_rhosts(). */
            ssh_xfree(client_hostname_copy);
            goto error;
          }
      }

      ctx->client_hostname = client_hostname_copy;

      if (ssh_buffer_len(packet))
        {
          /* There shouldn't be anything else in the packet, so this
             packet was invalid. */
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "Invalid packet. (extra data at end)");
          goto error;
        }

      /* Check that client pubkey matches previously saved. */












        {
          char *comment;
          unsigned char *key_blob;
          size_t key_blob_len;
          char *candidate;
          Boolean retry = FALSE;
          Boolean no_userdir = FALSE;
          
          /* Check that client pubkey algorithms are acceptable. */
          if (!ssh_public_key_name_ssh_to_cryptolib(client_pubkey_alg))
            {

              ssh_log_event(config->log_facility,
                            SSH_LOG_WARNING,
                            "Client's public key algorithms are not "
                            "supported by us. (client sent '%s')",
                            client_pubkey_alg);
              goto error;
            }

          /* Construct candidate filename. */

          /* Try first from user's directory. */

        retry_with_global_dir:

          if (!no_userdir && config->user_known_hosts && !retry)
            {
              char *user_ssh2_dir;

              user_ssh2_dir = ssh_userdir(uc, config, FALSE);

              if (user_ssh2_dir == NULL)
                {
                  ssh_log_event(config->log_facility,
                                SSH_LOG_WARNING,
                                "Couldn't find or create user %s's .ssh2 "
                                "directory.", ssh_user_name(uc));

                  /* Try again with global hostkey-dir. */
                  no_userdir = TRUE;
                  goto retry_with_global_dir;
                }

              ssh_dsprintf(&candidate, "%s%s%c%s%s%s",
                           user_ssh2_dir, SSH_KNOWNHOSTS_DIR, '/',
                           client_hostname, client_pubkey_alg, ".pub");
              ssh_xfree(user_ssh2_dir);
            }
          else
            {
              static Boolean tried = FALSE;

              if (tried)
                {
                  SSH_TRACE(1, ("Client host's pubkey not found."));

                  goto error;
                }

              tried = TRUE;

              ssh_dsprintf(&candidate, "%s%c%s%s%s",
                           SSH_GLOBAL_KNOWNHOSTS_DIR, '/',
                           client_hostname, client_pubkey_alg, ".pub");
            }

          /* Try to load pubkey-candidate into memory */

          SSH_DEBUG(2, ("Trying to read client host's pubkey from '%s'...",
                        candidate));

          if (ssh2_key_blob_read(uc, candidate, TRUE, &comment, &key_blob,
                                 &key_blob_len, NULL)
              == SSH_KEY_MAGIC_FAIL)
            {
              SSH_TRACE(1, ("Error occurred while reading in '%s' "
                            "(perhaps it doesn't exist?)", candidate));

              /* If there isn't a matching name, try global directory. */
              retry = TRUE;
              ssh_xfree(candidate);
              goto retry_with_global_dir;
            }
          /* Compare. */

          if (certs_len == key_blob_len)
            {
              if (memcmp(key_blob, certs, key_blob_len))
                {
                  ssh_log_event(config->log_facility,
                                SSH_LOG_WARNING,
                                "The public stored in %s and the given "
                                "by client were different.", candidate);
                  goto error;
                }
            }
          else
            {
              ssh_log_event(config->log_facility,
                            SSH_LOG_WARNING,
                            "The public key stored in %s and the given "
                            "by client were different.", candidate);
              goto error;
            }

          /* The blobs match*/
          SSH_TRACE(2, ("Found matching public key in file '%s'.",
                        candidate));

          ssh_xfree(candidate);

          /* If it is ok, continue. If not, it's an error. */

          if ((pubkey = ssh_decode_pubkeyblob(certs, certs_len)) == NULL)
            {
              ssh_log_event(config->log_facility,
                            SSH_LOG_WARNING,
                            "Importing client pubkey failed.");
              goto error;
            }

          hostbased_verify_sig(pubkey, ctx);
          break;
        }
    error:
      if (ctx)
        hostbased_destroy_ctx(ctx);

      ssh_userfile_uninit();
    error_no_uninit:
      if (



          !(*server->common->compat_flags->hostbased_loop_bug)

          )
        (*completion_proc)(SSH_AUTH_SERVER_REJECTED,
                           packet, completion_context);
      else
        (*completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                           packet, completion_context);

      break;
    case SSH_AUTH_SERVER_OP_ABORT:
      /* This shouldn't happen as we haven't sent
         SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK, so this is an
         error. */
      SSH_DEBUG(1, ("We received op SSH_AUTH_SERVER_OP_ABORT. This is "
                    "an error, as we haven't sent "
                    "SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK ."));

      /* XXX Should we log this? */
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED, packet,
                         completion_context);
      break;
    case SSH_AUTH_SERVER_OP_CONTINUE:
      /* This shouldn't happen as we haven't sent
         SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK, so this is an
         error.*/
      SSH_DEBUG(1, ("We received op SSH_AUTH_SERVER_OP_CONTINUE. This is "
                    "an error, as we haven't sent "
                    "SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK ."));

      /* XXX Should we log this? */
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED, packet,
                         completion_context);
      break;
    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      /* We return this just for fun, as this will be ignored. */
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet, completion_context);
      break;
    }
}
