/*

auths-kerberos.c

Author: Tatu Ylonen <ylo@ssh.com>

Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
All rights reserved.

Kerberos authentication, server side.

Special acknowledgements to Dug Song <dogsong@umic.edu> and Glenn
Machin (Sandia Natl Labs) for writing the original Kerberos support
code for SSH1.

*/

#include "sshincludes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshserver.h"
#include "sshconfig.h"
#include "auths-common.h"

#ifdef KERBEROS

#undef ctime
#undef free

/* The #%@!$(#$ krb5.h header redefines these.  ARRRGH! -ylo */
#undef SIZEOF_INT
#undef SIZEOF_LONG
#undef SIZEOF_SHORT
#undef HAVE_STDARG_H
#undef HAVE_SYS_TYPES_H
#include <krb5.h>

#define SSH_DEBUG_MODULE "Ssh2AuthKerberosServer"

/* Kerberos authentication. */

void ssh_server_auth_kerberos(SshAuthServerOperation op,
                              const char *user,
                              SshUser uc,
                              SshBuffer packet,
                              const unsigned char *session_id,
                              size_t session_id_len,
                              void **state_placeholder,
                              void **longtime_placeholder,
                              SshAuthServerCompletionProc completion_proc,
                              void *completion_context,
                              void *method_context)
{
  SshServer server = (SshServer)method_context;
  SshConfig config = server->config;
  unsigned char *ticket = NULL;
  size_t ticket_len;
  krb5_error_code kerr;
  krb5_context kcontext;
  krb5_auth_context kauth;
  krb5_data auth_data;
  krb5_ticket *kticket;
  char *server_name, *tkt_client_user;
  int ret = SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED;
  
  SSH_TRACE(6, ("kerberos auth."));
  
  SSH_DEBUG(6, ("op = %d  user = %s", op, user));

  /* Initialize kerberos data that we use in cleanup. */
  memset(&kcontext, 0, sizeof(kcontext));
  memset(&kauth, 0, sizeof(kauth));
  memset(&auth_data, 0, sizeof(auth_data));
  kticket = NULL;
  server_name = NULL;
  tkt_client_user = NULL;
  
  switch (op)
    {
    case SSH_AUTH_SERVER_OP_START:
      /* Check that the user is allowed to log in and that this authentication
         method is allowed. */
      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_KERBEROS))
        goto fail;
      
      /* Parse the password authentication request. */
      if (ssh_decode_buffer(packet,
                            SSH_FORMAT_UINT32_STR, &ticket, &ticket_len,
                            SSH_FORMAT_END) == 0)
        {
          SSH_DEBUG(2, ("bad packet"));
          goto fail;
        }
      
      /* Sanity check: do not pass excessively long tickets, just in case. */
      if (ticket_len > 2048)
        {
          SSH_DEBUG(2, ("ticket too long"));
          ssh_xfree(ticket);
          goto fail;
        }

      /* Initialize the Kerberos application context. */
      kerr = krb5_init_context(&kcontext);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_init_context %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Prepare the authentication data from the ticket. */
      auth_data.data = ticket;
      auth_data.length = ticket_len;

      /* Initialize the authentication context. */
      kerr = krb5_auth_con_init(kcontext, &kauth);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_auth_con_init %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Extract remote and local addresses from the socket used for
         communication. */
      kerr = krb5_auth_con_genaddrs(kcontext, kauth, config->ssh1_fd,
                                KRB5_AUTH_CONTEXT_GENERATE_REMOTE_FULL_ADDR);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_auth_con_genaddrs %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Parse the client's authentication request. */
      kerr = krb5_rd_req(kcontext, &kauth, &auth_data, NULL, NULL, NULL,
                         &kticket);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_rd_req %d: %s", kerr, error_message(kerr)));
          goto fail;
        }

      /* Parse the server name from the ticket. */
      kerr = krb5_unparse_name(kcontext, kticket->server, &server_name);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_unparse_name %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Make sure the server principal name beginst with "host/". */
      if (strncmp(server_name, "host/", 5) != 0)
        {
          SSH_DEBUG(2, ("server principal is not host"));
          goto fail;
        }

      /* Dig out the client user's name from the ticket. */
      kerr = krb5_unparse_name(kcontext, kticket->enc_part2->client,
                              &tkt_client_user);
      if (kerr)
        {
          SSH_DEBUG(2, ("could not unparse client user's name from ticket"));
          goto fail;
        }
      
      /* Check if the client user is allowed to log into the server as the
         specified user. */
      if (!krb5_kuserok(kcontext, kticket->enc_part2->client,
                        ssh_user_name(uc)))
        {
          ssh_log_event(config->log_facility, SSH_LOG_NOTICE,
                        "Kerberos authentication as %.100s denied for %.100s",
                        ssh_user_name(uc), tkt_client_user);
          goto fail;
        }
      
      /* Log the successful login. */
      ssh_log_event(config->log_facility, SSH_LOG_NOTICE,
                    "Kerberos authentication as user %.100s accepted for %.100s.",
                    ssh_user_name(uc), tkt_client_user);

      /* Authentication successful. */
      ret = SSH_AUTH_SERVER_ACCEPTED;

    fail:
      /* Authentication has failed.  Clear any allocated kerberos data, and
         fail. */
      if (ticket)
        ssh_xfree(ticket);
      if (server_name)
        free(server_name);
      if (tkt_client_user)
        free(tkt_client_user);
      if (kticket)
        krb5_free_ticket(kcontext, kticket);
      if (kauth)
        krb5_auth_con_free(kcontext, kauth);
      if (kcontext)
        krb5_free_context(kcontext);
      /* Fall to next case. */
      (*completion_proc)(ret, packet, completion_context);
      return;

    case SSH_AUTH_SERVER_OP_ABORT:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;
      
    case SSH_AUTH_SERVER_OP_CONTINUE:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;
      
    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;
      
    default:
      ssh_fatal("ssh_server_auth_kerberos: unknown op %d", (int)op);
    }
  
  SSH_NOTREACHED;
}

#endif /* KERBEROS */
