/*

  auths-pam.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000, 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  PAM (Pluggable Authentication Modules) authentication, server side.

*/

/*
  This is how PAM-authentication is implemented to the SSH2 protocol.

  client to server:

  byte     SSH_MSG_USERAUTH_REQUEST
  string   user name
  string   service
  string   "pam-1@ssh.com"
  boolean  FALSE

  server to client:
  one of the following:

  SSH_MSG_USERAUTH_SUCCESS

  or:

  SSH_MSG_USERAUTH_FAILURE

  or

  byte     SSH_MSG_USERAUTH_PAM_MSG
  uint32   number of messages
  byte     message type
  string   message
  ...      more messages (message type - message pairs), up to number of
           messages

  client MUST reply to this message with

  byte     SSH_MSG_USERAUTH_REQUEST
  string   user name
  string   service
  string   "pam-1@ssh.com"
  boolean  TRUE
  uint32   number of responses
  byte     response retcode
  string   response
  ...      more responses (response retcode - response pairs), up to number
           of responses

  Number of responses MUST be the same as the number of messages sent
  by the server, and the responses must be in the same order. If
  response isn't needed for a message, the response MUST be an empty
  string.

  There can be any number of SSH_MSG_USERAUTH_PAM_MSG and reply
  transactions.  This is because the server can't know what the PAM
  modules are asking the client, and the PAM modules can do the
  authentications in multiple phases.
*/
#include "sshincludes.h"
#include "sshpamserver.h"
#include "auth-pam-common.h"
#include "sshpacketstream.h"
#include "sshserver.h"
#include "ssholdfsm.h"
#include "sshencode.h"
#include "auths-common.h"
#include "sshunixpipestream.h"
#include "sshmsgs.h"
#ifdef DAEMON_WITH_PAM
#include <security/pam_appl.h>
#endif /* DAEMON_WITH_PAM */

#define SSH_DEBUG_MODULE "Ssh2AuthPAMServer"


#ifdef DAEMON_WITH_PAM

typedef enum
{
  SENT_NOTHING = 0,
  SENT_PAM_START,
  SENT_PAM_SET_RHOST,
  SENT_PAM_SET_TTY,
  SENT_PAM_SET_RUSER,
  SENT_PAM_AUTHENTICATE,
  SENT_PAM_ACCT_MGMT,
  SENT_PAM_OPEN_SESSION,
  SENT_PAM_SETCRED,
  SENT_PAM_GET_ITEM,
  SENT_PAM_CHAUTHTOK
} PAMAuthSentState;

typedef struct SshServerPAMAuthRec
{
  char *service_name;
  SshServer server;
  SshUser uc;
  PAMAuthSentState sent_state;
  /* Packetstream to ssh-pam-client. */
  SshPacketWrapper wrapper;

  /* Received packet from client. */
  SshBuffer client_packet;

  /* Set to TRUE, if we are waiting a response to a conversation
     message from the client side. */
  Boolean doing_conversation;

  /* Completion callback, used to inform userauth-layer. */
  SshAuthServerCompletionProc completion_proc;
  /* ... and it's context. */
  void *completion_context;

  void **state_placeholder;

  SshOldFSM fsm;
  SshOldFSMThread main_thread;
  SshOldFSMThread aux_thread;
} *SshServerPAMAuth, SshServerPAMAuthStruct;

void received_packet_cb(SshPacketType packet_type,
                        const unsigned char *data, size_t len,
                        void *context)
{
  SshServerPAMAuth gdata = (SshServerPAMAuth) context;
  unsigned int err_num;
  char *err_msg;

  SSH_PRECOND(gdata);

  SSH_DEBUG(4, ("Received packet %d.", packet_type));

  if (packet_type == SSH_PAM_OP_ERROR ||
      packet_type == SSH_PAM_ERROR)
    {
      if (ssh_decode_array(data, len,
                           SSH_FORMAT_CHAR, &err_num,
                           SSH_FORMAT_UINT32_STR, &err_msg, NULL,
                           SSH_FORMAT_END) !=
          len || len == 0)
        {
          /* Protocol error. */
          ssh_log_event(gdata->server->config->log_facility,
                        SSH_LOG_ERROR,
                        "auths-pam: ssh-pam-client sent a malformed "
                        "packet. (packet type: %d)", packet_type);

          *gdata->state_placeholder = NULL;

          (*gdata->completion_proc)
            (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
             gdata->client_packet,
             gdata->completion_context);
          ssh_oldfsm_set_next(gdata->main_thread, "ssh_pam_abort");
          ssh_oldfsm_continue(gdata->main_thread);
          return;
        }
    }

  switch (packet_type)
    {
    case SSH_PAM_OP_SUCCESS:
      /* Do nothing. */
      break;
    case SSH_PAM_OP_SUCCESS_WITH_PAYLOAD:
      /* Check, that we have sent a message, that expects a success
         message with payload. */
      if (gdata->sent_state == SENT_PAM_GET_ITEM)
        {
          /* XXX Store data (currently this isn't used). */
          SSH_NOTREACHED;
        }
      else
        {
          ssh_log_event(gdata->server->config->log_facility,
                        SSH_LOG_ERROR,
                        "auths-pam: ssh-pam-client returned packet "
                        "SSH_PAM_OP_SUCCESS_WITH_PAYLOAD, expecting "
                        "SSH_PAM_OP_SUCCESS.");

          *gdata->state_placeholder = NULL;
          (*gdata->completion_proc)
            (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
             gdata->client_packet,
             gdata->completion_context);
          ssh_oldfsm_set_next(gdata->main_thread, "ssh_pam_abort");
        }
      break;
    case SSH_PAM_OP_ERROR:
      /* Check if operation can fail, and what will we do about
         it. Otherwise, fall through to generic error.*/
      switch (gdata->sent_state)
        {
        case SSH_PAM_ACCT_MGMT:
          if (err_num == PAM_NEW_AUTHTOK_REQD)
            {
              ssh_oldfsm_set_next(gdata->main_thread, "ssh_pam_chauthtok");
              /* XX clean up. */
              ssh_xfree(err_msg);
              ssh_oldfsm_continue(gdata->main_thread);
            }
          break;
        default:
          /* Do nothing. */
          break;
        }
      /* FALL THROUGH */
    case SSH_PAM_ERROR:
      SSH_ASSERT(err_msg);
      ssh_log_event(gdata->server->config->log_facility,
                    SSH_LOG_ERROR,
                    "auths-pam: ssh-pam-client returned packet "
                    "SSH_PAM_OP_ERROR. (err_num: %d, err_msg: %s)",
                    err_num, err_msg);
      ssh_xfree(err_msg);

      *gdata->state_placeholder = NULL;

      (*gdata->completion_proc)
        (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
         gdata->client_packet,
         gdata->completion_context);
      ssh_oldfsm_set_next(gdata->main_thread, "ssh_pam_abort");

      break;
    case SSH_PAM_CONVERSATION_MSG:
      /* send conv message to client. */
      SSH_ASSERT(gdata->sent_state >= SENT_PAM_AUTHENTICATE);
      gdata->doing_conversation = TRUE;

      ssh_buffer_clear(gdata->client_packet);
      ssh_encode_buffer(gdata->client_packet,
                        SSH_FORMAT_CHAR, SSH_MSG_USERAUTH_PAM_MSG,
                        SSH_FORMAT_DATA, data, len,
                        SSH_FORMAT_END);
      (*gdata->completion_proc)(SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK,
                                gdata->client_packet,
                                gdata->completion_context);
      gdata->client_packet = NULL;

      /* Don't revive thread; we will come here again (after the
         client sends us back a response, the aux_thread will send it
         to ssh-pam-client, and the ssh-pam-client will send us the
         acknowledgement for the original message). */
      return;

    default:
      ssh_log_event(gdata->server->config->log_facility,
                    SSH_LOG_ERROR,
                    "auths-pam: ssh-pam-client returned packet "
                    "%d, which is invalid from the ssh-pam-client.",
                    packet_type);

      *gdata->state_placeholder = NULL;
      (*gdata->completion_proc)
        (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
         gdata->client_packet,
         gdata->completion_context);
      ssh_oldfsm_set_next(gdata->main_thread, "ssh_pam_abort");
      break;
    }
  ssh_oldfsm_continue(gdata->main_thread);
}

void received_eof_cb(void *context)
{
  SshServerPAMAuth gdata = (SshServerPAMAuth) context;
  SSH_TRACE(0, ("Received EOF."));

  *gdata->state_placeholder = NULL;

  (*gdata->completion_proc) (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                             gdata->client_packet,
                             gdata->completion_context);

  ssh_oldfsm_set_next(gdata->main_thread, "ssh_pam_abort");
  ssh_oldfsm_continue(gdata->main_thread);
}

SSH_OLDFSM_STEP(ssh_pam_start)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);

  SSH_OLDFSM_SET_NEXT("ssh_pam_set_item_rhost");

  gdata->sent_state = SENT_PAM_START;

  SSH_TRACE(4,("Sending service_name:%s, user:%s to ssh-pam-client.",
               gdata->service_name, ssh_user_name(gdata->uc)));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_START,
                                 SSH_FORMAT_UINT32_STR, gdata->service_name,
                                 strlen(gdata->service_name),
                                 SSH_FORMAT_UINT32_STR,
                                 ssh_user_name(gdata->uc),
                                 strlen(ssh_user_name(gdata->uc)),
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_set_item_rhost)
{
  char *client_host;
  SSH_OLDFSM_GDATA(SshServerPAMAuth);
  SSH_OLDFSM_SET_NEXT("ssh_pam_set_item_tty");

  if (gdata->server->authenticated_client_host)
    /* If set, use authenticated remote host name from "hostbased". */
    client_host = gdata->server->authenticated_client_host;
  else
    client_host = gdata->server->common->remote_host;

  SSH_ASSERT(client_host);

  gdata->sent_state = SENT_PAM_SET_RHOST;

  SSH_DEBUG(4, ("Sending a SSH_PAM_SET_ITEM (PAM_RHOST) packet to "
                "ssh-pam-client..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_SET_ITEM,
                                 SSH_FORMAT_UINT32, (SshUInt32)PAM_RHOST,
                                 SSH_FORMAT_UINT32_STR,
                                 client_host,
                                 strlen(client_host),
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_set_item_tty)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);
  SSH_OLDFSM_SET_NEXT("ssh_pam_set_item_ruser");

  gdata->sent_state = SENT_PAM_SET_TTY;

  SSH_DEBUG(4, ("Sending a SSH_PAM_SET_ITEM (PAM_RUSER) packet to "
                "ssh-pam-client..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_SET_ITEM,
                                 SSH_FORMAT_UINT32, (SshUInt32)PAM_TTY,
                                 SSH_FORMAT_UINT32_STR,
                                 "ssh", strlen("ssh"),
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_set_item_ruser)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);
  SSH_OLDFSM_SET_NEXT("ssh_pam_authenticate");

  if (!gdata->server->authenticated_client_user_name)
    {
      return SSH_OLDFSM_CONTINUE;
    }

  gdata->sent_state = SENT_PAM_SET_RUSER;

  SSH_DEBUG(4, ("Sending a SSH_PAM_SET_ITEM (PAM_RUSER) packet to "
                "ssh-pam-client..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_SET_ITEM,
                                 SSH_FORMAT_UINT32, (SshUInt32)PAM_RUSER,
                                 SSH_FORMAT_UINT32_STR,
                                 gdata->server->authenticated_client_user_name,
                                 strlen(gdata->server->
                                        authenticated_client_user_name),
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_authenticate)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);
  SSH_OLDFSM_SET_NEXT("ssh_pam_acct_mgmt");

  gdata->sent_state = SENT_PAM_AUTHENTICATE;

  SSH_DEBUG(4, ("Sending a SSH_PAM_AUTHENTICATE packet to ssh-pam-client..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_AUTHENTICATE,
                                 /* XXX PermitEmptyPasswords (etc.) ? */
                                 SSH_FORMAT_UINT32,
                                 (SshUInt32) PAM_DISALLOW_NULL_AUTHTOK,
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_acct_mgmt)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);
  SSH_OLDFSM_SET_NEXT("ssh_pam_open_session");

  gdata->sent_state = SENT_PAM_ACCT_MGMT;

  SSH_DEBUG(4, ("Sending a SSH_PAM_ACCT_MGMT packet to ssh-pam-client..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_ACCT_MGMT,
                                 /* XXX PermitEmptyPasswords (etc.) ? */
                                 SSH_FORMAT_UINT32,
                                 (SshUInt32) PAM_DISALLOW_NULL_AUTHTOK,
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_chauthtok)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);

  SSH_OLDFSM_SET_NEXT("ssh_pam_acct_mgmt");

  gdata->sent_state = SENT_PAM_CHAUTHTOK;

  SSH_DEBUG(4, ("Sending a SSH_PAM_CHAUTHTOK packet to ssh-pam-client..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_CHAUTHTOK,
                                 SSH_FORMAT_UINT32,
                                 (SshUInt32) PAM_CHANGE_EXPIRED_AUTHTOK,
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_open_session)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);
  SSH_OLDFSM_SET_NEXT("ssh_pam_setcred");

  gdata->sent_state = SENT_PAM_OPEN_SESSION;

  SSH_DEBUG(4, ("Sending a SSH_PAM_OPEN_SESSION packet to ssh-pam-client..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_OPEN_SESSION,
                                 SSH_FORMAT_UINT32, (SshUInt32) 0,
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_setcred)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);
  SSH_OLDFSM_SET_NEXT("ssh_pam_auth_successful");

  gdata->sent_state = SENT_PAM_SETCRED;

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_SETCRED,
                                 /* XXX Other flags? */
                                 SSH_FORMAT_UINT32,
                                 (SshUInt32) PAM_ESTABLISH_CRED,
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_auth_successful)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);

  ssh_packet_wrapper_can_receive(gdata->wrapper, FALSE);
  ssh_packet_wrapper_set_callbacks(gdata->wrapper,
                                   NULL, NULL, NULL, NULL);

  ssh_user_pam_set_handle(gdata->uc, gdata->wrapper);
  gdata->wrapper = NULL;

  ssh_oldfsm_kill_thread(gdata->aux_thread);
  gdata->aux_thread = NULL;

  ssh_oldfsm_destroy(gdata->fsm);

  *gdata->state_placeholder = NULL;
  /* inform userauth-layer of successful authentication. */
  (*gdata->completion_proc)(SSH_AUTH_SERVER_ACCEPTED,
                            gdata->client_packet,
                            gdata->completion_context);

  return SSH_OLDFSM_FINISH;
}

/* The aux_thread handles these. */
SSH_OLDFSM_STEP(ssh_pam_process_client_packet)
{
  Boolean msg_code;
  SSH_OLDFSM_GDATA(SshServerPAMAuth);

  SSH_PRECOND(gdata->client_packet);
  SSH_PRECOND(thread == gdata->aux_thread);

  if (!gdata->doing_conversation)
    {
      ssh_log_event(gdata->server->config->log_facility,
                    SSH_LOG_ERROR,
                    "auths-pam: client sent response packet, "
                    "and we weren't expecting any (yet).");
      goto error;
    }

  if (!ssh_decode_buffer(gdata->client_packet,
                         SSH_FORMAT_BOOLEAN, &msg_code,
                         SSH_FORMAT_END) || msg_code != TRUE)
    {
      if (msg_code != TRUE)
        SSH_TRACE(0, ("msg_code = %d (invalid)", msg_code));

      ssh_log_event(gdata->server->config->log_facility,
                    SSH_LOG_ERROR,
                    "auths-pam: client sent invalid packet as a "
                    "response to our conversation message.");
    error:
      *gdata->state_placeholder = NULL;
      (*gdata->completion_proc)
        (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
         gdata->client_packet,
         gdata->completion_context);
      ssh_oldfsm_set_next(gdata->main_thread, "ssh_pam_abort");
      ssh_oldfsm_continue(gdata->main_thread);
      return SSH_OLDFSM_FINISH;
    }

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_CONVERSATION_RESP,
                                 SSH_FORMAT_DATA,
                                 ssh_buffer_ptr(gdata->client_packet),
                                 ssh_buffer_len(gdata->client_packet),
                                 SSH_FORMAT_END);
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_suspend)
{
  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(ssh_pam_abort)
{
  SSH_OLDFSM_GDATA(SshServerPAMAuth);

  if (gdata->aux_thread)
    {
      ssh_oldfsm_kill_thread(gdata->aux_thread);
      gdata->aux_thread = NULL;
    }
  if (gdata->fsm)
    {
      ssh_oldfsm_destroy(gdata->fsm);
      gdata->fsm = NULL;
    }

  return SSH_OLDFSM_FINISH;
}

SshOldFSMStateMapItemStruct pam_states[] =
{
  { "ssh_pam_start", "Start PAM authentication", ssh_pam_start },
  { "ssh_pam_authenticate", "Request pam_authenticate()",
    ssh_pam_authenticate },
  { "ssh_pam_acct_mgmt", "Request pam_acct_mgmt()",
    ssh_pam_acct_mgmt },
  { "ssh_pam_chauthtok", "Request pam_chauthtok()",
    ssh_pam_chauthtok },
  { "ssh_pam_open_session", "Request pam_open_session()",
    ssh_pam_open_session },
  { "ssh_pam_setcred", "Request pam_setcred()",
    ssh_pam_setcred },
  { "ssh_pam_set_item_rhost", "Request pam_set_item() for RHOST",
    ssh_pam_set_item_rhost },
  { "ssh_pam_set_item_tty", "Request pam_set_item() for TTY",
    ssh_pam_set_item_tty },
  { "ssh_pam_set_item_ruser", "Request pam_set_item() for RUSER",
    ssh_pam_set_item_ruser },

  { "ssh_pam_process_client_packet", "Process packet received from client",
    ssh_pam_process_client_packet },

  { "ssh_pam_auth_successful", "Authentication was successful",
    ssh_pam_auth_successful },

  { "ssh_pam_suspend", "Suspend thread", ssh_pam_suspend },

  { "ssh_pam_abort", "Abort PAM ops", ssh_pam_abort }
};

void pam_destructor(void *gdata_ctx)
{
  SshServerPAMAuth gdata = (SshServerPAMAuth) gdata_ctx;
  SSH_PRECOND(gdata);

  ssh_xfree(gdata->service_name);

  if (gdata->wrapper)
    {
      ssh_packet_wrapper_can_receive(gdata->wrapper, FALSE);
      ssh_packet_wrapper_destroy(gdata->wrapper);
      gdata->wrapper = NULL;
    }

  *gdata->state_placeholder = NULL;
}

void ssh_server_auth_pam(SshAuthServerOperation op,
                         const char *user,
                         SshUser uc,
                         SshBuffer packet,
                         const unsigned char *session_id,
                         size_t session_id_len,
                         void **state_placeholder,
                         void **longtime_placeholder,
                         SshAuthServerCompletionProc completion_proc,
                         void *completion_context,
                         void *method_context)
{
  SshServer server = (SshServer)method_context;
  SshConfig config = server->config;
  SshServerPAMAuth state = *state_placeholder;
  SshOldFSM fsm;
  SshStream child_stdio;
  Boolean msg_code;

  SSH_TRACE(6, ("PAM auth."));

  SSH_DEBUG(6, ("op = %d  user = %s", op, user));

  switch (op)
    {
    case SSH_AUTH_SERVER_OP_START:
      /* This is the first operation for doing PAM authentication.
         We should not have any previous saved state when we come here. */
      SSH_ASSERT(*state_placeholder == NULL);

      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_PAM))
        {
          (*completion_proc) (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                              packet,
                              completion_context);
          return;
        }

      if (!ssh_decode_buffer(packet,
                             SSH_FORMAT_BOOLEAN, &msg_code,
                             SSH_FORMAT_END) || msg_code != FALSE)
        {
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "Client sent continuation packet when expecting "
                        "start packet for PAM-auth..",
                        ssh_user_name(uc));
          (*completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                             packet,
                             completion_context);
          return;
        }


      if (ssh_buffer_len(packet) != 0)
        {
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "Extra data at of SSH_MSG_USERAUTH_REQUEST packet "
                        "for PAM authentication.",
                        ssh_user_name(uc));
          (*completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                             packet,
                             completion_context);
          return;
        }

      switch (ssh_pipe_create_and_fork(&child_stdio, NULL))
        {
        case SSH_PIPE_ERROR:
          ssh_log_event(config->log_facility,
                        SSH_LOG_ERROR,
                        "ssh_server_auth_pam: pipe creation failed.");
          (*completion_proc) (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                              packet,
                              completion_context);
          return;
          break;
        case SSH_PIPE_PARENT_OK:
          SSH_TRACE(2, ("Fork successful."));
          break;
        case SSH_PIPE_CHILD_OK:
          {
            char *pam_argv[2];

            SSH_TRACE(0, ("Child: Execing ssh-pam-client. (path: %s)",
                          config->ssh_pam_client_path));

            pam_argv[0] = config->ssh_pam_client_path;
            pam_argv[1] = NULL;

            execvp(pam_argv[0], pam_argv);
            ssh_log_event(config->log_facility,
                          SSH_LOG_ERROR,
                          "ssh_server_auth_pam: execing ssh-pam-client "
                          "failed (system error message: %s)",
                          strerror(errno));

            ssh_fatal("ssh_server_auth_pam: execing ssh-pam-client "
                      "failed (system error message: %s)",
                      strerror(errno));
            break;
          }

          /* Continuing as parent. */

          break;
        }

      fsm = ssh_oldfsm_allocate(sizeof(*state),
                             pam_states,
                             SSH_OLDFSM_NUM_STATES(pam_states),
                             pam_destructor);

      state = ssh_oldfsm_get_gdata_fsm(fsm);

      memset(state, 0, sizeof(*state));

      state->fsm = fsm;

      state->server = server;
      state->uc = uc;
      state->client_packet = packet;
      state->completion_proc = completion_proc;
      state->completion_context = completion_context;
      /* Service name defaults to "sshd2". */
      state->service_name = ssh_xstrdup(SSH_PAM_SERVICE_NAME);
      state->sent_state = SENT_NOTHING;
      state->doing_conversation = FALSE;
      state->state_placeholder = state_placeholder;

      state->wrapper = ssh_packet_wrap(child_stdio,
                                       received_packet_cb,
                                       received_eof_cb,
                                       NULL, state);


      state->main_thread = ssh_oldfsm_spawn(state->fsm,
                                         0, "ssh_pam_start",
                                         NULL, NULL);

      state->aux_thread = ssh_oldfsm_spawn(state->fsm,
                                        0, "ssh_pam_suspend",
                                        NULL, NULL);

      *state_placeholder = state;
      /* Rest is done in callbacks. */
      return;

    case SSH_AUTH_SERVER_OP_ABORT:
      SSH_ASSERT(state && state->main_thread);
      ssh_oldfsm_set_next(state->main_thread, "ssh_pam_abort");
      ssh_oldfsm_continue(state->main_thread);
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet, completion_context);
      return;

    case SSH_AUTH_SERVER_OP_CONTINUE:
      SSH_ASSERT(state && state->aux_thread);
      state->client_packet = packet;
      state->completion_proc = completion_proc;
      state->completion_context = completion_context;
      if (uc != state->uc)
        state->uc = uc;

      ssh_oldfsm_set_next(state->aux_thread, "ssh_pam_process_client_packet");
      ssh_oldfsm_continue(state->aux_thread);
      return;

    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      /* Clear state. */
      if (state)
        {
          if (!server)
            state->server = NULL;
          if (state->wrapper)
            ssh_packet_wrapper_can_receive(state->wrapper, FALSE);

          ssh_oldfsm_set_next(state->main_thread, "ssh_pam_abort");
          ssh_oldfsm_continue(state->main_thread);
        }

      *state_placeholder = NULL;
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;

    default:
      ssh_fatal("ssh_server_auth_pam: unknown op %d", (int)op);
    }

  SSH_NOTREACHED;
}

#endif /* DAEMON_WITH_PAM */
