/*

auths-passwd.c

  Author: Tatu Ylonen <ylo@ssh.com>

  Copyright (C) 1997-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Password authentication, server-side.  This calls functions in 
  machine-specific files to perform the actual authentication.

*/

#include "sshincludes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshserver.h"
#include "sshconfig.h"
#include "auths-common.h"
#include "auths-passwd.h"
#include "sshdsprintf.h"
#include "sshuserfiles.h"

#define SSH_DEBUG_MODULE "Ssh2AuthPasswdServer"


/* Checks if the user is root and if root login is disabled.
   Returns TRUE if user is root and root login is disabled,
   FALSE otherwise. */

Boolean is_root_and_disabled(SshConfig config, SshUser uc,
                             SshBuffer packet,
                             SshAuthServerCompletionProc completion_proc,
                             void *completion_context)
{
  if(ssh_user_uid(uc) == UID_ROOT &&
     (config->permit_root_login == SSH_ROOTLOGIN_FALSE ||
      config->permit_root_login == SSH_ROOTLOGIN_NOPWD))
    {
      /* XXX Add client addresses etc. */
      SSH_DEBUG(2, ("root logins are not permitted."));
      ssh_log_event(config->log_facility,
                    SSH_LOG_WARNING,
                    "root login denied for user '%s'.",
                    ssh_user_name(uc));

      if (config->password_guesses > 0)
        {
          (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                             completion_context);
          return TRUE;
        }
      else
        {
          (*completion_proc)
            (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED, packet,
             completion_context);
          return TRUE;
        }
    }
  return FALSE;
}

/* Password authentication.  This handles all forms of password authentication,
   including local passwords, kerberos, and secure rpc passwords. */

void ssh_server_auth_passwd(SshAuthServerOperation op,
                            const char *user,
                            SshUser uc,
                            SshBuffer packet,
                            const unsigned char *session_id,
                            size_t session_id_len,
                            void **state_placeholder,
                            void **longtime_placeholder,
                            SshAuthServerCompletionProc completion_proc,
                            void *completion_context,
                            void *method_context)
{
  SshServer server = (SshServer)method_context;
  SshConfig config = server->config;
  Boolean change_request;
  char *password, *prompt;
  Boolean disable_method = FALSE;
  size_t pass_len = 0L;
  
  SSH_TRACE(6, ("password auth."));
  
  SSH_DEBUG(6, ("op = %d  user = %s", op, user));
  
  switch (op)
    {
    case SSH_AUTH_SERVER_OP_START:
      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_PASSWD))
        {
          config->password_guesses--;
          if (config->password_guesses <= 0)
            {
              /* If this attempt is not succesful, disable this method. */
              disable_method = TRUE;
            }
          goto password_bad;
        }
      
      config->password_guesses--;
      if (config->password_guesses <= 0)
        {
          /* If this attempt is not succesful, disable this method. */
          disable_method = TRUE;
        }


      if (is_root_and_disabled(config, uc, packet,
                               completion_proc, completion_context))
        return;


      /* Parse the password authentication request. */
      if (ssh_decode_buffer(packet,
                            SSH_FORMAT_BOOLEAN, &change_request,
                            SSH_FORMAT_UINT32_STR, &password, &pass_len,
                            SSH_FORMAT_END) == 0)
        {
          SSH_DEBUG(2, ("bad packet"));
          goto password_bad;
        }

      if (!config->permit_empty_passwords && pass_len == 0L)
        {
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "attempt to login as %s with empty password denied.",
                        ssh_user_name(uc));
          SSH_DEBUG(2, ("attempted login with empty password denied"));
          goto password_bad;
        }
      
      /* Password changing requests should only be received as continuation
         messages. */
      if (change_request)
        {
          SSH_DEBUG(2 ,("cannot change password before authentication"));
          goto password_bad;
        }
      
      /* Sanity check: do not pass excessively long passwords to system
         functions to avoid buffer overflows in operating system code. */
      if (strlen(password) > 256)
        {
          SSH_DEBUG(2, ("password too long."));
          ssh_xfree(password);
          goto password_bad;
        }


      /* Try SECURE RPC passwords.  We do this first, as this might be
         needed to access disks. */
      if (ssh_user_validate_secure_rpc_password(uc, password))
        {
          ssh_log_event(config->log_facility,
                        SSH_LOG_NOTICE,
                        "User %s's secure rpc password accepted.",
                        ssh_user_name(uc));
          SSH_DEBUG(5, ("ssh_server_auth_passwd: accepted by secure rpc"));
          goto password_ok;
        }

      /* Try KERBEROS passwords.  This might also be needed to access
         disks. */
      if (ssh_user_validate_kerberos_password(uc, password))
        {
          ssh_log_event(config->log_facility,
                        SSH_LOG_NOTICE,
                        "Kerberos password accepted for user %s (%s).",
                        ssh_user_name(uc),
                        ssh_user_kerberos_name(uc));
          SSH_DEBUG(5, ("ssh_server_auth_passwd: accepted by " \
                        "kerberos passwd"));
          goto password_ok;
        }


      /* Try a local password (either normal or shadow). */
      if (ssh_user_validate_local_password(uc, 
                                           password,
                                           server->common->remote_host))
        {









          ssh_log_event(config->log_facility,
                        SSH_LOG_NOTICE,
                        "User %s's local password accepted.",
                        ssh_user_name(uc));

          SSH_DEBUG(5, ("ssh_server_auth_passwd: accepted by local passwd"));
          goto password_ok;
        }

      ssh_log_event(config->log_facility,
                    SSH_LOG_WARNING,
                    "Wrong password given for user '%s'.",
                    ssh_user_name(uc));

      ssh_xfree(password);
      goto password_bad;

    password_bad:
      if (uc)
        ssh_user_record_login_failure(uc, server->common->remote_host);
      
      (*completion_proc)((disable_method ?
                          SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED :
                          SSH_AUTH_SERVER_REJECTED),
                         packet, completion_context);
      return;
    password_ok:
      /* Password authentication passed, but we still need to check whether
         the password needs to be changed. */
      ssh_xfree(password);

      ssh_log_event(config->log_facility, SSH_LOG_NOTICE,
                    "Password authentication for user %.100s accepted.",
                    ssh_user_name(uc));
      
      /* Check if the user's password needs to be changed. */
      if (ssh_user_password_must_be_changed(uc, &prompt))
        {
          ssh_log_event(config->log_facility, SSH_LOG_INFORMATIONAL,
                        "User %s forced to change password.",
                        ssh_user_name(uc));
          if (server->common->publickey_options)
            ssh_xfree(server->common->publickey_options);
          else
            server->common->publickey_options =
              ssh_xcalloc(1, sizeof(*server->common->publickey_options));

          ssh_xfree(server->common->publickey_options->command);

          /* XXX This is a problem, if the same uid is shared by many
             user's. In that case, invoking "passwd" without arguments
             will only change the _first_ password in the passwd
             db. OTOH, things are made trickier with RH Linux, which
             doesn't like a <username> argument with passwd when
             executer is non-root... Fantastic. */
          server->common->publickey_options->command =
            ssh_xstrdup(PASSWD_PATH);
        }

      (*completion_proc)(SSH_AUTH_SERVER_ACCEPTED, packet, completion_context);
      return;

    case SSH_AUTH_SERVER_OP_ABORT:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet, completion_context);
      return;
      
    case SSH_AUTH_SERVER_OP_CONTINUE:
      SSH_DEBUG(1, ("ssh_server_auth_passwd: XXX CONTINUE not yet "\
                    "implemented"));
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet, completion_context);
      return;
      
    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      (*completion_proc) (SSH_AUTH_SERVER_REJECTED, packet,
                          completion_context);
      return;
      
    default:
      ssh_fatal("ssh_server_auth_passwd: unknown op %d", (int)op);
    }
 
  SSH_NOTREACHED;
}
