/*
  ssh-keygen.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  A tool for generating and manipulating private keys.
*/

#include "sshincludes.h"
#include "ssh2includes.h"
#include "sshkeyfile.h"
#include "sshuserfiles.h"
#include "sshreadline.h"
#include "sshfileio.h"
#include "readpass.h"
#include "sshuser.h"
#include "sshcrypt.h"
#include "sshcipherlist.h"
#include "sshgetopt.h"
#include "sshappcommon.h"
#include <sys/types.h>
#include "sshmp.h"
#include "sshfingerprint.h"
#include "math.h"
#include "sshsignals.h"









#ifndef VXWORKS
#include <pwd.h>
#endif /* VXWORKS */
















#define SSH_DEBUG_MODULE "SshKeyGen"

/* Standard (assumed) choices.. */

#ifndef KEYGEN_ASSUMED_PUBKEY_LEN
#define KEYGEN_ASSUMED_PUBKEY_LEN 1024
#endif /* KEYGEN_ASSUMED_PUBKEY_LEN */

/* helptext */

const char keygen_helptext[] =
  "Usage: ssh-keygen [options] [key1 key2 ...]\n"
  "\n"
  "Where `options' are:\n"
  " -b nnn         Specify key strength in bits (e.g. 1024)\n"
#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  " -t dsa | rsa   Choose the key type.\n"
#else /* WITH_RSA */
  " -t dsa         Choose the key type (only dsa available).\n"
#endif /* WITH_RSA */


#endif /* SSHDIST_CRYPT_RSA */
  " -c comment     Provide the comment.\n"
  " -e file        Edit the comment/passphrase of the key.\n"
  " -p passphrase  Provide passphrase.\n"
  " -P             Assume empty passphrase.\n"
  " -?\n"
  " -h             Print this help text.\n"
  " -q             Suppress the progress indicator.\n"
  " -1             Convert a SSH 1.x key.\n"
  " -i file        Load and display information on `file'.\n"
  " -D file        Derive the public key from the private key 'file'.\n"
  " -B number      The number base for displaying key information (default 10).\n"
  " -V             Print ssh-keygen version number.\n"
  " -r file        Stir data from file to random pool.\n"






  " -F file        Dump fingerprint of file.\n\n";

/* A context structure -- we don't like global variables. */

typedef struct
{
  int keybits;
  Boolean newkey;
  Boolean convert;





  Boolean status;
  Boolean edit_key;

  char *keytype;
  char *keytypecommon;
  char *comment;
  char *in_filename;
  char *out_filename;
  char *passphrase;
  char *info_file;
  char *derive_file;
  Boolean pass_empty;

  unsigned int base;

  SshPrivateKey private_key;
  SshPublicKey public_key;
  SshUser user;

  Boolean prog_ind;
  Boolean have_prog_ind;

  Boolean read_file;
  Boolean dump_fingerprint;
  SshFingerPrintType fingerprint_type;
  char *entropy_file_name;
} KeyGenCtx;

/* mapping of common names and CryptoLib names. Common names are
   not case sensitive.

   The first entry in this list will be the preferred (standard)
   choice. */

const char *keygen_common_names[][2] =
{
  /* Digital Signature Standard */
  { "dsa", SSH_CRYPTO_DSS },
  { "dss", SSH_CRYPTO_DSS },

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  /* RSA */
  { "rsa", SSH_CRYPTO_RSA },
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

  /* Last entry */
  { NULL, NULL }
};

/* allocate the context */

KeyGenCtx *keygen_init_ctx(void)
{
  KeyGenCtx *kgc;

  kgc = ssh_xcalloc(1, sizeof (KeyGenCtx));
  kgc->keybits = FALSE;
  kgc->newkey = FALSE;
  kgc->convert = FALSE;





  kgc->status = FALSE;
  kgc->read_file = FALSE;
  kgc->edit_key = FALSE;

  kgc->keytype = NULL;
  kgc->keytypecommon = NULL;
  kgc->comment = NULL;
  kgc->out_filename = NULL;
  kgc->in_filename = NULL;
  kgc->passphrase = NULL;
  kgc->pass_empty = FALSE;
  kgc->info_file = NULL;
  kgc->derive_file = NULL;
  kgc->dump_fingerprint = FALSE;
  kgc->fingerprint_type = SSH_FINGERPRINT_BABBLE;

  kgc->base = 10;

  kgc->user = ssh_user_initialize(NULL, FALSE);
  kgc->public_key = NULL;
  kgc->private_key = NULL;

  kgc->prog_ind = FALSE;
  kgc->have_prog_ind = TRUE;

  kgc->entropy_file_name = NULL;
  return kgc;
}

/* Zeroize and free a NULL-terminated string, assuming that the pointer
   is non-null */

void keygen_free_str(char *s)
{
  if (s == NULL)
    return;
  memset(s, 0, strlen(s));
  ssh_xfree(s);
}

/* free the context */

void keygen_free_ctx(KeyGenCtx *kgc)
{
  keygen_free_str(kgc->keytype);
  keygen_free_str(kgc->keytypecommon);
  keygen_free_str(kgc->comment);
  keygen_free_str(kgc->in_filename);
  keygen_free_str(kgc->out_filename);
  keygen_free_str(kgc->passphrase);
  keygen_free_str(kgc->info_file);
  keygen_free_str(kgc->derive_file);
  keygen_free_str(kgc->entropy_file_name);

  if (kgc->public_key != NULL)
    ssh_public_key_free(kgc->public_key);
  if (kgc->private_key != NULL)
    ssh_private_key_free(kgc->private_key);
  ssh_randseed_update(kgc->user, NULL);
  if (kgc->user != NULL)
    ssh_user_free(kgc->user, FALSE);
  if (kgc->prog_ind)
    ssh_crypto_library_register_progress_func(NULL, NULL);

  memset(kgc, 0, sizeof (KeyGenCtx));
  ssh_xfree(kgc);
}


/* Ask for a passphrase twice */

char *ssh_read_passphrase_twice(char *prompt1, char *prompt2, int from_stdin)
{
  char *r1, *r2;

  for (;;)
    {
      r1 = ssh_read_passphrase(prompt1, from_stdin);
      r2 = ssh_read_passphrase(prompt2, from_stdin);

      if (strcmp(r1, r2) != 0)
        {
          keygen_free_str(r1);
          keygen_free_str(r2);
          fprintf(stderr, "Passphrases do not match.\n");
        }
      else
        break;
    }
  keygen_free_str(r2);

  return r1;
}


/* Read a string (with echo) from stdin */

char *ssh_askpass_read_stdin(char *prompt)
{
  char buf[1024], *p;

  if (prompt != NULL)
    {
      printf("%s", prompt);
      fflush(stdout);
    }
  if (fgets(buf, sizeof (buf)-1, stdin) == NULL)
    return ssh_xstrdup("");

  for(p = buf; *p >= 32 || *p < 0; p++);
  *p = '\0';

  p = ssh_xstrdup(buf);
  memset(buf, 0, sizeof (buf));

  return p;
}



/* Keygen error message */

void keygen_error(KeyGenCtx *kgc, char *expl)
{
  fprintf(stderr, "\nError: %s\n", expl);
  keygen_free_ctx(kgc);
  exit(-1);
}

/* The progress indicator */

void keygen_prog_ind(SshCryptoProgressID id, unsigned int time_value,
                     void *incr)
{
  int i;

  i = *((int *) incr);
  (*((int *) incr))++;

  if (i % 13 == 0)
    {
      printf("\r %3d ", i / 13 + 1);
    }
  else
    {
      switch( i % 4 )
        {
        case 0:
          printf(".");
          break;
        case 1:
        case 3:
          printf("o");
          break;
        case 2:
          printf("O");
          break;
        }
    }
  fflush(stdout);
}

/* Generate a filename for the private key */

void keygen_choose_filename(KeyGenCtx *kgc)
{
  int i;
  char buf[1024], *udir;
  struct stat st;

  if (kgc->out_filename != NULL)
    return;

  if((udir = ssh_userdir(kgc->user, NULL, TRUE)) == NULL)
    {
      ssh_warning("Unable to open user ssh2 directory. "
                  "Saving to current directory.");
      udir = ssh_xstrdup(".");
    }

  for (i = 'a'; i <= 'z'; i++)
    {
      ssh_snprintf(buf, sizeof (buf), "%s/id_%s_%d_%c",
                   udir, kgc->keytypecommon, kgc->keybits, i);
      if (stat(buf, &st) == 0)
        continue;
      kgc->out_filename = ssh_xstrdup(buf);
      goto done;
    }
  ssh_fatal("Could not find a suitable file name.");

done:
  ssh_xfree(udir);
}

/* Read private key. */

SshPrivateKey keygen_read_privkey(KeyGenCtx *kgc, char *filename)
{
  SshPrivateKey prvkey = NULL;
  /* Guess that it is a private key. */

  if (kgc->passphrase)
    {
      if (ssh_privkey_read(kgc->user, filename, kgc->passphrase, NULL,
                           &prvkey) == SSH_PRIVKEY_KEY_UNREADABLE)
        {
          ssh_warning("Private key %s is unreadable.", filename);
          return NULL;
        }
    }

  if (prvkey == NULL)
    {
      if (ssh_privkey_read(kgc->user, filename,
                           "", NULL, &prvkey) == SSH_PRIVKEY_KEY_UNREADABLE)
        {
          ssh_warning("Private key %s is unreadable.", filename);
          return NULL;
        }

      if (prvkey == NULL &&
          !kgc->passphrase && kgc->pass_empty == FALSE)
        {
          char *pass;
          pass = ssh_read_passphrase("Passphrase : ", FALSE);

          if (ssh_privkey_read(kgc->user, filename, pass, NULL,
                               &prvkey) == SSH_PRIVKEY_KEY_UNREADABLE)
            {
              ssh_warning("Private key %s is unreadable.", filename);
              return NULL;
            }

          ssh_xfree(pass);
        }
    }

  return prvkey;
}

Boolean keygen_out_number(KeyGenCtx *kgc, SshMPInt number)
{
  char *buf;
  buf = ssh_mp_get_str(NULL, kgc->base, number);
  printf("%s (%u bits)\n", buf, ssh_mp_get_size(number, 2));
  ssh_xfree(buf);
  return FALSE;
}

void keygen_check_primality(KeyGenCtx *kgc, SshMPInt number)
{
  if (ssh_mp_is_probable_prime(number, 20) == 1)
    return;

  ssh_warning("The number should be prime, but it is not. This is a security violation, and the key should NOT be used to security sensitive traffic.");
}

#define ATTACK_ESTIMATED_IN        "July, 2000"
#define FAST_MACHINE_OPS_LOG2      54 /* log2 ops per year */
#define FAST_MACHINE_SPEED         1  /* Ghz */
#define ATTACK_CONSTANT            1.9
#define ATTACK_METHOD              "NFS"

static unsigned int l_func(double alpha, double beta,
                           double bits)
{
  return alpha * pow(bits, beta) * pow(log(bits)/log(2), 1.0 - beta);
}

static unsigned int rsa_strength(unsigned int bits)
{
  return l_func(ATTACK_CONSTANT, 1.0/3.0, bits);
}

static unsigned int dsa_strength(unsigned int q, unsigned int p)
{
  unsigned int f, s;

  f = l_func(ATTACK_CONSTANT, 1.0/3.0, p);
  s = q / 2;

  if (f < s)
    return f;
  else
    return s;
}

static const char *compute_years(int e)
{
  SshMPIntStruct g;
  unsigned int b10, i, neg;
  static char buf[20];

  if (e < 0)
    {
      ssh_snprintf(buf, 20, "<1");
      return buf;
    }

  ssh_mp_init(&g);

  /* Compute quickly 2^e. */
  ssh_mp_set_bit(&g, e);

  /* Now deduce the two first digits in base 10, and exponent. */
  b10 = ssh_mp_get_size(&g, 10);

  neg = 1;
  if (b10 < 2)
    {
      ssh_mp_mul_ui(&g, &g, 100);
      b10 += 2;
      neg += 2;
    }

  for (i = 2; i < b10; i++)
    ssh_mp_div_ui(&g, &g, 10);

  buf[2] = '0' + ssh_mp_div_ui(&g, &g, 10);
  buf[1] = '.';
  buf[0] = '0' + ssh_mp_div_ui(&g, &g, 10);

  ssh_snprintf(&buf[3], 17, " * 10^%d", ((int)b10 - (int)neg));

  ssh_mp_clear(&g);

  return buf;
}

static void pubkey_print_estimate(SshPublicKey pub)
{
  const char *name, *remarks = "", *method = "", *years = "~0";
  unsigned int log2 = 0;

  if (ssh_public_key_get_info(pub, SSH_PKF_KEY_TYPE, &name,
                               SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      printf("Public key has no type (corrupted)\n");
      return;
    }

  if (strcmp(name, "if-modn") == 0)
    {
      SshMPIntStruct n, e;

      /* RSA? */
      ssh_mp_init(&n);
      ssh_mp_init(&e);

      /* Get the necessary information of the SSH style RSA key. */
      if (ssh_public_key_get_info(pub,
                                  SSH_PKF_MODULO_N,  &n,
                                  SSH_PKF_PUBLIC_E,  &e,
                                  SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&n);
          ssh_mp_clear(&e);
          printf("RSA public key does not have all the fields.");
          return;
        }

      log2 = rsa_strength(ssh_mp_get_size(&n,2));
      method = "NFS";
      if (ssh_mp_cmp_ui(&e, 15) < 0)
        {
          remarks = "\n  Use of the key is not recommended for encryption.";
        }

      ssh_mp_clear(&n);
      ssh_mp_clear(&e);
    }

  if (strcmp(name, "dl-modp") == 0)
    {
      SshMPIntStruct p, q;

      ssh_mp_init(&p);
      ssh_mp_init(&q);

      if (ssh_public_key_get_info(pub,
                                  SSH_PKF_PRIME_P, &p,
                                  SSH_PKF_PRIME_Q, &q,
                                  SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&p);
          ssh_mp_clear(&q);
          printf("DSA public key does not have all the fields.");
          return;
        }

      log2 = dsa_strength(ssh_mp_get_size(&q, 2),
                          ssh_mp_get_size(&p, 2));
      method = "NFS and Pollard rho";
      ssh_mp_clear(&p);
      ssh_mp_clear(&q);
    }

  if (log2 + 5 < FAST_MACHINE_OPS_LOG2)
    {
      printf("[WARNING: Key is might be weak---it should not be used for"
             " communication of sensitive material.]\n");
    }

  years = compute_years(log2 - FAST_MACHINE_OPS_LOG2);

  printf("[Strength estimation as of %s considering %s:"
         " Attack requires O(2^%u) steps, which is roughly"
         " equivalent to %s years of effort with %uGHz machine.%s]\n",
         ATTACK_ESTIMATED_IN,
         method,
         log2,
         years,
         FAST_MACHINE_SPEED, remarks);
}

static void prvkey_print_estimate(SshPrivateKey prv)
{
  const char *name, *remarks = "", *method = "", *years = "~0";
  unsigned int log2 = 0;

  if (ssh_private_key_get_info(prv, SSH_PKF_KEY_TYPE, &name,
                               SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      printf("Private key has no type (corrupted)\n");
      return;
    }

  if (strcmp(name, "if-modn") == 0)
    {
      SshMPIntStruct n, e;

      /* RSA? */
      ssh_mp_init(&n);
      ssh_mp_init(&e);
      /* Get the necessary information of the SSH style RSA key. */
      if (ssh_private_key_get_info(prv,
                                   SSH_PKF_MODULO_N,  &n,
                                   SSH_PKF_PUBLIC_E,  &e,
                                   SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&n);
          printf("RSA private key does not have all the fields.");
          return;
        }

      log2 = rsa_strength(ssh_mp_get_size(&n,2));
      method = "NFS";
      if (ssh_mp_cmp_ui(&e, 15) < 0)
        {
          remarks = "\n  Use of the key is not recommended for encryption.";
        }

      ssh_mp_clear(&n);
      ssh_mp_clear(&e);
    }

  if (strcmp(name, "dl-modp") == 0)
    {
      SshMPIntStruct p, q;

      ssh_mp_init(&p);
      ssh_mp_init(&q);

      if (ssh_private_key_get_info(prv,
                                   SSH_PKF_PRIME_P, &p,
                                   SSH_PKF_PRIME_Q, &q,
                                   SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&p);
          ssh_mp_clear(&q);

          printf("DSA private key does not have all the fields.");
          return;
        }

      log2 = dsa_strength(ssh_mp_get_size(&q, 2),
                          ssh_mp_get_size(&p, 2));
      method = "NFS and Pollard rho";

      ssh_mp_clear(&p);
      ssh_mp_clear(&q);
    }

  if (log2 + 5 < FAST_MACHINE_OPS_LOG2)
    {
      printf("[WARNING: Key is might be weak---it should not be used for"
             " communication of sensitive material.]\n");
    }

  years = compute_years(log2 - FAST_MACHINE_OPS_LOG2);

  printf("[Strength estimation as of %s considering %s:"
         " Attack requires O(2^%u) steps, which is roughly"
         " equivalent to %s years of effort with %uGHz machine.%s]\n",
         ATTACK_ESTIMATED_IN,
         method,
         log2,
         years,
         FAST_MACHINE_SPEED, remarks);
}

Boolean keygen_view_prvkey(KeyGenCtx *kgc, SshPrivateKey prv)
{
  const char *name;

  if (ssh_private_key_get_info(prv, SSH_PKF_KEY_TYPE, &name,
                               SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      printf("Private key has no type (corrupted)\n");
      goto failed;
    }

  if (strcmp(name, "if-modn") == 0)
    {
      SshMPIntStruct n, e, d, p, q, u;

      /* RSA? */
      ssh_mp_init(&n);
      ssh_mp_init(&e);
      ssh_mp_init(&d);
      ssh_mp_init(&p);
      ssh_mp_init(&q);
      ssh_mp_init(&u);

      /* Get the necessary information of the SSH style RSA key. */
      if (ssh_private_key_get_info(prv,
                                   SSH_PKF_MODULO_N,  &n,
                                   SSH_PKF_PUBLIC_E,  &e,
                                   SSH_PKF_SECRET_D,  &d,
                                   SSH_PKF_PRIME_P,   &p,
                                   SSH_PKF_PRIME_Q,   &q,
                                   SSH_PKF_INVERSE_U, &u,
                                   SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&n);
          ssh_mp_clear(&e);
          ssh_mp_clear(&d);
          ssh_mp_clear(&p);
          ssh_mp_clear(&q);
          ssh_mp_clear(&u);
          printf("RSA private key does not have all the fields.");
          goto failed;
        }

      printf("RSA Private Key\n");
      prvkey_print_estimate(prv);

      printf("  e = [Encryption exponent]\n");
      keygen_out_number(kgc, &e);

      printf("  n = [Large composite modulus]\n");
      keygen_out_number(kgc, &n);

      printf("  d = [Inverse of e modulo (p-1)*(q-1) (Euler's phi function of n)]\n");
      keygen_out_number(kgc, &d);

      printf("  p = [Large random prime]\n");
      keygen_check_primality(kgc, &p);
      keygen_out_number(kgc, &p);

      printf("  q = [Large random prime]\n");
      keygen_check_primality(kgc, &q);
      keygen_out_number(kgc, &q);

      printf("  u = [Multiplicative inverse of p modulo q, used to speed RSA]\n");
      keygen_out_number(kgc, &u);

      ssh_mp_clear(&n);
      ssh_mp_clear(&e);
      ssh_mp_clear(&d);
      ssh_mp_clear(&p);
      ssh_mp_clear(&q);
      ssh_mp_clear(&u);

      return FALSE;
    }

  if (strcmp(name, "dl-modp") == 0)
    {
      SshMPIntStruct p, q, g, y, x;

      ssh_mp_init(&p);
      ssh_mp_init(&q);
      ssh_mp_init(&g);
      ssh_mp_init(&y);
      ssh_mp_init(&x);

      if (ssh_private_key_get_info(prv,
                                   SSH_PKF_PRIME_P, &p,
                                   SSH_PKF_PRIME_Q, &q,
                                   SSH_PKF_GENERATOR_G, &g,
                                   SSH_PKF_PUBLIC_Y, &y,
                                   SSH_PKF_SECRET_X, &x,
                                   SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&p);
          ssh_mp_clear(&g);
          ssh_mp_clear(&q);
          ssh_mp_clear(&y);
          ssh_mp_clear(&x);

          printf("DSA private key does not have all the fields.");
          goto failed;
        }

      /* DSA */
      printf("DSA Private Key\n");
      prvkey_print_estimate(prv);

      printf("  p = [Large prime, characteristic of the finite field]\n");
      keygen_check_primality(kgc, &p);
      keygen_out_number(kgc, &p);

      printf("  g = [Generator of a cyclic subgroup of the multiplicative group F_p]\n");
      keygen_out_number(kgc, &g);

      printf("  q = [Prime number, order of the cyclic group]\n");
      keygen_check_primality(kgc, &q);
      keygen_out_number(kgc, &q);

      printf("  y = [Random power of the generator, the 'actual' public key]\n");
      keygen_out_number(kgc, &y);

      printf("  x = [A random number, the 'actual' secret key]\n");
      keygen_out_number(kgc, &x);

      ssh_mp_clear(&p);
      ssh_mp_clear(&g);
      ssh_mp_clear(&q);
      ssh_mp_clear(&y);
      ssh_mp_clear(&x);

      return FALSE;
    }

failed:
  return TRUE;
}

Boolean keygen_view_pubkey(KeyGenCtx *kgc, SshPublicKey pub)
{
  const char *name;

  if (ssh_public_key_get_info(pub, SSH_PKF_KEY_TYPE, &name,
                               SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      printf("Public key has no type (corrupted)\n");
      goto failed;
    }

  if (strcmp(name, "if-modn") == 0)
    {
      SshMPIntStruct n, e, d, p, q, u;

      /* RSA? */
      ssh_mp_init(&n);
      ssh_mp_init(&e);
      ssh_mp_init(&d);
      ssh_mp_init(&p);
      ssh_mp_init(&q);
      ssh_mp_init(&u);

      /* Get the necessary information of the SSH style RSA key. */
      if (ssh_public_key_get_info(pub,
                                  SSH_PKF_MODULO_N,  &n,
                                  SSH_PKF_PUBLIC_E,  &e,
                                  SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&n);
          ssh_mp_clear(&e);
          ssh_mp_clear(&d);
          ssh_mp_clear(&p);
          ssh_mp_clear(&q);
          ssh_mp_clear(&u);
          printf("RSA public key does not have all the fields.");
          goto failed;
        }

      printf("RSA Public Key\n");
      pubkey_print_estimate(pub);
      printf("  e = [Encryption exponent]\n");
      keygen_out_number(kgc, &e);

      printf("  n = [Large composite modulus]\n");
      keygen_out_number(kgc, &n);

      ssh_mp_clear(&n);
      ssh_mp_clear(&e);
      ssh_mp_clear(&d);
      ssh_mp_clear(&p);
      ssh_mp_clear(&q);
      ssh_mp_clear(&u);

      return FALSE;
    }

  if (strcmp(name, "dl-modp") == 0)
    {
      SshMPIntStruct p, q, g, y, x;

      ssh_mp_init(&p);
      ssh_mp_init(&q);
      ssh_mp_init(&g);
      ssh_mp_init(&y);
      ssh_mp_init(&x);

      if (ssh_public_key_get_info(pub,
                                  SSH_PKF_PRIME_P, &p,
                                  SSH_PKF_PRIME_Q, &q,
                                  SSH_PKF_GENERATOR_G, &g,
                                  SSH_PKF_PUBLIC_Y, &y,
                                  SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&p);
          ssh_mp_clear(&g);
          ssh_mp_clear(&q);
          ssh_mp_clear(&y);
          ssh_mp_clear(&x);

          printf("DSA public key does not have all the fields.");
          goto failed;
        }

      /* DSA */
      printf("DSA Public Key\n");
      pubkey_print_estimate(pub);

      printf("  p = [Large prime, characteristic of the finite field]\n");
      keygen_check_primality(kgc, &q);
      keygen_out_number(kgc, &p);

      printf("  g = [Generator of a cyclic subgroup of the multiplicative group F_p]\n");
      keygen_out_number(kgc, &g);

      printf("  q = [Prime number, order of the cyclic group]\n");
      keygen_check_primality(kgc, &q);
      keygen_out_number(kgc, &q);

      printf("  y = [Random power of the generator, the 'actual' public key]\n");
      keygen_out_number(kgc, &y);

      ssh_mp_clear(&p);
      ssh_mp_clear(&g);
      ssh_mp_clear(&q);
      ssh_mp_clear(&y);
      ssh_mp_clear(&x);

      return FALSE;
    }

failed:
  return TRUE;
}


/* Generate the key. this is done when kgc->newkey is TRUE. */

int keygen_keygen(KeyGenCtx *kgc)
{
  SshCryptoStatus code;
  char buf[1024];
  int incr;
  char *pass = NULL;
  int r = 0;

  /* Register our progress indicator. */

  incr = 0;
  if (kgc->prog_ind == FALSE && kgc->have_prog_ind == TRUE)
    {
      ssh_crypto_library_register_progress_func(keygen_prog_ind, &incr);
      kgc->prog_ind = TRUE;
    }

  printf("Generating %d-bit %s key pair\n",
         kgc->keybits,
         kgc->keytypecommon);

  if ((code = ssh_private_key_generate(&(kgc->private_key),
                                       kgc->keytype,
                                       SSH_PKF_SIZE, kgc->keybits,
                                       SSH_PKF_END)) != SSH_CRYPTO_OK)
    {
      keygen_error(kgc, (char *) ssh_crypto_status_message(code));
    }
  printf("\nKey generated.\n");

  printf("%s\n", kgc->comment);

  /* Ok, now save the private key. */

  keygen_choose_filename(kgc);

  if ((! kgc->passphrase) || (! (*(kgc->passphrase)))) {
    if (!(kgc->pass_empty))
      {
        pass = ssh_read_passphrase_twice("Passphrase : ",
                                         "Again      : ",
                                         FALSE);
      }
    else
      {
        pass = ssh_xstrdup("");
      }
    keygen_free_str(kgc->passphrase);
    kgc->passphrase = pass ? pass : ssh_xstrdup("");
  }

  if (!(*(kgc->passphrase)) && !kgc->pass_empty)
    {
      ssh_informational("Key is stored with NULL passphrase.\n");
      ssh_informational(" (You can ignore the following warning if you are "
                        "generating hostkeys.)\n");
      ssh_informational(" This is not recommended.\n");
      ssh_informational(" Don't do this unless you know what you're doing.\n");
      ssh_informational(" If file system protections fail (someone can "
                        "access the keyfile), \n");
      ssh_informational(" or if the super-user is malicious, your key can "
                        "be used without \n");
      ssh_informational(" the deciphering effort.\n");
    }

  if (ssh_privkey_write(kgc->user,
                        kgc->out_filename,
                        kgc->passphrase,
                        kgc->comment,
                        kgc->private_key, NULL))
    {
      ssh_warning("Private key not written !");
      r++;
    }
  else
    {
      printf("Private key saved to %s\n", kgc->out_filename);
    }

  /* Save the public key */

  ssh_snprintf(buf, sizeof (buf), "%s.pub", kgc->out_filename);
  kgc->public_key = ssh_private_key_derive_public_key(kgc->private_key);
  if (kgc->public_key == NULL)
    {
      ssh_warning("Could not derive public key from private key.");
      return r + 1;
    }

  if (ssh_pubkey_write(kgc->user, buf, kgc->comment, kgc->public_key, NULL))
    {
      ssh_warning("Public key not written !");
      r++;
    }
  else
    {
      printf("Public key saved to %s\n", buf);
    }
  return r;
}

/* Stir in data from file */

void stir_from_file(KeyGenCtx *kgc)
{
  unsigned char *buffer;
  size_t buffer_len;

  if (FALSE == ssh_read_file(kgc->entropy_file_name, &buffer, &buffer_len))
    {
      ssh_fatal("Cannot read file %s.", kgc->entropy_file_name);
      return;
    }
  ssh_random_add_noise(buffer, buffer_len);
  memset(buffer, 0, buffer_len);
  ssh_xfree(buffer);
  ssh_random_stir();
  if (kgc->have_prog_ind)
    printf("Stirred in %lu bytes.\n", (unsigned long) buffer_len);
}

/* Saves a private key with an optional extention */
int keygen_write_private_key(KeyGenCtx *kgc, SshPrivateKey key,
                             char *comment, char *ext)
{
  char newfilename[127];
  int retval;
  char *passphrase;
  char *empty_string = "";

  if (ext == NULL)
    ext = empty_string;

  /* make passphrase empty if none supplied */
  if (kgc->passphrase != NULL)
    passphrase = kgc->passphrase;
  else
    passphrase = empty_string;

  ssh_snprintf(newfilename, sizeof(newfilename),
               "%s%s",kgc->in_filename, ext);

  if (ssh_privkey_write(kgc->user, newfilename,
                        passphrase, comment, key,
                        NULL) == TRUE)
    {
      printf("Unable to write private key.\r\n");
      retval = 1;
    }
  else
    {
      printf("Successfully saved private key to %s\r\n",newfilename);
      retval = 0;
    }

  return retval;
}
































































































































































































































































































































































/* converts and saves public key to SSH2 format
   returns 1 on error, 0 otherwise */
int keygen_convert_public_key(KeyGenCtx *kgc)
{
  char *base_name;
  char newfilename[1024];
  char *comment = NULL;
  SshPublicKey pubkey = NULL;
  Boolean retval = TRUE;

  pubkey = ssh_pubkey_read(kgc->user, kgc->in_filename,
                           &comment, NULL);

  if (pubkey == NULL)
    {
      printf("Cannot open key %s\r\n",kgc->in_filename);
      return 1;
    }

  /* remove the ".pub" extension if it is there */
  if ((base_name = strstr(kgc->in_filename, ".pub")) != NULL)
    {
      base_name[0] = '\0';
    }

  /* name new key name_ssh2.pub */

  ssh_snprintf( newfilename, sizeof(newfilename),
                "%s_ssh2.pub",kgc->in_filename);

  if (ssh_pubkey_write(kgc->user, newfilename,
                       comment, pubkey, NULL) == TRUE)
    {
      /* pubkey write failed for some reason */
      printf("Unable to convert public key.\r\n");
      retval = 1;
    }
  else
    {
      printf("Successfully converted public key to %s\r\n",newfilename);
      retval = 0;
    }

  if (pubkey)
    ssh_public_key_free( pubkey );

  if (comment)
    ssh_xfree(comment);

  return retval;
}


int keygen_convert_private_key(KeyGenCtx *kgc)
{

  char *comment = NULL;
  SshPrivateKey prvkey = NULL;
  int retval = 1;

  /* maybe it's a private key */
  kgc->passphrase = ssh_read_passphrase("Passphrase : ", FALSE);
  if (ssh_privkey_read(kgc->user, kgc->in_filename, kgc->passphrase,
                       &comment, &prvkey) == SSH_PRIVKEY_KEY_UNREADABLE)
    {
      printf("Private key is unreadable.\r\n");
      retval = 1;
    }

  else if (prvkey != NULL)
    {
      retval = keygen_write_private_key(kgc, prvkey, comment, "_ssh2");
    }
  else
    {
      printf("Cannot determine the type of the key or "
             "passphrase incorrect.\r\n");
      retval = 1;
    }

  if (kgc->passphrase)
    ssh_xfree(kgc->passphrase);

  kgc->passphrase = NULL;

  if (comment)
    ssh_xfree(comment);

  if (prvkey)
    ssh_private_key_free( prvkey );

  return retval;
}


/* converts SSH1 keys to SSH2 format */
int keygen_convert_key(KeyGenCtx *kgc)
{
  Boolean ret;
  char *base_name;
  char newfilename[1024];

  if ((base_name = strstr(kgc->in_filename, ".pub")))
    {
      /* It's a public key */
      if ((ret = keygen_convert_public_key(kgc)) == 0)
        {
          /* pub key convert worked so try priv key as well */

          base_name[0] = '\0';
          keygen_convert_private_key(kgc);
        }
    }

  else
    {
      /* user supplied private key */
      if ((ret = keygen_convert_private_key(kgc)) == 0)
        {
          /* priv key convert worked so add .pub to name and
             try to convert pub key */
          ssh_snprintf(newfilename, sizeof(newfilename),
                       "%s.pub", kgc->in_filename);
          ssh_xfree(kgc->in_filename);

          kgc->in_filename = ssh_xstrdup(newfilename);
          keygen_convert_public_key(kgc);
        }
    }

  return 0;
}

/* Returns FALSE when successful, TRUE otherwise. */
Boolean keygen_edit_key(KeyGenCtx *kgc)
{
  SshPrivateKey seckey = NULL;
  SshPublicKey pubkey = NULL;
  char pubfn[1024], outpubfn[1024], pubbu[1024], secbu[1024];
  char *secfn, *outsecfn, *oldcomment = NULL, *newcomment = NULL;
  char *newpass = NULL;
  Boolean ok;
  Boolean no_public_key = FALSE;

  if (!(*(kgc->in_filename)))
    {
      ssh_warning("Invalid keyfile.");
      return TRUE;
    }
  else
    {
      secfn = kgc->in_filename;
      ssh_snprintf(pubfn, sizeof (pubfn), "%s.pub", secfn);
    }

  if (!(kgc->out_filename))
    {
      kgc->out_filename = ssh_xstrdup(kgc->in_filename);
    }

  outsecfn = kgc->out_filename;
  ssh_snprintf(outpubfn, sizeof (outpubfn), "%s.pub", outsecfn);

  ssh_snprintf(secbu, sizeof (secbu), "%s~", kgc->in_filename);
  ssh_snprintf(pubbu, sizeof (pubbu), "%s~", pubfn);
  (void)unlink(secbu);
  (void)unlink(pubbu);

  pubkey = ssh_pubkey_read(kgc->user, pubfn, &oldcomment, NULL);

  if (!pubkey)
    {
      /* no public key exists, but we can continue editing the private
         key anyway */
      ssh_warning("Cannot read public keyfile %s.", pubfn);
      no_public_key  = TRUE;
    }


  if (kgc->passphrase)
    {
      if (ssh_privkey_read(kgc->user, secfn, kgc->passphrase,
                           oldcomment ? NULL : &oldcomment,
                           &seckey) == SSH_PRIVKEY_KEY_UNREADABLE)
        {
          ssh_warning("Cannot read private keyfile %s.", secfn);
          return TRUE;
        }
    }

  if (!seckey)
    {
      if (ssh_privkey_read(kgc->user, secfn, "",
                           oldcomment ? NULL : &oldcomment,
                           &seckey) == SSH_PRIVKEY_KEY_UNREADABLE)
        {
          ssh_warning("Cannot read private keyfile %s.", secfn);
          return TRUE;
        }

      if (seckey)
        {
          keygen_free_str(kgc->passphrase);
          kgc->passphrase = ssh_xstrdup("");
        }
    }

  if (!oldcomment)
    oldcomment = ssh_xstrdup("");

  if (!kgc->pass_empty)
    {
      if (!seckey)
        {
          keygen_free_str(kgc->passphrase);
          printf("Passphrase needed for key \"%s\".\n", oldcomment);
          kgc->passphrase = ssh_read_passphrase("Passphrase : ", FALSE);
          if (kgc->passphrase)
            {
              if (ssh_privkey_read(kgc->user, secfn, kgc->passphrase, NULL,
                                   &seckey) == SSH_PRIVKEY_KEY_UNREADABLE)
                {
                  ssh_warning("Cannot read private keyfile %s.", secfn);
                  return TRUE;
                }
            }
        }
    }

  if (!seckey)
    {
      ssh_warning("Cannot read private keyfile %s.", secfn);
      return TRUE;
    }


  printf("Do you want to edit key \"%s\" ", oldcomment);
  if (!ssh_read_confirmation("(yes or no)? ", FALSE))
    {
      printf("Key unedited and unsaved.\n");
      keygen_free_str(oldcomment);
      return FALSE;
    }

  ok = FALSE;

  while (!ok)
    {
      if (!newcomment)
        newcomment = ssh_xstrdup(oldcomment);
      printf("Your key comment is \"%s\". ", newcomment);
      if (ssh_read_confirmation("Do you want to edit it (yes or no)? ", FALSE))
        {
          if (ssh_readline("New key comment: ", &newcomment,
                           fileno(stdin), fileno(stdout)) < 0)
            {
              fprintf(stderr, "Abort!  Key unedited and unsaved.\n");
              keygen_free_str(newpass);
              keygen_free_str(newcomment);
              keygen_free_str(oldcomment);
              return TRUE;
            }
          putchar('\n');
        }
      if (!newpass)
        newpass = ssh_xstrdup(kgc->passphrase);
      if (!kgc->pass_empty)
        {
          if (ssh_read_confirmation("Do you want to edit passphrase (yes "
                                    "or no)? ", FALSE))
            {
              keygen_free_str(newpass);
              newpass = ssh_read_passphrase_twice("New passphrase : ",
                                                  "Again          : ",
                                                  FALSE);
              if (!newpass)
                {
                  fprintf(stderr, "Abort!  Key unedited and unsaved.\n");
                  keygen_free_str(newcomment);
                  keygen_free_str(oldcomment);
                  return FALSE;
                }
            }
        }
      printf("Do you want to continue editing key \"%s\" ", newcomment);
      if (ssh_read_confirmation("(yes or no)? ", FALSE))
        ok = FALSE;
      else
        ok = TRUE;
    }

  if ((strcmp(newpass, kgc->passphrase) == 0) &&
      (strcmp(newcomment, oldcomment) == 0))
    {
      printf("Key unedited and unsaved.\n");
      keygen_free_str(newpass);
      keygen_free_str(newcomment);
      keygen_free_str(oldcomment);
      return FALSE;
    }
  printf("Do you want to save key \"%s\" to file %s ", newcomment, outsecfn);
  if (! ssh_read_confirmation("(yes or no)? ", FALSE))
    {
      printf("Key unsaved.\n");
      keygen_free_str(newpass);
      keygen_free_str(newcomment);
      keygen_free_str(oldcomment);
      return FALSE;
    }

  if (strcmp(outsecfn, kgc->in_filename) == 0)
    {
      if (rename(outsecfn, secbu) != 0)
        {
          ssh_warning("Unable to backup private key.");
          keygen_free_str(newpass);
          keygen_free_str(newcomment);
          keygen_free_str(oldcomment);
          fprintf(stderr, "Abort!\n");
          return TRUE;
        }
    }

  if (no_public_key == FALSE)
    {
      if (strcmp(outpubfn, pubfn) == 0)
        {
          if (rename(outpubfn, pubbu) != 0)
            {
              ssh_warning("Unable to backup public key.");
              if (strcmp(outsecfn, secfn) == 0)
                (void)rename(secbu, outsecfn);
              keygen_free_str(newpass);
              keygen_free_str(newcomment);
              keygen_free_str(oldcomment);
              fprintf(stderr, "Abort!\n");
              return TRUE;
            }
        }
    }

  if (ssh_privkey_write(kgc->user,
                        outsecfn,
                        newpass,
                        newcomment,
                        seckey,
                        NULL))
    {
      ssh_warning("Unable to write private key.!");
      if (strcmp(outsecfn, secfn) == 0)
        (void)rename(secbu, outsecfn);
      if (strcmp(outpubfn, pubfn) == 0)
        (void)rename(pubbu, outpubfn);
      keygen_free_str(newpass);
      keygen_free_str(newcomment);
      keygen_free_str(oldcomment);
      fprintf(stderr, "Abort!\n");
      return TRUE;
    }

  if (no_public_key == FALSE)
    {
      if (ssh_pubkey_write(kgc->user,
                           outpubfn,
                           newcomment,
                           pubkey,
                           NULL))
        {
          ssh_warning("Unable to write public key.!");
          unlink(outsecfn);
          if (strcmp(outsecfn, secfn) == 0)
            (void)rename(secbu, outsecfn);
          if (strcmp(outpubfn, pubfn) == 0)
            (void)rename(pubbu, outpubfn);
          keygen_free_str(newpass);
          keygen_free_str(newcomment);
          keygen_free_str(oldcomment);
          fprintf(stderr, "Abort!\n");
          return TRUE;
        }
    }

  keygen_free_str(newpass);
  keygen_free_str(newcomment);
  keygen_free_str(oldcomment);

  return FALSE;
}

Boolean keygen_dump_pubkey_fingerprint(KeyGenCtx *kgc, SshPublicKey pubkey)
{
  char *hash_name = "sha1";
  unsigned char *digest;
  SshHash hash_ctx;
  size_t digest_len;
  unsigned char *blob;
  size_t blob_len;
  char *fingerprint;

  if (!(blob_len = ssh_encode_pubkeyblob(pubkey, &blob)))
    {
      ssh_warning("keygen_dump_pubkey_fingerprint: Couldn't encode publickey "
                  "to blob.");
      return FALSE;
    }

  if (!ssh_hash_supported(hash_name))
    {
      ssh_warning("keygen_dump_pubkey_fingerprint: Hash \"%s\" not supported, "
                  "even though it is required in the draft. Contact "
                  "ssh2-bugs@ssh.com.", hash_name);
      return FALSE;
    }

  if (ssh_hash_allocate(hash_name, &hash_ctx) != SSH_CRYPTO_OK)
    {
      ssh_warning("keygen_dump_pubkey_fingerprint: Couldn't initialize hash "
                  "function.", hash_name);
      return FALSE;
    }

  digest_len = ssh_hash_digest_length(hash_ctx);
  digest = ssh_xcalloc(digest_len, sizeof(unsigned char));
  ssh_hash_update(hash_ctx, blob, blob_len);
  ssh_hash_final(hash_ctx, digest);
  ssh_hash_free(hash_ctx);

  fingerprint =
    ssh_fingerprint(digest, digest_len, kgc->fingerprint_type);

  fprintf(stderr, "Fingerprint for key:\r\n");
  printf("%s\r\n", fingerprint);

  ssh_xfree(blob);
  ssh_xfree(fingerprint);
  ssh_xfree(digest);
  return TRUE;
}

void keygen_fatal_cb(const char *message, void *context)
{
  fprintf(stderr, "%s: FATAL: %s\r\n", ssh_get_program_name(), message);
  exit(255);
}

/* main for ssh-keygen */

int main(int argc, char **argv)
{
  int ch, i;
  KeyGenCtx *kgc;
  char *t;
  SshTime now;
  int r = 0, rr = 0;
  char *progname;

  ssh_debug_register_callbacks(keygen_fatal_cb, NULL, NULL, NULL);

  /* check for --help before anything */
  if (argc > 1)
    if (strcmp(argv[1], "--help") == 0)
      {
        printf("%s", keygen_helptext);
        exit(0);
      }

  if (strchr(argv[0], '/'))
    progname = strrchr(argv[0], '/') + 1;
  else
    progname = argv[0];

#ifdef SSHDIST_CRYPT_RSA
  ssh_pk_provider_register(&ssh_pk_if_modn_generator);
#endif /* SSHDIST_CRYPT_RSA */
#ifdef SSHDIST_CRYPT_DSA
  ssh_pk_provider_register(&ssh_pk_dl_modp_generator);
#endif /* SSHDIST_CRYPT_DSA */

  /* Initialize the context */
  kgc = keygen_init_ctx();


  ssh_signals_prevent_core(FALSE, kgc);




  /* try to find the user's home directory */

  ssh_register_program_name(progname);
  
  while ((ch = ssh_getopt(argc, argv,
                          "vhqr:?P1:7:x:t:b:p:i:D:B:e:c:VF:k:",
                          NULL))
         != EOF)
    {
      if (!ssh_optval)
        {
          printf("%s", keygen_helptext);
          keygen_free_ctx(kgc);
          exit(1);
        }
      switch (ch)
        {
          /* -b: specify the strength of the key */

        case 'b':
          if (kgc->keybits != 0)
            keygen_error(kgc, "Multiple -b parameters");
          kgc->keybits = atoi(ssh_optarg);
          if (kgc->keybits < 512 || kgc->keybits > 0x10000)
            keygen_error(kgc, "Illegal key size.");
          break;

          /* -t: specify the key type. */
        case 't':
          /* the kgc->keytype gets the crypto library name of the alg. */

          if (kgc->keytype != NULL)
            keygen_error(kgc, "multiple key types specified.");

          for (i = 0; keygen_common_names[i][0] != NULL; i++)
            {
              if (strcasecmp(keygen_common_names[i][0], ssh_optarg) == 0)
                {
                  kgc->keytypecommon = ssh_xstrdup(keygen_common_names[i][0]);
                  kgc->keytype = ssh_xstrdup(keygen_common_names[i][1]);
                  break;
                }
            }
          if (keygen_common_names[i][0] == NULL)
            keygen_error(kgc, "unknown key type.");
          break;

          /* -c: Comment string. */
        case 'c':
          kgc->comment = ssh_xstrdup(ssh_optarg);
          break;

          /* -e: Edit key file. */
        case 'e':
          kgc->edit_key = TRUE;
          kgc->in_filename = ssh_xstrdup(ssh_optarg);

          /* -p: Provide passphrase. */
        case 'p':
          kgc->passphrase = ssh_xstrdup(ssh_optarg);
          break;

          /* -P: Don't provide passphrase. */
        case 'P':
          kgc->pass_empty = TRUE;
          keygen_free_str(kgc->passphrase);
          kgc->passphrase = NULL;
          break;

          /* -1: Convert a key from SSH1 format to SSH2 format. */
        case '1':
          kgc->convert = TRUE;
          kgc->in_filename = ssh_xstrdup(ssh_optarg);
          break;



















          /* -V: print the version number */
        case 'V':
          printf("%s version " SSH2_VERSION
                 ", compiled "__DATE__".\n", progname);



          /* XXX more stuff here possibly */
          keygen_free_ctx(kgc);
          exit(0);

        case '?':
        case 'h':
          printf("%s", keygen_helptext);
          keygen_free_ctx(kgc);
          exit(0);

          /* -q: supress the progress indicator */
        case 'q':
          kgc->have_prog_ind = FALSE;
          break;

          /* -r: stir in data from file to the random pool */
        case 'r':
          kgc->read_file = TRUE;
          kgc->entropy_file_name = ssh_xstrdup(ssh_optarg);
          break;

          /* XXX Following are experimental options. */

          /* -i: display (all) information about a key */
        case 'i':
          kgc->info_file = ssh_xstrdup(ssh_optarg);
          break;
        case 'D':
          kgc->derive_file = ssh_xstrdup(ssh_optarg);
          break;
        case 'B':
          kgc->base = atoi(ssh_optarg);
          break;
        case 'F':
          kgc->dump_fingerprint = TRUE;
          ssh_xfree(kgc->info_file);
          kgc->info_file = ssh_xstrdup(ssh_optarg);
          break;
        default:
          keygen_error(kgc, "Invalid option.");
          break;
        }
    }

























  /* Stir in random data from file, if requested */
  if (kgc->read_file)
    {
      stir_from_file(kgc);
    }
  else
    {
      ssh_randseed_open(kgc->user, NULL);
    }

  /* Dump fingerprint */
  if (kgc->dump_fingerprint)
    {
      SshPublicKey pubkey = NULL;

      if (!kgc->info_file)
        ssh_fatal("-F option needs public key file as argument.");

      pubkey = ssh_pubkey_read(kgc->user, kgc->info_file,
                               NULL, NULL);
      if (pubkey != NULL)
        {
          if (keygen_dump_pubkey_fingerprint(kgc, pubkey))
            r = 0;
          else
            r = 1;

          goto finish;
        }
      else
        {
          ssh_fatal("Couldn't read public key \"%s\".", kgc->info_file);
        }
    }

  /* Perhaps we should derive a key? */
  if (kgc->derive_file)
    {
      SshPrivateKey prvkey = NULL;
      SshPublicKey  pubkey = NULL;
      char buf[256];

      pubkey = ssh_pubkey_read(kgc->user, kgc->derive_file,
                               NULL, NULL);
      if (pubkey != NULL)
        {
          /* Attempt at deriving the private key? */
          printf("\n  Derivation of a private key from the public key.\n"
                 "   Please wait...\n");

          sleep(1);

          printf("\n  Just kidding,\n"
                 "   cryptographically strong keys cannot be broken!\n");

          pubkey_print_estimate(pubkey);

          goto finish;
        }

      /* Perhaps its a private key? */
      prvkey = keygen_read_privkey(kgc, kgc->derive_file);

      if (prvkey != NULL)
        {
          /* Derive the appropriate public key. */
          ssh_snprintf(buf, sizeof (buf), "%s.pub", kgc->derive_file);
          pubkey = ssh_private_key_derive_public_key(prvkey);
          if (pubkey == NULL)
            {
              keygen_error(kgc, "Could not derive public key from private key.");
            }

          if (ssh_pubkey_write(kgc->user, buf, kgc->comment, pubkey, NULL))
            {
              keygen_error(kgc, "Public key not written !");
            }
          printf("Public key saved to %s\n", buf);

          ssh_public_key_free(pubkey);
          ssh_private_key_free(prvkey);
          goto finish;
        }

      keygen_error(kgc, "Cannot determine the type of the key.");
    }

  if (kgc->info_file)
    {
      SshPrivateKey prvkey = NULL;
      SshPublicKey  pubkey = NULL;

      pubkey = ssh_pubkey_read(kgc->user, kgc->info_file,
                               NULL, NULL);
      if (pubkey != NULL)
        {
          keygen_view_pubkey(kgc, pubkey);
          goto finish;
        }

      /* Perhaps its a private key? */
      prvkey = keygen_read_privkey(kgc, kgc->info_file);

      if (prvkey != NULL)
        {
          keygen_view_prvkey(kgc, prvkey);
          goto finish;
        }

      keygen_error(kgc, "Cannot determine the type of the key.");
    }

  if (kgc->convert)
    {
      r = keygen_convert_key(kgc);
    }














  else if (kgc->edit_key)
    {
      r = keygen_edit_key(kgc);
    }
  else
    {
      kgc->newkey = TRUE;

      if (kgc->keybits == 0)
        {
          kgc->keybits = KEYGEN_ASSUMED_PUBKEY_LEN;
        }

      if (kgc->keytype == NULL)
        {
          kgc->keytypecommon = ssh_xstrdup(keygen_common_names[0][0]);
          kgc->keytype = ssh_xstrdup(keygen_common_names[0][1]);
        }

      if (kgc->comment == NULL)
        {
          char *time_str;
          const char *username;

          struct passwd *pw;
          pw = getpwuid(getuid());
          if (!pw)
            keygen_error(kgc, "Could not get user's password structure.");
          username = pw->pw_name;
          t = ssh_xmalloc(64);
          gethostname(t, 64);





          kgc->comment = ssh_xmalloc(256);
          now = ssh_time();
          time_str = ssh_readable_time_string(now, TRUE);
          ssh_snprintf(kgc->comment, 256, "%d-bit %s, %s@%s, %s",
                       kgc->keybits, kgc->keytypecommon,
                       username, t, time_str);
          ssh_xfree(time_str);
          ssh_xfree(t);
        }

      if (ssh_optind >= argc)
        {
          /* generate single key. if no file names given, make up one. */
          if (kgc->out_filename == NULL)
            keygen_choose_filename(kgc);
          r = keygen_keygen(kgc);
        }
      else
        {
          if (kgc->out_filename != NULL)
            keygen_keygen(kgc);

          /* iterate over additional filenames */

          for (i = ssh_optind; i < argc; i++)
            {
              if (kgc->out_filename != NULL)
                ssh_xfree(kgc->out_filename);
              kgc->out_filename = ssh_xstrdup(argv[i]);
              rr = keygen_keygen(kgc);
              if (rr != 0)
                {
                if (r == 0)
                  r = rr;
                else
                  r = -1;
                }
            }
        }
    }

finish:

  keygen_free_ctx(kgc);

  return r;
}















































/*
 *  $Log: ssh-keygen2.c,v $ *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  
 *  $EndLog$
 */
