/*

  ssh1bufferstream.c

  Author:
        Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Stream object with which write never blocks.

*/

#include "sshincludes.h"
#include "ssh2includes.h"
#ifdef WITH_INTERNAL_SSH1_EMULATION
#include "sshbuffer.h"
#include "ssh1bufferstream.h"

typedef struct {
  SshStream stream;
  SshBufferStruct buffer[1];
  SshStreamCallback stream_callback;
  void *stream_context;
  Ssh1BufferStreamFlushedCallback flushed_callback;
  void *flushed_context;
  Boolean disconnected;
  Boolean destroyed;
  Boolean in_callback;
} Ssh1BufferStreamRec, *Ssh1BufferStream;

extern const SshStreamMethodsStruct ssh1_buffer_methods;

void ssh1_stream_callback(SshStreamNotification notification,
                          void *context);
void ssh1_buffer_stream_destroy(void *context);

SshStream ssh1_buffer_stream_wrap(SshStream stream)
{
  Ssh1BufferStream buffer_stream;

  buffer_stream = ssh_xcalloc(1, sizeof(*buffer_stream));
  buffer_stream->stream = stream;
  ssh_buffer_init(buffer_stream->buffer);
  ssh_stream_set_callback(buffer_stream->stream,
                          ssh1_stream_callback,
                          (void *)buffer_stream);
  return ssh_stream_create((SshStreamMethods)&ssh1_buffer_methods,
                           (void *)buffer_stream);
}

void ssh1_buffer_stream_wait_flush(SshStream stream,
                                   Ssh1BufferStreamFlushedCallback callback,
                                   void *context)
{
  Ssh1BufferStream buffer_stream;

  if (ssh_stream_get_methods(stream) != (void *)&ssh1_buffer_methods)
    ssh_fatal("ssh_serial_stream_params: not a buffer stream");
  buffer_stream = ssh_stream_get_context(stream);
  buffer_stream->flushed_callback = callback;
  buffer_stream->flushed_context = context;
  if ((ssh_buffer_len(buffer_stream->buffer) == 0) &&
      (buffer_stream->flushed_callback))
    {
      buffer_stream->in_callback = TRUE;
      (*buffer_stream->flushed_callback)(TRUE, 
                                         buffer_stream->flushed_context);
      buffer_stream->in_callback = FALSE;
      buffer_stream->flushed_callback = NULL;
      buffer_stream->flushed_context = NULL;
    }
  else if (buffer_stream->disconnected)
    {
      buffer_stream->in_callback = TRUE;
      (*buffer_stream->flushed_callback)(FALSE, 
                                         buffer_stream->flushed_context);
      buffer_stream->in_callback = FALSE;
      buffer_stream->flushed_callback = NULL;
      buffer_stream->flushed_context = NULL;
    }
  if (buffer_stream->destroyed)
    ssh1_buffer_stream_destroy(buffer_stream);
  return;
}

void ssh1_buffer_stream_flush_write_buffer(Ssh1BufferStream buffer_stream)
{
  int r;
  
  while (ssh_buffer_len(buffer_stream->buffer) > 0)
    {
      r = ssh_stream_write(buffer_stream->stream,
                           ssh_buffer_ptr(buffer_stream->buffer),
                           ssh_buffer_len(buffer_stream->buffer));
      if (r > 0)
        {
          ssh_buffer_consume(buffer_stream->buffer, r);
        }
      else if (r < 0)
        {
          break;
        }
      else
        {
          buffer_stream->disconnected = TRUE;
          if ((ssh_buffer_len(buffer_stream->buffer) == 0) &&
              (buffer_stream->flushed_callback))
            {
              buffer_stream->in_callback = TRUE;
              (*buffer_stream->flushed_callback)(
                                            TRUE, 
                                            buffer_stream->flushed_context);
              buffer_stream->in_callback = FALSE;
              buffer_stream->flushed_callback = NULL;
              buffer_stream->flushed_context = NULL;
            }
          break;
        }
      if ((ssh_buffer_len(buffer_stream->buffer) == 0) &&
          (buffer_stream->flushed_callback))
        {
          buffer_stream->in_callback = TRUE;
          (*buffer_stream->flushed_callback)(TRUE, 
                                             buffer_stream->flushed_context);
          buffer_stream->in_callback = FALSE;
          buffer_stream->flushed_callback = NULL;
          buffer_stream->flushed_context = NULL;
        }
    }
  if (buffer_stream->destroyed)
    ssh1_buffer_stream_destroy(buffer_stream);
  return;
}

void ssh1_stream_callback(SshStreamNotification notification,
                          void *context)
{
  Ssh1BufferStream buffer_stream = (Ssh1BufferStream)context;

  switch (notification)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      buffer_stream->in_callback = TRUE;
      if (buffer_stream->stream_callback)
        (*buffer_stream->stream_callback)(notification,
                                          buffer_stream->stream_context);
      buffer_stream->in_callback = FALSE;
      break;

    case SSH_STREAM_CAN_OUTPUT:
      /* This never goes to the upper level, because upper 
         level write never blocks. */
      ssh1_buffer_stream_flush_write_buffer(buffer_stream);
      break;

    case SSH_STREAM_DISCONNECTED:
      buffer_stream->disconnected = TRUE;
      buffer_stream->in_callback = TRUE;
      if (buffer_stream->stream_callback)
        (*buffer_stream->stream_callback)(notification,
                                          buffer_stream->stream_context);
      buffer_stream->in_callback = FALSE;
      break;

    default:
      /* We assume that future notification types can be passed to the
         upper level directly. */
      buffer_stream->in_callback = TRUE;
      if (buffer_stream->stream_callback)
        (*buffer_stream->stream_callback)(notification,
                                          buffer_stream->stream_context);
      buffer_stream->in_callback = FALSE;
    }
  if (buffer_stream->destroyed)
    ssh1_buffer_stream_destroy(buffer_stream);
  return;
}

int ssh1_buffer_stream_read(void *context, unsigned char *buf, size_t size)
{
  Ssh1BufferStream buffer_stream = (Ssh1BufferStream)context;

  return ssh_stream_read(buffer_stream->stream, 
                         buf,
                         size);
}

int ssh1_buffer_stream_write(void *context, 
                             const unsigned char *buf,
                             size_t size)
{
  Ssh1BufferStream buffer_stream = (Ssh1BufferStream)context;

  if (buffer_stream->disconnected)
    return 0;
  ssh_xbuffer_append(buffer_stream->buffer, buf, size);
  ssh1_buffer_stream_flush_write_buffer(buffer_stream);
  return size;
}

void ssh1_buffer_stream_output_eof(void *context)
{
  Ssh1BufferStream buffer_stream = (Ssh1BufferStream)context;

  ssh_stream_output_eof(buffer_stream->stream);
}

void ssh1_buffer_stream_set_callback(void *context, 
                                     SshStreamCallback callback,
                                     void *callback_context)
{
  Ssh1BufferStream buffer_stream = (Ssh1BufferStream)context;

  buffer_stream->stream_callback = callback;
  buffer_stream->stream_context = callback_context;
}

void ssh1_buffer_stream_destroy(void *context)
{
  Ssh1BufferStream buffer_stream = (Ssh1BufferStream)context;

  if (buffer_stream->in_callback)
    {
      buffer_stream->destroyed = TRUE;
      return;
    }
  ssh_stream_destroy(buffer_stream->stream);
  ssh_buffer_uninit(buffer_stream->buffer);
  ssh_xfree(buffer_stream);
}

const SshStreamMethodsStruct ssh1_buffer_methods = {
  ssh1_buffer_stream_read,
  ssh1_buffer_stream_write,
  ssh1_buffer_stream_output_eof,
  ssh1_buffer_stream_set_callback,
  ssh1_buffer_stream_destroy
};

#endif /* WITH_INTERNAL_SSH1_EMULATION */
