/*

  ssh1proto.c

  Author:
        Timo J. Rinne <tri@ssh.com>
        Tomi Mickelsson <tomi@ssh.com> (Windows specific code)

  Copyright (C) 1999-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Ssh1 protocol implementation.

*/
#include "ssh2includes.h"

#ifdef WITH_INTERNAL_SSH1_EMULATION
#include "sshencode.h"
#include "sshkeyfile.h"
#include "ssh1encode.h"
#include "sshkeyblob1.h"
#include "ssh1proto.h"
#include "ssh1packet.h"
#include "sshclient.h"
#include "sshappcommon.h"
#include "sshstream.h"
#include "sshbuffer.h"
#include "sshcrypt.h"
#include "bufzip.h"
#include "sshcrc32.h"
#include "readpass.h"
#include "sshdsprintf.h"
#include "ssheloop.h"
#include "sshtimeouts.h"
#include "sshfilterstream.h"
#include "sshfingerprint.h"

#include "sshtty.h"
#include "sshttyflags.h"

#include "sshagent.h"
#include "sshagentint.h"
#include "sshuserfiles.h"
#include "sshsnlist.h"
#include "sshreadline.h"
#ifdef WITH_PGP
#include "sshpgp.h"
#include "ssh2pgp.h"
#endif /* WITH_PGP */
#include "sshtcp.h"
#include "ssh1bufferstream.h"











/* These headers are needed for Unix domain sockets. */
#if defined(XAUTH_PATH) || defined(HPUX_NONSTANDARD_X11_KLUDGE)
#include <sys/socket.h>
#include <netinet/in.h>
#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#else /* Some old linux systems at least have in_system.h instead. */
#include <netinet/in_system.h>
#endif /* HAVE_NETINET_IN_SYSTM_H */
#if !defined(__PARAGON__)
#include <netinet/ip.h>
#endif /* !__PARAGON__ */
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif /* HAVE_SYS_SELECT_H */
#ifdef HAVE_SYS_UTSNAME_H
#include <sys/utsname.h>
#endif /* HAVE_SYS_UTSNAME_H */
#endif /* XAUTH_PATH || HPUX_NONSTANDARD_X11_KLUDGE */

#define SSH_DEBUG_MODULE "Ssh1Protocol"

#define SSH1_PROTOCOL_ID_STRING "SSH-%d.%d-" SSH2_VERSION " (compat mode)\n"
#define SSH1_DEFAULT_TERMINAL_TYPE "vt100"
#define SSH1_CAN_ASK_PASSWORD 1
#define SSH1_SUPPORTED_AUTHENTICATIONS_MASK                                 \
                              ((((SshUInt32)1) << SSH1_AUTH_PASSWORD) |     \
                               (((SshUInt32)1) << SSH1_AUTH_RSA))
#define SSH1_MAXIMUM_TTY_BUFFER 0x10000

#ifndef X11_DIR
#define X11_DIR "/tmp/.X11-unix"
#endif /* !X11_DIR */
#define X11_MAX_AUTH_PACKET_SIZE 10000

typedef struct Ssh1ClientRec *Ssh1Client;

typedef enum {
  SSH1_MODE_NONE = 0,
  SSH1_MODE_VERSION_SENT,
  SSH1_MODE_SESSION_KEY_SENT,
  SSH1_MODE_USER_NAME_SENT,
  SSH1_PASSWORD_SENT,
  SSH1_RSA_AUTHENTICATION_REQUESTED,
  SSH1_AGENT_AUTHENTICATION_REQUESTED,
  SSH1_AGENT_OPERATION_IN_PROGRESS,
  SSH1_RSA_AUTHENTICATION_SENT,
  SSH1_MODE_AUTHENTICATION_COMPLETE,
  SSH1_MODE_COMPRESSION_REQUESTED,
  SSH1_MODE_REMOTE_FORWARD_REQUESTED,
  SSH1_MODE_AGENT_FORWARD_REQUESTED,
  SSH1_MODE_X11_FORWARD_REQUESTED,
  SSH1_MODE_PTY_REQUESTED,
  SSH1_MODE_INTERACTIVE,
  SSH1_MODE_INTERACTIVE_AFTER_EOF,
  SSH1_MODE_DESTROYING
} Ssh1ClientMode;

typedef enum {
  SSH1_AUTH_TYPE_PASSWORD,
  SSH1_AUTH_TYPE_AGENT,
#ifdef WITH_PGP
  SSH1_AUTH_TYPE_PGP_KEY,
#endif /* WITH_PGP */
  SSH1_AUTH_TYPE_RSA_FILE
} Ssh1AuthType;

typedef struct Ssh1AuthTokenRec *Ssh1AuthToken;

struct Ssh1AuthTokenRec {
  Ssh1AuthToken next;
  Ssh1AuthType auth_type;
  char *password;
  char *filename;
#ifdef WITH_PGP
  char *pgp_keyring;
  char *pgp_name;
  char *pgp_fingerprint;
  SshUInt32 pgp_id;
  unsigned char *pgp_keyblob;
  size_t pgp_keyblob_len;
#endif /* WITH_PGP */
  SshAgent agent;
  unsigned char *certs;
  size_t certs_len;
  char *comment;
};

typedef struct Ssh1LocalTcpListener {
  struct Ssh1LocalTcpListener *next;
  Ssh1Client ssh1;
  SshTcpListener listener;
  SshForward forward;
} Ssh1LocalTcpListenerRec, *Ssh1LocalTcpListener;

typedef struct Ssh1RemoteTcpListener {
  struct Ssh1RemoteTcpListener *next;
  SshForward forward;
} Ssh1RemoteTcpListenerRec, *Ssh1RemoteTcpListener;

typedef struct Ssh1RemoteChannel {
  struct Ssh1RemoteChannel *next;
  Ssh1Client ssh1;
  SshUInt32 remote_id;
  SshUInt32 local_id;
  char *host;
  char *host_info;
  char *port;
  char *originator;
  SshOperationHandle open_handle; /* non-null only if open is in progress */
  SshStream stream;
  Boolean output_closed;
  Boolean input_closed;
  Boolean eof_received;
  Boolean eof_sent;
  Boolean dying;
  Boolean forming;
} Ssh1RemoteChannelRec, *Ssh1RemoteChannel;

struct Ssh1ClientRec {
  SshOperationHandle op;       /* Operation handle of the ssh1 emulation */
  Boolean aborted;             /* True if received abort from the user. */
  Ssh1ClientMode mode;         /* State of the ssh1 protocol engine */
  Ssh1ClientMode prev_mode;    /* Previous state, if relevant */
  SshClient client;            /* Ssh2 configuration data */
  SshForward requested_remote_forward; /* Pointer to remote forward
                                          that was last requested */
  SshForward requested_local_forward;  /* Pointer to remote forward
                                          that was last initialized */
  SshStream stream;            /* Transport stream */
  SshCipher in_cipher;         /* Inbound traffic cipher context */
  SshCipher in_cipher2;        /* Used only in ssh1 inner mode CBC type 3DES */
  SshCipher in_cipher3;        /* Used only in ssh1 inner mode CBC type 3DES */
  SshCompression in_compress;  /* Compression context */
  SshCipher out_cipher;        /* Inbound traffic cipher context */
  SshCipher out_cipher2;       /* Used only in ssh1 inner mode CBC type 3DES */
  SshCipher out_cipher3;       /* Used only in ssh1 inner mode CBC type 3DES */
  SshCompression out_compress; /* Compression context */
  Boolean have_forwarding;     /* TRUE if we can make forwards. */
  Ssh1LocalTcpListener local_tcp_listeners; /* Local forward listeners
                                               that are active. */
  Ssh1RemoteTcpListener remote_tcp_listeners; /* Remote forward listeners
                                                 that are active. */
  Ssh1RemoteChannel remote_channels; /* Channels opened from remote */
  SshUInt32 last_allocated_local_id;
  Boolean agent_forwarding_on;
  Boolean x11_forwarding_on;
  unsigned char *x11_real_auth_protocol;
  size_t x11_real_auth_protocol_len;
  unsigned char *x11_fake_auth_protocol;
  size_t x11_fake_auth_protocol_len;
  unsigned char *x11_fake_data;
  size_t x11_fake_data_len;
  char *x11_fake_data_hex;
  unsigned char *x11_real_data;
  size_t x11_real_data_len;
  SshUInt32 x11_screen_number;

  Boolean have_pty;            /* Have we succesfully allocated a pty? */
  unsigned char *input_buf;    /* Buffer for partial packet */
  size_t input_offset;         /* Bytes stored in input_buf */
  size_t input_len;            /* Decoded input length. */
  size_t input_size;           /* Size of the input buffer.  0 is a
                                  special case indicating 4 byte
                                  buffer in state where we are reading
                                  the packet length.  If input size is
                                  greater than zero, it is indicating
                                  the total amount of bytes (+
                                  input_offset) that is needed before
                                  we have a complete packet. */
  SshBufferStruct output_buf[1];     /* Buffer of characters to be sent to
                                  the server when the server can
                                  receive. */
  SshBufferStruct stdout_buf[1];     /* Data waiting to be written to the
                                  terminal. */
  SshBufferStruct stderr_buf[1];     /* Error data waiting to be written to
                                        the terminal. */
  SshBufferStruct stdin_pre_buf[1];  /* Buffer for data read from stdin
                                  before entering interactive mode session. */
  Boolean tty_saturated;       /* Have we put the protocol conversion to
                                  ``wait'' until we have managed to flush
                                  some output to the tty. */
  Boolean have_stdin_eof;      /* This is true if eof has arrived from the
                                  stdin before entering interactive mode. */
  SshAgent agent;              /* Connection to authentication agent. */
  Ssh1AuthToken auth_list;     /* List of available authentication methods. */
  Ssh1AuthToken auth_last;     /* Last available authentication method. */
  Ssh1AuthToken auth_current;  /* Currently active authentication method. */
  SshUInt32 auth_round;        /* Round number of auth list passes. */
  Boolean begin_authentication_after_adding_auth_tokens;

  /* This information received from the remote server in the beginning
     of the protocol conversation. */
  SshPublicKey host_key;       /* Server side host key. */
  SshPublicKey server_key;     /* Server side server key. */
  unsigned char anti_spoofing_cookie[8];
  SshUInt32 protocol_flags;
  SshUInt32 supported_ciphers_mask;
  SshUInt32 supported_authentications_mask;
  int remote_server_major_protocol_version; /* 1 */
  int remote_server_minor_protocol_version; /* 0 - 5 */
  int remote_server_protocol_version;       /* 10 - 15 */

  /* Statistics of the ongoing connection */
  SshTransportStatisticsStruct statistics;

  /* Session key generated by the client */
  unsigned char session_id[16];
  unsigned char session_key[32];
  SshMPIntStruct encrypted_session_key;
  int chosen_cipher;

  /* Callback that is called, after ssh1 protocol context is destroyed
     and protocol is disconnected.  Before this call a wrapped stream
     is destroyed and all callbacks that possibly has been set by the
     ssh1 protocol are cancelled. */
  Ssh1DestroyCallback destroy_callback;
















  SshUInt16 *crc_deattack_h;
  SshUInt32 crc_deattack_n;

  Boolean remote_server_allows_ignore_packets_in_startup;

  /* Context for all N callbacks above, NOT JUST for destroy_callback */
  void *destroy_context;
  SshUInt32 exit_value;
};

Boolean ssh1_session_id(Ssh1Client ssh1, unsigned char id[16]);
void ssh1_stdio_event(SshStreamNotification event, void *context);
void ssh1_stderr_event(SshStreamNotification event, void *context);
void ssh1_stdin_available(Ssh1Client ssh1);
void ssh1_stdin_available_timeout(void *context);
void ssh1_stream_event(SshStreamNotification event, void *context);
Boolean ssh1_output_available(Ssh1Client ssh1);
void ssh1_input_available_timeout(void *context);
Boolean ssh1_input_available(Ssh1Client ssh1);
Boolean ssh1_send_buffer(SshBuffer buffer, Ssh1Client ssh1);
void ssh1_protocol_error(const char *msg, Ssh1Client ssh1);
void ssh1_protocol_destroy(Ssh1Client ssh1, Boolean immediate);
void ssh1_protocol_abort(void *context); /* Abort callback */
void ssh1_exit_timeout(void *context);
Boolean ssh1_flush_stdout_and_stderr(Ssh1Client ssh1);
Boolean ssh1_flush_stdout(Ssh1Client ssh1);
Boolean ssh1_flush_stderr(Ssh1Client ssh1);
void ssh1_add_auth_tokens(Boolean begin_authentication, Ssh1Client ssh1);
void ssh1_add_auth_tokens_completion(Ssh1Client ssh1);
Boolean ssh1_host_key_check(Ssh1Client ssh1);
void ssh1_host_key_write(Ssh1Client ssh1, const char *filename);
void CBHostKeyResult(Boolean result, void *result_context);
void ssh1_local_forward_start(SshIpError error,
                              SshStream stream,
                              void *context);

/* Authentication token handling. */
void ssh1_add_password_auth_token(const char *password, Ssh1Client ssh1);
void ssh1_add_rsa_file_auth_token(const char *filename, Ssh1Client ssh1);
void ssh1_add_pgp_key_auth_token(const char *pgp_keyring,
                                 const char *pgp_name,
                                 const char *pgp_fingerprint,
                                 SshUInt32 pgp_id,
                                 Ssh1Client ssh1);
void ssh1_add_agent_auth_token(SshAgent agent,
                               const unsigned char *certs,
                               size_t certs_len,
                               Ssh1Client ssh1);
void ssh1_add_auth_token(Ssh1AuthToken tok, Ssh1Client ssh1);
void ssh1_flush_auth_tokens(Ssh1Client ssh1);
void ssh1_try_auth_method(Ssh1Client ssh1);
int ssh1_canonical_cipher_name_to_number(const char *cipher_name);
int ssh1_choose_cipher(SshUInt32 server_cipher_mask,
                       const char *config_cipher_list);
void ssh1_buffer_reverse_byte_order(unsigned char *buf, size_t buflen);
void ssh1_agent_open_done(SshAgent agent, void *context);
void ssh1_agent_list_done(SshAgentError error,
                          unsigned int num_keys,
                          SshAgentKeyInfo keys,
                          void *context);
void ssh1_agent_op_done(SshAgentError error,
                        const unsigned char *result,
                        size_t result_len,
                        void *context);
void ssh1_authenticated(Ssh1Client ssh1, Boolean result);
void ssh1_uninit_local_forward(Ssh1Client ssh1);

/* Preparation mode steps. */
void ssh1_request_compression(Ssh1Client ssh1);
void ssh1_request_pty(Ssh1Client ssh1);
void ssh1_request_remote_forward(Ssh1Client ssh1);
void ssh1_request_agent_forward(Ssh1Client ssh1);
void ssh1_request_x11_forward(Ssh1Client ssh1);
void ssh1_init_local_forward(Ssh1Client ssh1);
void ssh1_request_command(Ssh1Client ssh1);
Boolean ssh1_send_random_ignore_packet(SshUInt32 len,
                                       Ssh1Client ssh1);

/* Packet handlers.  These ones uninit the packet buffer. */
void ssh1_handle_packet_msg_disconnect(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_smsg_public_key(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_smsg_auth_rsa_challenge(SshBuffer packet,
                                                Ssh1Client ssh1);
void ssh1_handle_packet_smsg_success(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_smsg_failure(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_smsg_stdout_data(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_smsg_stderr_data(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_smsg_exitstatus(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_msg_channel_open_confirmation(SshBuffer packet,
                                                      Ssh1Client ssh1);
void ssh1_handle_packet_msg_channel_open_failure(SshBuffer packet,
                                                 Ssh1Client ssh1);
void ssh1_handle_packet_msg_channel_data(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_msg_channel_input_eof(SshBuffer packet,
                                              Ssh1Client ssh1);
void ssh1_handle_packet_msg_channel_output_closed(SshBuffer packet,
                                                  Ssh1Client ssh1);
void ssh1_handle_packet_smsg_x11_open(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_msg_port_open(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_smsg_agent_open(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_msg_ignore(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_msg_debug(SshBuffer packet, Ssh1Client ssh1);
void ssh1_handle_packet_smsg_auth_tis_challenge(SshBuffer packet,
                                                Ssh1Client ssh1);
void ssh1_handle_packet_smsg_auth_kerberos_response(SshBuffer packet,
                                                    Ssh1Client ssh1);
void ssh1_send_window_size_change(int xcharacters,
                                  int ycharacters,
                                  int xpixels,
                                  int ypixels,
                                  Ssh1Client ssh1);

/* Channel stuff */
SshUInt32 ssh1_allocate_local_channel_id(Ssh1Client ssh1);
void ssh1_remote_channel_open(SshIpError error,
                              SshStream stream,
                              void *context);
void ssh1_remote_agent_channel_open(SshStream stream,
                                    void *context);
void ssh1_delete_remote_channel(Ssh1RemoteChannel remote_channel);
void ssh1_delete_remote_channel_if_completed(
                                         Ssh1RemoteChannel remote_channel);
Ssh1RemoteChannel ssh1_find_remote_channel_with_local_id(Ssh1Client ssh1,
                                                        SshUInt32 channel_id);
void ssh1_remote_channel_flush(Ssh1RemoteChannel remote_channel);
void ssh1_remote_channel_stream_callback(SshStreamNotification notification,
                                         void *context);
void ssh1_init_x11_auth_data(Ssh1Client ssh1, const char *display);
void ssh1_remote_x11_channel_open_tcp(SshIpError error,
                                      SshStream stream,
                                      void *context);
void ssh1_remote_x11_channel_open(SshStream stream,
                                  void *context);
void ssh1_x11_channel_filter(void *context,
                             SshFilterGetCB get_data,
                             SshFilterCompletionCB completed,
                             void *internal_context);
void ssh1_x11_channel_filter_destroy(void *context);

Boolean ssh1_crc_compensation_attack_detect(Ssh1Client ssh1,
                                            unsigned char *buf,
                                            size_t len,
                                            unsigned char *iv);
void ssh1_crc_compensation_attack_init(Ssh1Client ssh1);
void ssh1_crc_compensation_attack_uninit(Ssh1Client ssh1);
void ssh1_crc_compensation_attack_crc_update(SshUInt32 *a,
                                             SshUInt32 b);
Boolean ssh1_crc_compensation_attack_check_crc(unsigned char *s,
                                               unsigned char *buf,
                                               size_t len,
                                               unsigned char *iv);

#ifndef WIN32
/* A signal handler. */
void ssh1_window_size_change_sig(int sig, void *context);
#endif /* WIN32 */

Boolean ssh1_challenge_reply(SshMPInt challenge,
                             SshPrivateKey key,
                             unsigned char **reply,
                             size_t *reply_len,
                             Ssh1Client ssh1);

SshOperationHandle ssh1_stream_wrap(SshStream stream,
                                    const char *remote_version_string,
                                    SshClient client,
                                    Ssh1DestroyCallback destroy_callback,






                                    void *destroy_context)
{
  Ssh1Client ssh1;
  char *protocol_id_string;
  int i;

  ssh1 = ssh_xcalloc(1, sizeof (*ssh1));
  ssh1->remote_server_major_protocol_version = 0;
  ssh1->remote_server_minor_protocol_version = 0;
  ssh1->remote_server_protocol_version = 0;
  for (i = 0; i < 10; i++)
    {
      ssh_dsprintf(&protocol_id_string, "SSH-1.%d-", i);
      if (strncmp(remote_version_string, protocol_id_string, 8) == 0)
        {
          ssh_xfree(protocol_id_string);
          ssh1->remote_server_major_protocol_version = 1;
          ssh1->remote_server_minor_protocol_version = i;
          ssh1->remote_server_protocol_version =
            ((ssh1->remote_server_major_protocol_version * 10) +
             ssh1->remote_server_minor_protocol_version);
          break;
        }
      ssh_xfree(protocol_id_string);
    }
  if ((ssh1->remote_server_protocol_version == 10) ||
      (ssh1->remote_server_protocol_version == 11) ||
      (ssh1->remote_server_protocol_version == 12) ||
      (ssh1->remote_server_protocol_version == 13) ||
      (ssh1->remote_server_protocol_version == 14))
    {
      ssh1->have_forwarding = FALSE;
      ssh_warning("Remote server talks SSH-%d.%d protocol.  "
                  "(terminal mode emulation only)",
                  ssh1->remote_server_major_protocol_version,
                  ssh1->remote_server_minor_protocol_version);
      ssh1->remote_server_allows_ignore_packets_in_startup = FALSE;
    }
  else if (ssh1->remote_server_protocol_version == 15)
    {
      ssh1->have_forwarding = TRUE;
      ssh_warning("Remote server talks SSH-%d.%d protocol.",
                  ssh1->remote_server_major_protocol_version,
                  ssh1->remote_server_minor_protocol_version);
      ssh1->remote_server_allows_ignore_packets_in_startup = TRUE;
    }
  else
    {
      if (ssh1->remote_server_protocol_version == 0)
        ssh_warning("Unrecognized server version.");
      else
        ssh_warning("Protocol version %d.%d is not supported.",
                    ssh1->remote_server_major_protocol_version,
                    ssh1->remote_server_minor_protocol_version);
      memset(ssh1, 'F', sizeof (*ssh1));
      ssh_xfree(ssh1);
      if (stream)
        {
          ssh_stream_destroy(stream);
        }

      if (client->stdio_stream || client->stderr_stream)
        {
          ssh_leave_non_blocking(-1);

          if (client->stdio_stream)
            {
              ssh_stream_destroy(client->stdio_stream);
              client->stdio_stream = NULL;
            }
          if (client->stderr_stream)
            {
              ssh_stream_destroy(client->stderr_stream);
              client->stderr_stream = NULL;
            }
        }

      if (destroy_callback)
        {
          destroy_callback(-1, destroy_context);
        }
      else
        {
          ssh_register_timeout(0L, 10000L, ssh1_exit_timeout, (void *)(-1));
        }
      return NULL;
    }
  ssh1->mode = SSH1_MODE_NONE;
  ssh1->prev_mode = SSH1_MODE_NONE;
  ssh1->op = NULL;
  ssh1->aborted = FALSE;
  ssh1->stream = stream;
  ssh1->client = client;
  ssh1->have_pty = FALSE;
  ssh1->in_cipher = NULL;
  ssh1->in_cipher2 = NULL;
  ssh1->in_cipher3 = NULL;
  ssh1->in_compress = NULL;
  ssh1->out_cipher = NULL;
  ssh1->out_cipher2 = NULL;
  ssh1->out_cipher3 = NULL;
  ssh1->out_compress = NULL;
  ssh1->input_buf = NULL;
  ssh1->input_offset = 0;
  ssh1->input_len = 0;
  ssh1->input_size = 0;
  ssh_buffer_init(ssh1->output_buf);
  ssh_buffer_init(ssh1->stdout_buf);
  ssh_buffer_init(ssh1->stderr_buf);
  ssh_buffer_init(ssh1->stdin_pre_buf);
  ssh1->tty_saturated = FALSE;
  ssh1->have_stdin_eof = FALSE;
  ssh1->host_key = NULL;
  ssh1->server_key = NULL;
  ssh1->begin_authentication_after_adding_auth_tokens = FALSE;
  ssh1->statistics.compressed_incoming_bytes = 0;
  ssh1->statistics.uncompressed_incoming_bytes = 0;
  ssh1->statistics.incoming_packets = 0;
  ssh1->statistics.compressed_outgoing_bytes = 0;
  ssh1->statistics.uncompressed_outgoing_bytes = 0;
  ssh1->statistics.outgoing_packets = 0;
  ssh1->statistics.kex_name = "ssh-rsa";
  ssh1->statistics.host_key_name = "ssh-rsa";
  ssh1->statistics.host_key_names = "ssh-rsa";
  ssh1->statistics.cipher_c_to_s = "none";
  ssh1->statistics.mac_c_to_s = "crc32";
  ssh1->statistics.compression_c_to_s = "none";
  ssh1->statistics.cipher_s_to_c = "none";
  ssh1->statistics.mac_s_to_c = "crc32";
  ssh1->statistics.compression_s_to_c = "none";

  for (i = 0; i < sizeof (ssh1->session_key); i++)
    ssh1->session_key[i] = ssh_random_get_byte();
  ssh_mp_init(&(ssh1->encrypted_session_key));
  ssh1->destroy_callback = destroy_callback;
  ssh1->destroy_context = destroy_context;









  ssh1_crc_compensation_attack_init(ssh1);

  client->session_requested = TRUE;

  ssh_stream_set_callback(ssh1->stream, ssh1_stream_event, (void *)ssh1);

  if (ssh1->client->stdio_stream)
    ssh_stream_set_callback(ssh1->client->stdio_stream,
                            ssh1_stdio_event,
                            (void *)ssh1);
  if (ssh1->client->stderr_stream)
    ssh_stream_set_callback(ssh1->client->stderr_stream,
                            ssh1_stderr_event,
                            (void *)ssh1);

  ssh_dsprintf(&protocol_id_string,
               SSH1_PROTOCOL_ID_STRING,
               ssh1->remote_server_major_protocol_version,
               ssh1->remote_server_minor_protocol_version);
  ssh_xbuffer_append(ssh1->output_buf,
                     (unsigned char *)protocol_id_string,
                     strlen(protocol_id_string));
  memset(protocol_id_string, 'F', strlen (protocol_id_string));
  ssh_xfree(protocol_id_string);
  ssh1->mode = SSH1_MODE_VERSION_SENT;
  if (ssh1_output_available(ssh1) == TRUE)
    {
      ssh1->op = ssh_operation_register(ssh1_protocol_abort, (void *)ssh1);
      return ssh1->op;
    }
  return NULL;
}

void ssh1_get_statistics(SshOperationHandle handle,
                         SshTransportStatistics statistics_return)
{
  Ssh1Client ssh1;

  ssh1 = (Ssh1Client)(ssh_operation_get_context(handle));
  *statistics_return = ssh1->statistics;
}

void ssh1_protocol_abort(void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;

  ssh1->aborted = TRUE;
  ssh1->op = NULL; /* We don't want protocol destroy to unregister this. */
  ssh1_protocol_destroy(ssh1, TRUE);
  return;
}

void ssh1_exit_timeout(void *context)
{
  SshUInt32 exit_value = (SshUInt32)context;

  exit((int)exit_value);
  /*NOTREACHED*/
}

void ssh1_protocol_destroy_timeout(void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;

  ssh1_protocol_destroy(ssh1, FALSE);
}

/* Destroy stream, delete context, call destroy callback. */
void ssh1_protocol_destroy(Ssh1Client ssh1, Boolean immediate)
{
  SshUInt32 exit_value = ssh1->exit_value;

  if (ssh1->mode != SSH1_MODE_DESTROYING)
    {
      SSH_DEBUG(5, ("Attempting to destroy the protocol object."));
      ssh1_uninit_local_forward(ssh1); /* This is safe for multiple calls */
#ifndef WIN32
#ifdef SIGWINCH
      if (ssh1->have_pty)
        ssh_unregister_signal(SIGWINCH);
#endif /* SIGWINCH */
#endif /* WIN32 */
      /* cancel all timeouts using `ssh1' here. (we don't use
         SSH_ALL_CALLBACKS, as it is deemed deprecated by the SSH AB.) */
      ssh_cancel_timeouts(ssh1_stdin_available_timeout, (void *)ssh1);
      ssh_stream_set_callback(ssh1->stream, NULL, NULL);
      ssh_stream_destroy(ssh1->stream);
      ssh1->stream = NULL;
      ssh1_flush_auth_tokens(ssh1);
      ssh1->prev_mode = ssh1->mode;
      ssh1->mode = SSH1_MODE_DESTROYING;
    }
  else
    {
      SSH_DEBUG(5, ("Reattempting to destroy the protocol object."));
    }
  if (! immediate)
    {
      if (ssh1_flush_stdout_and_stderr(ssh1) == FALSE)
        {
          SSH_DEBUG(5, ("Stdout and stderr not flushed; rescheduling."));
          ssh_register_timeout(0L, 100000L,
                               ssh1_protocol_destroy_timeout, (void *)ssh1);
          return;
        }
      else
        {
          SSH_DEBUG(5, ("All flushed; let's flush the protocol."));
        }
    }
  if (ssh1->op != NULL)
    {
      ssh_operation_unregister(ssh1->op);
      ssh1->op = NULL;
    }

  if (ssh1->client->stdio_stream)
    ssh_stream_set_callback(ssh1->client->stdio_stream, NULL, NULL);
  if (ssh1->client->stderr_stream)
    ssh_stream_set_callback(ssh1->client->stderr_stream, NULL, NULL);

  ssh_leave_non_blocking(-1);

  if (((ssh1->prev_mode == SSH1_MODE_INTERACTIVE) ||
       (ssh1->prev_mode == SSH1_MODE_INTERACTIVE_AFTER_EOF)) &&
      ssh1->have_pty)
    {
      ssh_leave_raw_mode(-1);
    }


  if (ssh1->client->stderr_stream)
    {
      ssh_stream_destroy(ssh1->client->stderr_stream);
      ssh1->client->stderr_stream = NULL;
    }

  if (ssh1->client->stdio_stream)
    {
      ssh_stream_destroy(ssh1->client->stdio_stream);
      ssh1->client->stdio_stream = NULL;
    }
  while (ssh1->remote_channels)
    ssh1_delete_remote_channel(ssh1->remote_channels);

  SSH_DEBUG(4, ("incoming: %lu packets, %lu bytes (%lu bytes compressed)",
                ssh1->statistics.incoming_packets,
                ssh1->statistics.uncompressed_incoming_bytes,
                ssh1->statistics.compressed_incoming_bytes));
  SSH_DEBUG(4, ("outgoing: %lu packets, %lu bytes (%lu bytes compressed)",
                ssh1->statistics.outgoing_packets,
                ssh1->statistics.uncompressed_outgoing_bytes,
                ssh1->statistics.compressed_outgoing_bytes));
  ssh1_crc_compensation_attack_uninit(ssh1);
  if (ssh1->in_cipher)
    ssh_cipher_free(ssh1->in_cipher);
  if (ssh1->in_cipher2)
    ssh_cipher_free(ssh1->in_cipher2);
  if (ssh1->in_cipher3)
    ssh_cipher_free(ssh1->in_cipher3);
  if (ssh1->in_compress)
    ssh_compress_free(ssh1->in_compress);
  if (ssh1->out_cipher)
    ssh_cipher_free(ssh1->out_cipher);
  if (ssh1->out_cipher2)
    ssh_cipher_free(ssh1->out_cipher2);
  if (ssh1->out_cipher3)
    ssh_cipher_free(ssh1->out_cipher3);
  if (ssh1->out_compress)
    ssh_compress_free(ssh1->out_compress);
  if (ssh1->host_key)
    ssh_public_key_free(ssh1->host_key);
  if (ssh1->server_key)
    ssh_public_key_free(ssh1->server_key);
  ssh_xfree(ssh1->input_buf);
  ssh_buffer_uninit(ssh1->output_buf);
  ssh_buffer_uninit(ssh1->stdout_buf);
  ssh_buffer_uninit(ssh1->stderr_buf);
  ssh_buffer_uninit(ssh1->stdin_pre_buf);
  memset(ssh1->session_key, 0, sizeof (ssh1->session_key));
  memset(ssh1->session_id, 0, sizeof (ssh1->session_id));
  ssh_mp_clear(&(ssh1->encrypted_session_key));
  if (ssh1->destroy_callback)
    {









      if (! ssh1->aborted)
        ssh1->destroy_callback(ssh1->exit_value, ssh1->destroy_context);
      memset(ssh1, 'F', sizeof (*ssh1));
      ssh_xfree(ssh1);
    }
  else
    {
      memset(ssh1, 'F', sizeof (*ssh1));
      ssh_xfree(ssh1);
      ssh_register_timeout(0L, 10000L, ssh1_exit_timeout, (void *)exit_value);
    }
  return;
}

/* Handle received ssh1 protocol packet.  Packet is passed to this function
   without the length bytes.  This function always frees the packet
   buffer. */
void ssh1_packet_received(unsigned char *data,
                          size_t data_len,
                          size_t packet_len,
                          Ssh1Client ssh1)
{
  SshBufferStruct packet[1];
  SshUInt8 tmpbyte;
  SshUInt32 packet_crc, local_crc;
  Ssh1PacketType packet_type;
  SshCryptoStatus cr;
  size_t pad_len;
  Boolean crc_compensation_attack;
  unsigned char idea_iv[8];

  SSH_DEBUG(5, ("Received packet (dlen = %d plen = %d).",
                (int)data_len, (int)packet_len));

  /* Various sanity checks */
  if (packet_len > 262144) /* Defined in RFC */
    {
      ssh_xfree(data);
      ssh1_protocol_error("Packet too long.", ssh1);
      return;
    }
  if (packet_len < 5) /* At least packet type and CRC */
    {
      ssh_xfree(data);
      ssh1_protocol_error("Packet too short.", ssh1);
      return;
    }
  if ((data_len % 8) != 0)
    {
      ssh_xfree(data);
      ssh1_protocol_error("Packet size not multiple of 8.", ssh1);
      return;
    }
  pad_len = (8 - (packet_len % 8));
  SSH_ASSERT(((packet_len + pad_len) == data_len));
  if (ssh1->in_cipher)
    {
      /* Run CRC compensation attack detector. */
      if (ssh1->chosen_cipher == SSH1_CIPHER_IDEA)
        {
          SSH_ASSERT(ssh_cipher_get_iv_length(ssh1->in_cipher) == 8);
          cr = ssh_cipher_get_iv(ssh1->in_cipher, idea_iv);
          if (cr != SSH_CRYPTO_OK)
            {
              ssh1_protocol_error("Cipher IV retrieval failed.", ssh1);
              return;
            }
          crc_compensation_attack =
            ssh1_crc_compensation_attack_detect(ssh1,
                                                data,
                                                data_len,
                                                idea_iv);
        }
      else
        {
          crc_compensation_attack =
            ssh1_crc_compensation_attack_detect(ssh1,
                                                data,
                                                data_len,
                                                NULL);
        }
      /* Cipher transformation(s) are made to entire packet containing
         padding in the beginning and CRC in the end. */
      if (crc_compensation_attack)
        {
          ssh1_protocol_error("CRC compensation attack detected.", ssh1);
          return;
        }
      if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
        ssh1_buffer_reverse_byte_order(data, data_len);
      cr = ssh_cipher_transform(ssh1->in_cipher, data, data, data_len);
      if (cr != SSH_CRYPTO_OK)
        {
          ssh_xfree(data);
          ssh1_protocol_error("Cipher failed.", ssh1);
          return;
        }
      else
        {
          SSH_DEBUG(5, ("ssh_cipher_transform #1 for %d bytes",
                        (int)data_len));
        }
      if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
        ssh1_buffer_reverse_byte_order(data, data_len);
      if (ssh1->in_cipher2)
        {
          if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
            ssh1_buffer_reverse_byte_order(data, data_len);
          cr = ssh_cipher_transform(ssh1->in_cipher2, data, data, data_len);
          if (cr != SSH_CRYPTO_OK)
            {
              ssh_xfree(data);
              ssh1_protocol_error("Cipher failed.", ssh1);
              return;
            }
          else
            {
              SSH_DEBUG(5, ("ssh_cipher_transform #2 for %d bytes",
                            (int)data_len));
            }
          if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
            ssh1_buffer_reverse_byte_order(data, data_len);
          if (ssh1->in_cipher3)
            {
              if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
                ssh1_buffer_reverse_byte_order(data, data_len);
              cr = ssh_cipher_transform(ssh1->in_cipher3,
                                        data, data, data_len);
              if (cr != SSH_CRYPTO_OK)
                {
                  ssh_xfree(data);
                  ssh1_protocol_error("Cipher failed.", ssh1);
                  return;
                }
              else
                {
                  SSH_DEBUG(5, ("ssh_cipher_transform #3 for %d bytes",
                                (int)data_len));
                }
              if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
                ssh1_buffer_reverse_byte_order(data, data_len);
            }
        }
    }

  /* CRC can now been obtained from the packet tail.  Modify packet_len
     to the length without CRC, since CRC is not compressed and obviously
     it is not used in the CRC calculation either. */
  packet_crc = ((((SshUInt32)data[data_len - 4]) << 24) |
                (((SshUInt32)data[data_len - 3]) << 16) |
                (((SshUInt32)data[data_len - 2]) << 8) |
                (((SshUInt32)data[data_len - 1])));
  packet_len -= 4;

  /* CRC check.  This is not very secure way to guarantee packet
     integrity.  In ssh2 protocol, real MAC is used instead of CRC. */
  local_crc = crc32_buffer(data, packet_len + pad_len);
  if (packet_crc != local_crc)
    {
      SSH_DEBUG(5, ("crc mismatch (packet=0x%08lx, local=0x%08lx)",
                    (unsigned long)packet_crc,
                    (unsigned long)local_crc));
      ssh_xfree(data);
      ssh1_protocol_error("CRC doesn't match.", ssh1);
      return;
    }
  else
    {
      SSH_DEBUG(5, ("crc ok (crc=0x%08lx)", (unsigned long)packet_crc));
    }

  /* Update statistics compressed part. */
  ssh1->statistics.compressed_incoming_bytes += packet_len;

  /* Packet is now decompresed or just copied to the buffer. */
  ssh_buffer_init(packet);
  if (ssh1->in_compress && (ssh1->mode != SSH1_MODE_COMPRESSION_REQUESTED))
    ssh_compress_buffer(ssh1->in_compress,
                        data + pad_len, packet_len,  packet);
  else
    ssh_xbuffer_append(packet, data + pad_len, packet_len);

  /* Data buffer is no longer needed and this packet frees it in any case. */
  ssh_xfree(data);

  /* Update statistics uncompressed part and packet count. */
  ssh1->statistics.uncompressed_incoming_bytes += ssh_buffer_len(packet);
  ssh1->statistics.incoming_packets++;

  /* Decode the packet type. */
  if (ssh1_decode_byte(packet, &tmpbyte))
    {
      packet_type = (Ssh1PacketType)tmpbyte;
    }
  else
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("No packet type in empty packet.", ssh1);
      return;
    }

  SSH_DEBUG(5, ("packet type = %d, len = %d",
                (int)packet_type, (int)(ssh_buffer_len(packet))));

  switch ((int)packet_type) {
  case SSH1_PACKET_MSG_NONE:                       /* no message */
    ssh_buffer_uninit(packet);
    break;

  case SSH1_PACKET_MSG_DISCONNECT:                 /* cause (string) */
    ssh1_handle_packet_msg_disconnect(packet, ssh1);
    break;

  case SSH1_PACKET_SMSG_PUBLIC_KEY:
    ssh1_handle_packet_smsg_public_key(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_AUTH_RSA_CHALLENGE:        /* int (MP_INT) */
    ssh1_handle_packet_smsg_auth_rsa_challenge(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_SUCCESS:                   /* */
    ssh1_handle_packet_smsg_success(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_FAILURE:                   /* */
    ssh1_handle_packet_smsg_failure(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_STDOUT_DATA:               /* data (string) */
    ssh1_handle_packet_smsg_stdout_data(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_STDERR_DATA:               /* data (string) */
    ssh1_handle_packet_smsg_stderr_data(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_EXITSTATUS:                /* status (int) */
    ssh1_handle_packet_smsg_exitstatus(packet, ssh1);
    return;

  case SSH1_PACKET_MSG_CHANNEL_OPEN_CONFIRMATION:  /* channel (int) */
    ssh1_handle_packet_msg_channel_open_confirmation(packet, ssh1);
    return;

  case SSH1_PACKET_MSG_CHANNEL_OPEN_FAILURE:       /* channel (int) */
    ssh1_handle_packet_msg_channel_open_failure(packet, ssh1);
    return;

  case SSH1_PACKET_MSG_CHANNEL_DATA:               /* ch,data (int,str) */
    ssh1_handle_packet_msg_channel_data(packet, ssh1);
    return;

  case SSH1_PACKET_MSG_CHANNEL_INPUT_EOF:          /* ch */
    ssh1_handle_packet_msg_channel_input_eof(packet, ssh1);
    return;

  case SSH1_PACKET_MSG_CHANNEL_OUTPUT_CLOSED:      /* ch */
    ssh1_handle_packet_msg_channel_output_closed(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_X11_OPEN:                  /* channel (int) */
    ssh1_handle_packet_smsg_x11_open(packet, ssh1);
    return;

  case SSH1_PACKET_MSG_PORT_OPEN:                  /* ch,h,p (i,s,i) */
    ssh1_handle_packet_msg_port_open(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_AGENT_OPEN:                /* port (int) */
    ssh1_handle_packet_smsg_agent_open(packet, ssh1);
    return;

  case SSH1_PACKET_MSG_IGNORE:                     /* string */
    ssh1_handle_packet_msg_ignore(packet, ssh1);
    return;

  case SSH1_PACKET_MSG_DEBUG:                      /* string */
    ssh1_handle_packet_msg_debug(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_AUTH_TIS_CHALLENGE:        /* string */
    ssh1_handle_packet_smsg_auth_tis_challenge(packet, ssh1);
    return;

  case SSH1_PACKET_SMSG_AUTH_KERBEROS_RESPONSE:    /* string (KTEXT) */
    ssh1_handle_packet_smsg_auth_kerberos_response(packet, ssh1);
    return;

  case SSH1_PACKET_CMSG_SESSION_KEY:
  case SSH1_PACKET_CMSG_USER:
  case SSH1_PACKET_CMSG_AUTH_RHOSTS:
  case SSH1_PACKET_CMSG_AUTH_RSA:
  case SSH1_PACKET_CMSG_AUTH_RSA_RESPONSE:
  case SSH1_PACKET_CMSG_AUTH_PASSWORD:
  case SSH1_PACKET_CMSG_REQUEST_PTY:
  case SSH1_PACKET_CMSG_WINDOW_SIZE:
  case SSH1_PACKET_CMSG_EXEC_SHELL:
  case SSH1_PACKET_CMSG_EXEC_CMD:
  case SSH1_PACKET_CMSG_STDIN_DATA:
  case SSH1_PACKET_CMSG_EOF:
  case SSH1_PACKET_CMSG_PORT_FORWARD_REQUEST:
  case SSH1_PACKET_CMSG_AGENT_REQUEST_FORWARDING:
  case SSH1_PACKET_CMSG_EXIT_CONFIRMATION:
  case SSH1_PACKET_CMSG_X11_REQUEST_FORWARDING:
  case SSH1_PACKET_CMSG_AUTH_RHOSTS_RSA:
  case SSH1_PACKET_CMSG_REQUEST_COMPRESSION:
  case SSH1_PACKET_CMSG_MAX_PACKET_SIZE:
  case SSH1_PACKET_CMSG_AUTH_TIS:
  case SSH1_PACKET_CMSG_AUTH_TIS_RESPONSE:
  case SSH1_PACKET_CMSG_AUTH_KERBEROS:
  case SSH1_PACKET_CMSG_HAVE_KERBEROS_TGT:
    /* Client messages from the server side are always protocol
       errors.  They may also be an indication of a malicious hacked
       server trying to abuse broken client implementation.  We don't
       fall into this. */
    ssh1_protocol_error("Server sent a client packet.", ssh1);
    ssh_buffer_uninit(packet);
    return;

  default:
    ssh1_protocol_error("Unknown packet type.", ssh1);
    ssh_buffer_uninit(packet);
    return;
  }
  /*NOTREACHED*/
}

/* Make transformations.  Add padding, CRC and length tag.  Send to
   the remote server. */
Boolean ssh1_send_buffer(SshBuffer buffer, Ssh1Client ssh1)
{
  size_t packet_len;
  size_t pad_len;
  SshBufferStruct packet[1];
  SshUInt32 crc;
  SshCryptoStatus cr;
  int i;
  unsigned char *dummy;

  SSH_DEBUG_HEXDUMP(9,
                    ("packet payload (%d bytes)",
                     (int)(ssh_buffer_len(buffer))),
                    ssh_buffer_ptr(buffer), ssh_buffer_len(buffer));
  ssh_buffer_init(packet);

  /* Update statistics uncompressed part. */
  ssh1->statistics.uncompressed_outgoing_bytes += ssh_buffer_len(buffer);

  if (ssh1->out_compress)
    ssh_compress_buffer(ssh1->out_compress,
                        ssh_buffer_ptr(buffer),
                        ssh_buffer_len(buffer),
                        packet);
  else
    ssh_xbuffer_append(packet,
                       ssh_buffer_ptr(buffer),
                       ssh_buffer_len(buffer));

  /* Update statistics compressed part and packet counter. */
  ssh1->statistics.compressed_outgoing_bytes += ssh_buffer_len(packet);
  ssh1->statistics.outgoing_packets++;

  packet_len = ssh_buffer_len(packet) + 4;
  pad_len = (8 - (packet_len % 8));
  ssh_xbuffer_append_space(packet, &dummy, pad_len + 4);
  memmove(ssh_buffer_ptr(packet) + pad_len + 4,
          ssh_buffer_ptr(packet),
          packet_len);
  if (ssh1->out_cipher != NULL)
    {
      for (i = 0; i < pad_len; i++)
        *(ssh_buffer_ptr(packet) + 4 + i) = ssh_random_get_byte();
    }
  else
    {
      memset(ssh_buffer_ptr(packet) + 4, 0, pad_len);
    }
  crc = crc32_buffer(ssh_buffer_ptr(packet) + 4, ssh_buffer_len(packet) - 4);
  ssh1_encode_int(packet, crc);
  *(ssh_buffer_ptr(packet)) =
    (unsigned char)((((SshUInt32)packet_len) >> 24) & 0xff);
  *(ssh_buffer_ptr(packet) + 1) =
    (unsigned char)((((SshUInt32)packet_len) >> 16) & 0xff);
  *(ssh_buffer_ptr(packet) + 2) =
    (unsigned char)((((SshUInt32)packet_len) >> 8) & 0xff);
  *(ssh_buffer_ptr(packet) + 3) =
    (unsigned char)((((SshUInt32)packet_len)) & 0xff);
  if (ssh1->out_cipher)
    {
      if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
        ssh1_buffer_reverse_byte_order(ssh_buffer_ptr(packet) + 4,
                                       ssh_buffer_len(packet) - 4);
      cr = ssh_cipher_transform(ssh1->out_cipher,
                                ssh_buffer_ptr(packet) + 4,
                                ssh_buffer_ptr(packet) + 4,
                                ssh_buffer_len(packet) - 4);
      if (cr != SSH_CRYPTO_OK)
        {
          ssh_buffer_uninit(packet);
          ssh1_protocol_error("Cipher failed.", ssh1);
          return FALSE;
        }
      else
        {
          SSH_DEBUG(5, ("ssh_cipher_transform #1 for %d bytes",
                        (int)(ssh_buffer_len(packet) - 4)));
        }
      if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
        ssh1_buffer_reverse_byte_order(ssh_buffer_ptr(packet) + 4,
                                       ssh_buffer_len(packet) - 4);
      if (ssh1->out_cipher2)
        {
          if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
            ssh1_buffer_reverse_byte_order(ssh_buffer_ptr(packet) + 4,
                                           ssh_buffer_len(packet) - 4);
          cr = ssh_cipher_transform(ssh1->out_cipher2,
                                    ssh_buffer_ptr(packet) + 4,
                                    ssh_buffer_ptr(packet) + 4,
                                    ssh_buffer_len(packet) - 4);
          if (cr != SSH_CRYPTO_OK)
            {
              ssh_buffer_uninit(packet);
              ssh1_protocol_error("Cipher failed.", ssh1);
              return FALSE;
            }
          else
            {
              SSH_DEBUG(5, ("ssh_cipher_transform #2 for %d bytes",
                            (int)(ssh_buffer_len(packet) - 4)));
            }
          if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
            ssh1_buffer_reverse_byte_order(ssh_buffer_ptr(packet) + 4,
                                           ssh_buffer_len(packet) - 4);
          if (ssh1->out_cipher3)
            {
              if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
                ssh1_buffer_reverse_byte_order(ssh_buffer_ptr(packet) + 4,
                                               ssh_buffer_len(packet) - 4);
              cr = ssh_cipher_transform(ssh1->out_cipher3,
                                        ssh_buffer_ptr(packet) + 4,
                                        ssh_buffer_ptr(packet) + 4,
                                        ssh_buffer_len(packet) - 4);
              if (cr != SSH_CRYPTO_OK)
                {
                  ssh_buffer_uninit(packet);
                  ssh1_protocol_error("Cipher failed.", ssh1);
                  return FALSE;
                }
              else
                {
                  SSH_DEBUG(5, ("ssh_cipher_transform #3 for %d bytes",
                                (int)(ssh_buffer_len(packet) - 4)));
                }
              if (ssh1->chosen_cipher == SSH1_CIPHER_BLOWFISH)
                ssh1_buffer_reverse_byte_order(ssh_buffer_ptr(packet) + 4,
                                               ssh_buffer_len(packet) - 4);
            }
        }
    }
  SSH_DEBUG_HEXDUMP(9,
                    ("complete packet data (%d bytes)",
                     (int)(ssh_buffer_len(packet))),
                    ssh_buffer_ptr(packet), ssh_buffer_len(packet));
  ssh_xbuffer_append(ssh1->output_buf,
                     ssh_buffer_ptr(packet),
                     ssh_buffer_len(packet));
  ssh_buffer_uninit(packet);
  return ssh1_output_available(ssh1);
}

void ssh1_protocol_error(const char *msg, Ssh1Client ssh1)
{
  ssh_warning("SSH1 PROTOCOL ERROR: %s", msg);
  ssh1->exit_value = -1;
  ssh1_protocol_destroy(ssh1, FALSE);
  return;
}

Boolean ssh1_session_id(Ssh1Client ssh1, unsigned char id[16])
{
  SshHash hash;
  SshCryptoStatus cr;
  unsigned char *mp_buf;
  size_t mp_len, mp_buf_len;
  SshMPIntStruct n;

  cr = ssh_hash_allocate("md5", &hash);
  if (cr != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(5, ("Unable to allocate md5 hash for session id."));
      return FALSE;
    }
  ssh_mp_init(&n);
  cr = ssh_public_key_get_info(ssh1->host_key,
                               SSH_PKF_MODULO_N, &n,
                               SSH_PKF_END);
  if (cr != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(5, ("Unable to get N from host key."));
      ssh_mp_clear(&n);
      ssh_hash_free(hash);
      return FALSE;
    }
  mp_len = ssh_mp_get_size(&n, 2);
  mp_buf_len = ((mp_len + 7) >> 3) & 0xffff;
  mp_buf = ssh_xmalloc(mp_buf_len);
  ssh_mp_get_buf(mp_buf, mp_buf_len, &n);
  ssh_hash_update(hash, mp_buf, mp_buf_len);
  ssh_xfree(mp_buf);
  ssh_mp_init(&n);
  cr = ssh_public_key_get_info(ssh1->server_key,
                               SSH_PKF_MODULO_N, &n,
                               SSH_PKF_END);
  if (cr != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(5, ("Unable to get N from server key."));
      ssh_mp_clear(&n);
      ssh_hash_free(hash);
      return FALSE;
    }
  mp_len = ssh_mp_get_size(&n, 2);
  mp_buf_len = ((mp_len + 7) >> 3) & 0xffff;
  mp_buf = ssh_xmalloc(mp_buf_len);
  ssh_mp_get_buf(mp_buf, mp_buf_len, &n);
  ssh_mp_clear(&n);
  ssh_hash_update(hash, mp_buf, mp_buf_len);
  ssh_xfree(mp_buf);
  ssh_hash_update(hash, ssh1->anti_spoofing_cookie, 8);
  if (ssh_hash_digest_length(hash) != 16)
    {
      SSH_DEBUG(5, ("Hash digest length != 16."));
      ssh_hash_free(hash);
      return FALSE;
    }
  ssh_hash_final(hash, id);
  ssh_hash_free(hash);
  return TRUE;
}

Boolean ssh1_flush_stdout_and_stderr(Ssh1Client ssh1)
{
  return ((ssh1_flush_stdout(ssh1)) &&
          (ssh1_flush_stderr(ssh1)));
}

Boolean ssh1_flush_stdout(Ssh1Client ssh1)
{
  int n;

  if (ssh_buffer_len(ssh1->stdout_buf) > 0)
    {
      do {
        n = ssh_stream_write(ssh1->client->stdio_stream,
                             ssh_buffer_ptr(ssh1->stdout_buf),
                             ssh_buffer_len(ssh1->stdout_buf));
        if (n > 0)
          ssh_buffer_consume(ssh1->stdout_buf, n);
      } while (n > 0);
    }
  return (ssh_buffer_len(ssh1->stdout_buf) == 0);
}

Boolean ssh1_flush_stderr(Ssh1Client ssh1)
{
  int n;

  if (ssh_buffer_len(ssh1->stderr_buf) > 0)
    {
      do {
        n = ssh_stream_write(ssh1->client->stderr_stream,
                             ssh_buffer_ptr(ssh1->stderr_buf),
                             ssh_buffer_len(ssh1->stderr_buf));
        if (n > 0)
          ssh_buffer_consume(ssh1->stderr_buf, n);
      } while (n > 0);
    }
  return (ssh_buffer_len(ssh1->stderr_buf) == 0);
}

void ssh1_stdio_event(SshStreamNotification event, void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;

  switch (event)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      SSH_DEBUG(5, ("input available"));
      ssh1_stdin_available(ssh1);
      break;

    case SSH_STREAM_CAN_OUTPUT:
      SSH_DEBUG(5, ("output available"));
      ssh1_flush_stdout(ssh1);
      break;

    case SSH_STREAM_DISCONNECTED:
      SSH_DEBUG(5, ("stdin disconnected"));
      ssh1_protocol_error("StdIO stream unexpectedly broken.", ssh1);
      break;

    default:
      ssh_warning("Unexpected stream event (%d).", (int)event);
      break;
    }
  return;
}

void ssh1_stderr_event(SshStreamNotification event, void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;

  switch (event)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      break;

    case SSH_STREAM_CAN_OUTPUT:
      SSH_DEBUG(5, ("output available"));
      ssh1_flush_stderr(ssh1);
      break;

    case SSH_STREAM_DISCONNECTED:
      SSH_DEBUG(5, ("stderr disconnected"));
      ssh1_protocol_error("StdErr stream unexpectedly broken.", ssh1);
      break;

    default:
      ssh_warning("Unexpected stream event (%d).", (int)event);
      break;
    }
  return;
}

void ssh1_stdin_available_timeout(void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;

  ssh1_stdin_available(ssh1);
  return;
}

void ssh1_stdin_available(Ssh1Client ssh1)
{
  unsigned char *buf;
  size_t buf_len, buf_offset;
  int n;
  SshBufferStruct reply[1];

  if (ssh1->have_pty)
    {
      buf_len = 0x100;
      buf = ssh_xmalloc(buf_len);
      buf_offset = 0;
      n = ssh_stream_read(ssh1->client->stdio_stream, buf, buf_len);
      if (n > 0)
        buf_offset += n;
    }
  else
    {
      buf_len = 0x8000;
      buf = ssh_xmalloc(buf_len);
      buf_offset = 0;
      n = ssh_stream_read(ssh1->client->stdio_stream, buf, buf_len);
      if (n > 0)
        buf_offset += n;
    }
  if (buf_offset > 0)
    {
      if (ssh1->mode == SSH1_MODE_INTERACTIVE)
        {
          ssh_buffer_init(reply);
          ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_STDIN_DATA);
          ssh1_encode_string(reply, (char *)buf, buf_offset);
          if (ssh1_send_buffer(reply, ssh1) == FALSE)
            {
              ssh_buffer_uninit(reply);
              return;
            }
          ssh_buffer_uninit(reply);
        }
      else if (ssh1->mode != SSH1_MODE_INTERACTIVE_AFTER_EOF)
        {
          ssh_xbuffer_append(ssh1->stdin_pre_buf, buf, buf_offset);
        }
    }
  ssh_xfree(buf);
  if ((n == 0) && (ssh1->mode == SSH1_MODE_INTERACTIVE))
    {
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_EOF);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1->mode = SSH1_MODE_INTERACTIVE_AFTER_EOF;
      ssh1->have_stdin_eof = TRUE;
      SSH_DEBUG(5, ("Sent stdin EOF to the server."));
    }
  else if (n == 0)
    {
      ssh1->have_stdin_eof = TRUE;
      SSH_DEBUG(5, ("Marked stdin EOF received."));
    }
  if (n > 0)
    {
      ssh_register_timeout(0L, 10L,
                           ssh1_stdin_available_timeout, (void *)ssh1);
    }
  return;
}


void ssh1_stream_event(SshStreamNotification event, void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;

  switch (event)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      SSH_DEBUG(5, ("input available"));
      ssh1_input_available(ssh1);
      break;

    case SSH_STREAM_CAN_OUTPUT:
      SSH_DEBUG(5, ("input available"));
      ssh1_output_available(ssh1);
      break;

    case SSH_STREAM_DISCONNECTED:
      SSH_DEBUG(5, ("stream disconnected"));
      ssh1_protocol_error("Unexpected EOF from the server.", ssh1);
      break;

    default:
      ssh_warning("Unexpected stream event (%d).", (int)event);
      break;
    }
  return;
}

Boolean ssh1_output_available(Ssh1Client ssh1)
{
  int n;

  if (ssh_buffer_len(ssh1->output_buf) == 0)
    {
      /* We have nothing to write.  This is an error if we came from
         the event loop bottom.  There is however no way to verify
         this. */
      return TRUE;
    }

  n = ssh_stream_write(ssh1->stream,
                       ssh_buffer_ptr(ssh1->output_buf),
                       ssh_buffer_len(ssh1->output_buf));
  if (n == 0)
    {
      SSH_DEBUG(5, ("write permanently failed, %d bytes left.",
                    (int)(ssh_buffer_len(ssh1->output_buf))));
      ssh1_protocol_error("Write failed because of the remote server.", ssh1);
      return FALSE;
    }
  else if (n < 0)
    {
      /* Write would have blocked.  This can happen, if this function
         is initiated from somewhere else than the event loop
         bottom. */
      SSH_DEBUG(5, ("write would block, %d bytes pending.",
                    (int)(ssh_buffer_len(ssh1->output_buf))));
      return TRUE;
    }
  ssh_buffer_consume(ssh1->output_buf, n);
  SSH_DEBUG(5, ("write (%d bytes) completed, %d bytes pending.",
                n, (int)(ssh_buffer_len(ssh1->output_buf))));
  /* We recall ourselves here, if we have pending packet bytes and we
     were able to write at least some bytes. */
  if (ssh_buffer_len(ssh1->output_buf) > 0)
    return ssh1_output_available(ssh1);
  return TRUE;
}

void ssh1_input_available_timeout(void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;

  ssh1->tty_saturated = FALSE;
  ssh1_input_available(ssh1);
  return;
}

Boolean ssh1_input_available(Ssh1Client ssh1)
{
  int n;
  unsigned char *tmp_input_buf;

  if (ssh1->tty_saturated)
    return TRUE;
  if ((ssh_buffer_len(ssh1->stdout_buf) > SSH1_MAXIMUM_TTY_BUFFER) ||
      (ssh_buffer_len(ssh1->stderr_buf) > SSH1_MAXIMUM_TTY_BUFFER))
    {
      ssh1->tty_saturated = TRUE;
      ssh_register_timeout(0L, 100000L,
                           ssh1_input_available_timeout, (void *)ssh1);
      return TRUE;
    }

  if (ssh1->input_buf == NULL)
    {
      ssh1->input_buf = ssh_xmalloc(4);
      ssh1->input_offset = 0;
      ssh1->input_len = 0;
      ssh1->input_size = 0;
    }
  if (ssh1->input_size == 0)
    {
      /* We are reading a packet length. */
      SSH_DEBUG(5, ("Trying to read packet %d bytes of header.",
                    (int)(4 - ssh1->input_offset)));
      n = ssh_stream_read(ssh1->stream,
                          ssh1->input_buf + ssh1->input_offset,
                          4 - ssh1->input_offset);
      SSH_DEBUG(5, ("ssh_stream_read returns %d", n));
      if (n == 0)
        {
          ssh1_protocol_error("Unexpected EOF from the server.", ssh1);
          return FALSE;
        }
      else if (n < 0)
        {
          /* Read would have blocked.  This can legally happen, if we
             have read a complete packet and recall this function just
             in order to avoid going to event loop bottom with each
             packet. */
          return TRUE;
        }
      ssh1->input_offset += n;
      SSH_ASSERT(ssh1->input_offset <= 4);
      if (ssh1->input_offset == 4)
        {
          ssh1->input_len = (size_t)((((SshUInt32)ssh1->input_buf[0]) << 24) |
                                     (((SshUInt32)ssh1->input_buf[1]) << 16) |
                                     (((SshUInt32)ssh1->input_buf[2]) << 8) |
                                     (((SshUInt32)ssh1->input_buf[3])));
          ssh1->input_offset = 0;
          ssh1->input_size = ssh1->input_len + (8 - (ssh1->input_len % 8));
          ssh_xfree(ssh1->input_buf);
          ssh1->input_buf = ssh_xmalloc(ssh1->input_size);
          SSH_DEBUG(4, ("packet header arrived for %d byte packet",
                        (int)ssh1->input_len));
        }
    }
  SSH_DEBUG(5, ("Trying to read %d bytes %d byte packet to offset %d.",
                (int)(ssh1->input_size - ssh1->input_offset),
                (int)(ssh1->input_size),
                (int)(ssh1->input_offset)));
  n = ssh_stream_read(ssh1->stream,
                      ssh1->input_buf + ssh1->input_offset,
                      ssh1->input_size - ssh1->input_offset);
  SSH_DEBUG(5, ("ssh_stream_read returns %d", n));
  if (n == 0)
    {
      ssh1_protocol_error("Unexpected EOF from the server.", ssh1);
      return FALSE;
    }
  else if (n < 0)
    {
      /* Read would have blocked.  This can legally happen, if we just
         read the packet length field and by doing so drained the
         input.  However, it is not wise to go through the bottom of
         the event loop twice for every packet. */
      return TRUE;
    }
  ssh1->input_offset += n;
  SSH_ASSERT(ssh1->input_offset <= ssh1->input_size);
  if (ssh1->input_offset == ssh1->input_size)
    {
      tmp_input_buf = ssh1->input_buf;
      ssh1->input_buf = NULL;  /* Packet handler frees the buffer. */
      ssh1_packet_received(tmp_input_buf,
                           ssh1->input_size,
                           ssh1->input_len,
                           ssh1);
      tmp_input_buf = NULL;
      return TRUE;
    }
  /* We try to complete packet with this recursive call. */
  return ssh1_input_available(ssh1);
}

int ssh1_choose_cipher(SshUInt32 server_cipher_mask,
                       const char *config_cipher_list)
{
  char *list, *rest, *current;
  int cipher_number;

  if ((config_cipher_list == NULL) ||
      (strlen(config_cipher_list) == 0))
    {
      SSH_DEBUG(5, ("Empty cipher list."));
      return -1;
    }
  SSH_DEBUG(5, ("Choosing cipher from list \"%s\".", config_cipher_list));
  list = ssh_xstrdup(config_cipher_list);
  rest = list;
  while ((current = ssh_snlist_get_name(rest)) != NULL)
    {
      rest += strlen(current);
      if (*rest == ',')
        rest++;
      cipher_number = ssh1_canonical_cipher_name_to_number(current);
      ssh_xfree(current);
      if ((cipher_number >= 0) &&
               (((((SshUInt32)1) << cipher_number) & server_cipher_mask) != 0))
        {
          if (cipher_number == SSH1_CIPHER_ARCFOUR)
            {
              ssh_warning("Arcfour cipher is not allowed in ssh1 emulation.");
            }
          else
            {
              ssh_xfree(list);
              SSH_DEBUG(5, ("Choosing cipher number %d.", cipher_number));
              return cipher_number;
            }
        }
      if (strlen(rest) == 0)
        break;
    }
  ssh_xfree(list);
  SSH_DEBUG(5, ("No cipher found."));
  return -1;
}

int ssh1_canonical_cipher_name_to_number(const char *cipher_name)
{
  if (strcmp(cipher_name, "none") == 0)
    {
      SSH_DEBUG(5, ("Mapping cipher name \"%s\" to SSH_CIPHER_NONE",
                    cipher_name));
      return SSH1_CIPHER_NONE;
    }
  else if (((strcmp(cipher_name, "idea") == 0) ||
            (strncmp(cipher_name, "idea-", 5) == 0)) &&
           (ssh_cipher_supported("idea-cfb")))
    {
      SSH_DEBUG(5, ("Mapping cipher name \"%s\" to SSH_CIPHER_IDEA",
                    cipher_name));
      return SSH1_CIPHER_IDEA;
    }
  else if (((strcmp(cipher_name, "des") == 0) ||
            (strncmp(cipher_name, "des-", 4) == 0)) &&
           (ssh_cipher_supported("des-cbc")))
    {
      SSH_DEBUG(5, ("Mapping cipher name \"%s\" to SSH_CIPHER_DES",
                    cipher_name));
      return SSH1_CIPHER_DES;
    }
  else if (((strcmp(cipher_name, "3des") == 0) ||
            (strncmp(cipher_name, "3des-", 5) == 0)) &&
           (ssh_cipher_supported("des-cbc"))) /* des-cbc here is NOT a bug! */
    {
      SSH_DEBUG(5, ("Mapping cipher name \"%s\" to SSH_CIPHER_3DES",
                    cipher_name));
      return SSH1_CIPHER_3DES;
    }
  else if ((strcmp(cipher_name, "arcfour") == 0) &&
           (ssh_cipher_supported("arcfour")))
    {
      SSH_DEBUG(5, ("Mapping cipher name \"%s\" to SSH_CIPHER_ARCFOUR",
                    cipher_name));
      return SSH1_CIPHER_ARCFOUR;
    }
  else if (((strcmp(cipher_name, "blowfish") == 0) ||
            (strncmp(cipher_name, "blowfish-", 9) == 0)) &&
           (ssh_cipher_supported("blowfish-cbc")))
    {
      SSH_DEBUG(5, ("Mapping cipher name \"%s\" to SSH_CIPHER_BLOWFISH",
                 cipher_name));
      return SSH1_CIPHER_BLOWFISH;
    }
  return -1;
}

void ssh1_buffer_reverse_byte_order(unsigned char *buf, size_t buflen)
{
  size_t i;
  unsigned char tmp;

  SSH_ASSERT((buflen % 4) == 0);
  for (i = 0; i < buflen; i += 4)
    {
      tmp = buf[i];
      buf[i] = buf[i + 3];
      buf[i + 3] = tmp;
      tmp = buf[i + 1];
      buf[i + 1] = buf[i + 2];
      buf[i + 2] = tmp;
    }
  return;
}

void ssh1_add_password_auth_token(const char *password, Ssh1Client ssh1)
{
  Ssh1AuthToken tok;

  tok = ssh_xcalloc(1, sizeof (*tok));
  tok->auth_type = SSH1_AUTH_TYPE_PASSWORD;
  tok->password = password ? ssh_xstrdup(password) : NULL;
  ssh1_add_auth_token(tok, ssh1);
  return;
}

void ssh1_add_rsa_file_auth_token(const char *filename, Ssh1Client ssh1)
{
  Ssh1AuthToken tok;

  SSH_ASSERT(filename != NULL);
  if (ssh1->remote_server_protocol_version == 10)
    return; /* RSA authentication is insecure in 1.0 protocol. */
  tok = ssh_xcalloc(1, sizeof (*tok));
  tok->auth_type = SSH1_AUTH_TYPE_RSA_FILE;
  tok->filename = ssh_xstrdup(filename);
  ssh1_add_auth_token(tok, ssh1);
  return;
}

#ifdef WITH_PGP
void ssh1_add_pgp_key_auth_token(const char *pgp_keyring,
                                 const char *pgp_name,
                                 const char *pgp_fingerprint,
                                 SshUInt32 pgp_id,
                                 Ssh1Client ssh1)
{
  Ssh1AuthToken tok;

  SSH_ASSERT(pgp_keyring != NULL);
  tok = ssh_xcalloc(1, sizeof (*tok));
  tok->auth_type = SSH1_AUTH_TYPE_PGP_KEY;
  tok->pgp_keyring = ssh_xstrdup(pgp_keyring);
  if (pgp_name)
    tok->pgp_name = ssh_xstrdup(pgp_name);
  else if (pgp_fingerprint)
    tok->pgp_fingerprint = ssh_xstrdup(pgp_fingerprint);
  else
    tok->pgp_id = pgp_id;
  ssh1_add_auth_token(tok, ssh1);
}
#endif /* WITH_PGP */
void ssh1_add_agent_auth_token(SshAgent agent,
                               const unsigned char *certs,
                               size_t certs_len,
                               Ssh1Client ssh1)
{
  Ssh1AuthToken tok;

  SSH_ASSERT(agent != NULL);
  if (ssh1->remote_server_protocol_version == 10)
    return; /* RSA authentication is insecure in 1.0 protocol. */
  tok = ssh_xcalloc(1, sizeof (*tok));
  tok->auth_type = SSH1_AUTH_TYPE_AGENT;
  tok->agent = agent;
  tok->certs = ssh_xmemdup(certs, certs_len);
  tok->certs_len = certs_len;
  ssh1_add_auth_token(tok, ssh1);
  return;
}

void ssh1_add_auth_token(Ssh1AuthToken tok, Ssh1Client ssh1)
{
  if (ssh1->auth_list == NULL)
    {
      ssh1->auth_list = tok;
      ssh1->auth_last = tok;
      ssh1->auth_current = tok;
    }
  else
    {
      ssh1->auth_last->next = tok;
      ssh1->auth_last = tok;
    }
  return;
}

void ssh1_flush_auth_tokens(Ssh1Client ssh1)
{
  Ssh1AuthToken tok;

  if (ssh1->agent != NULL)
    {
      SSH_DEBUG(4, ("Shutting down the agent connection."));
      ssh_agent_close(ssh1->agent);
      ssh1->agent = NULL;
    }
  while (ssh1->auth_list != NULL)
    {
      tok = ssh1->auth_list;
      ssh1->auth_list = ssh1->auth_list->next;
      if (tok->password != NULL)
        {
          memset(tok->password, 0, strlen(tok->password));
          ssh_xfree(tok->password);
        }
      if (tok->filename != NULL)
        {
          memset(tok->filename, 0, strlen(tok->filename));
          ssh_xfree(tok->filename);
        }
#ifdef WITH_PGP
      if (tok->pgp_keyring != NULL)
        {
          memset(tok->pgp_keyring, 0, strlen(tok->pgp_keyring));
          ssh_xfree(tok->pgp_keyring);
        }
      if (tok->pgp_name != NULL)
        {
          memset(tok->pgp_name, 0, strlen(tok->pgp_name));
          ssh_xfree(tok->pgp_name);
        }
      if (tok->pgp_keyblob != NULL)
        {
          memset(tok->pgp_keyblob, 0, tok->pgp_keyblob_len);
          ssh_xfree(tok->pgp_keyblob);
        }
      if (tok->pgp_fingerprint != NULL)
        {
          memset(tok->pgp_fingerprint, 0, strlen(tok->pgp_fingerprint));
          ssh_xfree(tok->pgp_fingerprint);
        }
#endif /* WITH_PGP */
      if (tok->certs != NULL)
        {
          memset(tok->certs, 0, tok->certs_len);
          ssh_xfree(tok->certs);
        }
      if (tok->comment != NULL)
        {
          memset(tok->comment, 0, strlen(tok->comment));
          ssh_xfree(tok->comment);
        }
      memset(tok, 'F', sizeof (*tok));
      ssh_xfree(tok);
    }
  ssh1->auth_last = NULL;
  ssh1->auth_current = NULL;
  return;
}

void ssh1_add_auth_tokens_completion(Ssh1Client ssh1)
{
  struct SshConfigPrivateKey **auth_keys;
  char *remote_server;
  int i;

  if (ssh1->remote_server_protocol_version == 10)
    {
      ssh_warning("No public key authentication available for 1.0 protocol.");
    }
  else
    {

      /* Add private keys we can try. */
      remote_server = ssh_xstrdup(ssh1->client->remote_server);
      auth_keys = ssh_privkey_list(ssh1->client->user_data,
                                   remote_server,
                                   ssh1->client->config);
      ssh_xfree(remote_server);






      if (auth_keys != NULL)
        {
          for (i = 0; auth_keys[i]; i++)
            {
              if (auth_keys[i]->keyfile)
                {
                  ssh1_add_rsa_file_auth_token(auth_keys[i]->keyfile, ssh1);
                }
#ifdef WITH_PGP
              else if (auth_keys[i]->pgp_keyring && auth_keys[i]->pgp_name)
                {
                  ssh1_add_pgp_key_auth_token(auth_keys[i]->pgp_keyring,
                                              auth_keys[i]->pgp_name,
                                              NULL,
                                              0,
                                              ssh1);
                }
              else if (auth_keys[i]->pgp_keyring && auth_keys[i]->pgp_fingerprint)
                {
                  ssh1_add_pgp_key_auth_token(auth_keys[i]->pgp_keyring,
                                              NULL,
                                              auth_keys[i]->pgp_fingerprint,
                                              0,
                                              ssh1);
                }
              else if (auth_keys[i]->pgp_keyring && auth_keys[i]->pgp_id)
                {
                  ssh1_add_pgp_key_auth_token(auth_keys[i]->pgp_keyring,
                                              NULL,
                                              NULL,
                                              auth_keys[i]->pgp_id,
                                              ssh1);
                }
#endif /* WITH_PGP */
              ssh_xfree(auth_keys[i]->keyfile);
#ifdef WITH_PGP
              ssh_xfree(auth_keys[i]->pgp_keyring);
              ssh_xfree(auth_keys[i]->pgp_name);
              ssh_xfree(auth_keys[i]->pgp_fingerprint);
#endif /* WITH_PGP */
              ssh_xfree(auth_keys[i]);
            }
          ssh_xfree(auth_keys);
        }
    }
#ifdef SSH1_CAN_ASK_PASSWORD
  /* Add password. */

  ssh1_add_password_auth_token(NULL, ssh1);



#endif /* SSH1_CAN_ASK_PASSWORD */
  ssh1->auth_current = ssh1->auth_list;
  ssh1->auth_round = 1;
  if (ssh1->begin_authentication_after_adding_auth_tokens)
    {
      if (ssh1->auth_current == NULL)
        {
          ssh_warning("Authentication failed.  No authentication methods.");
          ssh1->exit_value = -1;
          ssh1_protocol_destroy(ssh1, FALSE);
          return;
        }
      ssh1_try_auth_method(ssh1);
    }
  return;
}

void ssh1_add_auth_tokens(Boolean begin_authentication, Ssh1Client ssh1)
{
  const char *agent_path;
  ssh1->begin_authentication_after_adding_auth_tokens = begin_authentication;

  if (ssh1->remote_server_protocol_version == 10)
    {
      /* Agent authentication phase is skipped for protocol 1.0 */
      ssh_warning("No agent authentication available for 1.0 protocol.");
      ssh1_add_auth_tokens_completion(ssh1);
      return;
    }

  agent_path = getenv(SSH_AGENT_VAR);
  if (agent_path == NULL)
    agent_path = getenv(SSH_AA_VAR);
  if (agent_path == NULL)
    {
      ssh1_add_auth_tokens_completion(ssh1);
      return;
    }



  ssh_agent_open(ssh1_agent_open_done, (void *)ssh1);
  return;
}

void ssh1_agent_open_done(SshAgent agent, void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;

  if (agent == NULL)
    {
      SSH_DEBUG(5, ("Unable to connect authentication agent."));
      ssh1_add_auth_tokens_completion(ssh1);
      return;
    }
  ssh1->agent = agent;
  ssh_agent_list(ssh1->agent, ssh1_agent_list_done, (void *)ssh1);
  return;
}

void ssh1_agent_list_done(SshAgentError error,
                          unsigned int num_keys,
                          SshAgentKeyInfo keys,
                          void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;
  unsigned int i;

  if (error != SSH_AGENT_ERROR_OK)
    {
      SSH_DEBUG(5, ("Authentication agent can't list keys (%s).",
                    (int)error));
      ssh_agent_close(ssh1->agent);
      ssh1->agent = NULL;
      ssh1_add_auth_tokens_completion(ssh1);
      return;
    }
  for (i = 0; i < num_keys; i++)
    ssh1_add_agent_auth_token(ssh1->agent,
                              keys[i].certs,
                              keys[i].certs_len,
                              ssh1);
  ssh1_add_auth_tokens_completion(ssh1);
  return;
}

void ssh1_agent_op_done(SshAgentError error,
                        const unsigned char *result,
                        size_t result_len,
                        void *context)
{
  Ssh1Client ssh1 = (Ssh1Client)context;
  unsigned char *response;
  size_t response_len;
  SshBufferStruct reply[1];

  if ((error != SSH_AGENT_ERROR_OK) || (result_len != 16))
    {
      response_len = 16;
      response = ssh_xcalloc(response_len, sizeof (char));
    }
  else
    {
      response_len = result_len;
      response = ssh_xmemdup(result, result_len);
    }
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_AUTH_RSA_RESPONSE);
  ssh1_encode_data(reply, (unsigned char *)response, response_len);
  ssh_xfree(response);
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1->mode = SSH1_RSA_AUTHENTICATION_SENT;
  ssh1_input_available(ssh1);
  return;
}

void ssh1_try_auth_method(Ssh1Client ssh1)
{
  char *password, *password_prompt;
  SshMPIntStruct modulus;
  unsigned long magic;
  char *name;
  size_t password_len, random_limit;
  SshUInt32 x;
  SshPublicKey tmpkey;
  SshBufferStruct reply[1];
  SshCryptoStatus cr;

  SSH_ASSERT(ssh1->auth_current != NULL);
  switch (ssh1->auth_current->auth_type)
    {
    case SSH1_AUTH_TYPE_PASSWORD:










      SSH_DEBUG(5, ("trying SSH1_AUTH_TYPE_PASSWORD"));
      if (((((SshUInt32)1) << SSH1_AUTH_PASSWORD) &
           SSH1_SUPPORTED_AUTHENTICATIONS_MASK &
           ssh1->supported_authentications_mask) == 0)
        {
          goto skip_to_the_next_method;
        }
      if (ssh1->auth_current->password != NULL)
        {
          SSH_DEBUG(5, ("Attempting authentication with known password."));
          /* password = ssh_xstrdup(ssh1->auth_current->password); */
          /* Set the password NULL so we'll ask it the next time. */
          password = ssh1->auth_current->password;
          ssh1->auth_current->password = NULL;
        }
      else
        {
          if (!ssh1->remote_server_allows_ignore_packets_in_startup ||
              ssh1->client->config->ssh1_no_ignore_packets_in_password_auth)
            {
              ssh_warning
                ("Not sending ignore packets during authentication in "
                 "ssh1-emulation. %s",
                 ssh1->remote_server_allows_ignore_packets_in_startup ?
                 "" : "(remote protocol version too old)");
              ssh_warning("Password length can be deduced from packets!");
            }

          SSH_DEBUG(5, ("Attempting interactive password authentication."));
          if (ssh1->client->config->password_prompt == NULL)
            {
              ssh_dsprintf(&password_prompt,
                           "%s's password: ",
                           ssh1->client->remote_user);
            }
          else
            {
              password_prompt =
                ssh_xstrdup(ssh1->client->config->password_prompt);
            }
          if (!ssh1->client->config->batch_mode)
            password = ssh_read_passphrase(password_prompt, FALSE);
          else
            password = NULL;

          ssh_xfree(password_prompt);
















          if (password == NULL)
            password = ssh_xstrdup("");
        }
      ssh_buffer_init(reply);
      password_len = strlen(password);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_AUTH_PASSWORD);
      ssh1_encode_string(reply, password, password_len);
      memset(password, 0, password_len);
      ssh_xfree(password);
      if (ssh1->remote_server_allows_ignore_packets_in_startup &&
          !ssh1->client->config->ssh1_no_ignore_packets_in_password_auth)
        {
          for (x = 0; x < password_len; x++)
            {
              if (ssh1_send_random_ignore_packet(x, ssh1) == FALSE)
                {
                  ssh_buffer_uninit(reply);
                  return;
                }
            }
        }
      SSH_DEBUG(7, ("Sending %u byte password packet",
                    (unsigned int)(ssh_buffer_len(reply) - 5)));
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1->mode = SSH1_PASSWORD_SENT;
      if (ssh1->remote_server_allows_ignore_packets_in_startup &&
          !ssh1->client->config->ssh1_no_ignore_packets_in_password_auth)
        {
          random_limit =
            ((password_len + 32) & (~0xf)) + (ssh_random_get_byte() % 16);
          for (x = password_len + 1; x < random_limit; x++)
            {
              if (ssh1_send_random_ignore_packet(x, ssh1) == FALSE)
                {
                  return;
                }
            }
        }
      ssh1_input_available(ssh1);
      return;

    case SSH1_AUTH_TYPE_RSA_FILE:
#ifdef WITH_PGP
    case SSH1_AUTH_TYPE_PGP_KEY:
#endif /* WITH_PGP */
      if (ssh1->auth_current->auth_type == SSH1_AUTH_TYPE_RSA_FILE)
        {










          SSH_DEBUG(5, ("trying SSH1_AUTH_TYPE_RSA_FILE %s",
                        ssh1->auth_current->filename));
          if (((((SshUInt32)1) << SSH1_AUTH_RSA) &
               SSH1_SUPPORTED_AUTHENTICATIONS_MASK &
               ssh1->supported_authentications_mask) == 0)
            {
              goto skip_to_the_next_method;
            }
          if (ssh1->auth_current->filename)
            {
              if (ssh1->auth_current->certs == NULL)
                {
                  ssh_dsprintf(&name, "%s.pub", ssh1->auth_current->filename);
                  magic = ssh2_key_blob_read(ssh1->client->user_data,
                                             name,
                                             TRUE,
                                             &(ssh1->auth_current->comment),
                                             &(ssh1->auth_current->certs),
                                             &(ssh1->auth_current->certs_len),
                                             NULL);
                  ssh_xfree(name);
                  if (magic != SSH_KEY_MAGIC_PUBLIC)
                    {
                      ssh_warning("Could not read public key file %s.pub",
                                  ssh1->auth_current->filename);
                      ssh_xfree(ssh1->auth_current->certs);
                      ssh1->auth_current->certs = NULL;
                      ssh_xfree(ssh1->auth_current->comment);
                      ssh1->auth_current->comment = NULL;
                      goto skip_to_the_next_method;
                    }
                }
            }
        }
#ifdef WITH_PGP
      else if (ssh1->auth_current->auth_type == SSH1_AUTH_TYPE_PGP_KEY)
        {
          SshPgpSecretKey pgpkey;

          if (ssh1->auth_current->certs == NULL)
            {
              if (ssh1->auth_current->pgp_keyblob == NULL)
                {
                  if (ssh1->auth_current->pgp_name)
                    ssh2_find_pgp_secret_key_with_name
                      (ssh1->client->user_data,
                       ssh1->auth_current->pgp_keyring,
                       ssh1->auth_current->pgp_name,
                       &(ssh1->auth_current->pgp_keyblob),
                       &(ssh1->auth_current->pgp_keyblob_len),
                       &(ssh1->auth_current->comment));
                  else if (ssh1->auth_current->pgp_fingerprint)
                    ssh2_find_pgp_secret_key_with_fingerprint
                      (ssh1->client->user_data,
                       ssh1->auth_current->pgp_keyring,
                       ssh1->auth_current->pgp_fingerprint,
                       &(ssh1->auth_current->pgp_keyblob),
                       &(ssh1->auth_current->pgp_keyblob_len),
                       &(ssh1->auth_current->comment));
                  else
                    ssh2_find_pgp_secret_key_with_id
                      (ssh1->client->user_data,
                       ssh1->auth_current->pgp_keyring,
                       ssh1->auth_current->pgp_id,
                       &(ssh1->auth_current->pgp_keyblob),
                       &(ssh1->auth_current->pgp_keyblob_len),
                       &(ssh1->auth_current->comment));
                }
              if (ssh1->auth_current->pgp_keyblob != NULL)
                {
                  if (ssh_pgp_secret_key_decode
                      (ssh1->auth_current->pgp_keyblob,
                       ssh1->auth_current->pgp_keyblob_len,
                       &pgpkey) > 0)
                    {
                      if (pgpkey->public_key->key != NULL)
                        {
                          ssh1->auth_current->certs_len =
                            ssh_encode_pubkeyblob
                            (pgpkey->public_key->key,
                             &(ssh1->auth_current->certs));
                        }
                      ssh_pgp_secret_key_free(pgpkey);
                    }
                }
            }
        }
#endif /* WITH_PGP */
      else
        {
          SSH_NOTREACHED;
        }
      if (ssh1->auth_current->certs == NULL)
        {
          goto skip_to_the_next_method;
        }
      tmpkey = ssh_decode_pubkeyblob(ssh1->auth_current->certs,
                                     ssh1->auth_current->certs_len);
      if (tmpkey == NULL)
        {
          goto skip_to_the_next_method;
        }
      name = ssh_public_key_name(tmpkey);
      if ((name == NULL) || (strstr(name, "{rsa-") == NULL))
        {
          ssh_xfree(name);
          goto skip_to_the_next_method;
        }
      ssh_xfree(name);
      ssh_mp_init(&modulus);
      cr = ssh_public_key_get_info(tmpkey,
                                   SSH_PKF_MODULO_N, &modulus,
                                   SSH_PKF_END);
      ssh_public_key_free(tmpkey);
      if (cr != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&modulus);
          goto skip_to_the_next_method;
        }
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_AUTH_RSA);
      ssh1_encode_mp(reply, &modulus);
      ssh_mp_clear(&modulus);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1->mode = SSH1_RSA_AUTHENTICATION_REQUESTED;
      ssh1_input_available(ssh1);
      return;
      /*NOTREACHED*/

    case SSH1_AUTH_TYPE_AGENT:
      if (((((SshUInt32)1) << SSH1_AUTH_RSA) &
           SSH1_SUPPORTED_AUTHENTICATIONS_MASK &
           ssh1->supported_authentications_mask) == 0)
        {
          goto skip_to_the_next_method;
        }
      SSH_ASSERT(ssh1->auth_current->auth_type == SSH1_AUTH_TYPE_AGENT);
      SSH_ASSERT(ssh1->auth_current->certs != NULL);
      SSH_ASSERT(ssh1->auth_current->agent != NULL);
      SSH_ASSERT(ssh1->auth_current->agent == ssh1->agent);
      tmpkey = ssh_decode_pubkeyblob(ssh1->auth_current->certs,
                                     ssh1->auth_current->certs_len);
      if (tmpkey == NULL)
        {
          goto skip_to_the_next_method;
        }
      name = ssh_public_key_name(tmpkey);
      if ((name == NULL) || (strstr(name, "{rsa-") == NULL))
        {
          ssh_xfree(name);
          goto skip_to_the_next_method;
        }
      ssh_xfree(name);
      ssh_mp_init(&modulus);
      cr = ssh_public_key_get_info(tmpkey,
                                   SSH_PKF_MODULO_N, &modulus,
                                   SSH_PKF_END);
      ssh_public_key_free(tmpkey);
      if (cr != SSH_CRYPTO_OK)
        {
          ssh_mp_clear(&modulus);
          goto skip_to_the_next_method;
        }
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_AUTH_RSA);
      ssh1_encode_mp(reply, &modulus);
      ssh_mp_clear(&modulus);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1->mode = SSH1_AGENT_AUTHENTICATION_REQUESTED;
      ssh1_input_available(ssh1);
      return;
      /*NOTREACHED*/

    default:
      ssh1_protocol_error("Internal error in authentication.", ssh1);
      return;
    }
  /*NOTREACHED*/

 skip_to_the_next_method:
  if ((ssh1->auth_current->next == NULL) &&
      (ssh1->auth_round > 2))
    {
      ssh_warning("Authentication failed.");
      ssh1->exit_value = -1;
      ssh1_protocol_destroy(ssh1, FALSE);
      return;
    }
  if (ssh1->auth_current->next != NULL)
    {
      ssh1->auth_current = ssh1->auth_current->next;
    }
  else
    {
      ssh1->auth_current = ssh1->auth_list;
      ssh1->auth_round++;
    }
  ssh1_try_auth_method(ssh1);
  return;
}

void ssh1_request_compression(Ssh1Client ssh1)
{
  SshBufferStruct reply[1];
  SshCompression in_compress, out_compress;

  if (ssh1->client->config->compression)
    {
      out_compress =
        ssh_compress_allocate("zlib",
                              ssh1->client->config->compression_level, TRUE);
      in_compress =
        ssh_compress_allocate("zlib",
                              ssh1->client->config->compression_level, FALSE);
      if ((out_compress == NULL) || (in_compress == NULL))
        {
          ssh_warning("local client can't handle compression.");
          if (out_compress != NULL)
            {
              ssh_compress_free(out_compress);
              out_compress = NULL;
            }
          if (in_compress != NULL)
            {
              ssh_compress_free(in_compress);
              in_compress = NULL;
            }
        }
    }
  else
    {
      out_compress = NULL;
      in_compress = NULL;
    }

  if ((out_compress != NULL) && (in_compress != NULL))
    {
      SSH_DEBUG(5, ("Requesting compression."));
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_REQUEST_COMPRESSION);
      ssh1_encode_int(reply,
                      (SshUInt32)ssh1->client->config->compression_level);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1->mode = SSH1_MODE_COMPRESSION_REQUESTED;
      ssh1->in_compress = in_compress;
      ssh1->out_compress = out_compress;
    }
  else
    {
      SSH_DEBUG(5, ("Not requesting compression."));
      ssh1_request_pty(ssh1);
    }
  ssh1_input_available(ssh1);
  return;
}

void ssh1_request_pty(Ssh1Client ssh1)
{
  SshBufferStruct reply[1];
  unsigned char *tty_modes;
  size_t tty_modes_len;

  if (ssh1->client->allocate_pty)
    {
      SSH_DEBUG(5, ("Requesting pty."));
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_REQUEST_PTY);
      ssh1_encode_string(reply,
                         (ssh1->client->terminal_type ?
                          ssh1->client->terminal_type :
                          SSH1_DEFAULT_TERMINAL_TYPE),
                         (ssh1->client->terminal_type ?
                          strlen(ssh1->client->terminal_type) :
                          strlen(SSH1_DEFAULT_TERMINAL_TYPE)));
      ssh1_encode_int(reply, (SshUInt32)24);
      ssh1_encode_int(reply, (SshUInt32)80);
      ssh1_encode_int(reply, (SshUInt32)0);
      ssh1_encode_int(reply, (SshUInt32)0);
      if (isatty(fileno(stdin)))
        {







          ssh1_encode_tty_flags(fileno(stdin), &tty_modes, &tty_modes_len);
          ssh1_encode_data(reply, tty_modes, tty_modes_len);
          ssh_xfree(tty_modes);

        }
      else
        {
          /* Terminal mode end marker byte (0). */
          ssh1_encode_byte(reply, (SshUInt8)0);
        }
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1->mode = SSH1_MODE_PTY_REQUESTED;
    }
  else
    {
      SSH_DEBUG(5, ("Not requesting pty."));
      ssh1_request_remote_forward(ssh1);
    }
  ssh1_input_available(ssh1);
  return;
}

void ssh1_request_remote_forward(Ssh1Client ssh1)
{
  SshBufferStruct reply[1];

  if (ssh1->requested_remote_forward)
    ssh1->requested_remote_forward = ssh1->requested_remote_forward->next;
  else
    ssh1->requested_remote_forward = ssh1->client->config->remote_forwards;
  if (! ssh1->requested_remote_forward)
    {
      SSH_DEBUG(5, ("No more remote forwards."));
      ssh1_request_agent_forward(ssh1);
      return;
    }
  SSH_DEBUG(5, ("Requesting remote forward %u:%s:%u.",
                strtoul(ssh1->requested_remote_forward->port,
                        NULL, 0),
                ssh1->requested_remote_forward->connect_to_host,
                strtoul(ssh1->requested_remote_forward->connect_to_port,
                        NULL, 0)));
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_PORT_FORWARD_REQUEST);
  ssh1_encode_int(reply,
                  strtoul(ssh1->requested_remote_forward->port,
                  NULL,
                  0));
  ssh1_encode_string(reply,
                     ssh1->requested_remote_forward->connect_to_host,
                     strlen(ssh1->requested_remote_forward->connect_to_host));
  ssh1_encode_int(reply,
                  strtoul(ssh1->requested_remote_forward->connect_to_port,
                          NULL,
                          0));
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1->mode = SSH1_MODE_REMOTE_FORWARD_REQUESTED;
  ssh1_input_available(ssh1);
  return;
}

void ssh1_request_agent_forward(Ssh1Client ssh1)
{
  SshBufferStruct reply[1];

  if (! ssh1->client->config->forward_agent)
    {
      ssh1_request_x11_forward(ssh1);
      return;
    }
  SSH_DEBUG(4, ("Requesting agent forward."));
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply,
                   (SshUInt8)SSH1_PACKET_CMSG_AGENT_REQUEST_FORWARDING);
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1->mode = SSH1_MODE_AGENT_FORWARD_REQUESTED;
  ssh1_input_available(ssh1);
}

void ssh1_request_x11_forward(Ssh1Client ssh1)
{
  SshBufferStruct reply[1];
  char *display, *help;

  if (! ssh1->client->config->forward_agent)
    {
      ssh1_init_local_forward(ssh1);
      return;
    }

  display = getenv("DISPLAY");



  if (! display)
    {
      SSH_DEBUG(4, ("DISPLAY not set.  No X11 forwarding."));
      ssh1_init_local_forward(ssh1);
      return;
    }
  help = display;
  help = strchr(help, ':');
  if (help)
    help = strchr(help, '.');
  if (help)
    ssh1->x11_screen_number = atoi(help + 1);
  else
    ssh1->x11_screen_number = 0;
  ssh1_init_x11_auth_data(ssh1, display);
  SSH_DEBUG(4, ("Requesting X11 forward."));
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply,
                   (SshUInt8)SSH1_PACKET_CMSG_X11_REQUEST_FORWARDING);
  ssh1_encode_string(reply,
                     ssh1->x11_real_auth_protocol,
                     ssh1->x11_real_auth_protocol_len);
  ssh1_encode_string(reply,
                     ssh1->x11_fake_data_hex,
                     strlen(ssh1->x11_fake_data_hex));
  if (ssh1->protocol_flags & SSH1_PROTOFLAG_SCREEN_NUMBER)
    {
      ssh1_encode_int(reply,
                      ssh1->x11_screen_number);
    }
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1->mode = SSH1_MODE_X11_FORWARD_REQUESTED;
  ssh1_input_available(ssh1);
}

void ssh1_init_local_forward(Ssh1Client ssh1)
{
  Ssh1LocalTcpListener ssh1_listener;

  while (1)
    {
      if (ssh1->requested_local_forward)
        ssh1->requested_local_forward = ssh1->requested_local_forward->next;
      else
        ssh1->requested_local_forward = ssh1->client->config->local_forwards;
      if (! ssh1->requested_local_forward)
        {
          SSH_DEBUG(5, ("No more local forwards."));
          ssh1_request_command(ssh1);
          return;
        }
      if (ssh1->requested_local_forward->protocol &&
          (strcasecmp(ssh1->requested_local_forward->protocol, "tcp") != 0))
        {
          ssh_warning("forwarding protocol \"%s\" defaults to \"tcp\" "
                      "in ssh1 compatibility mode",
                      ssh1->requested_local_forward->protocol);
        }
      SSH_DEBUG(5, ("Initializing local forward %u:%s:%u.",
                    strtoul(ssh1->requested_local_forward->port,
                            NULL, 0),
                    ssh1->requested_local_forward->connect_to_host,
                    strtoul(ssh1->requested_local_forward->connect_to_port,
                            NULL, 0)));
      ssh1_listener = ssh_xcalloc(1, sizeof (*ssh1_listener));
      ssh1_listener->ssh1 = ssh1;
      ssh1_listener->forward = ssh1->requested_local_forward;
      ssh1_listener->listener = ssh_tcp_make_listener(
                                 (ssh1->client->config->gateway_ports ?
                                  ssh1->requested_local_forward->local_addr :
                                  "127.0.0.1"),
                                 ssh1->requested_local_forward->port,
                                 NULL,
                                 ssh1_local_forward_start,
                                 ssh1_listener);
      if (ssh1_listener->listener == NULL)
        {
          ssh_warning("Local TCP/IP forwarding for port %s failed.",
                      ssh1->requested_local_forward->port);
          ssh_xfree(ssh1_listener);
        }
      else
        {
          ssh1_listener->next = ssh1->local_tcp_listeners;
          ssh1->local_tcp_listeners = ssh1_listener;
        }
    }
}

void ssh1_uninit_local_forward(Ssh1Client ssh1)
{
  Ssh1LocalTcpListener next;

  while (ssh1->local_tcp_listeners)
    {
      next = ssh1->local_tcp_listeners->next;
      ssh_tcp_destroy_listener(ssh1->local_tcp_listeners->listener);
      ssh_xfree(ssh1->local_tcp_listeners);
      ssh1->local_tcp_listeners = next;
    }
}

void ssh1_request_command(Ssh1Client ssh1)
{
  SshBufferStruct reply[1];
  size_t x;

  ssh_buffer_init(reply);
  if (ssh1->client->remote_command)
    {
      SSH_DEBUG(5, ("Requesting command."));
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_EXEC_CMD);
      ssh1_encode_string(reply,
                         ssh1->client->remote_command,
                         strlen(ssh1->client->remote_command));
    }
  else
    {
      SSH_DEBUG(5, ("Requesting shell."));
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_EXEC_SHELL);
    }
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1->mode = SSH1_MODE_INTERACTIVE;

  if (ssh1->have_pty)
    {
      ssh_enter_raw_mode(-1, FALSE);
    }

  while (ssh_buffer_len(ssh1->stdin_pre_buf) > 0)
    {
      SSH_DEBUG(5, ("Initial data %d bytes.",
                    ssh_buffer_len(ssh1->stdin_pre_buf)));
      x = ((ssh_buffer_len(ssh1->stdin_pre_buf) <= 0x1000) ?
           ssh_buffer_len(ssh1->stdin_pre_buf) : 0x1000);
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_STDIN_DATA);
      ssh1_encode_string(reply,
                         (char *)(ssh_buffer_ptr(ssh1->stdin_pre_buf)),
                         x);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh_buffer_consume(ssh1->stdin_pre_buf, x);
    }
  if (ssh1->have_stdin_eof)
    {
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_EOF);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1->mode = SSH1_MODE_INTERACTIVE_AFTER_EOF;
      SSH_DEBUG(5, ("Sent stdin EOF to the server."));
    }
  else
    {
      SSH_DEBUG(5, ("Entering interactive mode."));
      ssh1_stdin_available(ssh1);
    }
  ssh1_input_available(ssh1);
  if (ssh1->have_pty)
    {
#ifndef WIN32
#ifdef SIGWINCH
      ssh_register_signal(SIGWINCH, ssh1_window_size_change_sig, (void *)ssh1);
#endif /* SIGWINCH */
      ssh1_window_size_change_sig(0, (void *)ssh1);
#endif /* WIN32 */
    }
  return;
}

Boolean ssh1_send_random_ignore_packet(SshUInt32 len,
                                       Ssh1Client ssh1)
{
  SshBufferStruct reply[1];
  unsigned char *buf;
  SshUInt32 j;

  ssh_buffer_init(reply);
  buf = ssh_xmalloc(len);
  for (j = 0; j < len; j++)
    buf[j] = ssh_random_get_byte();
  ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_MSG_IGNORE);
  ssh1_encode_string(reply, (char *)buf, len);
  SSH_DEBUG(7, ("Sending %u byte ignore packet", (unsigned int)len));
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_xfree(buf);
      ssh_buffer_uninit(reply);
      return FALSE;
    }
  ssh_xfree(buf);
  ssh_buffer_uninit(reply);
  return TRUE;
}

void ssh1_handle_packet_msg_disconnect(SshBuffer packet, Ssh1Client ssh1)
{
  char *message;

  SSH_DEBUG(5, ("Remote server disconnected."));
  if (ssh1_decode_string(packet, &message, NULL))
    {
      ssh_warning("Server disconnected (%s).", message);
      ssh_xfree(message);
    }
  else
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed disconnect packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  ssh1->exit_value = -1;
  ssh1_protocol_destroy(ssh1, FALSE);
  return;
}

void ssh1_handle_packet_smsg_public_key(SshBuffer packet, Ssh1Client ssh1)
{
  /* Incoming protocol packet contents */
  unsigned char *anti_spoofing_cookie;
  SshUInt32 server_key_bits;
  SshMPIntStruct server_key_public_exponent;
  SshMPIntStruct server_key_public_modulus;
  SshUInt32 host_key_bits;
  SshMPIntStruct host_key_public_exponent;
  SshMPIntStruct host_key_public_modulus;
  SshUInt32 protocol_flags;
  SshUInt32 supported_ciphers_mask;
  SshUInt32 supported_authentications_mask;
  /* Other stuff */
  unsigned char *pk_buffer1;
  size_t pk_buffer1_len;
  unsigned char *pk_buffer2;
  size_t pk_buffer2_len;
  SshBufferStruct reply[1];
  SshCryptoStatus cr;
  int i;
  SshPublicKey k1, k2;







  SSH_DEBUG(5, ("mode = %d", ssh1->mode));

  if (ssh1->mode != SSH1_MODE_VERSION_SENT)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Unexpected smsg_public_key.", ssh1);
      return;
    }

  ssh_mp_init(&server_key_public_exponent);
  ssh_mp_init(&server_key_public_modulus);
  ssh_mp_init(&host_key_public_exponent);
  ssh_mp_init(&host_key_public_modulus);
  if (! (ssh1_decode_data(packet, &anti_spoofing_cookie, 8) &&
         ssh1_decode_int(packet, &server_key_bits) &&
         ssh1_decode_mp(packet, &server_key_public_exponent) &&
         ssh1_decode_mp(packet, &server_key_public_modulus) &&
         ssh1_decode_int(packet, &host_key_bits) &&
         ssh1_decode_mp(packet, &host_key_public_exponent) &&
         ssh1_decode_mp(packet, &host_key_public_modulus) &&
         ssh1_decode_int(packet, &protocol_flags) &&
         ssh1_decode_int(packet, &supported_ciphers_mask) &&
         ssh1_decode_int(packet, &supported_authentications_mask)))
    {
      ssh_mp_clear(&server_key_public_exponent);
      ssh_mp_clear(&server_key_public_modulus);
      ssh_mp_clear(&host_key_public_exponent);
      ssh_mp_clear(&host_key_public_modulus);
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed public_key packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  SSH_DEBUG(5, ("server key len = %d, host key len = %d",
                (int)ssh_mp_get_size(&server_key_public_modulus, 2),
                (int)ssh_mp_get_size(&host_key_public_modulus, 2)));
  if (ssh_mp_cmp_ui(&server_key_public_exponent, 2) <= 0)
    {
      ssh1_protocol_error("Server key exponent < 2.  Possible attack.", ssh1);
      return;
    }
  if (ssh_mp_get_size(&server_key_public_modulus, 2) < 512)
    {
      ssh1_protocol_error("Server key size < 512.  Possible attack.", ssh1);
      return;
    }
  if (ssh_mp_cmp_ui(&host_key_public_exponent, 2) <= 0)
    {
      ssh1_protocol_error("Host key exponent < 2.  Possible attack.", ssh1);
      return;
    }
  if (ssh_mp_get_size(&host_key_public_modulus, 2) < 512)
    {
      ssh1_protocol_error("Host key size < 512.  Possible attack.", ssh1);
      return;
    }
  memcpy(ssh1->anti_spoofing_cookie, anti_spoofing_cookie, 8);
  ssh_xfree(anti_spoofing_cookie);

  cr = ssh_public_key_define(&(ssh1->server_key),
                             SSH_CRYPTO_RSA,
                             SSH_PKF_MODULO_N, &server_key_public_modulus,
                             SSH_PKF_PUBLIC_E, &server_key_public_exponent,
                             SSH_PKF_ENCRYPT, "rsa-pkcs1-none",
                             SSH_PKF_END);

  if (cr != SSH_CRYPTO_OK)
    {
      ssh_mp_clear(&server_key_public_exponent);
      ssh_mp_clear(&server_key_public_modulus);
      ssh_mp_clear(&host_key_public_exponent);
      ssh_mp_clear(&host_key_public_modulus);
      ssh1_protocol_error("Unable to define server key.  "
                          "Most likely, your cryptographic library "
                          "doesn't support the RSA algorithm.", ssh1);
      return;
    }

  cr = ssh_public_key_define(&(ssh1->host_key),
                             SSH_CRYPTO_RSA,
                             SSH_PKF_MODULO_N, &host_key_public_modulus,
                             SSH_PKF_PUBLIC_E, &host_key_public_exponent,
                             SSH_PKF_ENCRYPT, "rsa-pkcs1-none",
                             SSH_PKF_END);

  if (cr != SSH_CRYPTO_OK)
    {
      ssh_mp_clear(&server_key_public_exponent);
      ssh_mp_clear(&server_key_public_modulus);
      ssh_mp_clear(&host_key_public_exponent);
      ssh_mp_clear(&host_key_public_modulus);
      ssh1_protocol_error("Unable to define host key.  "
                          "Most likely, your cryptographic library "
                          "doesn't support the RSA algorithm.", ssh1);
      return;
    }
  if (ssh_mp_cmp(&host_key_public_modulus, &server_key_public_modulus) > 0)
    {
      k1 = ssh1->server_key;
      k2 = ssh1->host_key;
    }
  else
    {
      k1 = ssh1->host_key;
      k2 = ssh1->server_key;
    }
  ssh_mp_clear(&server_key_public_exponent);
  ssh_mp_clear(&server_key_public_modulus);
  ssh_mp_clear(&host_key_public_exponent);
  ssh_mp_clear(&host_key_public_modulus);
  ssh1->protocol_flags = protocol_flags;
  ssh1->supported_ciphers_mask = supported_ciphers_mask;
  ssh1->supported_authentications_mask = supported_authentications_mask;



































  if (ssh1_host_key_check(ssh1) == FALSE)
    {
      ssh_warning("Host key check failed.");
      ssh1->exit_value = -1;
      ssh1_protocol_destroy(ssh1, FALSE);
      return;
    }



  pk_buffer1_len = ssh_public_key_max_encrypt_output_len(k1);
  SSH_DEBUG(5, ("max encrypt output len for k1 is %ld.", pk_buffer1_len));
  pk_buffer1 = ssh_xmalloc(pk_buffer1_len);
  cr = ssh_public_key_encrypt(k1,
                              ssh1->session_key,
                              sizeof (ssh1->session_key),
                              pk_buffer1,
                              pk_buffer1_len,
                              &pk_buffer1_len);
  SSH_DEBUG(5, ("ssh_public_key_encrypt returns %d.", (int)cr));
  if (cr != SSH_CRYPTO_OK)
    {
      ssh_xfree(pk_buffer1);
      ssh1_protocol_error("Unable to encrypt session key.", ssh1);
      return;
    }
  pk_buffer2_len = ssh_public_key_max_encrypt_output_len(k2);
  SSH_DEBUG(5, ("max encrypt output len for k2 is %ld.", pk_buffer2_len));

  pk_buffer2 = ssh_xmalloc(pk_buffer2_len);
  cr = ssh_public_key_encrypt(k2,
                              pk_buffer1,
                              pk_buffer1_len,
                              pk_buffer2,
                              pk_buffer2_len,
                              &pk_buffer2_len);
  SSH_DEBUG(5, ("ssh_public_key_encrypt returns %d.", (int)cr));
  ssh_xfree(pk_buffer1);
  if (cr != SSH_CRYPTO_OK)
    {
      ssh_xfree(pk_buffer2);
      ssh1_protocol_error("Unable to encrypt session key.", ssh1);
      return;
    }
  ssh_mp_set_buf(&(ssh1->encrypted_session_key),
                 pk_buffer2,
                 pk_buffer2_len);
  ssh_xfree(pk_buffer2);
  ssh1->chosen_cipher = ssh1_choose_cipher(ssh1->supported_ciphers_mask,
                                           ssh1->client->config->ciphers);
  if (ssh1->chosen_cipher < 0)
    {
      ssh1_protocol_error("Can't select cipher type.", ssh1);
      return;
    }
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_SESSION_KEY);
  ssh1_encode_byte(reply, (SshUInt8)ssh1->chosen_cipher);
  ssh1_encode_data(reply, ssh1->anti_spoofing_cookie, 8);
  ssh1_encode_mp(reply, &(ssh1->encrypted_session_key));
  protocol_flags = 0x00000000;
  if (ssh1->protocol_flags & SSH1_PROTOFLAG_SCREEN_NUMBER)
    protocol_flags |= SSH1_PROTOFLAG_SCREEN_NUMBER;
  if (ssh1->protocol_flags & SSH1_PROTOFLAG_HOST_IN_FWD_OPEN)
    protocol_flags |= SSH1_PROTOFLAG_HOST_IN_FWD_OPEN;
  ssh1_encode_int(reply, protocol_flags);
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh1->mode = SSH1_MODE_SESSION_KEY_SENT;
  ssh_buffer_uninit(reply);
  /* Session key is defined from protocol versions > 1.0 */
  if (ssh1->remote_server_protocol_version != 10)
    {
      if (ssh1_session_id(ssh1, ssh1->session_id) == FALSE)
        {
          ssh1_protocol_error("Can't calculate session id.", ssh1);
          return;
        }
      for (i = 0; i < 16; i++)
        ssh1->session_key[i] ^= ssh1->session_id[i];
    }
  if (ssh1->remote_server_protocol_version == 10)
    {
      /* Protocol 1.0 decoded the session key from the wrong end. */
      for (i = 0; i < 16; i++)
        {
          ssh1->session_key[i] ^= ssh1->session_key[31 - i];
          ssh1->session_key[31 - i] ^= ssh1->session_key[i];
          ssh1->session_key[i] ^= ssh1->session_key[31 - i];
        }
    }
  SSH_DEBUG_HEXDUMP(9,
                    ("session key:"),
                    ssh1->session_key, sizeof (ssh1->session_key));
  /* Set the cipher keys. */
  switch (ssh1->chosen_cipher) {
  case SSH1_CIPHER_NONE:
    break;

  case SSH1_CIPHER_IDEA:
    cr = ssh_cipher_allocate("idea-cfb",
                             ssh1->session_key,
                             16,
                             TRUE,
                             &(ssh1->out_cipher));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    cr = ssh_cipher_allocate("idea-cfb",
                             ssh1->session_key,
                             16,
                             FALSE,
                             &(ssh1->in_cipher));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    ssh1->statistics.cipher_c_to_s = "idea-cfb";
    ssh1->statistics.cipher_s_to_c = "idea-cfb";
    break;

  case SSH1_CIPHER_DES:
    cr = ssh_cipher_allocate("des-cbc",
                             ssh1->session_key,
                             8,
                             TRUE,
                             &(ssh1->out_cipher));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    cr = ssh_cipher_allocate("des-cbc",
                             ssh1->session_key,
                             8,
                             FALSE,
                             &(ssh1->in_cipher));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    ssh1->statistics.cipher_c_to_s = "des-cbc";
    ssh1->statistics.cipher_s_to_c = "des-cbc";
    break;

  case SSH1_CIPHER_3DES:
    cr = ssh_cipher_allocate("des-cbc",
                             ssh1->session_key,
                             8,
                             TRUE,
                             &(ssh1->out_cipher));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    cr = ssh_cipher_allocate("des-cbc",
                             ssh1->session_key + 8,
                             8,
                             FALSE,
                             &(ssh1->out_cipher2));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    cr = ssh_cipher_allocate("des-cbc",
                             ssh1->session_key + 16,
                             8,
                             TRUE,
                             &(ssh1->out_cipher3));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    cr = ssh_cipher_allocate("des-cbc",
                             ssh1->session_key + 16,
                             8,
                             FALSE,
                             &(ssh1->in_cipher));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    cr = ssh_cipher_allocate("des-cbc",
                             ssh1->session_key + 8,
                             8,
                             TRUE,
                             &(ssh1->in_cipher2));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    cr = ssh_cipher_allocate("des-cbc",
                             ssh1->session_key,
                             8,
                             FALSE,
                             &(ssh1->in_cipher3));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    ssh1->statistics.cipher_c_to_s = "3des-icbc";
    ssh1->statistics.cipher_s_to_c = "3des-icbc";
    break;

  case SSH1_CIPHER_ARCFOUR:
    if (ssh1->remote_server_protocol_version != 10)
      {
        cr = ssh_cipher_allocate("arcfour",
                                 ssh1->session_key + 16,
                                 16,
                                 TRUE,
                                 &(ssh1->out_cipher));
        if (cr != SSH_CRYPTO_OK)
          {
            ssh1_protocol_error("Unable to create cipher context.", ssh1);
            return;
          }
        cr = ssh_cipher_allocate("arcfour",
                                 ssh1->session_key,
                                 16,
                                 FALSE,
                                 &(ssh1->in_cipher));
        if (cr != SSH_CRYPTO_OK)
          {
            ssh1_protocol_error("Unable to create cipher context.", ssh1);
            return;
          }
        ssh_warning("Stream cipher with ssh1 protocol is not secure.");
      }
    else
      {
        /* Protocol version 1.0 used the same key for arcfour in both
           directions.  This is of course totally insecure due to the
           stream cipher property. */
        ssh1_protocol_error("Can't use Arcfour cipher with 1.0 protocol.",
                            ssh1);
        return;
      }
    ssh1->statistics.cipher_c_to_s = "arcfour";
    ssh1->statistics.cipher_s_to_c = "arcfour";
    break;

  case SSH1_CIPHER_BLOWFISH:
    cr = ssh_cipher_allocate("blowfish-cbc",
                             ssh1->session_key,
                             32,
                             TRUE,
                             &(ssh1->out_cipher));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    cr = ssh_cipher_allocate("blowfish-cbc",
                             ssh1->session_key,
                             32,
                             FALSE,
                             &(ssh1->in_cipher));
    if (cr != SSH_CRYPTO_OK)
      {
        ssh1_protocol_error("Unable to create cipher context.", ssh1);
        return;
      }
    ssh1->statistics.cipher_c_to_s = "blowfish-cbc";
    ssh1->statistics.cipher_s_to_c = "blowfish-cbc";
    break;

  default:
    ssh1_protocol_error("Internal error in cipher selection.", ssh1);
    return;
  }
  ssh1_input_available(ssh1);
}


/* These host key check functions are not used in Windows because the
   key check is performed in a user-defined callback function. */
char *ssh_host_key_file_name(SshUser user,
                             SshConfig config,
                             const char *host,
                             const char *port,
                             Boolean global_directory)
{
  char *user_dir, *key_dir, *r, host_stripped[1024];
  int i, j;
  struct stat st;

  if (global_directory)
    {
      key_dir = NULL;
    }
  else
    {
      if ((user_dir = ssh_userdir(user, config, TRUE)) == NULL)
        {
          ssh_warning("ssh_client_key_check: no user directory.");
          return NULL;
        }
      ssh_dsprintf(&key_dir, "%s/hostkeys", user_dir);
      ssh_xfree(user_dir);
      if (stat(key_dir, &st) < 0)
        {

          if (mkdir(key_dir, 0700) < 0)



            {
              ssh_warning("ssh_userdir: could not create user's ssh hostkey"
                          "directory %s", key_dir);
            }
        }
    }
  for ((j = 0, i = 0); host[i] != '\0'; i++)
    {
      if (j > sizeof(host_stripped) - 10)
        {
          break;
        }
      if (isalpha(host[i]))
        {
          host_stripped[j++] = tolower(host[i]);
          continue;
        }
      if ((isdigit(host[i])) || (host[i] == '.') || (host[i] == '-'))
        {
          host_stripped[j++] = host[i];
          continue;
        }
      /* escape this character in octal */
      host_stripped[j++] = '_';
      host_stripped[j++] = '0' + (host[i] >> 6);
      host_stripped[j++] = '0' + ((host[i] >> 3) & 7);
      host_stripped[j++] = '0' + (host[i] & 7);
    }
  host_stripped[j] = '\0';
  /* produce a file name from the server name */
  if (global_directory)
    {
      ssh_dsprintf(&r, "%s/key_%s_%s.pub",
                   SSH_GLOBAL_HOSTKEYS_DIR, port, host_stripped);
    }
  else
    {
      ssh_dsprintf(&r, "%s/key_%s_%s.pub", key_dir, port, host_stripped);
      ssh_xfree(key_dir);
    }
  return r;
}

void ssh1_host_key_write(Ssh1Client ssh1, const char *filename)
{
  SshTime now;
  char *time_str, *comment;
  unsigned char *blob;
  size_t blob_len;


  if ((blob_len = ssh_encode_pubkeyblob(ssh1->host_key, &blob)) == 0)
    {
      ssh_warning("Unable to encode host key %s", filename);
      return;
    }
  now = ssh_time();
  time_str = ssh_readable_time_string(now, TRUE);
  ssh_dsprintf(&comment,
               "host key for %s, accepted by %s %s",
               ssh1->client->remote_server,
               ssh_user_name(ssh1->client->user_data),
               time_str);
  ssh_xfree(time_str);
  if (ssh2_key_blob_write(ssh1->client->user_data,
                          filename,
                          0600, SSH_KEY_MAGIC_PUBLIC,
                          comment,
                          blob,
                          blob_len,
                          NULL))
    {
      ssh_warning("Unable to write host key %s", filename);
    }
  else
    {
      ssh_informational("Host key saved to %s\n", filename);
      ssh_informational("%s\n", comment);
    }
  ssh_xfree(comment);
  ssh_xfree(blob);
  return;
}

Boolean ssh1_host_key_check(Ssh1Client ssh1)
{
  unsigned char *nblob = NULL, *sblob = NULL, *digest;
  size_t nblob_len, sblob_len, digest_len;
  char *filename, *fingerprint;
  unsigned long magic;
  Boolean conf;
  SshHash hash;
  SshCryptoStatus cr;
  Boolean use_global_dir = FALSE;
  
  if ((nblob_len = ssh_encode_pubkeyblob(ssh1->host_key, &nblob)) == 0)
    {
      ssh_warning("Unable to encode received host key.");
      return FALSE;
    }

  cr = ssh_hash_allocate("sha1", &hash);
  if (cr == SSH_CRYPTO_OK)
    {
      ssh_hash_update(hash, nblob, nblob_len);
      digest_len = ssh_hash_digest_length(hash);
      digest = ssh_xmalloc(digest_len);
      ssh_hash_final(hash, digest);
      ssh_hash_free(hash);
      fingerprint = ssh_fingerprint(digest,
                                    digest_len,
                                    SSH_FINGERPRINT_BABBLE);
    }
  else
    {
      fingerprint = NULL;
    }
 retry:
  filename = ssh_host_key_file_name(ssh1->client->user_data,
                                    ssh1->client->config,
                                    ssh1->client->remote_server,
                                    ssh1->client->config->port,
                                    use_global_dir);
  if (filename == NULL)
    {
      ssh_xfree(nblob);
      ssh_xfree(fingerprint);
      return FALSE;
    }
  magic = ssh2_key_blob_read(ssh1->client->user_data,
                             filename,
                             TRUE,
                             NULL,
                             &sblob,
                             &sblob_len,
                             NULL);
  switch (magic)
    {
    case SSH_KEY_MAGIC_FAIL:
      if (!use_global_dir)
        {
          use_global_dir = TRUE;
          ssh_xfree(filename);
          goto retry;
        }
      ssh_informational("Host key not found from database.\r\n");
      if (fingerprint)
        {
          ssh_informational("Key fingerprint:\r\n");
          ssh_informational("%s\r\n", fingerprint);
          ssh_informational("You can get a public key's fingerprint "
                            "by running\r\n");
          ssh_informational("%% ssh-keygen -F publickey.pub\r\n");
          ssh_informational("on the keyfile.\r\n");
        }
      if (ssh1->client->config->strict_host_key_checking ==
          SSH_STRICT_HOSTKEY_CHECKING_YES)
        {
          /* User has requested strict host key checking.  We will not add
             the host key automatically.  The only alternative left is to
             abort. */
          ssh_informational("No host key is known for %.200s and "
                            "you have requested strict checking.",
                            ssh1->client->remote_server);
          ssh_xfree(filename);
          ssh_xfree(nblob);
          return FALSE;
        }

      if (!ssh1->client->config->batch_mode)
        conf = ssh_read_confirmation("Are you sure you want to continue "
                                     "connecting (yes/no)? ",
                                     FALSE);
      else
        conf = FALSE;

      if (conf == FALSE)
        {
          ssh_xfree(filename);
          ssh_xfree(nblob);
          ssh_xfree(fingerprint);
          return FALSE;
        }
      ssh1_host_key_write(ssh1, filename);
      ssh_xfree(filename);
      ssh_xfree(nblob);
      ssh_xfree(fingerprint);
      return TRUE;

    case SSH_KEY_MAGIC_PUBLIC:
      if ((sblob_len == nblob_len) && (memcmp(sblob, nblob, sblob_len) == 0))
        {
          ssh_xfree(fingerprint);
          ssh_xfree(filename);
          ssh_xfree(nblob);
          ssh_xfree(sblob);
          return TRUE;
        }
      /*FALLTHROUGH*/

    default:
      ssh_informational
        ("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\r\n"
         "@       WARNING: HOST IDENTIFICATION HAS CHANGED!         @\r\n"
         "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\r\n"
         "IT IS POSSIBLE THAT SOMEONE IS DOING SOMETHING NASTY!\r\n"
         "Someone could be eavesdropping on you right now "
         "(man-in-the-middle attack)!\r\n"
         "It is also possible that the host key has just been changed.\r\n"
         "Please contact your system administrator.\r\n"
         "Add correct host key to \"%s\"\r\n"
         "to get rid of this message or remove the keyfile, "
         "and connect again.\r\n",
         filename);
      ssh_xfree(sblob);
      if (ssh1->client->config->strict_host_key_checking ==
          SSH_STRICT_HOSTKEY_CHECKING_YES)
        {
          /* User has requested strict host key checking. We have to abort
             authentication. */
          ssh_informational("Host key has changed for %.200s and "
                            "you have requested strict checking.",
                            ssh1->client->remote_server);
          ssh_xfree(fingerprint);
          ssh_xfree(filename);
          ssh_xfree(nblob);
          return FALSE;
        }
      if (fingerprint)
        {
          ssh_informational("Received server key's fingerprint:\r\n");
          ssh_informational("%s\r\n", fingerprint);
          ssh_informational("You can get a public key's fingerprint by "
                            "running\r\n");
          ssh_informational("%% ssh-keygen -F publickey.pub\r\n");
          ssh_informational("on the keyfile.\r\n");
        }
      if (!ssh1->client->config->batch_mode)
        conf = ssh_read_confirmation( "Are you sure you want to continue "
                                      "connecting (yes/no)? ",
                                      FALSE);
      else
        conf = FALSE;

      if (conf == FALSE)
        {
          ssh_xfree(fingerprint);
          ssh_xfree(filename);
          ssh_xfree(nblob);
          return FALSE;
        }
      if (!ssh1->client->config->batch_mode)
        conf = ssh_read_confirmation("Do you want to update the new host key "
                                     "to the local database (yes/no)? ",
                                     FALSE);
      else
        conf = FALSE;

      if (conf == TRUE)
        {
          ssh1_host_key_write(ssh1, filename);
        }
      ssh_xfree(filename);
      ssh_xfree(nblob);
      ssh_xfree(fingerprint);
      return TRUE;
    }
  /*NOTREACHED*/
}











void ssh1_handle_packet_smsg_auth_rsa_challenge(SshBuffer packet,
                                                Ssh1Client ssh1)
{
  SshMPIntStruct challenge;
  SshBufferStruct reply[1];
  SshPrivateKey key = NULL;
  unsigned char *reply_buf;
  size_t reply_buf_len;
  char *passphrase_prompt;
  char *passphrase;
  unsigned char *challenge_buf;
  size_t challenge_len;

  SSH_DEBUG(5, ("mode = %d", ssh1->mode));

  if ((ssh1->mode != SSH1_RSA_AUTHENTICATION_REQUESTED) &&
      (ssh1->mode != SSH1_AGENT_AUTHENTICATION_REQUESTED))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Unexpected rsa_challenge.", ssh1);
      return;
    }
  ssh_mp_init(&challenge);
  if (! ssh1_decode_mp(packet, &challenge))
    {
      ssh_mp_clear(&challenge);
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed rsa_challenge packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  switch (ssh1->mode) {
  case SSH1_RSA_AUTHENTICATION_REQUESTED:
    if (ssh1->auth_current->auth_type == SSH1_AUTH_TYPE_RSA_FILE)
      {
        if (ssh_privkey_read(ssh1->client->user_data,
                             ssh1->auth_current->filename,
                             "", NULL, &key) == SSH_PRIVKEY_KEY_UNREADABLE)
          {
            key = NULL;
            ssh_warning("Couldn't read private key %s.",
                        ssh1->auth_current->filename);
          }
        else if (key == NULL)
          {

            if ((ssh1->auth_current->comment != NULL) &&
                (strlen(ssh1->auth_current->comment) > 0))
              {
                ssh_dsprintf(&passphrase_prompt,
                             "Passphrase for key \"%s\" with comment \"%s\":",
                             ssh1->auth_current->filename,
                             ssh1->auth_current->comment);
              }
            else
              {
                ssh_dsprintf(&passphrase_prompt,
                             "Passphrase for key \"%s\":",
                             ssh1->auth_current->filename);
              }
            if (!ssh1->client->config->batch_mode)
              passphrase = ssh_read_passphrase(passphrase_prompt , FALSE);
            else
              passphrase = NULL;

            ssh_xfree(passphrase_prompt);



















            if (passphrase && *passphrase)
              {
                (void) ssh_privkey_read(ssh1->client->user_data,
                                        ssh1->auth_current->filename,
                                        passphrase,
                                        NULL,
                                        &key);
                memset(passphrase, 'F', strlen(passphrase));
                ssh_xfree(passphrase);
              }
          }
      }
#ifdef WITH_PGP
    else if (ssh1->auth_current->auth_type == SSH1_AUTH_TYPE_PGP_KEY)
      {
        if (ssh1->auth_current->pgp_keyblob != NULL)
          {
            key = ssh2_pgp_privkey_import(ssh1->auth_current->pgp_keyblob,
                                          ssh1->auth_current->pgp_keyblob_len,
                                          NULL,
                                          ssh1->auth_current->comment,
                                          ssh1->client->config);
          }
        else
          {
            key = NULL;
          }
      }
#endif /* WITH_PGP */
    else
      {
        SSH_NOTREACHED;
      }
    if ((key == NULL) ||
        (ssh1_challenge_reply(&challenge,
                              key,
                              &reply_buf,
                              &reply_buf_len,
                              ssh1) == FALSE))
      {
        reply_buf_len = 16;
        reply_buf = ssh_xcalloc(reply_buf_len, sizeof (char));
      }
    ssh_mp_clear(&challenge);
    if (key != NULL)
      ssh_private_key_free(key);
    ssh_buffer_init(reply);
    ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_AUTH_RSA_RESPONSE);
    ssh1_encode_data(reply, (unsigned char *)reply_buf, reply_buf_len);
    if (ssh1_send_buffer(reply, ssh1) == FALSE)
      {
        ssh_buffer_uninit(reply);
        return;
      }
    ssh_buffer_uninit(reply);
    ssh1->mode = SSH1_RSA_AUTHENTICATION_SENT;
    ssh1_input_available(ssh1);
    return;

  case SSH1_AGENT_AUTHENTICATION_REQUESTED:
    challenge_len = ((ssh_mp_get_size(&challenge, 2) + 7) >> 3) & 0xffff;
    if (challenge_len == 0)
      {
        reply_buf_len = 16;
        reply_buf = ssh_xcalloc(reply_buf_len, sizeof (char));
        ssh_buffer_init(reply);
        ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_AUTH_RSA_RESPONSE);
        ssh1_encode_data(reply, (unsigned char *)reply_buf, reply_buf_len);
        ssh_xfree(reply_buf);
        if (ssh1_send_buffer(reply, ssh1) ==  FALSE)
          {
            ssh_buffer_uninit(reply);
            return;
          }
        ssh_buffer_uninit(reply);
        ssh1->mode = SSH1_RSA_AUTHENTICATION_SENT;
        ssh1_input_available(ssh1);
      }
    else
      {
        challenge_buf = ssh_xmalloc(challenge_len);
        ssh_mp_get_buf(challenge_buf, challenge_len, &challenge);
        ssh_buffer_init(reply);
        ssh_encode_buffer(reply,
                          SSH_FORMAT_UINT32_STR,
                          challenge_buf, challenge_len,
                          SSH_FORMAT_UINT32_STR,
                          ssh1->session_id, 16,
                          SSH_FORMAT_END);
        ssh_xfree(challenge_buf);
        ssh1->mode = SSH1_AGENT_OPERATION_IN_PROGRESS;
        SSH_ASSERT(ssh1->auth_current->auth_type == SSH1_AUTH_TYPE_AGENT);
        SSH_ASSERT(ssh1->auth_current->certs != NULL);
        SSH_ASSERT(ssh1->auth_current->agent != NULL);
        SSH_ASSERT(ssh1->auth_current->agent == ssh1->agent);
        ssh_agent_op(ssh1->auth_current->agent,
                     SSH_AGENT_SSH1_CHALLENGE_RESPONSE,
                     ssh1->auth_current->certs,
                     ssh1->auth_current->certs_len,
                     ssh_buffer_ptr(reply),
                     ssh_buffer_len(reply),
                     ssh1_agent_op_done,
                     (void *)ssh1);
        ssh_buffer_uninit(reply);
      }
    return;

  default:
    ssh1_protocol_error("Internal error.", ssh1);
    return;
  }
  /*NOTREACHED*/
}

void ssh1_handle_packet_smsg_success(SshBuffer packet, Ssh1Client ssh1)
{
  SshBufferStruct reply[1];

  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  switch (ssh1->mode)
    {
    case SSH1_MODE_SESSION_KEY_SENT:
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_USER);
      ssh1_encode_string(reply,
                         ssh1->client->remote_user,
                         strlen(ssh1->client->remote_user));
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1->mode = SSH1_MODE_USER_NAME_SENT;
      ssh1_input_available(ssh1);
      return;

    case SSH1_MODE_USER_NAME_SENT:
    case SSH1_RSA_AUTHENTICATION_SENT:
    case SSH1_PASSWORD_SENT:
      ssh1_flush_auth_tokens(ssh1);
      ssh1->mode = SSH1_MODE_AUTHENTICATION_COMPLETE;
      ssh1_authenticated(ssh1, TRUE);
      ssh1_request_compression(ssh1);
      return;

    case SSH1_MODE_COMPRESSION_REQUESTED:
      SSH_DEBUG(5, ("Compression turned on."));
      ssh1->statistics.compression_c_to_s = "zlib";
      ssh1->statistics.compression_s_to_c = "zlib";
      ssh1_request_pty(ssh1);
      return;

    case SSH1_MODE_PTY_REQUESTED:
      SSH_DEBUG(5, ("Pseudo terminal allocated."));
      ssh1->have_pty = TRUE;
      if (ssh1->have_forwarding)
        ssh1_request_remote_forward(ssh1);
      else
        ssh1_request_command(ssh1);
      return;

    case SSH1_MODE_REMOTE_FORWARD_REQUESTED:
      {
        Ssh1RemoteTcpListener ssh1_listener;

        ssh1_listener = ssh_xcalloc(1, sizeof (*ssh1_listener));
        ssh1_listener->forward = ssh1->requested_remote_forward;
        ssh1_listener->next = ssh1->remote_tcp_listeners;
        ssh1->remote_tcp_listeners = ssh1_listener;
      }
      SSH_DEBUG(5, ("Remote forward %u:%s:%u initialized.",
                    strtoul(ssh1->requested_remote_forward->port,
                            NULL, 0),
                    ssh1->requested_remote_forward->connect_to_host,
                    strtoul(ssh1->requested_remote_forward->connect_to_port,
                            NULL, 0)));
      ssh1_request_remote_forward(ssh1);
      return;

    case SSH1_MODE_AGENT_FORWARD_REQUESTED:
      SSH_DEBUG(4, ("Agent forward on."));
      ssh1->agent_forwarding_on = TRUE;
      ssh1_request_x11_forward(ssh1);
      return;

    case SSH1_MODE_X11_FORWARD_REQUESTED:
      SSH_DEBUG(4, ("X11 forward on."));
      ssh1->x11_forwarding_on = TRUE;
      ssh1_init_local_forward(ssh1);
      return;

    default:
      ssh1_protocol_error("Unexpected smsg_success.", ssh1);
      return;
    }
  /*NOTREACHED*/
}

void ssh1_handle_packet_smsg_failure(SshBuffer packet, Ssh1Client ssh1)
{
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  switch (ssh1->mode)
    {
    case SSH1_MODE_USER_NAME_SENT:
      ssh1_add_auth_tokens(TRUE, ssh1);
      return;

    case SSH1_PASSWORD_SENT:                  /* Wrong password */
    case SSH1_RSA_AUTHENTICATION_REQUESTED:   /* Denied public key */
    case SSH1_AGENT_AUTHENTICATION_REQUESTED: /* Denied agent public key */
    case SSH1_RSA_AUTHENTICATION_SENT:        /* Bad response to challenge */
      if ((ssh1->auth_current->next == NULL) &&
          (ssh1->auth_round > 2))
        {
          ssh1_authenticated(ssh1, FALSE);
          ssh_warning("Authentication failed.");
          ssh1->exit_value = -1;
          ssh1_protocol_destroy(ssh1, FALSE);
          return;
        }
      if (ssh1->auth_current->next != NULL)
        {
          ssh1->auth_current = ssh1->auth_current->next;
        }
      else
        {
          ssh1->auth_current = ssh1->auth_list;
          ssh1->auth_round++;
        }
      ssh1_try_auth_method(ssh1);
      return;

    case SSH1_MODE_COMPRESSION_REQUESTED:
      ssh_warning("Remote server denied compression.");
      ssh_compress_free(ssh1->out_compress);
      ssh1->out_compress = NULL;
      ssh_compress_free(ssh1->in_compress);
      ssh1->in_compress = NULL;
      ssh1_request_pty(ssh1);
      return;

    case SSH1_MODE_PTY_REQUESTED:
      ssh_warning("Remote server can't allocate pty.");
      if (ssh1->have_forwarding)
        ssh1_request_remote_forward(ssh1);
      else
        ssh1_request_command(ssh1);
      return;

    case SSH1_MODE_REMOTE_FORWARD_REQUESTED:
      SSH_DEBUG(5, ("Remote server denied remote forward %u:%s:%u.",
                    strtoul(ssh1->requested_remote_forward->port,
                            NULL, 0),
                    ssh1->requested_remote_forward->connect_to_host,
                    strtoul(ssh1->requested_remote_forward->connect_to_port,
                            NULL, 0)));
      ssh1_request_remote_forward(ssh1);
      return;

    case SSH1_MODE_AGENT_FORWARD_REQUESTED:
      SSH_DEBUG(4, ("Agent forward request failed."));
      ssh1_request_x11_forward(ssh1);
      return;

    case SSH1_MODE_X11_FORWARD_REQUESTED:
      SSH_DEBUG(4, ("X11 forward request failed."));
      ssh1_init_local_forward(ssh1);
      return;

    default:
      ssh1_protocol_error("Unexpected smsg_failure.", ssh1);
      return;
    }
  /*NOTREACHED*/
}

void ssh1_handle_packet_smsg_stdout_data(SshBuffer packet, Ssh1Client ssh1)
{
  char *data;
  size_t data_len;

  if ((ssh1->mode != SSH1_MODE_INTERACTIVE) &&
      (ssh1->mode != SSH1_MODE_INTERACTIVE_AFTER_EOF))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Unexpected stdout_data.", ssh1);
      return;
    }
  if (! ssh1_decode_string(packet, &data, &data_len))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed stdout_data packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  ssh_xbuffer_append(ssh1->stdout_buf, (unsigned char *)data, data_len);
  ssh_xfree(data);
  ssh1_flush_stdout(ssh1);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_smsg_stderr_data(SshBuffer packet, Ssh1Client ssh1)
{
  char *data;
  size_t data_len;

  /* If stderr is not defined, use stdio */
  if (ssh1->client->stderr_stream == NULL)
  {
    ssh1_handle_packet_smsg_stdout_data(packet, ssh1);
    return;
  }

  if ((ssh1->mode != SSH1_MODE_INTERACTIVE) &&
      (ssh1->mode != SSH1_MODE_INTERACTIVE_AFTER_EOF))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Unexpected stderr_data.", ssh1);
      return;
    }
  if (! ssh1_decode_string(packet, &data, &data_len))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed stderr_data packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  ssh_xbuffer_append(ssh1->stderr_buf, (unsigned char *)data, data_len);
  ssh_xfree(data);
  ssh1_flush_stderr(ssh1);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_smsg_exitstatus(SshBuffer packet, Ssh1Client ssh1)
{
  SshUInt32 exit_value;
  SshBufferStruct reply[1];

  if ((ssh1->mode != SSH1_MODE_INTERACTIVE) &&
      (ssh1->mode != SSH1_MODE_INTERACTIVE_AFTER_EOF))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Unexpected exitstatus.", ssh1);
      return;
    }
  if (! ssh1_decode_int(packet, &exit_value))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed exitvalue packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);

  if (ssh1->have_pty)
    ssh_leave_raw_mode(-1);

  ssh_buffer_init(reply);
  ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_EXIT_CONFIRMATION);
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1->exit_value = exit_value;
  ssh1_protocol_destroy(ssh1, FALSE);
  return;
}

void ssh1_handle_packet_msg_channel_open_confirmation(SshBuffer packet,
                                                      Ssh1Client ssh1)
{
  SshUInt32 channel_id, remote_id;
  Ssh1RemoteChannel remote_channel;

  if (! ssh1_decode_int(packet, &channel_id))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_channel_open_failure packet.", ssh1);
      return;
    }
  if (! ssh1_decode_int(packet, &remote_id))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_channel_open_failure packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  remote_channel = ssh1_find_remote_channel_with_local_id(ssh1, channel_id);
  if (! remote_channel)
    {
      ssh1_protocol_error("Server denied nonexistent channel.", ssh1);
      return;
    }
  if (! remote_channel->forming)
    {
      ssh1_protocol_error("Server denied channel that is not forming.", ssh1);
      return;
    }
  remote_channel->remote_id = remote_id;
  remote_channel->forming = FALSE;
  ssh_stream_set_callback(remote_channel->stream,
                          ssh1_remote_channel_stream_callback,
                          remote_channel);
  ssh1_remote_channel_flush(remote_channel);
  SSH_DEBUG(4, ("Channel open confirmed by the server."));
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_msg_channel_open_failure(SshBuffer packet,
                                                 Ssh1Client ssh1)
{
  SshUInt32 channel_id;
  Ssh1RemoteChannel remote_channel;

  if (! ssh1_decode_int(packet, &channel_id))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_channel_open_failure packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  remote_channel = ssh1_find_remote_channel_with_local_id(ssh1, channel_id);
  if (! remote_channel)
    {
      ssh1_protocol_error("Server denied nonexistent channel.", ssh1);
      return;
    }
  if (! remote_channel->forming)
    {
      ssh1_protocol_error("Server denied channel that is not forming.", ssh1);
      return;
    }
  SSH_DEBUG(4, ("Channel open failure from the server."));
  ssh1_delete_remote_channel(remote_channel);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_msg_channel_data(SshBuffer packet, Ssh1Client ssh1)
{
  SshUInt32 channel_id;
  unsigned char *channel_data;
  size_t channel_data_len;
  Ssh1RemoteChannel remote_channel;
  SshBufferStruct reply[1];
  int srv;

  SSH_DEBUG(4, ("Channel data"));
  if (! ssh1_decode_int(packet, &channel_id))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_channel_data packet.", ssh1);
      return;
    }
  if (! ssh1_decode_string(packet,
                           (char **)(&channel_data),
                           &channel_data_len))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_channel_data packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  remote_channel = ssh1_find_remote_channel_with_local_id(ssh1, channel_id);
  if (! remote_channel)
    {
      ssh1_protocol_error("Server sent channel_data to nonexistent channel.",
                          ssh1);
      return;
    }
  if (remote_channel->dying)
    {
      ssh1_protocol_error("Server sent channel_data to a dying channel.",
                          ssh1);
      return;
    }
  if (remote_channel->forming)
    {
      ssh1_protocol_error("Server sent channel_data to a forming channel.",
                          ssh1);
      return;
    }
  if (channel_data_len > 0)
    {
      srv = ssh_stream_write(remote_channel->stream,
                             channel_data,
                             channel_data_len);
      ssh_xfree(channel_data);
      if (srv == 0)
        {
          if (! remote_channel->input_closed)
            {
              remote_channel->input_closed = TRUE;
              ssh_buffer_init(reply);
              ssh1_encode_byte(reply,
                             (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OUTPUT_CLOSED);
              ssh1_encode_int(reply, remote_channel->remote_id);
              if (ssh1_send_buffer(reply, ssh1) == FALSE)
                {
                  ssh_buffer_uninit(reply);
                  return;
                }
              ssh_buffer_uninit(reply);
            }
        }
    }
  ssh1_delete_remote_channel_if_completed(remote_channel);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_msg_channel_input_eof(SshBuffer packet,
                                              Ssh1Client ssh1)
{
  SshUInt32 channel_id;
  Ssh1RemoteChannel remote_channel;
  SshBufferStruct reply[1];

  SSH_DEBUG(4, ("packet received"));
  if (! ssh1_decode_int(packet, &channel_id))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed channel_input_eof packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  remote_channel = ssh1_find_remote_channel_with_local_id(ssh1, channel_id);
  if (! remote_channel)
    {
      ssh1_protocol_error("Server sent eof to nonexistent channel.",
                          ssh1);
      return;
    }
  ssh_stream_output_eof(remote_channel->stream);
  remote_channel->eof_received = TRUE;
  if (! remote_channel->input_closed)
    {
      remote_channel->input_closed = TRUE;
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply,
                       (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OUTPUT_CLOSED);
      ssh1_encode_int(reply, remote_channel->remote_id);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
    }
  ssh1_delete_remote_channel_if_completed(remote_channel);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_msg_channel_output_closed(SshBuffer packet,
                                                  Ssh1Client ssh1)
{
  SshUInt32 channel_id;
  Ssh1RemoteChannel remote_channel;
  SshBufferStruct reply[1];

  SSH_DEBUG(4, ("packet received"));
  if (! ssh1_decode_int(packet, &channel_id))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed output_closed packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  remote_channel = ssh1_find_remote_channel_with_local_id(ssh1, channel_id);
  if (! remote_channel)
    {
      ssh1_protocol_error("Server sent output_closed to nonexistent channel.",
                          ssh1);
      return;
    }
  remote_channel->output_closed = TRUE;
  if (! remote_channel->eof_sent)
    {
      remote_channel->eof_sent = TRUE;
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply,
                       (SshUInt8)SSH1_PACKET_MSG_CHANNEL_INPUT_EOF);
      ssh1_encode_int(reply, remote_channel->remote_id);
      if (ssh1_send_buffer(reply, remote_channel->ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
    }
  ssh1_delete_remote_channel_if_completed(remote_channel);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_smsg_x11_open(SshBuffer packet, Ssh1Client ssh1)
{
  SshUInt32 channel;
  char *originator, *display, *cp;
  Ssh1RemoteChannel remote_channel;
  SshBufferStruct reply[1];
  int display_number;
  char buf[256], port[20];
  Boolean x11_tcp;

  SSH_DEBUG(4, ("packet received"));
  if (! ssh1_decode_int(packet, &channel))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_port_open packet.", ssh1);
      return;
    }
  if (ssh1->protocol_flags & SSH1_PROTOFLAG_HOST_IN_FWD_OPEN)
    {
      if (! ssh1_decode_string(packet, &originator, NULL))
        {
          ssh_buffer_uninit(packet);
          ssh1_protocol_error("Malformed msg_port_open packet.", ssh1);
          return;
        }
    }
  else
    {
      originator = ssh_xstrdup("UNKNOWN");
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  if (! ssh1->x11_forwarding_on)
    {
      ssh1_protocol_error("Unexpected x11_open.", ssh1);
      return;
    }

  display = getenv("DISPLAY");



  if (! display)
    goto no_connection_to_x11_server;
  ssh_buffer_uninit(packet);


  if (strncmp(display, "unix:", 5) == 0 ||
      display[0] == ':')
    {
      /* Connect to the unix domain socket. */
      x11_tcp = FALSE;
      if (sscanf(strrchr(display, ':') + 1, "%d", &display_number) != 1)
        {
          ssh_warning("Could not parse display number from DISPLAY: %.100s",
                      display);
          goto no_connection_to_x11_server;
        }

      /* Determine the name to use for the socket. */
#ifdef HPUX_NONSTANDARD_X11_KLUDGE
      {
        /* HP-UX release 10.X uses /var/spool/sockets/X11/0 for the
           unix-domain sockets, while earlier releases stores the
           socket in /usr/spool/sockets/X11/0 with soft-link from
           /tmp/.X11-unix/`uname -n`0 */

        struct stat st;

        if (stat("/var/spool/sockets/X11", &st) == 0)
          {
            ssh_snprintf(buf, sizeof(buf), "%s/%d",
                         "/var/spool/sockets/X11", display_number);
          }
        else
          {
            if (stat("/usr/spool/sockets/X11", &st) == 0)
              {
                ssh_snprintf(buf, sizeof(buf), "%s/%d",
                             "/usr/spool/sockets/X11", display_number);
              }
            else
              {
                struct utsname utsbuf;
                /* HP-UX stores unix-domain sockets in
                   /tmp/.X11-unix/`hostname`0
                   instead of the normal /tmp/.X11-unix/X0. */
                if (uname(&utsbuf) < 0)
                  ssh_fatal("uname: %.100s", strerror(errno));
                ssh_snprintf(buf, sizeof(buf), "%.20s/%.64s%d",
                             X11_DIR, utsbuf.nodename, display_number);
              }
          }
      }
#else /* HPUX_NONSTANDARD_X11_KLUDGE */
      ssh_snprintf(buf, sizeof(buf), "%.80s/X%d", X11_DIR, display_number);
#endif /* HPUX_NONSTANDARD_X11_KLUDGE */
    }
  else

    {
      /* Connect to an inet socket.  The DISPLAY value is supposedly
         hostname:d[.s], where hostname may also be numeric IP address. */
      x11_tcp = TRUE;
      strncpy(buf, display, sizeof(buf));
      buf[sizeof(buf) - 1] = 0;
      cp = strchr(buf, ':');
      if (!cp)
        {
          ssh_warning("Could not find ':' in DISPLAY: %.100s", display);
          goto no_connection_to_x11_server;
        }
      *cp = 0;
      /* buf now contains the host name.
         But first we parse the display number. */
      if (sscanf(cp + 1, "%d", &display_number) != 1)
        {
          ssh_warning("Could not parse display number from DISPLAY: %.100s",
                      display);
          goto no_connection_to_x11_server;
        }

      /* Host name is now in ``buf''.  Format port number in ``port''. */
      ssh_snprintf(port, sizeof(port), "%d", 6000 + display_number);
    }
  remote_channel = ssh_xcalloc(1, sizeof (*remote_channel));
  remote_channel->ssh1 = ssh1;
  remote_channel->remote_id = channel;
  if (x11_tcp)
    {
      remote_channel->host = ssh_xstrdup(buf);
      remote_channel->port = ssh_xstrdup(port);
    }
  else
    {
      remote_channel->host = ssh_xstrdup("X11 connection");
      remote_channel->port = ssh_xstrdup(buf);
    }
  remote_channel->host_info = ssh_xstrdup("X11 client");
  remote_channel->originator = originator;
  remote_channel->next = ssh1->remote_channels;
  ssh1->remote_channels = remote_channel;
  if (x11_tcp)
      ssh_tcp_connect(buf, port, NULL,
                      ssh1_remote_x11_channel_open_tcp,
                      remote_channel);
  else
      ssh_local_connect(buf,
                        ssh1_remote_x11_channel_open,
                        remote_channel);

  return;

 no_connection_to_x11_server:
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_FAILURE);
  ssh1_encode_int(reply, channel);
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_msg_port_open(SshBuffer packet, Ssh1Client ssh1)
{
  SshUInt32 channel, port;
  char *host, *originator;
  Ssh1RemoteTcpListener remote_tcp;
  Ssh1RemoteChannel remote_channel;
  SshBufferStruct reply[1];

  SSH_DEBUG(4, ("packet received"));
  if (! ssh1_decode_int(packet, &channel))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_port_open packet.", ssh1);
      return;
    }
  if (! ssh1_decode_string(packet, &host, NULL))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_port_open packet.", ssh1);
      return;
    }
  if (! ssh1_decode_int(packet, &port))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed msg_port_open packet.", ssh1);
      return;
    }
  if (ssh1->protocol_flags & SSH1_PROTOFLAG_HOST_IN_FWD_OPEN)
    {
      if (! ssh1_decode_string(packet, &originator, NULL))
        {
          ssh_buffer_uninit(packet);
          ssh1_protocol_error("Malformed msg_port_open packet.", ssh1);
          return;
        }
    }
  else
    {
      originator = ssh_xstrdup("UNKNOWN");
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  for (remote_tcp = ssh1->remote_tcp_listeners;
       remote_tcp;
       remote_tcp = remote_tcp->next)
    {
      if ((strcmp(host, remote_tcp->forward->connect_to_host) == 0) &&
          (strtoul(remote_tcp->forward->connect_to_port, NULL, 0) == port))
        break;
    }
  if (! remote_tcp)
    {
      ssh_warning("Server tried to open channel to %s:%u from %s.  Denied.",
                  host, (unsigned int)port, originator);
      ssh_xfree(host);
      ssh_xfree(originator);
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_FAILURE);
      ssh1_encode_int(reply, channel);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1_input_available(ssh1);
      return;
    }
  remote_channel = ssh_xcalloc(1, sizeof (*remote_channel));
  remote_channel->ssh1 = ssh1;
  remote_channel->remote_id = channel;
  remote_channel->host = ssh_xstrdup(remote_tcp->forward->connect_to_host);
  remote_channel->port = ssh_xstrdup(remote_tcp->forward->connect_to_port);
  remote_channel->host_info = host;
  remote_channel->originator = originator;
  remote_channel->next = ssh1->remote_channels;
  ssh1->remote_channels = remote_channel;
  remote_channel->open_handle = ssh_tcp_connect(remote_channel->host,
                                                remote_channel->port,
                                                NULL,
                                                ssh1_remote_channel_open,
                                                remote_channel);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_smsg_agent_open(SshBuffer packet, Ssh1Client ssh1)
{
  SshUInt32 channel;
  char *agent_path;
  SshBufferStruct reply[1];
  Ssh1RemoteChannel remote_channel;

  SSH_DEBUG(4, ("packet received"));
  if (! ssh1_decode_int(packet, &channel))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed smsg_agent_open packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  if (! ssh1->agent_forwarding_on)
    {
      ssh1_protocol_error("Unexpected agent_open.", ssh1);
      return;
    }

  agent_path = getenv(SSH1_AGENT_VAR);
  if (agent_path == NULL)
    agent_path = getenv(SSH_AA_VAR);
  if (agent_path == NULL)
    agent_path = getenv(SSH_AGENT_VAR);
  if (agent_path == NULL)
    {
      SSH_DEBUG(4, ("Can't define agent path.  Agent open failed."));
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_FAILURE);
      ssh1_encode_int(reply, channel);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1_input_available(ssh1);
      return;
    }



  remote_channel = ssh_xcalloc(1, sizeof (*remote_channel));
  remote_channel->ssh1 = ssh1;
  remote_channel->remote_id = channel;
  remote_channel->host = ssh_xstrdup("Agent connection");
  remote_channel->port = ssh_xstrdup("SSH_AUTH_SOCK");
  remote_channel->host_info = ssh_xstrdup("Agent client");
  remote_channel->originator = ssh_xstrdup("UNKNOWN");
  remote_channel->next = ssh1->remote_channels;
  ssh1->remote_channels = remote_channel;
  ssh_local_connect(agent_path,
                    ssh1_remote_agent_channel_open,
                    remote_channel);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_msg_ignore(SshBuffer packet, Ssh1Client ssh1)
{
  if (! ssh1_decode_string(packet, NULL, NULL))
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Malformed ignore packet.", ssh1);
      return;
    }
  if (ssh_buffer_len(packet) != 0)
    {
      ssh_buffer_uninit(packet);
      ssh1_protocol_error("Trash in the tail of the packet.", ssh1);
      return;
    }
  ssh_buffer_uninit(packet);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_msg_debug(SshBuffer packet, Ssh1Client ssh1)
{
  char *message;

  SSH_DEBUG(5, ("mode = %d", ssh1->mode));
  if (ssh1_decode_string(packet, &message, NULL))
    {
      SSH_DEBUG(5, ("SERVER DEBUG: %s", message));
      ssh_xfree(message);
    }
  else
    {
      SSH_DEBUG(5, ("Empty debug packet from the server."));
    }
  ssh_buffer_uninit(packet);
  ssh1_input_available(ssh1);
  return;
}

void ssh1_handle_packet_smsg_auth_tis_challenge(SshBuffer packet,
                                                Ssh1Client ssh1)
{
  ssh_buffer_uninit(packet);
  ssh1_protocol_error("Unexpected tis_challenge.", ssh1);
  return;
}

void ssh1_handle_packet_smsg_auth_kerberos_response(SshBuffer packet,
                                                    Ssh1Client ssh1)
{
  ssh_buffer_uninit(packet);
  ssh1_protocol_error("Unexpected kerberos_response.", ssh1);
  return;
}

void ssh1_send_window_size_change(int xcharacters,
                                  int ycharacters,
                                  int xpixels,
                                  int ypixels,
                                  Ssh1Client ssh1)
{
  SshBufferStruct reply[1];

  ssh_buffer_init(reply);
  ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_CMSG_WINDOW_SIZE);
  ssh1_encode_int(reply, (SshUInt32)(ycharacters));
  ssh1_encode_int(reply, (SshUInt32)(xcharacters));
  ssh1_encode_int(reply, (SshUInt32)(xpixels));
  ssh1_encode_int(reply, (SshUInt32)(ypixels));
  if (ssh1_send_buffer(reply, ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1_input_available(ssh1);
  return;
}





















void ssh1_window_size_change_sig(int sig, void *context)
{
#ifdef TIOCGWINSZ
  Ssh1Client ssh1 = (Ssh1Client)context;
  struct winsize ws;

  if (! ssh1->have_pty)
      return;
  if (ioctl(fileno(stdin), TIOCGWINSZ, &ws) < 0)
    {
      ssh_warning("Can't read window size.");
      return;
    }
  ssh1_send_window_size_change((int)(ws.ws_col),
                               (int)(ws.ws_row),
                               (int)(ws.ws_xpixel),
                               (int)(ws.ws_ypixel),
                               ssh1);
  return;
#endif /* TIOCGWINSZ */
  return;
}


Boolean ssh1_challenge_reply(SshMPInt challenge,
                             SshPrivateKey key,
                             unsigned char **reply,
                             size_t *reply_len,
                             Ssh1Client ssh1)
{
  unsigned char *challenge_buf, *output_buf, *output_digest;
  size_t challenge_len, output_len, output_digest_len;
  SshHash hash;
  SshCryptoStatus cr;
  SshPrivateKey tmp_key;

  if (ssh1->remote_server_protocol_version == 10)
    return FALSE;

  /* Linearize the challenge */
  challenge_len = ((ssh_mp_get_size(challenge, 2) + 7) >> 3) & 0xffff;
  if (challenge_len == 0)
    {
      SSH_DEBUG(6, ("failed to linearize the challenge"));
      return FALSE;
    }
  challenge_buf = ssh_xmalloc(challenge_len);
  ssh_mp_get_buf(challenge_buf, challenge_len, challenge);

  /* Decrypt challenge */
  output_len = ssh_private_key_max_decrypt_output_len(key);
  if (output_len == 0)
    {
      SSH_DEBUG(6, ("failed to define output length"));
      ssh_xfree(challenge_buf);
    }
  output_buf = ssh_xmalloc(output_len);
  cr = ssh_private_key_copy(key, &tmp_key);
  if (cr != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(6, ("ssh_private_key_copy failed (%d)", (int)cr));
      ssh_xfree(output_buf);
      return FALSE;
    }
  cr = ssh_private_key_select_scheme(tmp_key,
                                     SSH_PKF_ENCRYPT, "rsa-pkcs1-none",
                                     SSH_PKF_END);
  if (cr != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(6, ("ssh_private_key_select_scheme failed (%d)", (int)cr));
      ssh_private_key_free(tmp_key);
      ssh_xfree(output_buf);
      return FALSE;
    }
  
  cr = ssh_private_key_decrypt(tmp_key,
                               challenge_buf, challenge_len,
                               output_buf, output_len,
                               &output_len);
  ssh_private_key_free(tmp_key);
  ssh_xfree(challenge_buf);
  if (cr != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(6, ("failed to decrypt ssh1 challenge (%d)",
                    (int)cr));
      ssh_xfree(output_buf);
      return FALSE;
    }

  /* Calculate the reply */
  SSH_DEBUG(7, ("calculating type 1 ssh1 challenge"));
  cr = ssh_hash_allocate("md5", &hash);
  if (cr != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(6, ("ssh_hash_allocate failed (%d)", (int)cr));
      ssh_xfree(output_buf);
      return FALSE;
    }
  ssh_hash_update(hash, output_buf, (output_len < 32) ? output_len : 32);
  ssh_hash_update(hash, ssh1->session_id, 16);
  output_digest_len = ssh_hash_digest_length(hash);
  output_digest = ssh_xmalloc(output_digest_len);
  ssh_hash_final(hash, output_digest);
  ssh_hash_free(hash);
  if (reply)
    *reply = output_digest;
  else
    ssh_xfree(output_digest);
  if (reply_len)
    *reply_len = output_digest_len;
  return TRUE;
}

void ssh1_authenticated(Ssh1Client ssh1, Boolean result)
{
  if (ssh1->client->stream_wrapper_func)
    (*ssh1->client->stream_wrapper_func)
      (result, ssh1->client);


  if (ssh1->client->rl)
    {
      ssh_readline_eloop_unitialize(ssh1->client->rl);
      ssh1->client->rl = NULL;
    }



  if (ssh1->client->config->authentication_notify)
    {
      ssh_client_parent_notify_authentication(result,
                                              ssh1->client->stdio_stream);
    }









  if (ssh1->client->stdio_stream)
    ssh_stream_set_callback(ssh1->client->stdio_stream,
                            ssh1_stdio_event,
                            (void *)ssh1);
  if (ssh1->client->stderr_stream)
    ssh_stream_set_callback(ssh1->client->stderr_stream,
                            ssh1_stderr_event,
                            (void *)ssh1);

}

/* Channel stuff. */

void ssh1_local_forward_start(SshIpError error,
                              SshStream stream,
                              void *context)
{
  Ssh1LocalTcpListener ssh1_listener = (Ssh1LocalTcpListener)context;
  Ssh1RemoteChannel remote_channel;
  char host[256], port[20];
  SshBufferStruct reply[1];

  /* User connected to local forwarded tcp port */
  if (error != SSH_IP_NEW_CONNECTION)
    {
      if (stream)
        ssh_stream_destroy(stream);
      return;
    }
  remote_channel = ssh_xcalloc(1, sizeof (*remote_channel));
  remote_channel->ssh1 = ssh1_listener->ssh1;
  remote_channel->remote_id = 0;
  remote_channel->local_id =
    ssh1_allocate_local_channel_id(remote_channel->ssh1);
  remote_channel->host = ssh_xstrdup(ssh1_listener->forward->connect_to_host);
  remote_channel->port = ssh_xstrdup(ssh1_listener->forward->connect_to_port);
  if (ssh_tcp_get_remote_address(stream, host, sizeof (host)) == FALSE)
    {
      strcpy(host, "UNKNOWN");
    }
  if (ssh_tcp_get_remote_port(stream, port, sizeof (port)) == FALSE)
    {
      strcpy(port, "X");
    }
  remote_channel->host_info = ssh_xstrdup(host);
  ssh_dsprintf(&remote_channel->originator,
               "Local port %s, connection from %s port %s",
               ssh1_listener->forward->port, host, port);
  remote_channel->stream = ssh1_buffer_stream_wrap(stream);
  remote_channel->forming = TRUE;
  remote_channel->next = remote_channel->ssh1->remote_channels;
  remote_channel->ssh1->remote_channels = remote_channel;
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_MSG_PORT_OPEN);
  ssh1_encode_int(reply, remote_channel->local_id);
  ssh1_encode_string(reply,
                     remote_channel->host,
                     strlen(remote_channel->host));
  ssh1_encode_int(reply, strtoul(remote_channel->port, NULL, 0));
  if (remote_channel->ssh1->protocol_flags & SSH1_PROTOFLAG_HOST_IN_FWD_OPEN)
    {
      ssh1_encode_string(reply,
                         remote_channel->originator,
                         strlen(remote_channel->originator));
    }
  if (ssh1_send_buffer(reply, remote_channel->ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh1_input_available(remote_channel->ssh1);
  return;
}

Ssh1RemoteChannel ssh1_find_remote_channel_with_local_id(Ssh1Client ssh1,
                                                         SshUInt32 channel_id)
{
  Ssh1RemoteChannel current;

  for (current = ssh1->remote_channels;
       current;
       current = current->next)
    {
      if (current->local_id == channel_id)
        return current;
    }
  return NULL;
}

void ssh1_delete_remote_channel(Ssh1RemoteChannel remote_channel)
{
  Ssh1RemoteChannel current;

  if (remote_channel->ssh1->remote_channels == remote_channel)
    {
      remote_channel->ssh1->remote_channels =
        remote_channel->ssh1->remote_channels->next;
    }
  else
    {
      for (current = remote_channel->ssh1->remote_channels;
           current;
           current = current->next)
        {
          if (current->next == remote_channel)
            {
              current->next = current->next->next;
              break;
            }
        }
    }
  ssh_xfree(remote_channel->host);
  ssh_xfree(remote_channel->host_info);
  ssh_xfree(remote_channel->port);
  ssh_xfree(remote_channel->originator);
  if (remote_channel->open_handle)
    ssh_operation_abort(remote_channel->open_handle);
  if (remote_channel->stream)
    ssh_stream_destroy(remote_channel->stream);
  ssh_xfree(remote_channel);
}

void ssh1_buffer_stream_wait_flush(SshStream stream,
                                   Ssh1BufferStreamFlushedCallback callback,
                                   void *context);

void ssh1_remote_channel_flushed(Boolean ok, void *context)
{
  Ssh1RemoteChannel remote_channel = (Ssh1RemoteChannel)context;

  SSH_DEBUG(4, ("Channel flushed and will be deleted."));
  ssh1_delete_remote_channel(remote_channel);
  return;
}

void ssh1_delete_remote_channel_if_completed(Ssh1RemoteChannel remote_channel)
{
  if (remote_channel->eof_sent &&
      remote_channel->eof_received &&
      remote_channel->input_closed &&
      remote_channel->output_closed)
    {
      SSH_DEBUG(4, ("Scheduling the deletion of the channel."));
      remote_channel->dying = TRUE;
      ssh1_buffer_stream_wait_flush(remote_channel->stream,
                                    ssh1_remote_channel_flushed,
                                    remote_channel);
    }
}

void ssh1_remote_channel_open(SshIpError error,
                              SshStream stream,
                              void *context)
{
  Ssh1RemoteChannel remote_channel = (Ssh1RemoteChannel)context;
  SshBufferStruct reply[1];
  Ssh1Client ssh1;

  ssh1 = remote_channel->ssh1;
  remote_channel->open_handle = NULL;
  if (error != SSH_IP_OK)
    {
      SSH_DEBUG(4, ("Unable to open channel"));
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_FAILURE);
      ssh1_encode_int(reply, remote_channel->remote_id);
      ssh1_delete_remote_channel(remote_channel);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1_input_available(ssh1);
      return;
    }
  remote_channel->stream = ssh1_buffer_stream_wrap(stream);
  remote_channel->local_id =
    ssh1_allocate_local_channel_id(remote_channel->ssh1);
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply,
                   (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_CONFIRMATION);
  ssh1_encode_int(reply, remote_channel->remote_id);
  ssh1_encode_int(reply, remote_channel->local_id);
  if (ssh1_send_buffer(reply, remote_channel->ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh_stream_set_callback(remote_channel->stream,
                          ssh1_remote_channel_stream_callback,
                          remote_channel);
  ssh1_remote_channel_flush(remote_channel);
  SSH_DEBUG(4, ("Channel open"));
  ssh1_input_available(remote_channel->ssh1);
  return;
}

void ssh1_remote_agent_channel_open(SshStream stream,
                                    void *context)
{
  Ssh1RemoteChannel remote_channel = (Ssh1RemoteChannel)context;
  SshBufferStruct reply[1];
  unsigned char *help;
  Ssh1Client ssh1;

  ssh1 = remote_channel->ssh1;
  if (! stream)
    {
      SSH_DEBUG(4, ("Unable to open channel"));
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_FAILURE);
      ssh1_encode_int(reply, remote_channel->remote_id);
      ssh1_delete_remote_channel(remote_channel);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1_input_available(ssh1);
      return;
    }
  remote_channel->stream = ssh1_buffer_stream_wrap(stream);
  remote_channel->local_id =
    ssh1_allocate_local_channel_id(remote_channel->ssh1);
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply,
                   (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_CONFIRMATION);
  ssh1_encode_int(reply, remote_channel->remote_id);
  ssh1_encode_int(reply, remote_channel->local_id);
  if (ssh1_send_buffer(reply, remote_channel->ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  if (remote_channel->ssh1->client->config->ssh_agent_compat ==
      SSH_AGENT_COMPAT_SSH2)
    {
      ssh_buffer_init(reply);
      ssh_encode_buffer(
          reply,
          SSH_FORMAT_DATA, "1234", (size_t)4,
          SSH_FORMAT_CHAR,
          (unsigned int) SSH_AGENT_FORWARDING_NOTICE,
          SSH_FORMAT_UINT32_STR,
          remote_channel->ssh1->client->common->server_host_name,
          strlen(remote_channel->ssh1->client->common->server_host_name),
          SSH_FORMAT_UINT32_STR,
          remote_channel->ssh1->client->common->remote_ip,
          strlen(remote_channel->ssh1->client->common->remote_ip),
          SSH_FORMAT_UINT32,
          (SshUInt32) atol(remote_channel->ssh1->client->common->remote_port),
          SSH_FORMAT_END);
      help = ssh_buffer_ptr(reply);
      SSH_PUT_32BIT(help, ssh_buffer_len(reply) - 4);
      ssh_stream_write(remote_channel->stream,
                       ssh_buffer_ptr(reply),
                       ssh_buffer_len(reply));
      ssh_buffer_uninit(reply);
    }
  ssh_stream_set_callback(remote_channel->stream,
                          ssh1_remote_channel_stream_callback,
                          remote_channel);
  ssh1_remote_channel_flush(remote_channel);
  SSH_DEBUG(4, ("Agent channel open"));
  ssh1_input_available(remote_channel->ssh1);
  return;
}

void ssh1_remote_x11_channel_open_tcp(SshIpError error,
                                      SshStream stream,
                                      void *context)
{
  if (error != SSH_IP_OK)
    ssh1_remote_x11_channel_open(NULL, context);
  else
    ssh1_remote_x11_channel_open(stream, context);
  return;
}

void ssh1_remote_x11_channel_open(SshStream stream,
                                  void *context)
{
  SshBufferStruct reply[1];
  Ssh1RemoteChannel remote_channel = (Ssh1RemoteChannel)context;
  Ssh1Client ssh1;

  ssh1 = remote_channel->ssh1;
  if (! stream)
    {
      SSH_DEBUG(4, ("Unable to open channel to X11 server."));
      ssh_buffer_init(reply);
      ssh1_encode_byte(reply, (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_FAILURE);
      ssh1_encode_int(reply, remote_channel->remote_id);
      ssh1_delete_remote_channel(remote_channel);
      if (ssh1_send_buffer(reply, ssh1) == FALSE)
        {
          ssh_buffer_uninit(reply);
          return;
        }
      ssh_buffer_uninit(reply);
      ssh1_input_available(ssh1);
      return;
    }
  stream = ssh_stream_filter_create(stream,
                                    X11_MAX_AUTH_PACKET_SIZE,
                                    ssh1_x11_channel_filter,
                                    NULL,
                                    ssh1_x11_channel_filter_destroy,
                                    remote_channel);
  remote_channel->stream = ssh1_buffer_stream_wrap(stream);
  remote_channel->local_id =
    ssh1_allocate_local_channel_id(remote_channel->ssh1);
  ssh_buffer_init(reply);
  ssh1_encode_byte(reply,
                   (SshUInt8)SSH1_PACKET_MSG_CHANNEL_OPEN_CONFIRMATION);
  ssh1_encode_int(reply, remote_channel->remote_id);
  ssh1_encode_int(reply, remote_channel->local_id);
  if (ssh1_send_buffer(reply, remote_channel->ssh1) == FALSE)
    {
      ssh_buffer_uninit(reply);
      return;
    }
  ssh_buffer_uninit(reply);
  ssh_stream_set_callback(remote_channel->stream,
                          ssh1_remote_channel_stream_callback,
                          remote_channel);
  ssh1_remote_channel_flush(remote_channel);
  SSH_DEBUG(4, ("X11 channel open"));
  ssh1_input_available(remote_channel->ssh1);
  return;
}

void ssh1_remote_channel_stream_callback(SshStreamNotification notification,
                                         void *context)
{
  Ssh1RemoteChannel remote_channel = (Ssh1RemoteChannel)context;

  switch (notification)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      ssh1_remote_channel_flush(remote_channel);
      break;

    case SSH_STREAM_CAN_OUTPUT:
      /* This should never happen */
      break;

    case SSH_STREAM_DISCONNECTED:
      ssh1_remote_channel_flush(remote_channel);
      break;
    }
  return;
}

void ssh1_remote_channel_flush(Ssh1RemoteChannel remote_channel)
{
  unsigned char data[0x200];
  int rv;
  SshBufferStruct reply[1];

  while (1)
    {
      rv = ssh_stream_read(remote_channel->stream,
                           data,
                           sizeof (data));
      if (rv > 0)
        {
          if (! remote_channel->output_closed)
            {
              ssh_buffer_init(reply);
              ssh1_encode_byte(reply,
                               (SshUInt8)SSH1_PACKET_MSG_CHANNEL_DATA);
              ssh1_encode_int(reply, remote_channel->remote_id);
              ssh1_encode_string(reply, (char *)data, (size_t)rv);
              if (ssh1_send_buffer(reply, remote_channel->ssh1) == FALSE)
                {
                  ssh_buffer_uninit(reply);
                  return;
                }
              ssh_buffer_uninit(reply);
            }
        }
      else if (rv == 0)
        {
          if (! remote_channel->eof_sent)
            {
              remote_channel->eof_sent = TRUE;
              ssh_buffer_init(reply);
              ssh1_encode_byte(reply,
                               (SshUInt8)SSH1_PACKET_MSG_CHANNEL_INPUT_EOF);
              ssh1_encode_int(reply, remote_channel->remote_id);
              if (ssh1_send_buffer(reply, remote_channel->ssh1) == FALSE)
                {
                  ssh_buffer_uninit(reply);
                  return;
                }
              ssh_buffer_uninit(reply);
            }
          break;
        }
      else
        {
          break;
        }
    }
  ssh1_delete_remote_channel_if_completed(remote_channel);
  ssh1_input_available(remote_channel->ssh1);
  return;
}

SshUInt32 ssh1_allocate_local_channel_id(Ssh1Client ssh1)
{
  ssh1->last_allocated_local_id++;
  while (ssh1_find_remote_channel_with_local_id(
                                              ssh1,
                                              ssh1->last_allocated_local_id))
    ssh1->last_allocated_local_id++;
  return ssh1->last_allocated_local_id;
}

void ssh1_init_x11_auth_data(Ssh1Client ssh1, const char *display)
{
  char proto[512], data[512];
  size_t data_len;
  Boolean got_data;
#ifdef XAUTH_PATH
  char line[512];
  FILE *f;
#endif /* XAUTH_PATH */
  int i;
  unsigned int value;

  got_data = FALSE;
#ifdef XAUTH_PATH
  /* Try to get Xauthority information for the display. */
  ssh_snprintf(line, sizeof(line), "%s list %s 2>/dev/null",
               XAUTH_PATH, display);
  f = popen(line, "r");
  if (f && fgets(line, sizeof(line), f) &&
      sscanf(line, "%*s %s %s", proto, data) == 2)
    {
      SSH_DEBUG(4, ("Got local xauth data."));
      got_data = TRUE;
    }
  else
    {
      SSH_DEBUG(4, ("Failed to get local xauth data."));
      got_data = FALSE;
    }
  if (f)
    pclose(f);
#else /* XAUTH_PATH */

  SSH_DEBUG(4, ("No xauth program was found at configure time."));

#endif /* XAUTH_PATH */
  /* If we didn't get authentication data, just make up some data.  The
     forwarding code will check the validity of the response anyway, and
     substitute this data.  The X11 server, however, will ignore this
     fake data and use whatever authentication mechanisms it was using
     otherwise for the local connection. */
  if (! got_data)
    {
      strcpy(proto, "MIT-MAGIC-COOKIE-1");
      for (i = 0; i < 16; i++)
        ssh_snprintf(data + 2 * i, 3, "%02x",
                     ssh_random_get_byte());
    }

  /* Save protocol name. */
  ssh1->x11_real_auth_protocol = ssh_xstrdup(proto);
  ssh1->x11_real_auth_protocol_len = strlen(proto);
  ssh1->x11_fake_auth_protocol = ssh_xstrdup(proto);
  ssh1->x11_fake_auth_protocol_len = strlen(proto);

  /* Extract real authentication data and generate fake data of the same
     length. */
  data_len = strlen(data) / 2;
  ssh1->x11_real_data = ssh_xmalloc(data_len);
  ssh1->x11_real_data_len = data_len;
  ssh1->x11_fake_data = ssh_xmalloc(data_len);
  ssh1->x11_fake_data_len = data_len;

  /* Convert the real cookie into binary. */
  for (i = 0; i < data_len; i++)
    {
      if (sscanf(data + 2 * i, "%2x", &value) != 1)
        {
          ssh_warning("ssh_channel_x11_send_request: bad data: %s",
                      data);
          return;
        }
      ssh1->x11_real_data[i] = value;
    }

  /* Generate fake cookie. */
  for (i = 0; i < data_len; i++)
    ssh1->x11_fake_data[i] = ssh_random_get_byte();

  /* Convert the fake data into hex for transmission. */
  ssh1->x11_fake_data_hex = ssh_xmalloc(2 * ssh1->x11_fake_data_len + 1);
  for (i = 0; i < ssh1->x11_fake_data_len; i++)
    ssh_snprintf(ssh1->x11_fake_data_hex + 2 * i, 3, "%02x",
                 (unsigned int)(ssh1->x11_fake_data[i] & 0xff));
}

void ssh1_x11_channel_filter(void *context,
                             SshFilterGetCB get_data,
                             SshFilterCompletionCB completed,
                             void *internal_context)
{
  Ssh1RemoteChannel remote_channel = (Ssh1RemoteChannel)context;
  unsigned char *ucp;
  size_t received_len, data_len, proto_len, desired_len;
  SshBuffer data;
  size_t offset;
  Boolean eof_received;

  get_data(internal_context, &data, &offset, &eof_received);

  SSH_DEBUG(6, ("filter: data len %d, offset %d",
                (int)ssh_buffer_len(data), (int)offset));

  /* Compute the start and length of data that we have in the filter. */
  ucp = ssh_buffer_ptr(data);
  ucp += offset;
  received_len = ssh_buffer_len(data) - offset;

  /* Check if we have received enough to have the fixed packet header. */
  if (received_len < 12)
    {
      /* Not yet...  Keep receiving, unless we've got EOF. */
      if (eof_received)
        {
          completed(internal_context, SSH_FILTER_DISCONNECT); /* Abort the connection. */
          return;
        }
      else
        {
          completed(internal_context, SSH_FILTER_HOLD); /* Wait for more data. */
          return;
        }
    }

  /* Parse the lengths of variable-length fields. */
  if (ucp[0] == 0x42)
    { /* Byte order MSB first. */
      proto_len = 256 * ucp[6] + ucp[7];
      data_len = 256 * ucp[8] + ucp[9];
    }
  else
    if (ucp[0] == 0x6c)
      { /* Byte order LSB first. */
        proto_len = ucp[6] + 256 * ucp[7];
        data_len = ucp[8] + 256 * ucp[9];
      }
    else
      {
        ssh_warning("Initial X11 packet contains bad byte order byte: 0x%x",
                    ucp[0]);
        completed(internal_context, SSH_FILTER_DISCONNECT);
        return;
      }

  /* Compute the length of the packet we must receive. */
  desired_len = 12 + ((proto_len + 3) & ~3) + ((data_len + 3) & ~3);

  /* Sanity check: it must not be larger than we are willing to receive. */
  if (desired_len > X11_MAX_AUTH_PACKET_SIZE)
    {
      completed(internal_context, SSH_FILTER_DISCONNECT);
      return;
    }

  /* Check if the whole packet is in buffer. */
  if (received_len < desired_len)
    {
      if (eof_received)
        {
          completed(internal_context, SSH_FILTER_DISCONNECT);
          return;
        }
      else
        {
          completed(internal_context, SSH_FILTER_HOLD);
          return;
        }
    }

  /* Check if authentication protocol matches. */
  if (proto_len != remote_channel->ssh1->x11_fake_auth_protocol_len ||
      memcmp(ucp + 12,
             remote_channel->ssh1->x11_fake_auth_protocol,
             proto_len) != 0)
    {
      if (proto_len > 100)
        proto_len = 100; /* Limit length of output. */
      ssh_warning("X11 connection requests different authentication protocol:"
                  " '%.*s' vs. '%.*s'.",
                  remote_channel->ssh1->x11_fake_auth_protocol_len,
                  remote_channel->ssh1->x11_fake_auth_protocol,
                  proto_len,
                  (const char *)(ucp + 12));
      completed(internal_context, SSH_FILTER_DISCONNECT);
      return;
    }

  /* Check if authentication data matches our fake data. */
  if (data_len != remote_channel->ssh1->x11_fake_data_len ||
      memcmp(ucp + 12 + ((proto_len + 3) & ~3),
             remote_channel->ssh1->x11_fake_data,
             remote_channel->ssh1->x11_fake_data_len) != 0)
    {
      ssh_warning("X11 auth data does not match fake data.");
      completed(internal_context, SSH_FILTER_DISCONNECT);
      return;
    }

  /* Received authentication protocol and data match our fake data.
     Substitute the fake data with real data. */
  SSH_ASSERT(remote_channel->ssh1->x11_fake_data_len ==
             remote_channel->ssh1->x11_real_data_len);
  memcpy(ucp + 12 + ((proto_len + 3) & ~3),
         remote_channel->ssh1->x11_real_data,
         remote_channel->ssh1->x11_real_data_len);

  /* Otherwise, we accept and shortcircuit any further communications. */
  completed(internal_context, SSH_FILTER_SHORTCIRCUIT);
}

void ssh1_x11_channel_filter_destroy(void *context)
{
  return;
}

/* CRC compensation attack detector below is heavily modified from the
   ssh1 version that is Copyright (C) 1998 CORE-SDI, Buenos Aires,
   Argentina, Ariel Futoransky(futo@core-sdi.com)
   <http://www.core-sdi.com> */

/* Constants for attack detector. */
#define SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE  (8)
#define SSH1_CRC_DEATTACK_HASH_MINSIZE    (8 * 1024)
#define SSH1_CRC_DEATTACK_HASH_ENTRYSIZE  (2)
#define SSH1_CRC_DEATTACK_HASH_FACTOR(x)  ((x)*3/2)
#define SSH1_CRC_DEATTACK_HASH_UNUSED     (0xffff)
#define SSH1_CRC_DEATTACK_HASH_IV         (0xfffe)
#define SSH1_CRC_DEATTACK_HASH_MINBLOCKS  (7*SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE)

/* Hash function (Input keys are cipher results) */
#define SSH1_CRC_DEATTACK_HASH(x) SSH_GET_32BIT(x)

/* Convenience macro for comparing memory blocks */
#define SSH1_CRC_DEATTACK_CMP(a,b) \
        (memcmp(a, b, SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE))

void ssh1_crc_compensation_attack_init(Ssh1Client ssh1)
{
  SSH_DEBUG(5, ("Initializing CRC compensation attack detector."));
  ssh1->crc_deattack_n =
    SSH1_CRC_DEATTACK_HASH_MINSIZE / SSH1_CRC_DEATTACK_HASH_ENTRYSIZE;
  SSH_DEBUG(7, ("Initial CRC attack buffer is %u bytes.",
                (unsigned int)(ssh1->crc_deattack_n * sizeof (SshUInt16))));
  ssh1->crc_deattack_h = ssh_xmalloc(ssh1->crc_deattack_n *
                                     sizeof (SshUInt16));
  return;
}

void ssh1_crc_compensation_attack_uninit(Ssh1Client ssh1)
{
  SSH_DEBUG(5, ("Freeing CRC compensation attack detector."));
  ssh_xfree(ssh1->crc_deattack_h);
  ssh1->crc_deattack_h = NULL;
  ssh1->crc_deattack_n = 0;
  return;
}

void ssh1_crc_compensation_attack_crc_update(SshUInt32 *a,
                                             SshUInt32 b)
{
  b ^= *a;
  *a = crc32_buffer((unsigned char *)(&b), sizeof(b));
  return;
}

Boolean ssh1_crc_compensation_attack_check_crc(unsigned char *s,
                                               unsigned char *buf,
                                               size_t len,
                                               unsigned char *iv)
{
  SshUInt32 crc;
  unsigned char *c;

  crc = 0;
  if (iv && ! SSH1_CRC_DEATTACK_CMP(s, iv))
    {
      ssh1_crc_compensation_attack_crc_update(&crc, 1);
      ssh1_crc_compensation_attack_crc_update(&crc, 0);
    }
  for (c = buf; c < buf + len; c += SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE)
    {
      if (! SSH1_CRC_DEATTACK_CMP(s, c))
        {
          ssh1_crc_compensation_attack_crc_update(&crc, 1);
          ssh1_crc_compensation_attack_crc_update(&crc, 0);
        }
      else
        {
          ssh1_crc_compensation_attack_crc_update(&crc, 0);
          ssh1_crc_compensation_attack_crc_update(&crc, 0);
        }
    }
  return (crc == 0);
}

Boolean ssh1_crc_compensation_attack_detect(Ssh1Client ssh1,
                                            unsigned char *buf,
                                            size_t len,
                                            unsigned char *iv)
{
  SshUInt32 i, j;
  SshInt32 l;
  unsigned char *c, *d;

  SSH_ASSERT(ssh1->crc_deattack_h != NULL);

  for (l = ssh1->crc_deattack_n;
       l < SSH1_CRC_DEATTACK_HASH_FACTOR(len /
                                         SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE);
       l = l << 2)
    /*NOTHING*/;

  if (l > ssh1->crc_deattack_n)
    {
      SSH_DEBUG(7, ("Resizing CRC attack buffer from %u to %u bytes.",
                    (unsigned int)(ssh1->crc_deattack_n * sizeof (SshUInt16)),
                    (unsigned int)(l * sizeof (SshUInt16))));
      ssh1->crc_deattack_n = l;
      ssh1->crc_deattack_h = ssh_xrealloc(ssh1->crc_deattack_h,
                                          ssh1->crc_deattack_n *
                                          sizeof (SshUInt16));
    }

  if (len <= SSH1_CRC_DEATTACK_HASH_MINBLOCKS)
    {
      for (c = buf; c < buf + len; c += SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE)
        {
          if (iv && (! SSH1_CRC_DEATTACK_CMP(c, iv)))
            {
              if ((ssh1_crc_compensation_attack_check_crc(c, buf, len, iv)))
                {
                  SSH_DEBUG(5, ("CRC compensation attack detected."));
                  return TRUE;
                }
              else
                {
                  break;
                }
            }
          for (d = buf; d < c; d += SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE)
            {
              if (! SSH1_CRC_DEATTACK_CMP(c, d))
                {
                  if ((ssh1_crc_compensation_attack_check_crc(c,
                                                              buf,
                                                              len,
                                                              iv)))
                    {
                      SSH_DEBUG(5, ("CRC compensation attack detected."));
                      return TRUE;
                    }
                  else
                    {
                      break;
                    }
                }
            }
        }
      SSH_DEBUG(8, ("CRC compensation checked OK."));
      return FALSE;
    }

  for (i = 0; i < ssh1->crc_deattack_n; i++)
    ssh1->crc_deattack_h[i] = SSH1_CRC_DEATTACK_HASH_UNUSED;

  if (iv)
    ssh1->crc_deattack_h[(SSH1_CRC_DEATTACK_HASH(iv) &
                          (ssh1->crc_deattack_n - 1))] =
      SSH1_CRC_DEATTACK_HASH_IV;

  for (c = buf, j = 0;
       c < (buf + len);
       c += SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE, j++)
    {
      for (i = SSH1_CRC_DEATTACK_HASH(c) & (ssh1->crc_deattack_n - 1);
           ssh1->crc_deattack_h[i] != SSH1_CRC_DEATTACK_HASH_UNUSED;
           i = (i + 1) & (ssh1->crc_deattack_n - 1))
        {
          if (ssh1->crc_deattack_h[i] == SSH1_CRC_DEATTACK_HASH_IV)
            {
              if (! SSH1_CRC_DEATTACK_CMP(c, iv))
                {
                  if (ssh1_crc_compensation_attack_check_crc(c, buf, len, iv))
                    {
                      SSH_DEBUG(5, ("CRC compensation attack detected."));
                      return TRUE;
                    }
                  else
                    {
                      break;
                    }
                }
            }
          else if (! SSH1_CRC_DEATTACK_CMP(
                           c,
                           (buf + (ssh1->crc_deattack_h[i] *
                                   SSH1_CRC_DEATTACK_SSH1_BLOCKSIZE))))
            {
              if (ssh1_crc_compensation_attack_check_crc(c, buf, len, iv))
                {
                  SSH_DEBUG(5, ("CRC compensation attack detected."));
                  return TRUE;
                }
              else
                {
                  break;
                }
            }
        }
      ssh1->crc_deattack_h[i] = j;
    }
  SSH_DEBUG(8, ("CRC compensation checked OK."));
  return FALSE;
}

/* CRC compensation attack detector ends here. */

#endif /* WITH_INTERNAL_SSH1_EMULATION */

