/*
  sshappcommon.c

  Authors: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1999-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Common functions for all ssh-applications.
*/

#include "sshincludes.h"
#include "ssh2includes.h"
#include "sshappcommon.h"
#include "sshglob.h"

#include <syslog.h>
#ifdef NEED_SYS_SYSLOG_H
#include <sys/syslog.h>
#endif /* NEED_SYS_SYSLOG_H */


#define SSH_DEBUG_MODULE "SshAppCommon"

const char *ssh_app_registered_program_name = NULL;

SshRegexContext ssh_app_global_rex_ctx = NULL;

void ssh_register_program_name(const char *name)
{
  ssh_app_registered_program_name = name;
}

const char *ssh_get_program_name(void)
{
  return ssh_app_registered_program_name;
}

/* Prints version string to stderr. */
void ssh2_version(const char *name)
{
  if (name == NULL)
    {
      if (ssh_app_registered_program_name)
        name = ssh_app_registered_program_name;
      else
        return;
    }
#ifdef HOSTTYPE  
  fprintf(stderr, "%s: %s on %s\n", name, SSH2_VERSION_STRING, HOSTTYPE);
#else /* HOSTTYPE */
  fprintf(stderr, "%s: %s\n", name, SSH2_VERSION_STRING);
#endif /* HOSTTYPE */
}

#ifdef KERNEL
#define SSH_DEBUG_BUFFER_SIZE 512
#else /* KERNEL */
#define SSH_DEBUG_BUFFER_SIZE 4096
#endif /* KERNEL */

void (*ssh_informational_callback)(const char *message, void *context);
void *callback_context;

void ssh_informational_callback_register(void (*callback)
                                         (const char *message, void *context),
                                         void *context)
{
  ssh_informational_callback = callback;
  callback_context = context;
}

/* Function for conveying informational messages to user. The output
   is _not_ followed by a newline (by this function, that is; caller
   can add own newlines). */
void ssh_informational(const char *format, ...)
{
  char buf[SSH_DEBUG_BUFFER_SIZE];
  va_list args;
  va_start(args, format);
  ssh_vsnprintf(buf, sizeof(buf), format, args);

  if (ssh_informational_callback)
    (*ssh_informational_callback)(buf, callback_context);
  else
    fprintf(stderr, "%s", buf);
  
  va_end(args);
}

/* Helper function, that destroys a list, and frees it's contents,
   too. If delete_proc is non-NULL, it is used to delete the list
   item. Otherwise, the item is just ssh_xfree()d. */
void ssh_app_free_list(SshDlList list, SshAppListNodeDeleteProc delete_proc)
{
  char *to_be_deleted;

  if (list == NULL)
    return;
  
  ssh_dllist_rewind(list);
  while (!ssh_dllist_is_empty(list))
    {
      to_be_deleted = ssh_dllist_delete_current(list);
      if (delete_proc)
        delete_proc(to_be_deleted);
      else
        ssh_xfree(to_be_deleted);
    }    
  ssh_dllist_free(list);
}

char *ssh_app_param_list_get_next(const char *string)
{
  char *rest, *temp;
  
  if ((rest = ssh_glob_next_unescaped_char(string, ',')) == NULL)
    return ssh_xstrdup(string);
  
  temp = ssh_xcalloc(rest - string + 2, sizeof(char));

  strncpy(temp, string, rest - string);

  return temp;
}

/* Gets the global SshRegex context. If not already allocated, allocates it.
   This function will not fail. */
SshRegexContext ssh_app_get_global_regex_context(void)
{
  if (ssh_app_global_rex_ctx == NULL)
    {
      SSH_TRACE(2, ("Allocating global SshRegex context."));
      ssh_app_global_rex_ctx = ssh_regex_create_context();
    }

  return ssh_app_global_rex_ctx;
}

/* Frees the global regex context. */
void ssh_app_free_global_regex_context(void)
{
  if (ssh_app_global_rex_ctx == NULL)
    return;

  SSH_TRACE(2, ("Freeing global SshRegex context."));
  ssh_regex_free_context(ssh_app_global_rex_ctx);
  ssh_app_global_rex_ctx = NULL;
}


/* Return syslog-style severity from an SshLogSeverity variable. */
int ssh_app_log_severity(SshLogSeverity severity)
{
  switch(severity)
    {
    case SSH_LOG_INFORMATIONAL:
      return LOG_INFO;
    case SSH_LOG_NOTICE:
      return LOG_NOTICE;
    case SSH_LOG_WARNING:
      return LOG_WARNING;
    case SSH_LOG_ERROR:
      return LOG_ERR;
    case SSH_LOG_CRITICAL:
      return LOG_CRIT;
    }

  ssh_fatal("ssh_app_log_severity: Unknown severity.");
  return -1;
}

/* Return syslog-style facility from an SshLogFacility variable. */
int ssh_app_log_facility(SshLogFacility facility)
{
  switch (facility)
    {
    case SSH_LOGFACILITY_AUTH:
    case SSH_LOGFACILITY_SECURITY:
#ifdef LOG_AUTHPRIV
      return LOG_AUTHPRIV;
#else /* LOG_AUTHPRIV */
      return LOG_AUTH;
#endif /* LOG_AUTHPRIV */
    case SSH_LOGFACILITY_DAEMON:
      return LOG_DAEMON;
    case SSH_LOGFACILITY_USER:
      return LOG_USER;
    case SSH_LOGFACILITY_MAIL:
      return LOG_MAIL;
    case SSH_LOGFACILITY_LOCAL0:
      return LOG_LOCAL0;
    case SSH_LOGFACILITY_LOCAL1:
      return LOG_LOCAL1;
    case SSH_LOGFACILITY_LOCAL2:
      return LOG_LOCAL2;
    case SSH_LOGFACILITY_LOCAL3:
      return LOG_LOCAL3;
    case SSH_LOGFACILITY_LOCAL4:
      return LOG_LOCAL4;
    case SSH_LOGFACILITY_LOCAL5:
      return LOG_LOCAL5;
    case SSH_LOGFACILITY_LOCAL6:
      return LOG_LOCAL6;
    case SSH_LOGFACILITY_LOCAL7:
      return LOG_LOCAL7;
    }
  ssh_fatal("ssh_app_log_facility: Unknown facility (%d).", facility);
  return -1;
}


Boolean ssh_app_is_file_name_absolute(const char *filename)
{
  if (filename == NULL)
    return FALSE;


  return filename[0] == '/';









}
